package gov.va.med.mhv.hra.service.impl;

import java.math.BigInteger;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.UUID;

import javax.annotation.Resource;
import javax.xml.ws.BindingProvider;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.PropertySource;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.api.dto.PatientDTO;
import gov.va.med.mhv.common.api.dto.UserProfileDTO;
import gov.va.med.mhv.hla.client.BloodPressure;
import gov.va.med.mhv.hla.client.BloodSugar;
import gov.va.med.mhv.hla.client.BloodSugarMeasurementMethod;
import gov.va.med.mhv.hla.client.BodyTemperature;
import gov.va.med.mhv.hla.client.BodyWeight;
import gov.va.med.mhv.hla.client.Cholesterol;
import gov.va.med.mhv.hla.client.Demographics;
import gov.va.med.mhv.hla.client.Gender;
import gov.va.med.mhv.hla.client.HRAIntegrationService;
import gov.va.med.mhv.hla.client.HRAIntegrationService_Service;
import gov.va.med.mhv.hla.client.HealthExtract;
import gov.va.med.mhv.hla.client.HeartRate;
import gov.va.med.mhv.hla.client.INR;
import gov.va.med.mhv.hla.client.Identity;
import gov.va.med.mhv.hla.client.PulseOximetry;
import gov.va.med.mhv.hla.client.PulseOximetrySymptom;
import gov.va.med.mhv.hla.client.Session;
import gov.va.med.mhv.hla.client.TemperatureMeasurementMethod;
import gov.va.med.mhv.hla.client.TemperatureUnit;
import gov.va.med.mhv.hla.client.WeightUnit;
import gov.va.med.mhv.hra.data.model.HlaVital;
import gov.va.med.mhv.hra.data.repository.HlaVitalRepository;
import gov.va.med.mhv.hra.exception.MHVRuntimeException;
import gov.va.med.mhv.hra.service.HealthReadinessAssessmentService;
import gov.va.med.mhv.hra.util.DateUtil;
import gov.va.med.mhv.usermgmt.service.UserMgmtService;

@Component
@PropertySource("classpath:/${MHV_ENV_PROPERTY}.hra.application.properties")
public class HealthReadinessAssessmentServiceImpl implements HealthReadinessAssessmentService {

	private static Logger log = LogManager.getLogger(HealthReadinessAssessmentServiceImpl.class);
	
	private Boolean isPatient = null;
	private long patientId;
	
	@Resource(name="userMgmtServiceProxy")
	private UserMgmtService userMgmtService;

	@Value("${hra.osServerAddress}")
	private String hlaServiceAddress;
	
	@Autowired
	private HlaVitalRepository hlaVitalRepository;
	
	//use 'hlaClient' for local env
//	@Autowired
//	private HRAIntegrationService_Service hlaClient;
	
	@Override
	public String test(String sampleText) {
		return sampleText;
	}
	
	@Override
	public Session getHlaSession(String userName, String sessionId) {
		long start = System.currentTimeMillis();
		log.info("=====inside getHlaSession()========== start " + start);
		UserProfileDTO userProfileDto = null;
		Identity userIdentity = null;
		Demographics userDemographics = null;
		HealthExtract healthExtract = null;
		gov.va.med.mhv.hla.client.Session hraSession = null;
		
		try {
			userProfileDto = getCurrentUserProfile(userName);
			
			Long userprofileId = (null != userProfileDto) ? userProfileDto.getId() : null; //getUserProfileId();
			
			if (userprofileId != null) {
				userIdentity = new Identity();
				userIdentity.setMessageID(UUID.randomUUID().toString());
				userIdentity.setUserID(userProfileDto.getId());
				userIdentity.setUserName(userName);
				userIdentity.setSessionID(sessionId);

				userDemographics = new Demographics();
				
				userDemographics.setDateOfBirth(DateUtil.convertDate(userProfileDto.getBirthDate()));
				userDemographics.setFirstName((null != userProfileDto.getName()) ? userProfileDto.getName().getFirstName() : null);
				userDemographics.setLastName((null != userProfileDto.getName()) ? userProfileDto.getName().getLastName() : null);
				userDemographics.setGender(Gender.fromValue(userProfileDto.getGender()));
				userDemographics.setIsPatient(getIsPatient(userprofileId));
				userDemographics.setIsVeteran(userProfileDto.getIsVeteran());
				userDemographics.setHasPremiumAccount(getIsIPAedUser(userProfileDto));
				if (null != userProfileDto.getContact() 
						&& null != userProfileDto.getContact().getEmail())
					userDemographics.setEmailAddress(userProfileDto.getContact().getEmail());
				
				healthExtract = populateHealthExtract(userprofileId);
				
				userIdentity.setDemographics(userDemographics);
				userIdentity.setHealthExtract(healthExtract);
				
				// use 'hlaClient' for local env
//				hraSession = hlaClient.getHRAIntegrationService().newSession(userIdentity);
				hraSession = newHraSession(userIdentity);
			}
		} catch (Exception e) {
			log.error("Error in createUserIdentity", e);
			throw new MHVRuntimeException("Error in createUserIdentity", e);
		}
		
		if (log.isDebugEnabled()) {
			log.debug("===end getHlaSession=== elapsed time " + (System.currentTimeMillis() - start));
		}
		
		return hraSession;
	}
	
	@Override
	public HealthExtract populateHealthExtract(Long userProfileId) {
		long start = System.currentTimeMillis();
		if (log.isDebugEnabled()) {
			log.debug("===inside populateHealthExtract=== start : " + start);
		}
		Pageable topOne = new PageRequest(0, 1);
		List<HlaVital> hlaVitals = null;
		HlaVital hlaVital = null;
		HealthExtract healthExtract = new HealthExtract();
		try {
			hlaVitals = hlaVitalRepository.getVitals(userProfileId, topOne);
		} catch(Exception e) {
			log.error("Error reading vitals", e);
			throw new MHVRuntimeException("Error reading vitals", e);
		}
		
		if (null != hlaVitals && hlaVitals.size() > 0) {
			log.debug(hlaVitals.size());

			try {
				hlaVital = hlaVitals.get(0);
				
				//extract BloodPressureReading data
				if (null != hlaVital.getBloodPressureId()) {
					BloodPressure bp = new BloodPressure();
					log.debug("BloodPressure exists");

					if (hlaVital.getBpReading() != null) {
						if (isVitalLessThan1YearOld(hlaVital.getBpReading())) {
							bp.setDate(DateUtil.convertDate(hlaVital.getBpReading()));
							if (hlaVital.getDiastolic() != null && hlaVital.getSystolic() != null) {
								bp.setDiastolic(BigInteger.valueOf(hlaVital.getDiastolic().intValue()));
								bp.setSystolic(BigInteger.valueOf(hlaVital.getSystolic()));
								healthExtract.setBloodPressure(bp);
							}
						}
					}
				}
				
				// extract BloodSugarReading data
				if (null != hlaVital.getBloodSugarId()) {
					BloodSugar bs = new BloodSugar();
					log.debug("BloodSugar exists");
					if (hlaVital.getBsReading() != null) {
						if (isVitalLessThan1YearOld(hlaVital.getBsReading())) {
							bs.setDate(DateUtil.convertDate(hlaVital.getBsReading()));
							if (hlaVital.getBloodSugarCount() != null
									&& hlaVital.getTestingMethod() != null) {
								bs.setCount(BigInteger.valueOf(hlaVital.getBloodSugarCount().intValue()));
								bs.setMethod(BloodSugarMeasurementMethod.fromValue(hlaVital.getTestingMethod()));
								healthExtract.setBloodSugar(bs);
							}
						}
					}
				}
				
				// extract BodyTemperature data
				if (null != hlaVital.getBodyTempId()) {
					BodyTemperature bt = new BodyTemperature();
					log.debug("BodyTemperature exists");
					if (hlaVital.getBtReading() != null) {
						if (isVitalLessThan1YearOld(hlaVital.getBtReading())) {
							bt.setDate(DateUtil.convertDate(hlaVital.getBtReading()));
							if (hlaVital.getBodyTemperatureMethod() != null && hlaVital.getMeasure() != null
									&& hlaVital.getBodyTemperature() != null) {
								bt.setMethod(TemperatureMeasurementMethod.fromValue(hlaVital.getBodyTemperatureMethod()));
								bt.setUnit(TemperatureUnit.fromValue(hlaVital.getMeasure()));
								bt.setValue(new Float(hlaVital.getBodyTemperature().floatValue()));
								healthExtract.setBodyTemperature(bt);
							}
						}
					}
				}
				
				// extract BodyWeightReading data
				if (null != hlaVital.getBodyWeightId()) {
					BodyWeight bw = new BodyWeight();
					log.debug("BodyWeight exists");
					if (hlaVital.getBwReading() != null) {
						if (isVitalLessThan1YearOld(hlaVital.getBwReading())) {
							bw.setDate(DateUtil.convertDate(hlaVital.getBwReading()));
							if (hlaVital.getBodyweightMeasure() != null
									&& hlaVital.getBodyweight() != null) {
								bw.setUnit(WeightUnit.fromValue(hlaVital.getBodyweightMeasure()));
								bw.setValue(new Float(hlaVital.getBodyweight().floatValue()));
								healthExtract.setBodyWeight(bw);
							}
						}
					}
				}
				
				// extract HeartRateReading data
				if (null != hlaVital.getHeartRateId()) {
					HeartRate hr = new HeartRate();
					log.debug("HeartRate exists");
					if (hlaVital.getHtrReading() != null) {
						if (isVitalLessThan1YearOld(hlaVital.getHtrReading())) {
							hr.setDate(DateUtil.convertDate(hlaVital.getHtrReading()));
							if (hlaVital.getHeartRate() != null) {
								hr.setCount(BigInteger.valueOf(hlaVital.getHeartRate().intValue()));
								healthExtract.setHeartRate(hr);
							}
						}
					}
				}
				
				// extract InrReading data
				if (null != hlaVital.getInrId()) {
					INR inr = new INR();
					log.debug("inr exists");
					if (hlaVital.getInrReading() != null) {
						if (isVitalLessThan1YearOld(hlaVital.getInrReading())) {
							inr.setDate(DateUtil.convertDate(hlaVital.getInrReading()));
							try {
								if (hlaVital.getInr() != null) {
									inr.setValue(new Float(hlaVital.getInr().floatValue()));
									String hetr = (hlaVital.getHighendTragetRange() != null) ? convertHighEndTargetRange(hlaVital.getHighendTragetRange()) : null;
									if (hetr != null && !hetr.equalsIgnoreCase("None")) {
										inr.setHighEndTarget(new Float(hetr));
									}
									
									String letr = (hlaVital.getLowendTargetRange() != null) ? convertLowEndTargetRange(hlaVital.getLowendTargetRange()) : null;
									if (letr != null && !letr.equalsIgnoreCase("None")) {
										inr.setHighEndTarget(new Float(hetr));
									}
								}
								healthExtract.setINR(inr);
							} catch (NumberFormatException e) {
								log.error("Error in InrReading values " + e);
							}
						}
					}
				}
				
				// extract LipidsReading data
				if (null != hlaVital.getLipidsId() ) {
					Cholesterol chol = new Cholesterol();
					if (hlaVital.getLrReading() != null) {
						if (isVitalLessThan1YearOld(hlaVital.getLrReading())) {
							chol.setDate(DateUtil.convertDate(hlaVital.getLrReading()));
							if (hlaVital.getHdl() != null && hlaVital.getTotal() != null) {
								chol.setHdl(BigInteger.valueOf(hlaVital.getHdl().intValue()));
								chol.setTotal(BigInteger.valueOf(hlaVital.getTotal().intValue()));
								if (hlaVital.getLdl() != null)
									chol.setLdl(BigInteger.valueOf(hlaVital.getLdl().intValue()));
								if (hlaVital.getFast() != null)
									chol.setFasting(convertFasting(hlaVital.getFast()));
								if (hlaVital.getTriglycerides() != null)
									chol.setTriglycerides(BigInteger.valueOf(hlaVital.getTriglycerides().intValue()));
								healthExtract.setCholesterol(chol);
							}
						}
					}
				}
				
				//extract PulseOximetryReading data
				if (null != hlaVital.getPulseOximetryId()) {
					PulseOximetry po = new PulseOximetry();
					if (hlaVital.getPorReading() != null) {
						if (isVitalLessThan1YearOld(hlaVital.getPorReading())) {
							po.setDate(DateUtil.convertDate(hlaVital.getPorReading()));
							if (hlaVital.getOximeterReading() != null
									&& hlaVital.getRespiratoryRate() != null
									&& hlaVital.getSymptoms() != null) {
								po.setOximetryReading(BigInteger.valueOf(hlaVital.getOximeterReading()));
								po.setRespitoryRate(BigInteger.valueOf(hlaVital.getRespiratoryRate()));
								po.setSympton(PulseOximetrySymptom.fromValue(hlaVital.getSymptoms()));
								healthExtract.setPulseOximetry(po);
							}
						}
					}
				}
			} catch(Exception e) {
				log.error("Error extracting vitals data", e);
				throw new MHVRuntimeException("Error extracting vitals data", e);
			}
			
		} 
		
		
		if (log.isDebugEnabled()) {
			log.debug("===end populateHealthExtract=== elapsed time " + (System.currentTimeMillis()-start));
		}
		
		return healthExtract;
	}
	

	/**
	 * get PatientDTO for userProfileid
	 * 
	 * @param userId
	 * @return
	 */
	private PatientDTO getPatientForUser(Long userProfileid) {
		PatientDTO patientDTO = null;

		try {

			patientDTO = this.userMgmtService.getPatient(userProfileid);

		} catch (Exception e) {
			log.error("Error in fetching patient data " + e);
			throw new MHVRuntimeException(e);
		} 

		return patientDTO;
	}
	
	private boolean getIsPatient(Long userId) {
		
		PatientDTO existingPatient = getPatientForUser(userId);

		if (existingPatient != null) {
			isPatient = Boolean.TRUE;
			patientId = existingPatient.getId();
		} else {
			isPatient = Boolean.FALSE;
		}
		
		return isPatient;
	}

	private boolean getIsIPAedUser(UserProfileDTO userProfileDto) {
		return isPatientAuthenticated(patientId);
	}
	
	/**
	 * get PatientDTO for userProfileid
	 * 
	 * @param userId
	 * @return
	 */
	private boolean isPatientAuthenticated(Long patientId) {
		Boolean ipaedPatient = false;

		try {
			ipaedPatient = this.userMgmtService.isPatientIPA(patientId);
		} catch (Exception e) {
			log.error("Error in fetching patient data " + e);
			throw new MHVRuntimeException(e);
		} 

		return ipaedPatient;
	}
	
	
	private boolean isVitalLessThan1YearOld(Date readingDate) {
		Calendar readingDateCal = new GregorianCalendar();
		readingDateCal.setTimeInMillis(readingDate.getTime());
		Calendar now = Calendar.getInstance();
		now.add(Calendar.YEAR, -1);
		int diff = now.compareTo(readingDateCal);
		if (diff < 0)
			return true;
		else
			return false;
	}
	
	private boolean convertFasting(String fasting) {
    	if(fasting.equalsIgnoreCase("F"))
    		return true;
    	else
    		return false;
    }
	
	private String convertLowEndTargetRange(String value) {
		if(value.equalsIgnoreCase("NONE"))
			return "NONE";
		else if(value.equalsIgnoreCase("LETR"))
			return "1.5";
		else if(value.equalsIgnoreCase("METR"))
			return "2.0";
		else if(value.equalsIgnoreCase("HETR"))
			return "2.5";
		else
			return "NONE";
	}

	private String convertHighEndTargetRange(String value) {
		if(value.equalsIgnoreCase("NONE"))
			return "NONE";
		else if(value.equalsIgnoreCase("LETR"))
			return "3.0";
		else if(value.equalsIgnoreCase("METR"))
			return "3.5";
		else if(value.equalsIgnoreCase("HETR"))
			return "4.0";
		else
			return "NONE";
	}

	private UserProfileDTO getCurrentUserProfile(String userName) {
		UserProfileDTO userProfileDto = null;
		try {
			
			userProfileDto = this.userMgmtService.getUserProfile(userName);

		} catch (Exception e) {
			log.error("Error in fetching user data " + e);
			throw new MHVRuntimeException(e);
		} 

		return userProfileDto;
	}
	
	public String getHlaServiceAddress() {
		return hlaServiceAddress;
	}
	
	public void setHlaServiceAddress(String hlaServiceAddress) {
		this.hlaServiceAddress = hlaServiceAddress;
	}
	
	private gov.va.med.mhv.hla.client.Session newHraSession(Identity userIdentity) {
		long start = System.currentTimeMillis();
		if (log.isDebugEnabled()) {
			log.debug("inside newHraSession... start " + start);
		}
		gov.va.med.mhv.hla.client.Session hraSession = null;
		
		try{
			HRAIntegrationService_Service service = new HRAIntegrationService_Service();
			HRAIntegrationService port = service.getHRAIntegrationService();
			BindingProvider bindingProvider = (BindingProvider) port;
			
			if (log.isDebugEnabled()) {
				log.debug("HLA service Address : " + this.hlaServiceAddress);
			}
			
			bindingProvider.getRequestContext().put(BindingProvider.ENDPOINT_ADDRESS_PROPERTY, this.hlaServiceAddress);

			// Use the service
			hraSession = port.newSession(userIdentity);
			log.debug("Created HRA Session Token : " + hraSession.getToken());
		} catch(Exception e) {
			log.error("Error in accessing HLA service", e);
		}

		if (log.isDebugEnabled()) {
			log.debug("===end newHraSession=== elapsed time " + (System.currentTimeMillis() - start));
		}

		return hraSession;
	}
}
