package gov.va.med.mhv.login.hook.service;

import java.io.IOException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.servlet.http.HttpSession;
import javax.ws.rs.client.ClientRequestContext;
import javax.ws.rs.client.ClientRequestFilter;
import javax.ws.rs.core.Response;

import org.apache.commons.lang3.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import com.liferay.portal.kernel.servlet.PortalSessionThreadLocal;

import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.common.api.service.SessionAPIService;
import gov.va.med.mhv.login.hook.ApplicationContextProvider;

/**
 * 
 * THIS FILTER IS INTENDED TO BE ASSIGNED TO EACH SECURED API IN MHV SERVICES.
 * IT WILL CHECK THE HTTP SESSION FOR A SECURITY TOKEN AND CHECK THAT IT IS STILL IN GOOD STANDING.
 * (NOT EXPIRED AND EXISTING)
 * 
 * IF THE TOKEN IS NOT IN GOOD STANDING IT WILL AUTOMATICALLY SUBMIT A REQUEST TO CREATE A NEW TOKEN
 * AND THEN STORE IT IN THE HTTP SESSION.
 *
 */

public class HttpSessionApiSecurityClientFilterRequest implements ClientRequestFilter {
	private static final Logger LOGGER = LogManager.getLogger(HttpSessionApiSecurityClientFilterRequest.class);

	private static final String API_TOKEN_KEY = "Token";
	private static final String API_EXPIRES_KEY = "Expires";
	private static final String API_SESSION_TOKEN_KEY = "APISessionToken";
	private static final String API_SESSION_EXP_KEY = "APISessionExpiration";
	private static final String RFC1123_PATTERN  = "EEE, dd MMM yyyy HH:mm:ss z";
	private static final String SESSION_USERNAME_KEY = "LIFERAY_SHARED_userid";
	
	private SessionAPIService getSessionService() {
		return (SessionAPIService)ApplicationContextProvider.getApplicationContext().getBean("sessionServiceProxy");
	} 

	@Override
	public void filter(ClientRequestContext requestContext) throws IOException {
		
		try{
			HttpSession session = PortalSessionThreadLocal.getHttpSession();
			
			String token = (String)session.getAttribute(API_SESSION_TOKEN_KEY);
			String expires = (String)session.getAttribute(API_SESSION_EXP_KEY);
			
			Date expiresDt = null;

			if( expires != null ) {
				SimpleDateFormat sdf = new SimpleDateFormat(RFC1123_PATTERN);
				try {
					expiresDt = sdf.parse(expires);
				} catch (ParseException e1) {
					throw new IOException("API Session Expiration is invalid from http session");
				}
			}

			String mhvUsername = (String) session.getAttribute(SESSION_USERNAME_KEY);
			
			boolean needsNewAPIToken = false;
			
			if( token == null ||expiresDt == null) {
				needsNewAPIToken = true;
				if(LOGGER.isDebugEnabled()) {
					LOGGER.debug("The API Token for mhvUsername " + mhvUsername + " doesn't exist");
				}
			}
			if( !needsNewAPIToken && expiresDt.before(new Date()) ) {
				needsNewAPIToken = true;
				if( LOGGER.isDebugEnabled()) {
					LOGGER.debug("The API Token for mhvUsername " + mhvUsername + " expired");
				}
			}
			
			if( needsNewAPIToken ) {
				if( LOGGER.isDebugEnabled()) {
					LOGGER.debug("Fetching a new API Token for mhvUsername "+mhvUsername);
				}
				
				try {
					Response r = getSessionService().createSession();
					
					token = r.getHeaderString(API_TOKEN_KEY);
					expires = r.getHeaderString(API_EXPIRES_KEY);
					
					if(StringUtils.isEmpty(token) ||StringUtils.isEmpty(expires)) {
						throw new IOException("API Token response missing headers for 'Token' or 'Expires'");
					}
					
					session.setAttribute(API_SESSION_TOKEN_KEY,token);
					session.setAttribute(API_SESSION_EXP_KEY,expires);
				} catch (MHVException e) {
					throw new IOException(e);
				}
			}
			
			requestContext.getHeaders().add(API_TOKEN_KEY, token);
			
		} catch(Exception ex){
			throw new IOException("Unable to access the clients http session");
		} 
		
	}

}