package gov.va.med.mhv.login.hook;

import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.faces.context.FacesContext;
import javax.portlet.PortletRequest;
import javax.portlet.PortletSession;
import javax.servlet.http.HttpSession;

import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Component;

import com.liferay.portal.PasswordExpiredException;
import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.servlet.PortalSessionThreadLocal;
import com.liferay.portal.kernel.util.PropsUtil;
import com.liferay.portal.model.Role;
import com.liferay.portal.model.User;
import com.liferay.portal.security.auth.AuthException;
import com.liferay.portal.security.auth.Authenticator;
import com.liferay.portal.service.UserLocalServiceUtil;

import gov.va.med.mhv.common.api.dto.PatientDTO;
import gov.va.med.mhv.common.api.dto.UserProfileDTO;
import gov.va.med.mhv.common.api.enumeration.FieldTestEnum;
import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.login.hook.SystemDownException.ErrorCodeEnum;
import gov.va.med.mhv.usermgmt.service.AccountActivityLogService;
import gov.va.med.mhv.usermgmt.service.AccountValidatorService;
import gov.va.med.mhv.usermgmt.service.UserMgmtService;
import gov.va.med.mhv.usermgmt.common.dto.FieldTestDTO;

@Component
public class MhvAuthenticator implements Authenticator {

	private static final Log LOG = LogFactoryUtil.getLog(MhvAuthenticator.class);
	
	private static final Object HTTP_SESSION_ERROR = "java.io.IOException: Unable to access the clients http session";
	
	public static final String ATTR_SCREEN_NAME = "screenName";
	public static final String ATTR_USER_PROFILE = "userProfile";
	public static final String ATTR_PATIENT = "patient";
	public static final String ATTR_MHV_API_ERROR_CODE = "mhvapiErrorCode";
	public static final String ATTR_USER_PROFILE_TMP = "userProfileTmp";
	public static final String ATTR_PATIENT_TMP = "patientTmp";
	public static final String ATTR_SHOW_DELEGATE_FIELD_TESTER_TEXT = "showDelegateFieldTesterText";
	
	private static final String ATTR_CHANGE_PASSWORD = "change-temporary-password";
	private static final String ATTR_MHV_CONTEXT = "mhv.context";
	public static final String ATTR_REDIRECT_PATH = "redirectPath";

	
	private AccountValidatorService accountValidatorService;

	private UserMgmtService userMgmtService = null;

	public static final String VAR_USER_ROLE = "MHV_VAR";
	
	private AccountActivityLogService accountActivityLogService;

	
	@Override
	public int authenticateByScreenName(long companyId, String screenName, String password,
			Map<String, String[]> headerMap, Map<String, String[]> parameterMap) throws AuthException {
		
		HttpSession session = null;
		
		try {
			isUserIdValid(screenName);
			
			this.isUserInactive(screenName);

			session = PortalSessionThreadLocal.getHttpSession();
			session.setAttribute(ATTR_SCREEN_NAME, screenName);

			UserProfileDTO authenticatedMhvUserProfile = this.getUserProfile(screenName);
			if(authenticatedMhvUserProfile != null) {
				
				isTemporayarPasswordExpired(authenticatedMhvUserProfile.getId());
				
				boolean temporaryPaswordFlg =isLatestPasswordInHistoryTableTemporary(authenticatedMhvUserProfile.getId());

				if(temporaryPaswordFlg){
					session.setAttribute(ATTR_REDIRECT_PATH, getRedirectPath(screenName));
				}

				
				session.setAttribute(ATTR_USER_PROFILE, authenticatedMhvUserProfile);
				
				PatientDTO authenticateMhvPatient = (authenticatedMhvUserProfile != null)
						? this.getPatient(authenticatedMhvUserProfile.getId()) : null;
				
				if(authenticateMhvPatient != null) {
					session.setAttribute(ATTR_PATIENT, authenticateMhvPatient);
				}
				
				
				// Check whether you have VAR ACCESS.. IFF you are logged in thru DSLOGON
				//set Attribute MHV_VAR into session
//				if(checkDSloggedIn(authenticatedMhvUserProfile) && hasVARAccess(authenticateMhvPatient)) {
//					System.out.println(" **** ABOUT TO SET VAR ROLE ****");
//					session.setAttribute(VAR_USER_ROLE, VAR_USER_ROLE);
//					if(LOG.isInfoEnabled())
//						LOG.info("User Profile : " + authenticatedMhvUserProfile.getUserName() + " is set for VAR ACCESS ");
//				}
				
				
				createFailedLoginActivity(authenticatedMhvUserProfile);
			}
			
			boolean liferayAdmin = isUserLifeayAdmin(companyId,screenName);
			
			//The user has to be either a liferay admin user or should be in evault/ldap to be able to login
			if(!liferayAdmin && authenticatedMhvUserProfile == null){
				throw new AuthException();
			}


			
			// For Beta Testing only. Jazz Story 234879 - Analysis and Design of Delegation and Surrogacy BUC
			// hardcoded for delegator
			if (screenName.equals("mhvtbp")) {
				UserProfileDTO authenticatedMhvUserProfileTmp = this.getUserProfile("mhvtbp");
				if(authenticatedMhvUserProfileTmp != null) {
					session.setAttribute(ATTR_USER_PROFILE_TMP, authenticatedMhvUserProfileTmp);
					
					PatientDTO authenticateMhvPatientTmp = (authenticatedMhvUserProfileTmp != null)
							? this.getPatient(authenticatedMhvUserProfileTmp.getId()) : null;
					
					if(authenticateMhvPatientTmp != null) {
						session.setAttribute(ATTR_PATIENT_TMP, authenticateMhvPatientTmp);
					}
				}
			}
			
			// Set a flag attribute that determines whether some text, that signals that this user is a
			// 'Delegate Field Tester', should be show in that login portlet area of the UI.  
			session.setAttribute(ATTR_SHOW_DELEGATE_FIELD_TESTER_TEXT, isDelegateFieldTester(screenName));
		
		} catch (AuthException ae) {
			if(session != null && ae.getCause() instanceof SystemDownException) {
				session.setAttribute(ATTR_MHV_API_ERROR_CODE, ((SystemDownException)ae.getCause()).getErrorCode());
			}
			throw ae;
		} catch (Exception e) {
			LOG.error("API Server is down or User Management Web Service not available");
			LOG.error(e.getMessage());
			throw new AuthException(new SystemDownException(e));
		}
		return SUCCESS;
	}
	
	
	private void isUserIdValid(String screenName) throws AuthException {
		if(!StringUtils.isAlphanumeric(screenName)){
			throw new AuthException();
		}
	}

	
	private void createFailedLoginActivity(UserProfileDTO authenticatedMhvUserProfile) throws AuthException{
		try{
			getAccountActivityLogService().
			     createFailedLoginActivity(authenticatedMhvUserProfile.getId());
		}catch(MHVException e){
			LOG.error(ErrorCodeEnum.USER_MANAGMENT_FAILURE_POST_LOGIN_UPDATE.toString(), e);
			throw new AuthException("Error Creating Failed login Activity User Management Service.", new SystemDownException(ErrorCodeEnum.USER_MANAGMENT_FAILURE_POST_LOGIN_UPDATE, e));
		}
	}

	
	
	private String getRedirectPath(String screenName){
		StringBuffer sb = new StringBuffer();
		String context = PropsUtil.get(ATTR_MHV_CONTEXT);
		if (context != null && context.trim().length() > 0) {
			sb.append("/").append(context).append("/").append(ATTR_CHANGE_PASSWORD);
		}else{
			sb.append("/").append(ATTR_CHANGE_PASSWORD);
		}
		return sb.toString();

	}
	
	private boolean isLatestPasswordInHistoryTableTemporary(Long id) throws AuthException{
		try {
			 return  getUserManagementService().isLatestPasswordInHistTableTemporay(id) ;
		} catch (MHVException e) {
			// something failed with the web service. fail authentication.
			LOG.error(ErrorCodeEnum.LATEST_PASSWORD_IN_HISTORY_TABLE_TEMPORARY.toString(), e);
			throw new AuthException("Error determining latest password status in history table  from User Management Service.", new SystemDownException(ErrorCodeEnum.LATEST_PASSWORD_IN_HISTORY_TABLE_TEMPORARY, e));
		}
	}

	
	
	private void isTemporayarPasswordExpired(Long id) throws AuthException {
		try {
			if (getUserManagementService().isTempPasswdExpired(id)) {
				LOG.error(id + ":: Temp password expired");
				throw new AuthException("Error determining temporary password expired from User Management Service.", new PasswordExpiredException());
			}
		} catch (MHVException e) {
			// something failed with the web service. fail authentication.
			LOG.error(ErrorCodeEnum.TEMPORARY_PASSWORD_EXPIRED.toString(), e);
			throw new AuthException("Error determining temporary password expired from User Management Service.", new SystemDownException(ErrorCodeEnum.TEMPORARY_PASSWORD_EXPIRED, e));
		}
	}
	
	
	private boolean isUserLifeayAdmin(Long companyId,String screenName){
		boolean adminUser = false;
		List<Role> userRoles=null;
		try {
			User liferayUser = UserLocalServiceUtil.getUserByScreenName(companyId, screenName);
			userRoles = liferayUser.getRoles();
		} catch (Exception e) {
		}
		if(userRoles != null){
			for(Role r : userRoles){
               if("Administrator".equalsIgnoreCase(r.getName())){
            	   adminUser=true;
            	   break;
                }
			}  
		}
		return adminUser;
	}


	@Override
	public int authenticateByUserId(long companyId, long userId, String password, Map<String, String[]> headerMap,
			Map<String, String[]> parameterMap) throws AuthException {
		throw new AuthException("Unsupported authentication type: Authenticate by User ID.");
	}

	@Override
	public int authenticateByEmailAddress(long companyId, String emailAddress, String password,
			Map<String, String[]> headerMap, Map<String, String[]> parameterMap) throws AuthException {
		throw new AuthException("Unsupported authentication type: Authenticate by Email Address.");
	}
	
	private void isUserInactive(String screenName) throws AuthException {
		try {
			if (getAccountValidatorService().isUserInactive(screenName)) {
				LOG.error(screenName + ":: is deactivated");
				throw new AuthException(new UserDeactivatedException(String.format("User: %s is not active.", screenName)));
			}
		} catch (MHVException e) {
			// something failed with the web service. fail authentication.
			LOG.error(ErrorCodeEnum.ACCOUNT_VALIDATOR_FAILURE_IS_USER_INACTIVE.toString(), e);
			throw new AuthException("Error getting User Profile from User Management Service.", new SystemDownException(ErrorCodeEnum.ACCOUNT_VALIDATOR_FAILURE_IS_USER_INACTIVE, e));
		}
	}

	private UserProfileDTO getUserProfile(String screenName) throws AuthException {
		UserProfileDTO userProfile = null;
		try {
			userProfile = getUserManagementService().getUserProfile(screenName);
		} catch (Exception ex) {
			if( !HTTP_SESSION_ERROR.equals(ex.getMessage())) {
				LOG.error(ErrorCodeEnum.USER_MANAGMENT_FAILURE_USER_PROFILE.toString(), ex);
				throw new AuthException("Error getting User Profile from User Management Service.", new SystemDownException(ErrorCodeEnum.USER_MANAGMENT_FAILURE_USER_PROFILE, ex));
			}
		} 
		
		return userProfile;
	}

	private PatientDTO getPatient(Long userProfileId) throws AuthException {
		try {
			return getUserManagementService().getPatient(userProfileId);
		} catch (Exception ex) {
			LOG.error(ErrorCodeEnum.USER_MANAGMENT_FAILURE_PATIENT.toString(), ex);
			throw new AuthException("Error getting Patient from User Management Service.", new SystemDownException(ErrorCodeEnum.USER_MANAGMENT_FAILURE_PATIENT, ex));
		}
	}


	private UserMgmtService getUserManagementService() {
		if (this.userMgmtService == null) {
			this.userMgmtService = (UserMgmtService) ApplicationContextProvider.getApplicationContext()
					.getBean("userMgmtServiceProxy");
		}
		return this.userMgmtService;
	}

	private AccountValidatorService getAccountValidatorService() {
		if (this.accountValidatorService == null) {
			this.accountValidatorService = (AccountValidatorService) ApplicationContextProvider.getApplicationContext()
					.getBean("accountValidatorServiceProxy");
		}

		return this.accountValidatorService;
	}
	
	/**
	 * Determines whether this user is a 'Delegate Field Tester' and whether the 'Delegate Field Tester'
	 * role has gone national.
	 * @param userName The username of the user we are checking on
	 * @return	<code>true</code> if this user has the 'Delegate Field Tester' role and if that role
	 * 			has NOT gone national, or <code>false</code> otherwise.
	 * @throws AuthException
	 */
	private Boolean isDelegateFieldTester(String userName) throws AuthException  {
		Boolean isDelegateFieldTester = new Boolean(false);
		try {
			List<FieldTestDTO> fieldTesterRoles = getUserManagementService().getFieldTestGroupsByUserName(userName);
			if (fieldTesterRoles != null) {
				for (FieldTestDTO fieldTesterRole: fieldTesterRoles) {
					if (FieldTestEnum.DELEGATE_FIELD_TESTER.getRole().equals(fieldTesterRole.getRole())) {
						if (!fieldTesterRole.getIsNational()) {							
							isDelegateFieldTester = new Boolean(true);
						}
						break;
					}
				}
			}
		} catch (Exception e) {
			LOG.error(e.getMessage());
			throw new AuthException(new SystemDownException(e));
		}
		return isDelegateFieldTester;
	}
	
	private AccountActivityLogService getAccountActivityLogService() {
		if(this.accountActivityLogService == null) {
			this.accountActivityLogService = (AccountActivityLogService)ApplicationContextProvider.getApplicationContext().getBean("activityLogProxy");
		}
		
		return this.accountActivityLogService;
	}


}
