package gov.va.med.mhv.util.distributor;

import gov.va.med.mhv.util.distributor.message.BaseRequest;
import gov.va.med.mhv.util.distributor.message.HTTPRequestBean;
import gov.va.med.mhv.util.distributor.message.HTTPResponseBean;
import gov.va.med.mhv.util.distributor.message.MimeType;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.security.KeyStore;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.http.HttpResponse;
import org.apache.http.client.CookieStore;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.conn.scheme.Scheme;
import org.apache.http.conn.ssl.SSLSocketFactory;
import org.apache.http.cookie.Cookie;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.DefaultHttpClient;
import org.apache.http.impl.client.StandardHttpRequestRetryHandler;
import org.apache.http.params.BasicHttpParams;
import org.apache.http.params.CoreConnectionPNames;
import org.apache.http.params.HttpConnectionParams;
import org.apache.http.params.HttpParams;


/**
 * This is an all inclusive HTTP distributor class.  It gives the ability to set and retrieve information
 * about the HTTP distribution in a standardized way.
 *
 */
public class HttpClientDistributor implements Distributor {

	private static Logger logger = Logger.getLogger(HttpClientDistributor.class.getName());
	private static KeyStore trustStore, keyStore;
	private CookieStore cookies;
	private DefaultHttpClient httpClient;

    private static final String URI_REGEX = "^(http.*?)://(.*?)(?::(.*?))?$";
    @SuppressWarnings("unused")
	private static final int MODE = 1, HOST = 2, PORT    ;
    private static final Pattern URI_PATTERN = Pattern.compile(URI_REGEX, Pattern.CASE_INSENSITIVE);

    private HTTPResponseBean response ;
    private HTTPRequestBean request = new HTTPRequestBean();
   // private final MultiThreadedHttpConnectionManager connManager = new MultiThreadedHttpConnectionManager();

    /*
     * Building the SSL context for two way SSL
     */
    private void buildSSL(String trustStorePath, String trustStorePassword, String keyStorePath, String keyStorePassword) throws Exception{
    	try {
    		trustStore = KeyStore.getInstance(KeyStore.getDefaultType());
    		keyStore = KeyStore.getInstance(KeyStore.getDefaultType());
    		FileInputStream in = new FileInputStream(new File(trustStorePath));
    		FileInputStream ins = new FileInputStream(new File(keyStorePath));
    		try {
    			trustStore.load(in, trustStorePassword.toCharArray());
    			keyStore.load(ins, keyStorePassword.toCharArray());
    		} finally {
    			try {
    				in.close();
    				ins.close();
    			} catch (Exception ignore) {}
    		}
    	}catch(Exception e){
    		logger.log(Level.SEVERE, "Unable to load keystore", e);
    		throw new Exception("Unable to load keystore");
    	}
    }

    /**
     * Empty Constructor
     */
    HttpClientDistributor() {  }

    /**
     * Constructor to intialize with a full request bean
     * @param request
     * @throws DistributorException
     */
    public HttpClientDistributor(BaseRequest request) throws DistributorException{
    	this.request = (HTTPRequestBean)request;
    	httpClient = new DefaultHttpClient();
    	Matcher m = URI_PATTERN.matcher(request.getUri());

    	// Build the SSL connection for the HTTPS schema
    	if((request.getSslKeyStoreLocation() != null && !"".equals(request.getSslKeyStoreLocation())) &&
    			(request.getSslTrustStoreLocation() != null && !"".equals(request.getSslTrustStoreLocation()))){
    		try{
    			buildSSL(request.getSslTrustStoreLocation(), request.getSslKeyStorePassword().toString(),
    					request.getSslKeyStoreLocation(), request.getSslKeyStorePassword().toString());
    			SSLSocketFactory socketFactory = new SSLSocketFactory(keyStore, request.getSslKeyStorePassword().toString(), trustStore);
    			Scheme sch = new Scheme("https", Integer.parseInt(m.group(PORT)), socketFactory);
    			httpClient.getConnectionManager().getSchemeRegistry().register(sch);
    		}catch(Exception e){
    			logger.log(Level.SEVERE, "Error Distributing Message", e);
    			throw new DistributorException(e);
    		}
    	}
    }

    /**
     * Instantiation method for HTTP Client
     * @param port
     * @param trustStorePath
     * @param trustStorePassword
     * @throws Exception
     */
    public HttpClientDistributor(int port, String keyStorePath, String keyStorePassword, String trustStorePath, String trustStorePassword)throws DistributorException{
    	httpClient = new DefaultHttpClient();
    	// Build out the SSL creds
    	if(keyStorePath == null || "".equals(keyStorePath)){
    		try{
    			buildSSL(trustStorePath, trustStorePassword, keyStorePath, keyStorePassword);
    			SSLSocketFactory socketFactory = new SSLSocketFactory(keyStore, keyStorePassword, trustStore);
    			Scheme sch = new Scheme("https", port, socketFactory);
    			httpClient.getConnectionManager().getSchemeRegistry().register(sch);
    		}catch(Exception e){
    			throw new DistributorException(e);
    		}
    	}
    }

    /**
     * Add in cookies to be passed to the server
     * @param cookie
     */
    public void addCookie(Cookie cookie){
    	cookies.addCookie(cookie);
    }
    /**
     * Returns cookies from the httpClient
     * @return
     */
    public CookieStore getCookies(){
    	return cookies;
    }
    /**
     * Sets the entity that will be passed with the HTTP call out
     * @param string
     */
    public void setEntity(String entity) {
    	request.setBody(entity);
    }

    //##############################################################################################
    // AUTO-ADDED METHODS
    //##############################################################################################

	/* (non-Javadoc)
	 * @see gov.dtspo.oss.base.distributor.Distributor#setMimeType(java.lang.String)
	 */
	@Override
	public void setMimeType(String mimeType) {
		try{
			request.setMimeType(MimeType.valueOf(mimeType));
		}catch(Exception e){
			logger.log(Level.WARNING, "Illegal MIME Type in HTTP Client it has been defaulted to application/xml: "+mimeType);
			request.setMimeType(MimeType.XMLTEXT);
		}
	}

	/* (non-Javadoc)
	 * @see gov.dtspo.oss.base.distributor.Distributor#distribute()
	 */
	@Override
	public HTTPResponseBean distribute() throws DistributorException {
    	try{
    		HttpRequestBase distRequest;
    		response = new HTTPResponseBean();

    		switch(request.getHttpMethod()){
    			case POST:
    				distRequest = new HttpPost(request.getUri());
    				break;
    			case GET:
    				distRequest = new HttpGet(request.getUri());
    				break;
    			case PUT:
    				distRequest = new HttpPut(request.getUri());
    				break;
    			case DELETE:
    				distRequest = new HttpDelete(request.getUri());
    				break;
    			default:
    				distRequest = new HttpPost(request.getUri());
    				break;
    		}

    		if(distRequest instanceof HttpEntityEnclosingRequestBase){
    			((HttpEntityEnclosingRequestBase)distRequest).setEntity(new StringEntity(request.getBody()));
    		}

    		httpClient.setHttpRequestRetryHandler(new StandardHttpRequestRetryHandler(request.getRetryAttempts(),true));

    		HttpParams httpParams = new BasicHttpParams();
    		httpParams.setParameter(CoreConnectionPNames.CONNECTION_TIMEOUT, (request.getTimeout() > -1 ? (int)request.getTimeout() : 600000));

    		//Setting the HTTP Header information
    		if(request.getProperties() != null && !request.getProperties().isEmpty()){
    			for(String key : request.getProperties().keySet()){
    				distRequest.setHeader(key, (String)request.getProperties().get(key));
    			}
    		}
    		httpClient.setParams(httpParams);
    		HttpResponse httpResponse = httpClient.execute(distRequest);
    		cookies = httpClient.getCookieStore();

    		response.setHeaders(httpResponse.getAllHeaders());
    		response.setStatusLine(httpResponse.getStatusLine().getReasonPhrase());
    		response.setStatusCode(httpResponse.getStatusLine().getStatusCode());
    		response.setSuccessful(httpResponse.getStatusLine().getStatusCode() < 400);

    		//Build out a simple way to get the String response
    		BufferedReader reader = new BufferedReader(new InputStreamReader(httpResponse.getEntity().getContent()));
    		StringBuilder sb = new StringBuilder();
    		String line;
    		while((line = reader.readLine()) != null){
    			sb.append(line);
    		}
    		response.setResponseBody(sb.toString());
    		return response;
    	} catch(Exception e){
    		e.printStackTrace();
    		throw new DistributorException("During " + request.getHttpMethod() + ": " + request.getUri(), e);
    	}finally {
    		httpClient.getConnectionManager().shutdown();
    	}
	}

	/* (non-Javadoc)
	 * @see gov.dtspo.oss.base.distributor.Distributor#supports(java.lang.String)
	 */
	@Override
	public boolean supports(String url) {
		try {
            String scheme = url.substring(0, url.indexOf("://"));
            return ("http".equals(scheme) || "https".equals(scheme));
        } catch (IndexOutOfBoundsException ioobe) {
            return false;
        }
	}

	/* (non-Javadoc)
	 * @see gov.dtspo.oss.base.distributor.Distributor#create(gov.dtspo.oss.base.distributor.message.BaseRequest)
	 */
	@Override
	public Distributor create(BaseRequest request) throws DistributorException {
		// TODO Auto-generated method stub
		return null;
	}

	/* (non-Javadoc)
	 * @see gov.dtspo.oss.base.distributor.Distributor#isDestinationMutable()
	 */
	@Override
	public boolean isDestinationMutable() {
		return true;
	}

	/* (non-Javadoc)
	 * @see gov.dtspo.oss.base.distributor.Distributor#setMessageProperties(java.util.Map)
	 */
	@Override
	public void setMessageProperties(Map<String, String> properties) {
		request.setProperties(properties);
	}

	/* (non-Javadoc)
	 * @see gov.dtspo.oss.base.distributor.Distributor#setRequestBean(gov.dtspo.oss.base.distributor.message.BaseRequest)
	 */
	@Override
	public void setRequestBean(BaseRequest request) {
		this.request = (HTTPRequestBean)request;
	}
}