package gov.va.med.mhv.search;

import com.liferay.portal.kernel.log.Log;
import com.liferay.portal.kernel.log.LogFactoryUtil;
import com.liferay.portal.kernel.search.BaseOpenSearchImpl;
import com.liferay.portal.kernel.search.SearchContext;
import com.liferay.portal.kernel.search.SearchContextFactory;
import com.liferay.portal.kernel.search.SearchException;
import com.liferay.portal.kernel.util.HtmlUtil;
import com.liferay.portal.kernel.util.PropsUtil;
import com.liferay.portal.kernel.util.WebKeys;
import com.liferay.portal.kernel.xml.Element;
import com.liferay.portal.model.Group;
import com.liferay.portal.model.Layout;
import com.liferay.portal.theme.ThemeDisplay;

import gov.va.med.mhv.util.client.SimpleSender;
import gov.va.med.mhv.util.distributor.http.HTTPMethod;
import gov.va.med.mhv.util.distributor.message.HTTPResponseBean;
import gov.va.med.mhv.util.distributor.message.MimeType;
import gov.va.med.mhv.util.xml.XPathSelector;
import gov.va.med.mhv.util.xml.XmlHelper;

import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.Random;

import javax.servlet.http.HttpServletRequest;

import org.jdom2.Content;

/**
 * This is the base class used to integrate the VA Enterprise Search with Lifeary.  
 * This portlet project needs to be deployed to liferay as a portlet, and the Search funcionality will 
 * automatically call it when searches are conducted.
 * 
 * @author Raphael Silvestro
 * @author Jason Hodge
 *
 */
public class EnterpriseSearch extends BaseOpenSearchImpl {

	private static Log _log = LogFactoryUtil.getLog(EnterpriseSearch.class);
	private static final String ENTERPRISE_SEARCH_QUERY_STRING = "gov.va.med.mhv.util.EnterpriseSearch.query.string";
	private static final boolean DEBUG = false;
	
	public EnterpriseSearch() {
	}

	@Override
	public boolean isEnabled() {
		return true;
	}

	/**
	 * Executes the search given the search parameters and user.  The search service endpoint is retrieved from the environment.
	 */
	@Override
	public String search(
			HttpServletRequest request, long groupId, long userId,
			String keywords, int startPage, int itemsPerPage, String format)
					throws SearchException {
		
		_log.debug("startPage: " + startPage + " itemsPerPage: " + itemsPerPage);
		
		try {
			ThemeDisplay themeDisplay = (ThemeDisplay)request.getAttribute(
					WebKeys.THEME_DISPLAY);
			SearchContext searchContext = SearchContextFactory.getInstance(
					request);
			searchContext.setAttribute("paginationType", "simple"); // regular, none, simple, more
			if (groupId == 0) {
				searchContext.setGroupIds(null);
			}
			else {
				searchContext.setGroupIds(new long[] {groupId});
			}
			int end = startPage * itemsPerPage;
			searchContext.setEnd(end);
			Layout layout = themeDisplay.getLayout();
			Group layoutGroup = layout.getGroup();
			if (!layoutGroup.isStagingGroup() &&
					!layoutGroup.isControlPanel()) {
				searchContext.setIncludeStagingGroups(false);
			}
			searchContext.setKeywords(keywords);
			searchContext.setScopeStrict(false);
			int start = (startPage * itemsPerPage) - itemsPerPage;
			searchContext.setStart(start);
			searchContext.setUserId(userId);

			//TODO -- make this configurable
			String queryString = getQueryString();
			_log.debug("queryString = " + queryString);
			_log.debug("keywords = " + keywords);
			String escapedKeywords = HtmlUtil.escapeURL(keywords);
			_log.debug("escapedKeywords = " + escapedKeywords);
			if(queryString == null) {
				_log.error("Key 'gov.va.med.mhv.util.EnterpriseSearch.query.string' not found in portal-ext.properties or enterprise-search.properties.  " +
							"Enterprise Search assets will not be included in search results!");
				return null;
			}
			else {
				queryString = String.format(queryString, escapedKeywords);
				queryString += "&RPP=" + itemsPerPage;
				if (startPage > 1) {
					queryString += "&RS=" + (1 + ((startPage - 1) * itemsPerPage));
				}
			
				_log.debug("queryString = " + queryString);
				
				HTTPResponseBean bean = SimpleSender.send(HTTPMethod.GET, MimeType.XML, queryString, null, null);
	
				if (bean == null) {
					_log.error("Did not receive a response from Enterprise Search!");
	
					return null;
				}
				
				String xml = bean.getResponseBody();
				_log.debug("Response from Enterprise Search Appliance:\r\n\r\n" + xml + "\r\n\r\n");
				
				org.jdom2.Document xmlDoc = XmlHelper.parseString(xml);
				if (DEBUG) {
					writeToFile(xml, "EnterpriseSearchResults");
				}
				XPathSelector selector = new XPathSelector();
				Content matches = selector.selectFirst((org.jdom2.Element)xmlDoc.getRootElement(), "//matches");
				Integer totalMatches = new Integer(0);
				if( matches != null ) {
					String matchesValue = matches.getValue();
					if( matchesValue != null ) {
						totalMatches = new Integer(matchesValue);
						 
					}
				}
	
				String searchPath = "searchPath";
				format = "atom";
				String[] queryTerms = new String[]{keywords};
				if( totalMatches > 0 ) {
					selector = new XPathSelector();
					xmlDoc = XmlHelper.parseString(xml);
					
	
					List<Content> contentList = selector.selectAll((org.jdom2.Element)xmlDoc.getRootElement(), "//result");
					com.liferay.portal.kernel.xml.Document doc = null;
					
					if( contentList != null && contentList.size() > 0 ) {
						int total = contentList.size();//results.getLength();
						if (total == itemsPerPage) {
							total = 200;
						}
						Object[] values = addSearchResults(
								queryTerms, keywords, startPage, itemsPerPage, total, start,
								"Enterprise Search Title", searchPath, format, themeDisplay);
						
						doc = (com.liferay.portal.kernel.xml.Document)values[0];
						Element root = (Element)values[1];
						int count = 0;
						for( Content c : contentList ) {
							
							Date d = new Date();
	
							Content title = selector.selectFirst(c, "result_title");
							String titleValue = HtmlUtil.render(title.getValue());
							
							Content url = selector.selectFirst(c, "result_url");
							String urlValue = url.getValue();
							
							Content description = selector.selectFirst(c, "result_description");
							String descriptionValue = stripIllegalCharacters(description.getValue());

							_log.debug(title + " " + url);
							
							Random random = new Random(System.currentTimeMillis());
							addSearchResult(
									root, groupId, themeDisplay.getScopeGroupId(), "entryClassName",
									random.nextLong(), titleValue, urlValue, d, descriptionValue, 0.8, format);					
							count++;
							if( count == itemsPerPage ) {
								break;
							}
							_log.debug("Search Results Found: " + count);
						}
					} 
					
					String result = doc.asXML();
					_log.debug("Returning OpenSearch Results Doc to Liferay:\r\n\r\n" + result + "\r\n\r\n");
					if (DEBUG) {
						writeToFile(result, "OpenSearchDoc");
					}

					return result;			
				} else {
					int total = totalMatches;
					com.liferay.portal.kernel.xml.Document doc = null;
	
					Object[] values = addSearchResults(
							queryTerms, keywords, startPage, itemsPerPage, total, start,
							"Enterprise Search Title", searchPath, format, themeDisplay);
					doc = (com.liferay.portal.kernel.xml.Document)values[0];
					
					String responseXML = doc.asXML();
					_log.debug("Empty Respons:\r\n\r\n" + responseXML + "\r\n\r\n");
					return responseXML;		
				}
			}
		}
		catch (Exception e) {
			_log.error(e);
			throw new SearchException(e);
		}
	}
	
	/**
	 * There seems to be no way to safely escape the description results, so do a manual replacement of illegal
	 * characters with suitable, legal characters.  Note, /u0023 &amp; and &#187; formats are all illegal in search
	 * results.  %20 (URL) style replacements are of the wrong encoding type.  Thus, this leaves this as the only
	 * suitable option.
	 * 
	 * @param html
	 * @return
	 */
	private String stripIllegalCharacters(String html) {
		String result = html;
		
		// add here any additional replacements found to be required
		result = result.replaceAll("»", "-");
		
		return result;
	}
	
//	private String replaceToken(String source, String token, String with) {
//		String result = source;
//	}

	private String getQueryString() {
		// hard coded
		String queryString = null;
		
		// local properties
		Properties prop = new Properties();
		InputStream stream = null;
		try {
			stream = EnterpriseSearch.this.getClass().getResourceAsStream("/enterprise-search.properties");
			prop.load(stream);
			queryString = prop.getProperty(ENTERPRISE_SEARCH_QUERY_STRING);
			if (queryString == null || "".equals(queryString.trim())) {
				_log.debug("gov.va.med.mhv.search.EnterpriseSearch unable to read queryString from local properties file.  Checking portal-ext.properties...");
				// read from portal-ext.properties
				queryString = PropsUtil.get(ENTERPRISE_SEARCH_QUERY_STRING);
				_log.debug("gov.va.med.mhv.search.EnterpriseSearch read from portal-ext.properties: " + queryString);
			}
		} catch (IOException e) {
			_log.error("Unable to establish Enterprise Search Endpoint URL!", e);
			if (stream != null) {
				try {
					stream.close();
				} catch (IOException e1) {
				}
			}
		}
		
		return queryString;		
	}

	private void writeToFile(String contents, String filenamePrefix) {
		try {
			String fullPath = "/installs/temp/" + filenamePrefix + "-" + System.currentTimeMillis() + ".xml";
			FileWriter writer = new FileWriter(fullPath);
			writer.write(contents);
			writer.close();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

}
