package gov.va.med.mhv.core.crypto;

import java.io.Serializable;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import org.springframework.util.Assert;

/**
 * Provides calculation of a checksum, based on a specific algorithm, 
 * from a given input.
 * @author Rob Proper
 *
 */
public class ChecksumCalculator implements Serializable {
	
	/**
	 * Default checksum algorithm (MD5).
	 */
	public static final String DEFAULT_ALGORITHM = "MD5";

	private final MessageDigest digester;

	/**
	 * Creates a checksum calculator based on the default algorithm.
	 * @see #DEFAULT_ALGORITHM
	 * @throws NoSuchAlgorithmException When there is no provider for
	 * the default algorithm
	 */
	public ChecksumCalculator() throws NoSuchAlgorithmException {
		this(DEFAULT_ALGORITHM);
	}
	
	/**
	 * Creates a checksum calculator based on the given algorithm.
	 * @param algorithm The algorithm to use for calculations.
	 * @throws NoSuchAlgorithmException When there is no provider for
	 * the given algorithm
	 */
	public ChecksumCalculator(String algorithm) 
		throws NoSuchAlgorithmException 
    {
		digester = MessageDigest.getInstance(algorithm);
	}
	
	/**
	 * Calculates the checksum for a given input.
	 * @param input The input to calculate the checksum from
	 * @return The checksum calculated from the given input.
	 */
	public String calculate(String input) {
		Assert.notNull(input);
		//Precondition.assertNotNull("input", input);
		digester.update(input.getBytes());
		byte[] hash = digester.digest();
		StringBuffer buffer = new StringBuffer();
		for(int i = 0; i < hash.length; i++) {
			buffer.append(Integer.toHexString(hash[i]));
		}
		return buffer.toString();
	}

}
