package gov.va.med.mhv.common.api.security;

import java.net.URI;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Priority;
import javax.ws.rs.Priorities;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.container.PreMatching;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.ResponseBuilder;
import javax.ws.rs.core.UriBuilder;

import org.apache.cxf.interceptor.security.AuthenticationException;
import org.apache.cxf.jaxrs.impl.HttpHeadersImpl;
import org.apache.cxf.jaxrs.utils.HttpUtils;
import org.apache.cxf.jaxrs.utils.JAXRSUtils;
import org.apache.cxf.message.Message;
import org.springframework.beans.factory.annotation.Autowired;


@Priority(Priorities.AUTHENTICATION)
@PreMatching
public class ApiAuthenticationFilter implements ContainerRequestFilter {
	
	//see JAAS example: http://svn.apache.org/repos/asf/cxf/trunk/rt/frontend/jaxrs/src/main/java/org/apache/cxf/jaxrs/security/JAASAuthenticationFilter.java
    private static final List<MediaType> HTML_MEDIA_TYPES = 
            Arrays.asList(MediaType.APPLICATION_XHTML_XML_TYPE, MediaType.TEXT_HTML_TYPE);
    
	private URI redirectURI;
	
	@Autowired
	private BasicTokenAuthentication token;
	
	protected static final String HEADER_KEY_NAME = "X-Authorization-Key";
        
    public void filter(ContainerRequestContext context) {
        
    	Message m = JAXRSUtils.getCurrentMessage();
    	try {
        	doAuthentication(m);
        } catch (AuthenticationException ex) {
            context.abortWith(handleAuthenticationException(ex, m));
        } catch (SecurityException ex) {
            context.abortWith(handleAuthenticationException(ex, m));
        }
    }
    
    protected void doAuthentication(Message m) throws AuthenticationException, SecurityException {
    	HttpHeaders headers = new HttpHeadersImpl(m);
    	MultivaluedMap<String, String> hMap = headers.getRequestHeaders();
    	if (!hMap.containsKey(HEADER_KEY_NAME)) 
    	{
    		throw new SecurityException("Authorization header not found");
    	} else {
    		String authKey = hMap.getFirst(token.getRequiredHeader());
    		
    		if (!token.validate(authKey)) {
    			throw new SecurityException("Authorization key mismatch");
    		}
    	}
    }

    public BasicTokenAuthentication getToken() {
		return token;
	}

	public void setToken(BasicTokenAuthentication token) {
		this.token = token;
	}

	protected Response handleAuthenticationException(SecurityException ex, Message m) {
        HttpHeaders headers = new HttpHeadersImpl(m);
        if (redirectURI != null && isRedirectPossible(headers)) {
            
            URI finalRedirectURI = null;
     
            if (!redirectURI.isAbsolute()) {
                String endpointAddress = HttpUtils.getEndpointAddress(m);
                Object basePathProperty = m.get(Message.BASE_PATH);
                if (basePathProperty != null && !"/".equals(basePathProperty)) {
                    int index = endpointAddress.lastIndexOf(basePathProperty.toString());
                    if (index != -1) {
                        endpointAddress = endpointAddress.substring(0, index);
                    }
                }
                finalRedirectURI = UriBuilder.fromUri(endpointAddress).path(redirectURI.toString()).build();
            } else {
                finalRedirectURI = redirectURI;
            }
            
            return Response.status(getRedirectStatus()).
                    header(HttpHeaders.LOCATION, finalRedirectURI).build();
        } else {
            ResponseBuilder builder = Response.status(Response.Status.UNAUTHORIZED);
            return builder.build();
        }
    }
    
    protected Response.Status getRedirectStatus() {
        return Response.Status.TEMPORARY_REDIRECT;
    }
    
    protected boolean isRedirectPossible(HttpHeaders headers) {
        List<MediaType> clientTypes = headers.getAcceptableMediaTypes();
        return !JAXRSUtils.intersectMimeTypes(clientTypes, HTML_MEDIA_TYPES, false)
                          .isEmpty();
    }
}
