package gov.va.med.mhv.calendar.validator;

import gov.va.med.mhv.calendar.dto.CalendarEventDTO;
import gov.va.med.mhv.calendar.enums.ReminderIntervalEnumeration;
import gov.va.med.mhv.calendar.util.CommonUtility;
import gov.va.med.mhv.common.api.util.ResponseUtil;

import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.LinkedHashMap;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.time.DateUtils;
import org.springframework.stereotype.Component;


@Component
public class CalendarEventValidator extends Validator {
	
	private static final String START_START_DATE="Start Date/Time";
	private static final String START_END_DATE="End Date / Time";
	private static final String RECUR_UNTIL_DATE="Recur Until Date";
	private static final String VHA_EVENT=" VHA Event";
	private static final String CALENDAR_CATEGORY="Calendar Category";
	private static final String LINK="Link";
	
	private static final String EVNT_NAME="Event Name";
	private static final String EVNT_DESCR="Description";
	
	private static final String START_START_TIME ="Start Time";
	private static final String START_END_TIME ="End Time";
	private static final String START_AND_END_TIME ="Start Time and End Time";


	
	public ResponseUtil<CalendarEventDTO> fieldValidations(CalendarEventDTO calendarEventDTO) {
		ResponseUtil<CalendarEventDTO> response = new ResponseUtil<CalendarEventDTO>();
		
		LinkedHashMap<String, String> validationErrors = new LinkedHashMap<String, String>();
		
		validateEventName(calendarEventDTO, validationErrors);
		validateDescription(calendarEventDTO, validationErrors);
		validateStartEndDate(calendarEventDTO, validationErrors);
		validateRecurrenceInterval(calendarEventDTO, validationErrors);
		validateRecurUntilDate(calendarEventDTO, validationErrors);
		
		if (validationErrors.size() == 0) {
			complexValidation(calendarEventDTO, validationErrors);
		}
		response.setValidationErrors(validationErrors);
		return response;
	}

	
	private void complexValidation(CalendarEventDTO calendarEventDTO,LinkedHashMap<String, String> validationErrors) {
		validateMultiDayEvent(calendarEventDTO, validationErrors);
		validateReminderDate(calendarEventDTO, validationErrors);
	}


	/**
	 * Ensure the Event name is not blank
	 * 
	 */
	private void validateEventName(CalendarEventDTO calendarEventDTO, LinkedHashMap<String, String> validationErrors) {
		if ((calendarEventDTO.getSubject() == null) || (calendarEventDTO.getSubject().isEmpty())) {
			
				validationErrors.put(EVNT_NAME, EVNT_NAME + " is required.");
			
		}
		
	if (calendarEventDTO.getSubject() != null && calendarEventDTO.getSubject().length() > 50) {
			
			validationErrors.put(EVNT_NAME, EVNT_NAME + " has exceeded the maximun length.");
		}
	}
	

	/**
	 * Description validations
	 * 
	 */
	
	private void validateDescription(CalendarEventDTO calendarEventDTO, LinkedHashMap<String, String> validationErrors) {
		
		
	if (calendarEventDTO.getDescription() != null && calendarEventDTO.getDescription().length() > 255) {
			
			validationErrors.put(EVNT_DESCR, EVNT_DESCR + " has exceeded the maximun length.");
		}
	}
	
	/**
	 * Ensure the end date is greater than the start date
	 * 
	 */
	private void validateStartEndDate(CalendarEventDTO calendarEventDTO, LinkedHashMap<String, String> validationErrors) {

		
			if (calendarEventDTO.getStartDate() == null) {
			    
			    //validationErrors.put(START_END_DATE, "Please select the Start time and End time.&#60;br /&#62;Alternatively, you can make this an &#34;All Day Event&#34; which does not require a time.");
				validationErrors.put(START_END_DATE, "Please select the Start time and End time.\n Alternatively, you can make this an &#34;All Day Event&#34; which does not require a time.");
		       }
		
		
		if (!calendarEventDTO.getAllDay()){
			if (calendarEventDTO.getStartDate() != null ) {
			  
		
		//String dateTime = CommonUtility.dateToString(calendarEventDTO.getStartDate(), "HH:mm");
		
		String StartdateTimeHr = CommonUtility.dateToString(calendarEventDTO.getStartDate(), "HH");
		
		 /* if (StartdateTimeHr.equals("00")) {
				validationErrors.put(START_END_DATE, "Please select the Start time and End time.\n Alternatively, you can make this an &#34;All Day Event&#34; which does not require a time.");
 		    } */
		
		  
		       }
		}
		
		
		if (!calendarEventDTO.getAllDay()){
			if (calendarEventDTO.getEndDate() != null ) {
			  
		String EnddateTimeHr = CommonUtility.dateToString(calendarEventDTO.getEndDate(), "HH");
		
		  if (EnddateTimeHr.equals("00")) {
				validationErrors.put(START_END_DATE, "Please select the Start time and End time.\n Alternatively, you can make this an &#34;All Day Event&#34; which does not require a time.");		  }
		
		  
		       }
		}
		
		
		
			if (calendarEventDTO.getEndDate() == null) {
				validationErrors.put(START_END_DATE, "Please select the Start time and End time.\n Alternatively, you can make this an &#34;All Day Event&#34; which does not require a time.");		       }
		
		if (calendarEventDTO.getEndDate() != null && calendarEventDTO.getStartDate() != null) {
			if (calendarEventDTO.getEndDate().compareTo(calendarEventDTO.getStartDate()) < 0) {
				validationErrors.put(START_END_DATE, START_END_DATE + " must be later than or equal to the start date/time.");
			}
		}
		
		if (calendarEventDTO.getStartDate() != null ) {
			
			String startDate= dateToString(calendarEventDTO.getStartDate(),"yyyyMMdd");
			Calendar cal =Calendar.getInstance();
			String currentDate = dateToString(cal.getTime(),"yyyyMMdd");
			
			if(Integer.parseInt(startDate) < Integer.parseInt(currentDate)) {
				
		
			validationErrors.put(START_START_DATE, START_START_DATE + "  should be greater than today's date.");
			
		}
	}
	}
		
		private static String dateToString (Date date,String format){
			String stringDate="";
			try{
				if(date != null){
				   SimpleDateFormat sdf = new SimpleDateFormat(format);
				   stringDate =sdf.format(date);
				}
			}catch(Exception e){
				throw new RuntimeException("Invalid Date format");
			}
			return stringDate;
		}

	/**
	 * Ensure the recurrence interval is selected if until date is selected
	 * 
	 */
	protected void validateRecurrenceInterval(CalendarEventDTO calendarEventDTO, LinkedHashMap<String, String> validationErrors) {
		if (calendarEventDTO.getRecurUntilDate() != null && calendarEventDTO.getRecurrenceInterval() == null) {
			validationErrors.put(RECUR_UNTIL_DATE, "	Please specify the amount of time between repeating events.");
			
		}
	}

	/**
	 * Ensure the repeat until date is valid if recurrence interval is selected
	 * 
	 */
	protected void validateRecurUntilDate(CalendarEventDTO calendarEventDTO, LinkedHashMap<String, String> validationErrors) {
		if (calendarEventDTO.getRecurUntilDate() == null && calendarEventDTO.getRecurrenceInterval() != null) {
			validationErrors.put(RECUR_UNTIL_DATE, "	You must select a repeat until date.");
		}
		else if (calendarEventDTO.getRecurUntilDate() != null && calendarEventDTO.getRecurUntilDate().compareTo(calendarEventDTO.getStartDate()) < 0) {
			validationErrors.put(RECUR_UNTIL_DATE,  " The Repeats Until date must be greater than the Start Date.");

		}
	}

	/**
	 * Ensure the until date is null if event spans multiple days. An event
	 * cannot be both recurring and repeating.
	 * 
	 */
	protected void validateMultiDayEvent(CalendarEventDTO calendarEventDTO, LinkedHashMap<String, String> validationErrors) {
		Calendar startCal = Calendar.getInstance();
		Calendar endCal = Calendar.getInstance();
		startCal.setTimeInMillis(calendarEventDTO.getStartDate().getTime());
		endCal.setTimeInMillis(calendarEventDTO.getEndDate().getTime());
		if (!DateUtils.isSameDay(startCal, endCal)) {
			calendarEventDTO.setRepeating(new Boolean(true));
		} else {
			calendarEventDTO.setRepeating(new Boolean(false));
		}

		if (calendarEventDTO.getRecurrenceInterval() != null && calendarEventDTO.getRepeating().booleanValue()) {
			validationErrors.put(START_START_DATE,  "The start and end dates must be the same for repeating events.");
		}
	}

	/**
	 * First calculate the date the reminder should be sent. Then see if the
	 * date has already passed. If so, alert the user.
	 * 
	 */
	protected void validateReminderDate(CalendarEventDTO calendarEventDTO, LinkedHashMap<String, String> validationErrors) {
		if (calendarEventDTO.getReminderInterval() != null) {
			Calendar reminderCal = Calendar.getInstance();
			String remInt = calendarEventDTO.getReminderInterval();
			reminderCal.setTimeInMillis(calendarEventDTO.getStartDate().getTime());
			//CalendarUtil.setFirstSecondOfDay(reminderCal);
			if (remInt.equals(ReminderIntervalEnumeration.D1)) {
				reminderCal.add(Calendar.DATE, -1);
			} else if (remInt.equals(ReminderIntervalEnumeration.D2)) {
				reminderCal.add(Calendar.DATE, -2);
			} else if (remInt.equals(ReminderIntervalEnumeration.D3)) {
				reminderCal.add(Calendar.DATE, -3);
			} else if (remInt.equals(ReminderIntervalEnumeration.D4)) {
				reminderCal.add(Calendar.DATE, -4);
			} else if (remInt.equals(ReminderIntervalEnumeration.W1)) {
				reminderCal.add(Calendar.DATE, -7);
			} else if (remInt.equals(ReminderIntervalEnumeration.W2)) {
				reminderCal.add(Calendar.DATE, -14);
			}

			Calendar cutOffDateForReminders = Calendar.getInstance();
			cutOffDateForReminders.setTime(reminderCal.getTime());

			cutOffDateForReminders.set(Calendar.HOUR_OF_DAY, cutOffDateForReminders.getMinimum(Calendar.HOUR_OF_DAY));
			cutOffDateForReminders.set(Calendar.MINUTE, cutOffDateForReminders.getMinimum(Calendar.MINUTE));
			cutOffDateForReminders.set(Calendar.SECOND, cutOffDateForReminders.getMinimum(Calendar.SECOND));
			cutOffDateForReminders.set(Calendar.MILLISECOND, cutOffDateForReminders.getMinimum(Calendar.MILLISECOND));

			if (reminderCal.after(cutOffDateForReminders)) {
				calendarEventDTO.setReminderDate(new Timestamp(reminderCal.getTimeInMillis()));
			} else {
				// if reminder date is before today's date we should tell the user
				// that their reminder won't work because it has already passed.
				validationErrors.put(START_START_DATE, START_START_DATE + "The reminder date for this event has already passed. Please change the reminder interval or the due date.");
				
			}
		}
	}
	
	protected void validateAdminEvent(CalendarEventDTO calendarEventDTO, LinkedHashMap<String, String> validationErrors) {
		if (calendarEventDTO.getVhaEvent() != null && calendarEventDTO.getVhaEvent().booleanValue() == true && (calendarEventDTO.getVisn() != null && calendarEventDTO.getVisn().getVisnId().longValue()>0)) {
			// a vha global event, as indicated by the vha event flag, cannot
			// have a visn
			validationErrors.put(VHA_EVENT, VHA_EVENT + "The event cannot be both a regional and global event.");
		} else if (calendarEventDTO.getApiEventKey() != null && calendarEventDTO.getCalendarCategory() != null) {
			validationErrors.put(CALENDAR_CATEGORY, CALENDAR_CATEGORY + "The event cannot be assigned both an admin created category and a system generated category.");
			
		}  else if (((calendarEventDTO.getVhaEvent() != null && calendarEventDTO.getVhaEvent().booleanValue() == true) || calendarEventDTO.getVisn() != null) && calendarEventDTO.getApiEventKey() == null && calendarEventDTO.getCalendarCategory() == null) {
			// an admin entered event must have a category, but not an api event
			// key (api cat)
			
			//Error not found in legacy code. Does not appear that it's used. Add this later if found or update validation to include proper error.
			//addError("calendar.detail.error.emptyCategory");
		}
	}
	
	protected void validateLink(CalendarEventDTO calendarEventDTO, LinkedHashMap<String, String> validationErrors) {
		if (!StringUtils.isEmpty(calendarEventDTO.getLink())) {
			URI uri = null;
			// Create a URI
			try {
				uri = new URI(calendarEventDTO.getLink());
				// Convert an absolute URI to a URL
				uri.toURL();
			} catch (IllegalArgumentException e) {
				// URI was not absolute
				validationErrors.put(LINK, LINK + "The URL is malformed.  Please test the url in a new browser window and re-enter a valid url that follows the following pattern: [protocol://host/file].");
			} catch (MalformedURLException e) {
				validationErrors.put(LINK, LINK + "The URL is malformed.  Please test the url in a new browser window and re-enter a valid url that follows the following pattern: [protocol://host/file].");
			} catch (URISyntaxException e) {
				validationErrors.put(LINK, LINK + "The URL is malformed.  Please test the url in a new browser window and re-enter a valid url that follows the following pattern: [protocol://host/file].");
			}
		}
	}
	
}


