package gov.va.med.mhv.bluebutton.web.controller;

import gov.va.med.mhv.bluebutton.transfer.SendRequestsStatusDTO;
import gov.va.med.mhv.bluebutton.web.bean.SendRequestsStatusBean;
import gov.va.med.mhv.common.api.enumeration.UserTypeEnum;
import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.usermgmt.common.dto.DirectMessageDTO;
import gov.va.med.mhv.common.api.dto.UserProfileDTO;
import gov.va.med.mhv.usermgmt.common.enums.ActivityActionTypeEnumeration;
import gov.va.med.mhv.usermgmt.common.enums.ActivityActorTypeEnumeration;
import gov.va.med.mhv.usermgmt.common.enums.ActivityTypeEnumeration;
import gov.va.med.mhv.usermgmt.service.AccountActivityCreatorService;
import gov.va.med.mhv.usermgmt.service.DirectMessageService;
import gov.va.med.mhv.usermgmt.util.activity.ActivityHelper;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Serializable;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;
import java.util.Set;

import javax.annotation.Resource;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import javax.faces.bean.ManagedBean;
import javax.faces.context.FacesContext;
import javax.faces.event.ComponentSystemEvent;
import javax.portlet.PortletRequest;
import javax.portlet.PortletSession;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.PropertySource;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonParser;


@ManagedBean
@Component
@PropertySource("classpath:/${MHV_ENV_PROPERTY}.bluebutton.portlet.properties")
@Scope("request")
public class SendRequestsStatusController implements Serializable {

	private static final long serialVersionUID = -3579628966465330069L;
	private static Logger logger = LogManager.getLogger(SendRequestsStatusController.class);
	

	private static final String USER_ROLE_KEY = "LIFERAY_SHARED_accountType";
	private static final String USERPROFILE_DTO_KEY = "LIFERAY_SHARED_userprofiledto";
	
	@Autowired
	private SendRequestsStatusBean sendRequestsStatusBean;

	@Resource(name = "activityProxy")
	private AccountActivityCreatorService activityProxy;

	@Resource
	private DirectMessageService directMessageServiceProxy;
	
	@Autowired
	private ObjectMapper mapper;

	@Value("${direct.mailbox}")
	private String directMailbox;
	
	@Value("${direct.messageurl}")
	private String directMessageurl;
	
	@Value("${direct.messagepath}")
	private String directMessagepath;
	
	@Value("${direct.privatekey}")
	private String directPrivatekey;
	
	@Value("${direct.publickey}")
	private String directPublickey;
		
	
	public void init(ComponentSystemEvent event)  throws MHVException {
		// Get user profile info
		sendRequestsStatusBean.setUserProfileID(getUserProfileIDFromSession());
		sendRequestsStatusBean.setIsBasicUser(new Boolean(UserTypeEnum.isBasic(getUserRoleFromSession())));
		
		// Get the list of all direct messages for this user and determine if the message list is empty.
		List<DirectMessageDTO> allDirectMesages = directMessageServiceProxy.getDirectMessagesByUserProfileId(sendRequestsStatusBean.getUserProfileID());
		sendRequestsStatusBean.setIsRequestsStatusListEmpty(new Boolean(allDirectMesages != null ? (allDirectMesages.size() < 1) : true));
		if (logger.isDebugEnabled()) {
			logger.debug("Number of Direct messages for this user = " + (allDirectMesages != null ? allDirectMesages.size() : null));
		}
		
		// Set the flag to show the 'Nothing found' notice on the page
		sendRequestsStatusBean.setShowNothingFoundMsg(sendRequestsStatusBean.getIsRequestsStatusListEmpty());
		
		// Get a list of messages that have a status of 'Message Sent'
		List<DirectMessageDTO> messagesWithSentStatusList = getDirectMessagesWithSentStatus(allDirectMesages);
		if (logger.isDebugEnabled()) {
			logger.debug("Number of Direct messages with a status of 'Message Sent' = " + messagesWithSentStatusList.size());
		}
		
		// Check the status of each sent message and update its status in the database
		for (DirectMessageDTO directMessage : messagesWithSentStatusList) {
			updateStatusOfSentMessage(directMessage);
		}
		
		// Get the updated message list for display
		List<DirectMessageDTO> updatedDirectMesagesList = directMessageServiceProxy.getDirectMessagesByUserProfileId(sendRequestsStatusBean.getUserProfileID());
		sendRequestsStatusBean.setSendRequestsStatusListWithDirectMessages(updatedDirectMesagesList);
	}

		
	/**
	 * Uses the given list to create a new list of <code>DirectMessageDTO</code> objects that only contains
	 * Direct messages that have a 'Message sent' status.
	 * @param allDirectMessages	A list of all <code>DirectMessageDTO</code> objects for the logged in user.
	 * @return	A list of <code>DirectMessageDTO</code> objects, derived from the given list,  that represent
	 * 			Direct messages that have a status of 'Message sent', or an empty list. Never returns <code>null</code>.
	 */
	private List<DirectMessageDTO> getDirectMessagesWithSentStatus(List<DirectMessageDTO> allDirectMessages) {
		List<DirectMessageDTO> messagesWithSentStatus = new ArrayList<DirectMessageDTO>();
		
		if (allDirectMessages != null) {
			for (DirectMessageDTO directMessage : allDirectMessages) {
				if ( SendRequestsStatusDTO.MESSAGE_SENT.equals(directMessage.getDirectMessageStatus()) ) {
					messagesWithSentStatus.add(directMessage);
				}
			}
		}
		return messagesWithSentStatus;
	}
	
	
	/**
	 * Checks with VA Direct to determine the status of the given Direct message, and then updates
	 * that message's status in the database if the status is known.
	 * @param directMessage The Direct message to check on and update
	 * @throws MHVException 
	 */
	private void updateStatusOfSentMessage(DirectMessageDTO directMessage) {
		Long userProfileId = sendRequestsStatusBean.getUserProfileID();
		
		Boolean messageDelivered = isMessageDeliveredSuccessfully(directMessage.getMessageId());
		String emailAddress = directMessage.getSentToEmailAddress();
				
		if (messageDelivered != null) {
			// Record the success/failure status in the AAL
			try {
				activityProxy.createAccountActivityLog(ActivityHelper.createActivityDTO(userProfileId, ActivityActorTypeEnumeration.SYSTEM, messageDelivered.booleanValue(), ActivityTypeEnumeration.SEND_VA_HEALTH_SUMMARY, ActivityActionTypeEnumeration.DELIVERED_TO_SYSTEM, "Request to send VA Health Summary to " + emailAddress + " was delivered to "+ emailAddress.substring(emailAddress.indexOf("@"))));
			} catch (MHVException e) {
				if (logger.isErrorEnabled()) {
					logger.error(e);
				}
			}
			
    		// Update and save the status of the message
			directMessage.setDirectMessageStatus(messageDelivered.booleanValue() ? SendRequestsStatusDTO.MESSAGE_SUCCESSFUL : SendRequestsStatusDTO.MESSAGE_FAILED);
    		directMessageServiceProxy.saveDirectMessage(directMessage);
			if (logger.isDebugEnabled()) {
				logger.debug("Status of Direct message updated in database to '" + directMessage.getDirectMessageStatus() + "'.");
			}
		}
		else {
			if (logger.isDebugEnabled()) {
				logger.debug("Direct message not updated in database because the Http response codee from VA Direct was not 200.");
			}
		}
	}
	
	
	/**
	 * Connects to VA Direct and finds out whether the given Direct message with a 'Message sent' status was
	 * delivered or not.
	 * @param	messageID The ID of the message that we want to check the status of.
	 * @return	Returns <code>true</code> if the message was delivered, or <code>false</code> if the delivery failed,
	 * 			or <code>null</code> if the Http response code was not 200.
	 */
	private Boolean isMessageDeliveredSuccessfully(String messageID) {
		HttpURLConnection connectionToVADirect = null;
		Boolean deliveryStatus = null;
		
		try {
			connectionToVADirect = setUpHttpURLConnection(messageID);
			
            int responseCode = connectionToVADirect.getResponseCode();
			if (logger.isDebugEnabled()) {
				logger.debug("&&&&&&Response code from HTTP request to VA Direct = " + responseCode);
			}
            
	        if (responseCode == 200) {
				BufferedReader bufferedReader = new BufferedReader(new InputStreamReader((connectionToVADirect.getInputStream())));
				String output;
				String jsonResponsStr = "";
				while ((output = bufferedReader.readLine()) != null) {
					jsonResponsStr += output;
				}
				
				int[] statusValues = parseJsonResponseForMessageStatus(jsonResponsStr);
				
				// Follow the exact logic found in the legacy code
				
	        	if((statusValues != null) && (statusValues.length == 0)) {
	        		//We got back an empty message status value so we know for sure that the delivery status is 'failed'.
	        		deliveryStatus = new Boolean(false);
	        	}
	        	else{
	        		// We got back a non-empty message status value so we assume that the message was delivered.
	        		deliveryStatus = new Boolean(true);
	        	}		
			}
			else if (responseCode == 400) {
				// do nothing
			}
			else if (responseCode == 401) {
				
			}
			else {
				throw new RuntimeException("\nFailed : HTTP error code : " + responseCode + "\nHTTP response : " + connectionToVADirect.getResponseMessage());
			}
		}
		catch (MalformedURLException e) {
			if (logger.isErrorEnabled()) {
				logger.error(e);
			}
		}
		catch (IOException e) {
			if (logger.isErrorEnabled()) {
				logger.error(e);
			}
		}
		catch (NoSuchAlgorithmException e) {
			if (logger.isErrorEnabled()) {
				logger.error(e);
			}
		}
		catch (InvalidKeyException e) {
			if (logger.isErrorEnabled()) {
				logger.error(e);
			}
		}
		finally {
			if (connectionToVADirect != null) {
				connectionToVADirect.disconnect();
			}
		}
		
		// The Http response code was not 200
		return deliveryStatus;
	}

	
	@SuppressWarnings("restriction")
	private HttpURLConnection setUpHttpURLConnection(String messageID) throws MalformedURLException, IOException, NoSuchAlgorithmException, InvalidKeyException {
		String getQueryStr = "?mailbox="+ directMailbox + "&id=" + messageID + "&part=headers";
		SimpleDateFormat dateFormat = new SimpleDateFormat("MM/dd/yyyy HH:mm:ss z");
		String dateStr = dateFormat.format(new java.util.Date());
		String requestStr = "GET\n" + dateStr + "\n" + directMessagepath + getQueryStr;
		HttpURLConnection connectionToVADirect = null;
		
		URL urlDirect = new URL((directMessageurl + getQueryStr));
		connectionToVADirect = (HttpURLConnection)urlDirect.openConnection();
		
		Mac sha256_HMAC = Mac.getInstance("HmacSHA256");
        SecretKeySpec secret_key = new SecretKeySpec(directPrivatekey.getBytes(),"HmacSHA256");
		sha256_HMAC.init(secret_key);
		
        byte[] sha = sha256_HMAC.doFinal(requestStr.getBytes());
		String encsha = new sun.misc.BASE64Encoder().encode(sha);
		String auth = "DAAS " + directPublickey + ":" + encsha;

    	connectionToVADirect.setRequestProperty("Authorization", auth);
    	connectionToVADirect.setRequestProperty("Date", dateStr);
    	connectionToVADirect.setRequestProperty("X-DaaS-Date", dateStr);
    	connectionToVADirect.setRequestProperty("Accept", "application/json");
    	connectionToVADirect.setRequestMethod("GET");
		
		return connectionToVADirect;
	}
	
	/**
	 * Extracts the message status from the given JSON message <code>String</code>.
	 * @param jsonMessageString
	 * @return	An array of <code>int</code> values that are the status codes about a Direct Message. Returns
	 *			<code>null</code> if there was no 'message_status' element in the given JSON response message,
	 *			or if the given JSON message <code>String</code> is <code>null</code>.
	 * 			It is possible that the return value will be an empty array.
	 */
	public int[] parseJsonResponseForMessageStatus(String jsonMessageString) {
		
		if (jsonMessageString == null) {
			return null;
		}
		int[] outputArray = null;
		
		JsonParser jsonParser = new JsonParser();
		JsonElement jsonElement = jsonParser.parse(jsonMessageString);
		try{
			JsonElement messageStatusElement = jsonElement.getAsJsonObject().get("mail").getAsJsonObject().get("message_status");
			
			if (messageStatusElement != null)  {
				Set<Entry<String, JsonElement>> messageStatus = messageStatusElement.getAsJsonObject().entrySet();
				JsonArray valueArray = (JsonArray) messageStatus.iterator().next().getValue();
				
				if (valueArray != null) {
					outputArray = new int[valueArray.size()];
					
					for (int i=0; i<valueArray.size(); i++) {
						outputArray[i] = valueArray.get(i).getAsInt();
					}
				}
			}
		}catch(IllegalStateException e){
			if (logger.isErrorEnabled()) {
				logger.error(e);
			}
		}
		return outputArray;
	}
	
	
	protected Long getUserProfileIDFromSession() throws MHVException {
		Long userProfileID = null;
		try {
			PortletRequest request = (PortletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest();
			PortletSession session = request.getPortletSession();
			
			String patientStr =  (String)session.getAttribute(USERPROFILE_DTO_KEY, PortletSession.APPLICATION_SCOPE);
			UserProfileDTO userProfileDTO = mapper.readValue(patientStr, UserProfileDTO.class);
			userProfileID = userProfileDTO.getId();
		} catch(Exception e) {
			throw new MHVException("Unable to get UserProfileDTO from session");
		}
		
		return userProfileID;
	}

	
	protected UserTypeEnum getUserRoleFromSession() throws MHVException {
		UserTypeEnum userRole;
		PortletSession session = null;
		try {
			PortletRequest request = (PortletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest();
			session = request.getPortletSession();
			
			String userRoleStr = (String)session.getAttribute(USER_ROLE_KEY, PortletSession.APPLICATION_SCOPE);
			userRole = UserTypeEnum.valueOfByRole(userRoleStr);
			
		} catch(Exception e) {
			throw new MHVException("Unable to get user role from session");
		}
		return userRole;
	}
}
