package gov.va.med.mhv.bluebutton.web.controller;

import gov.va.med.mhv.bluebutton.StudyBusinessService;
import gov.va.med.mhv.bluebutton.StudyJobBusinessService;
import gov.va.med.mhv.bluebutton.transfer.BlueButtonReportDTO;
import gov.va.med.mhv.bluebutton.transfer.BlueButtonReportSelectionsDTO;
import gov.va.med.mhv.bluebutton.transfer.StudyDTO;
import gov.va.med.mhv.bluebutton.transfer.StudyJobDTO;
import gov.va.med.mhv.bluebutton.web.bean.BlueButtonReportBean;
import gov.va.med.mhv.bluebutton.web.bean.StudyListBean;
import gov.va.med.mhv.bluebutton.webservice.BlueButtonWebService;
import gov.va.med.mhv.common.api.enumeration.InformationSelectionTypeEnum;
import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.usermgmt.common.enums.ActivityActionTypeEnumeration;
import gov.va.med.mhv.usermgmt.common.enums.ActivityTypeEnumeration;
import gov.va.med.mhv.usermgmt.service.AccountActivityCreatorService;
import gov.va.med.mhv.usermgmt.util.activity.ActivityHelper;

import java.io.IOException;
import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.List;
import java.util.SimpleTimeZone;

import javax.annotation.Resource;
import javax.el.ValueExpression;
import javax.faces.application.FacesMessage;
import javax.faces.bean.ManagedBean;
import javax.faces.context.FacesContext;
import javax.faces.event.ComponentSystemEvent;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.primefaces.component.datatable.DataTable;
import org.primefaces.event.data.SortEvent;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

@ManagedBean(name = "imagesController")
@Component
@Scope("request")
public class ImagesController extends AbstractPhrController implements Serializable {
	private static final long serialVersionUID = 7201133542904862692L;

	private static Logger logger = LogManager.getLogger(ImagesController.class);

	private DataTable studyTable;
	protected ValueExpression sortColumn;
	protected String sortBy;
	protected int rowsPerPage;

	@Autowired
	private StudyListBean studies;

	@Autowired
	private BlueButtonReportBean reports;

	@Resource
	private StudyBusinessService studyBusinessServiceProxy;

	@Resource
	private StudyJobBusinessService studyJobBusinessServiceProxy;

	@Resource(name = "activityProxy")
	private AccountActivityCreatorService activityProxy;

	@Resource
	private BlueButtonWebService blueButtonWebserviceProxy;

	public void init(ComponentSystemEvent event) throws IOException {
		if (logger.isDebugEnabled()) {
			logger.debug("Initializing Refresh Status");
		}

		FacesContext context = FacesContext.getCurrentInstance();
		studyTable = (DataTable) FacesContext.getCurrentInstance()
				.getViewRoot()
				.findComponent("medicalImagesAndReports_refresh:study_table");

		if (!context.isPostback()) {
			setRowsPerPage(10);
		}
		findUser();
		checkAndRefresh();
		// if(studyTable.getSortColumn() != null && studyTable.getSortBy() != null){
		// studyTable.setValueExpression("sortBy", sortColumn);
		// studyTable.setSortOrder(studyTable.getSortBy());
		// }
		getStudyList();
		refreshStudyJobs();
	}

	public void refreshStatus() throws IOException {
		if (logger.isDebugEnabled()) {
			logger.debug("Checking Refresh Status");
		}
		checkAndRefresh();
	}

	public void getStudyList() {
		try {
			Long patientId = getPatientIdFromSession();
			if (logger.isDebugEnabled()) {
				logger.debug("Fetching Studies for patientId " + patientId);
			}

			if (!studies.getDataLoaded()) {
				// RECORD AAL WHEN THE INITIAL LIST IS LOADED
				activityProxy
						.createAccountActivityLog(ActivityHelper
								.createActivityDTOForSelf(
										getUserProfileIdFromSession(),
										true,
										ActivityTypeEnumeration.DOWNLOAD,
										ActivityActionTypeEnumeration.LIST_OF_VA_MEDICAL_IMAGES_AND_REPORTS_REQUESTED,
										""));
			}

			List<StudyDTO> studyList = studyBusinessServiceProxy
					.getStudies(patientId);
			studies.setStudies(studyList);
		} catch (MHVException e) {
			logger.error("Unable to get the list of studies", e);
			FacesContext.getCurrentInstance().addMessage(
					null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR,
							"Unable to get the list of studies",
							"Unable to get the list of studies"));
		}
		if (logger.isDebugEnabled()) {
			logger.debug("Returning " + studies.getStudies().size() + " studies");
		}

	}

	public String goBack() {
		return "selectReport";
	}

	public void refreshStudyJobs() {
		// Call the service to get the latest status list
		if (logger.isDebugEnabled()) {
			logger.debug("Refreshing StudyJobs");
		}

		try {
			Long patientId = getPatientIdFromSession();

			if (logger.isDebugEnabled()) {
				logger.debug("Fetching StudyJobs for patientId " + patientId);
			}
			List<StudyJobDTO> studyJobList = studyJobBusinessServiceProxy.getStudyJobs(patientId);
			if (logger.isDebugEnabled()) {
				logger.debug("Returning " + studyJobList.size() + " StudyJobs");
			}
			studies.setStudyJobList(studyJobList);
		} catch (MHVException e) {
			logger.error("Unable to refresh study list", e);
			FacesContext.getCurrentInstance().addMessage(
					null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR,
							"Unable to refresh the list of Studies",
							"Unable to refresh the list of Studies"));
		}
	}

	public boolean getAnyActiveStudyJobs() {
		return studies.getAnyInProgress();
	}

	public void requestStudy(StudyDTO study) {
		if (logger.isDebugEnabled()) {
			logger.debug("Requesting study: " + study.getStudyIdUrn());
		}
		if (study.getStudyJob() == null
				|| study.getStudyJob().getStatus().contains("ERROR")) {
			try {
				Long patientId = getPatientIdFromSession();
				String patientICN = getPatientICNFromSession();

				StudyJobDTO job = studyJobBusinessServiceProxy.createZipFile(
						patientId, patientICN, study.getStudyIdUrn());
				studies.setStudyJob(job);
			} catch (MHVException e) {
				logger.error("Unable to request study", e);
				FacesContext.getCurrentInstance().addMessage(
						null,
						new FacesMessage(FacesMessage.SEVERITY_ERROR,
								"Unable to request study", e
										.getFailureMessage()));
			}
		} else {
			logger.error("Trying to Request a study which is already in progress or new for study "
					+ study.getStudyIdUrn());
		}
	}

	public String showSelectedStudy(StudyDTO study) {
		studies.setSelectedStudy(study);
		try {
			Long patientId = getPatientIdFromSession();
			List<String> files = studyJobBusinessServiceProxy
					.getImagePreviewList(patientId, study.getStudyIdUrn());
			studies.setSelectedStudyFileList(files);

			if (logger.isDebugEnabled()) {
				logger.debug("Setting selected study: " + study.getStudyIdUrn());
			}
		} catch (MHVException e) {
			logger.error("Unable to retrieve study preview file", e);
			FacesContext
					.getCurrentInstance()
					.addMessage(
							null,
							new FacesMessage(
									FacesMessage.SEVERITY_ERROR,
									"Unable to retrieve study preview information",
									""));
		}

		try {
			SimpleDateFormat sdf = new SimpleDateFormat("dd MMM yyyy @ HHmm");
			String detail = study.getProcedureName() + " performed on "
					+ sdf.format(study.getPerformedDatePrecise());
			activityProxy
					.createAccountActivityLog(ActivityHelper
							.createActivityDTOForSelf(
									getUserProfileIdFromSession(),
									true,
									ActivityTypeEnumeration.DOWNLOAD,
									ActivityActionTypeEnumeration.VA_MEDICAL_IMAGES_AND_REPORT_VIEW,
									detail));
		} catch (MHVException e) {
			e.printStackTrace();
		}

		// reports.setRequestDate(new Date());
		SimpleDateFormat sdf = new SimpleDateFormat(
				"EEE, dd MMM yyyy HH:mm:ss z");
		BlueButtonReportSelectionsDTO selections = new BlueButtonReportSelectionsDTO();
		selections.setSelectionType(InformationSelectionTypeEnum.CUSTOM);
		selections.setDataClasses(new String[] { "varadiology" });

		Calendar cal = Calendar.getInstance(SimpleTimeZone.getDefault());
		cal.set(1900, 0, 1, 0, 0, 0);
		selections.setFromDate(sdf.format(cal.getTime())); // "Tue, 1 Aug 2015
															// 14:58:09 GMT
		selections.setToDate(sdf.format(cal.getTime()));
		try {
			// TODO: Connect to real report generator
			BlueButtonReportDTO report = blueButtonWebserviceProxy
					.generateReport(getUserProfileIdFromSession(), selections);
			studies.setBbDownloadReportId(report.getReportId());

			// TODO: Remove Testing:
			// studies.setBbDownloadReportId(new Long(13388307));
			// if(false) throw new MHVException();
			// BlueButtonReportDTO result = new BlueButtonReportDTO();

			SimpleDateFormat sdfDtFile = new SimpleDateFormat("ddMMMyyyy");
			String dtFile = sdfDtFile.format(study.getPerformedDatePrecise())
					.toUpperCase();

			String filename = "VA_IMG_RPT_"
					+ getLastName().toUpperCase()
					+ "_"
					+ (study.getProcedureName().replaceAll(" ", "_"))
							.replaceAll(",", "_") + "_" + dtFile + ".txt";

			report.setFilename(filename);
			report.setFilesize(3500L);
			reports.setReport(report);

		} catch (MHVException e) {
			e.printStackTrace();
		}

		return ("viewStudyTabs");
	}

	public String returnToViewStudy() {
		return ("viewStudyTabs");
	}

	public String returnToSummary() {
		return ("medicalImagesAndReports");
	}

	public String viewPrintPreviewForTXT() {

		try {
			SimpleDateFormat sdf = new SimpleDateFormat("dd MMM yyyy @ HHmm");
			String detail = studies.getSelectedStudy().getProcedureName()
					+ " performed on "
					+ sdf.format(studies.getSelectedStudy()
							.getPerformedDatePrecise());
			activityProxy
					.createAccountActivityLog(ActivityHelper
							.createActivityDTOForSelf(
									getUserProfileIdFromSession(),
									true,
									ActivityTypeEnumeration.DOWNLOAD,
									ActivityActionTypeEnumeration.VA_MEDICAL_IMAGES_AND_REPORT_VIEW_TEXT,
									detail));
		} catch (MHVException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		return ("viewPrintStudyReportTXT");
	}

	public String viewPrintPreviewForPDF() {

		try {
			SimpleDateFormat sdf = new SimpleDateFormat("dd MMM yyyy @ HHmm");
			String detail = studies.getSelectedStudy().getProcedureName()
					+ " performed on "
					+ sdf.format(studies.getSelectedStudy()
							.getPerformedDatePrecise());
			activityProxy
					.createAccountActivityLog(ActivityHelper
							.createActivityDTOForSelf(
									getUserProfileIdFromSession(),
									true,
									ActivityTypeEnumeration.DOWNLOAD,
									ActivityActionTypeEnumeration.VA_MEDICAL_IMAGES_AND_REPORT_VIEW_PDF,
									detail));
		} catch (MHVException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}

		return ("viewPrintStudyReportPDF");
	}

	public ValueExpression getSortColumn() {
		return sortColumn;
	}

	public void setSortColumn(ValueExpression sortColumn) {
		this.sortColumn = sortColumn;
	}

	public String getSortBy() {
		return sortBy;
	}

	public void setSortBy(String sortBy) {
		this.sortBy = sortBy;
	}

	public int getRowsPerPage() {
		return rowsPerPage;
	}

	public void setRowsPerPage(int rowsPerPage) {
		this.rowsPerPage = rowsPerPage;
	}

	public void onSort(SortEvent event) {
		sortColumn = event.getSortColumn().getValueExpression("sortBy");
		sortBy = event.isAscending() ? "ascending" : "descending";
	}
	
	public DataTable getStudyTable() {
		return studyTable;
	}

	public void setStudyTable(DataTable studyTable) {
		this.studyTable = studyTable;
	}

	public BlueButtonReportBean getReports() {
		return reports;
	}

	public void setReports(BlueButtonReportBean reports) {
		this.reports = reports;
	}
}
