package gov.va.med.mhv.bluebutton.web.controller;

import gov.va.med.mhv.bluebutton.web.bean.PhrStatusBean;
import gov.va.med.mhv.bluebutton.web.util.WebServiceClientUtil;
import gov.va.med.mhv.common.api.dto.UserProfileDTO;
import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.integration.phr.service.ServiceResponse;
import gov.va.med.mhv.integration.phr.transfer.FacilityExtractStatus;
import gov.va.med.mhv.integration.phr.transfer.PatientExtractStatus;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.Serializable;
import java.lang.reflect.Type;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.faces.application.FacesMessage;
import javax.faces.context.FacesContext;
import javax.portlet.PortletRequest;
import javax.portlet.PortletSession;
import javax.ws.rs.core.Response;

import org.apache.cxf.jaxrs.client.WebClient;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.google.gson.JsonDeserializationContext;
import com.google.gson.JsonDeserializer;
import com.google.gson.JsonElement;
import com.google.gson.JsonParseException;
import com.liferay.portal.kernel.util.PropsUtil;
import com.liferay.portal.model.User;
import com.liferay.portal.util.PortalUtil;

public class AbstractPhrController  implements Serializable {
	private static Logger logger = LogManager.getLogger(AbstractPhrController.class);
	private static final long serialVersionUID = 2641102847752789893L;
	private static final String CONTENT_TYPE = "application/json";
	
	private static final String USERPROFILE_DTO_KEY = "LIFERAY_SHARED_userprofiledto";
	private static final String PATIENT_ID_KEY = "LIFERAY_SHARED_patientid";
	private static final String PATIENT_ICN_KEY = "LIFERAY_SHARED_patienticn";
	private static final String USER_PROFILE_ID_KEY = "LIFERAY_SHARED_userprofileid";
	private static final String API_SESSION_TOKEN_KEY = "APISessionToken";

	private String firstName;
	private String lastName;
	private String middleName;

	static final String DATEONLY_FORMAT_STRING = "dd MMM yyyy";
	
	@Autowired
	protected PhrStatusBean status;
	
	@Autowired
	protected ObjectMapper mapper;

	@Value("${phrManagerendpointUrl}")
	private String phrManagerendpointUrl;

	@Value("${phrmgr.xauth.key}")
	private String authenticationHeaderValue;

	protected static final String authenticationHeader = "X-Authorization-Key";
		
	private static String GENERAL_ERROR = "Due to system issues, My HealtheVet is unable to refresh your data at this time.  Please try again later.";

	public String getMhvContext(){
		String context = PropsUtil.get("mhv.context");
		return  context;
	}

	public String getAuthenticationHeaderValue() {
		return authenticationHeaderValue;
	}

	public void setAuthenticationHeaderValue(String authenticationHeaderValue) {
		this.authenticationHeaderValue = authenticationHeaderValue;
	}

	public String getPhrManagerendpointUrl() {
		return phrManagerendpointUrl;
	}
	
	public void setPhrManagerendpointUrl(String phrManagerendpointUrl) {
		this.phrManagerendpointUrl = phrManagerendpointUrl;
	}
	
	private void refreshAllPhrData(String icn) throws MHVException {
		if(logger.isDebugEnabled()) {
			logger.debug("Submitting a PHR refresh request for " + icn);
		}
		
		WebClient client = getPHRManagerWebClient().path("refresh")
				.path(icn).accept(CONTENT_TYPE)
				.type(CONTENT_TYPE)
				.header(authenticationHeader, getAuthenticationHeaderValue());
		Response webserviceResponse = client.post(null);
		if (webserviceResponse.getStatus() == 200) {
			if(logger.isDebugEnabled()) {
				logger.debug("Response from PHR refresh service: ", webserviceResponse.getEntity().toString());				
			}
			Reader responseReader = new InputStreamReader((InputStream) webserviceResponse.getEntity());
			try {
				ServiceResponse r = mapper.readValue(responseReader, ServiceResponse.class);
				if(logger.isDebugEnabled()) {
					logger.debug("Refresh has been submitted with status of "  + r.getStatus());
				}
			} catch (Exception e) {
				logger.error("Unable to process the results from a PHR Refresh request",e);
				throw new MHVException("Unable to determine the refresh status");
			}
		} else {
			logger.error("Unable to submit the PHR Refresh to the webservice");
			throw new MHVException("Unable to submit a request to refresh");
		}
	}

	private PatientExtractStatus getPatientExtractStatus(String icn) throws MHVException {
		PatientExtractStatus patientExtractStatus = null;
		
		if (icn != null) {
			GsonBuilder builder = new GsonBuilder();
			builder.registerTypeAdapter(Date.class,
					new JsonDeserializer<Date>() {
						@Override
						public Date deserialize(JsonElement json, Type typeOfT, JsonDeserializationContext context) throws JsonParseException {
							return new Date(json.getAsJsonPrimitive().getAsLong());
						}
					});
			Gson gson = builder.create();
			WebClient client = getPHRManagerWebClient()
					.path("status")
					.path(icn)
					.accept(CONTENT_TYPE)
					.header(authenticationHeader, getAuthenticationHeaderValue());
			Response webserviceResponse = client.get();

			if (webserviceResponse.getStatus() == 200) {
				Reader responseReader = new InputStreamReader((InputStream) webserviceResponse.getEntity());
				patientExtractStatus = gson.fromJson(responseReader, PatientExtractStatus.class);
				if(logger.isDebugEnabled()) {
					logger.debug("Retrieved PHR status: " + patientExtractStatus);
				}
			} else {
				logger.error("Error invoking PHR status service: " + WebServiceClientUtil.formatErrorMessage(client, webserviceResponse));
				throw new MHVException("Error resulted from attempting to get a phr refresh status");
			}
		}
		return patientExtractStatus;
	}
	
	private WebClient getPHRManagerWebClient() {
		if(logger.isDebugEnabled()) {
			logger.debug("PHR Manager endpoint: " + phrManagerendpointUrl);
		}
		return WebClient.create(getPhrManagerendpointUrl());
	}

	protected void checkAndRefresh() {
		try {
			String icn = getPatientICNFromSession();
			
			//Load if the refresh status has not been loaded yet for this session
			if( !status.getIsPatientExtractStatusLoaded() ) {
				PatientExtractStatus patientExtractStatus = getPatientExtractStatus(icn);
				status.setPatientExtractStatus(patientExtractStatus);
			}
			
			//If the status is not complete go back and get an updated version
			if( !status.getIsRefreshComplete() ) {

				PatientExtractStatus patientExtractStatus = getPatientExtractStatus(icn);
				status.setPatientExtractStatus(patientExtractStatus);

				int hits = 0;
				for( FacilityExtractStatus s: patientExtractStatus.getFacilityExtractStatusList() ) { 
					if( s.inProgress() || isSameDate(s.getLastRequested(), new Date()) ) {
						//If any are in progress or were requested today the refresh is going
						hits++;
					}
				}

				if(logger.isDebugEnabled()) {
					logger.debug("Found " + hits + " extracts already requested today");
				}
				
				if( hits >= patientExtractStatus.getFacilityExtractStatusList().size() / 2 ) {
					status.setIsRefreshNeeded(false);
				}
				
				//If refresh has not been submitted submit a new request
				 if( status.getIsRefreshNeeded() ) {
					if(logger.isDebugEnabled()) {
						logger.debug("Refreshing " + icn + " PHR Data");
					}
					
					//Start the refresh
					refreshAllPhrData(icn);
					
				 } else {
					 if(logger.isDebugEnabled()) {
							logger.debug("No Refresh was needed for " + icn + " PHR Data");
						}
				 }
			} else {
				if(logger.isDebugEnabled()) {
					logger.debug("PHR Data Refresh is Complete for user " + icn);
				}
			}
			
		} catch(Exception e) {
			logger.error("Error in PHR Refresh", e);
			FacesContext.getCurrentInstance().addMessage(null, new FacesMessage(FacesMessage.SEVERITY_ERROR, GENERAL_ERROR, ""));
		}
 	}
	
	protected UserProfileDTO getUserProfileDTOFromSession() throws MHVException {
		UserProfileDTO userProfile;
		PortletSession session = null;
		try {
			PortletRequest request = (PortletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest();
			session = request.getPortletSession();
			
			String patientStr =  (String)session.getAttribute(USERPROFILE_DTO_KEY, PortletSession.APPLICATION_SCOPE);
			userProfile = mapper.readValue(patientStr, UserProfileDTO.class);
			
		} catch(Exception e) {
			throw new MHVException("Unable to get UserProfileDTO from session");
		}
		return userProfile;
	}
	
	protected Long getPatientIdFromSession() throws MHVException {
		Long patientId=0l;
		PortletSession session = null;
		try {
			PortletRequest request = (PortletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest();
			session = request.getPortletSession();
			patientId = (Long)session.getAttribute(PATIENT_ID_KEY, PortletSession.APPLICATION_SCOPE);
		} catch(Exception e) {
			throw new MHVException("Unable to get Patient Id from session");
		}
		return patientId;
	}
	
	protected Long getUserProfileIdFromSession() throws MHVException {
		Long userProfileId=0l;
		PortletSession session = null;
		try {
			PortletRequest request = (PortletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest();
			session = request.getPortletSession();
			userProfileId = (Long)session.getAttribute(USER_PROFILE_ID_KEY, PortletSession.APPLICATION_SCOPE);
		} catch(Exception e) {
			throw new MHVException("Unable to get UserProfile Id from session");
		}
		return userProfileId;
	}
	
	protected String getPatientICNFromSession() throws MHVException {
		String patientICN;
		PortletSession session = null;
		try {
			PortletRequest request = (PortletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest();
			session = request.getPortletSession();
			patientICN = (String)session.getAttribute(PATIENT_ICN_KEY, PortletSession.APPLICATION_SCOPE);
		} catch(Exception e) {
			throw new MHVException("Unable to get Patient ICN from session");
		}
		return patientICN;
	}

	public String getAPIToken() {
		PortletRequest request = (PortletRequest) FacesContext.getCurrentInstance().getExternalContext().getRequest();
		PortletSession session = request.getPortletSession();
		String token = (String)session.getAttribute(API_SESSION_TOKEN_KEY);
		
		return token;
	}
	
	public void findUser() {
		PortletRequest   request = (PortletRequest)FacesContext.getCurrentInstance().getExternalContext().getRequest();
		User user=null;
        try{
        	user = PortalUtil.getUser(request);
        }catch(Exception e){
        	
        }
		if(user != null){
			setFirstName(user.getFirstName());
			setLastName(user.getLastName());
			if(user.getMiddleName() != null) {
				setMiddleName(user.getMiddleName());
			} else {
				setMiddleName("");
			}
		}
	}
	
	public String getFirstName() {
		if(firstName ==  null) {
			findUser();
		}
		return firstName;
	}

	public void setFirstName(String firstName) {
		this.firstName = firstName;
	}

	public String getMiddleName() {
		if(middleName ==  null) {
			findUser();
		}
		return middleName;
	}

	public void setMiddleName(String middleName) {
		this.middleName = middleName;
	}
	
	public String getLastName() {
		if(lastName ==  null) {
			findUser();
		}
		return lastName;
	}

	public void setLastName(String lastName) {
		this.lastName = lastName;
	}
	
	private boolean isSameDate(Date date1, Date date2) {
        boolean result = false;
        try {
            SimpleDateFormat sdf = new SimpleDateFormat(DATEONLY_FORMAT_STRING);
            String temp1 = sdf.format(date1);
            String temp2 = sdf.format(date2);
            result = temp1.equals(temp2);
        } catch (Exception ignore) {
            //System.out.println(ignore.getMessage());
        }
        return result;
	 }

}

