package gov.va.med.mhv.bluebutton.service.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import javax.annotation.Resource;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.bluebutton.DownloadYourReportService;
import gov.va.med.mhv.bluebutton.StudyJobBusinessService;
import gov.va.med.mhv.bluebutton.bbdownload.util.PdfXHTMLWrapper;
import gov.va.med.mhv.bluebutton.bbmi.BbmiMdbSender;
import gov.va.med.mhv.bluebutton.bbmi.imaging.util.DecryptZipOutputStream;
import gov.va.med.mhv.bluebutton.bbmi.imaging.util.ImagingOutputStream;
import gov.va.med.mhv.bluebutton.bbmi.imaging.util.RadiologyReportHelper;
import gov.va.med.mhv.bluebutton.bbmi.imaging.util.StudyJobConstants;
import gov.va.med.mhv.bluebutton.converter.StudyJobConverter;
import gov.va.med.mhv.bluebutton.converter.StudyJobListConverter;
import gov.va.med.mhv.bluebutton.model.Study;
import gov.va.med.mhv.bluebutton.model.StudyJob;
import gov.va.med.mhv.bluebutton.repository.StudyJobRepository;
import gov.va.med.mhv.bluebutton.repository.StudyRepository;
import gov.va.med.mhv.bluebutton.transfer.BbmiEventMessage;
import gov.va.med.mhv.bluebutton.transfer.StudyJobDTO;
import gov.va.med.mhv.common.api.cache.CacheHandler;
import gov.va.med.mhv.common.api.dto.PatientDTO;
import gov.va.med.mhv.common.api.enumeration.ReportTypeEnum;
import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.common.api.transfer.Session;
import gov.va.med.mhv.core.crypto.MHVCipher;
import gov.va.med.mhv.usermgmt.common.enums.ActivityActionTypeEnumeration;
import gov.va.med.mhv.usermgmt.common.enums.ActivityTypeEnumeration;
import gov.va.med.mhv.usermgmt.service.AccountActivityCreatorService;
import gov.va.med.mhv.usermgmt.service.PatientWebService;
import gov.va.med.mhv.usermgmt.util.activity.ActivityHelper;

@Component
public class StudyJobBusinessServiceImpl implements StudyJobBusinessService {
	private static final Log log = LogFactory.getLog(StudyJobBusinessServiceImpl.class);

	@Resource
	private StudyJobRepository studyJobRepository;
	
	@Resource
	private StudyRepository studyRespository;
	
	@Autowired
	private BbmiMdbSender jmsBBMIStudySender;
	
	@Autowired
	private StudyJobListConverter studyJobListConverter;
	
	@Resource(name = "activityProxy")
	private AccountActivityCreatorService activityProxy; 
	
	@Autowired
	private StudyJobConverter studyJobConverter;
	
	@Resource
	private PatientWebService patientProxy;
	
	@Autowired
	private DownloadYourReportService bbProxy;

	@Value("${folder}")
	private String FOLDER_PROP;
	
	public StudyJobDTO createZipFile(Long patientId, String icn, String studyId) throws MHVException {
		StudyJob job = null;
		//Patient patient = new Patient();
		
		PatientDTO patDto = patientProxy.getPatientByPatientId(patientId);
		// CHECK IF THE USER HAS EXCEEDED THE CONCURRENT JOBS (3 INITIALLY)
		long count1 = studyJobRepository.getPhrStudyJobCountByStatus(patientId, StudyJobConstants.PROCESSING);
		long count2 = studyJobRepository.getPhrStudyJobCountByStatus(patientId, StudyJobConstants.NEW);
		if( (count1 + count2) >= StudyJobConstants.MAX_CONCURRENT_REQUESTS ) {
			throw new MHVException( "You have exceeded your limit of three (3) imaging report requests at one time.  Please allow one of your current requests to finish processing before requesting another report.");
		}
		
		// FIRST CHECK IF THE JOB IS ALREADY CREATED
		List<StudyJob> jobList = studyJobRepository.getPhrStudyJob(patientId, studyId);
		if(jobList.size() > 0)
			job = (StudyJob) jobList.get(0); 
		
		// CHECK IF THE JOB WAS CANCELLED THEN RE-QUEUE IT; IF IT IS ERRORED THEN ALLOW A RETRY
		if (job == null) {
			studyJobRepository.save(createStudyJob(patientId, icn, studyId ));
			List<StudyJob> jobL = studyJobRepository.getPhrStudyJob(patientId, studyId);
			if(jobL.size() > 0) {
				job = (StudyJob) jobL.get(0);
			}
			
			try {
				//Create Message OBJ....
				BbmiEventMessage bbmiEventMesg=null;
				if(job != null){
				  bbmiEventMesg = new BbmiEventMessage(patientId, studyId, icn, job.getId() );
				}
				jmsBBMIStudySender.sendMessage(bbmiEventMesg);
				
			} catch (Exception e) {
				log.error("exception occurred" + e);
				//RECORD AAL ENTRY FOR REQUEST FAILURE
				List<Study> studies = studyRespository.findByPatientIdAndStudyIdUrn(patientId, job.getStudyIdUrn());
				Study study = studies.get(0);
				SimpleDateFormat sdf = new SimpleDateFormat("dd MMM yyyy @ HHmm");
				String detail = study.getProcedureName() + " performed on " + sdf.format(study.getPerformedDate().getDatePrecise()); 
				activityProxy.createAccountActivityLog(ActivityHelper.createActivityDTOForSelf(patDto.getUserProfileId(), false, ActivityTypeEnumeration.DOWNLOAD, ActivityActionTypeEnumeration.VA_MEDICAL_IMAGES_AND_REPORT_REQUEST_SUBMITTED, detail));

				throw new MHVException("Unable to submit job for study " + job.getStudyIdUrn());
			}
		}
		else if (job != null) {
			if (job.getStatus().equals(StudyJobConstants.ERROR)) {
				try {
					job.setStatus(StudyJobConstants.NEW);
					job.setCreatedDate(new Date()); // reset the created date
					job.setTempFileName("");
					job.setFileName("");
					job.setServerNode(System.getProperty("weblogic.Name","None"));
					studyJobRepository.save(job);
					BbmiEventMessage bbmiEventMesg = new BbmiEventMessage(patientId, studyId, icn, job.getId() );
					jmsBBMIStudySender.sendMessage(bbmiEventMesg);
					
					List<Study> studies = studyRespository.findByPatientIdAndStudyIdUrn(patientId, job.getStudyIdUrn());
					Study study = studies.get(0);
					SimpleDateFormat sdf = new SimpleDateFormat("dd MMM yyyy @ HHmm");
					String detail = study.getProcedureName() + " performed on " + sdf.format(study.getPerformedDate().getDatePrecise()); 
					activityProxy.createAccountActivityLog(ActivityHelper.createActivityDTOForSelf(patDto.getUserProfileId(), false, ActivityTypeEnumeration.DOWNLOAD, ActivityActionTypeEnumeration.VA_MEDICAL_IMAGES_AND_REPORT_RETRY_REQUEST_SUBMITTED, detail));
					
				} catch (Exception e1) {
					log.error("exception occurred" + e1);
					throw new MHVException("Unable to submit job for study " + job.getStudyIdUrn());
				}
			} else {
				return studyJobConverter.convert(job);
			}
		}
		
		//RECORD AAL ENTRY FOR REQUEST SUCCESSFUL
		String studyUrn=null;
		if(job != null){
		   studyUrn = job.getStudyIdUrn();
		}
		List<Study> studies = studyRespository.findByPatientIdAndStudyIdUrn(patientId, studyUrn);
		Study study = studies.get(0);
		SimpleDateFormat sdf = new SimpleDateFormat("dd MMM yyyy @ HHmm");
		String detail = study.getProcedureName() + " performed on " + sdf.format(study.getPerformedDate().getDatePrecise()); 
		activityProxy.createAccountActivityLog(ActivityHelper.createActivityDTOForSelf(patDto.getUserProfileId(), true, ActivityTypeEnumeration.DOWNLOAD, ActivityActionTypeEnumeration.VA_MEDICAL_IMAGES_AND_REPORT_REQUEST_SUBMITTED, detail));

		return studyJobConverter.convert(job);
	}

	public Response getZipFileAsStream(Long patientId, String studyIdUrn) {
		StudyJob job;
		try {
			PatientDTO patDto = patientProxy.getPatientByPatientId(patientId);
			List<StudyJob> jobs = studyJobRepository.getPhrStudyJob(patientId, studyIdUrn);
			job = jobs.get(0);  		
			
			if (job == null) {
				//RECORD AAL ENTRY FOR DOWNLOAD REQUESTED
				List<Study> studies = studyRespository.findByPatientIdAndStudyIdUrn(patientId, studyIdUrn);
				Study study = studies.get(0);
				SimpleDateFormat sdf = new SimpleDateFormat("dd MMM yyyy @ HHmm");
				String detail = study.getProcedureName() + " performed on " + sdf.format(study.getPerformedDate().getDatePrecise()); 
				activityProxy.createAccountActivityLog(ActivityHelper.createActivityDTOForSelf(patDto.getUserProfileId(), false, ActivityTypeEnumeration.DOWNLOAD, ActivityActionTypeEnumeration.VA_MEDICAL_IMAGES_AND_REPORT_REQUEST_PROCESSED, detail));

				return Response.status(400).entity("No record found.").build();
			}
			else {
				List<Study> studies = studyRespository.findByPatientIdAndStudyIdUrn(patDto.getId(), job.getStudyIdUrn());
				Study study = studies.get(0);
				SimpleDateFormat sdf = new SimpleDateFormat("dd MMM yyyy @ HHmm");
				String detail = study.getProcedureName() + " performed on " + sdf.format(study.getPerformedDate().getDatePrecise());

				try {
					activityProxy.createAccountActivityLog(ActivityHelper.createActivityDTOForSelf(patDto.getUserProfileId(), true, ActivityTypeEnumeration.DOWNLOAD, ActivityActionTypeEnumeration.VA_MEDICAL_IMAGES_AND_REPORT_REQUEST_PROCESSED, detail));
				} catch (MHVException e) {
					log.error("An error occurred while getting the zipped study as a stream object.", e);
					throw e;
				}
			}
		} catch (Exception e1) {
			return Response.status(400).build();
		}

		DecryptZipOutputStream zipstream = new DecryptZipOutputStream(job){
			@Override
			public void write(final OutputStream os) throws IOException, FileNotFoundException {
				// BYTES READ from in will be DECRYPTED - FROM MOUNTED FOLDER FOR ALL MANAGED NODES
				if (log.isInfoEnabled()) {
					log.info("Reading " + getPhrStudyJob());
				}

				MHVCipher c = MHVCipher.createCipher(StudyJobConstants.PASSPHRASE, false);
				FileInputStream fis = new FileInputStream( FOLDER_PROP + "/" + getPhrStudyJob().getServerNode() +  "/" + getPhrStudyJob().getTempFileName());
				try {
					c.decrypt(fis, os, new byte[4096]);
				}finally{
					  if(fis != null){
					         fis.close();
					  }
				}	
				List<Study> studies = studyRespository.findByPatientIdAndStudyIdUrn(getPhrStudyJob().getPatientId(), getPhrStudyJob().getStudyIdUrn());
				Study study = studies.get(0);
				SimpleDateFormat sdf = new SimpleDateFormat("dd MMM yyyy @ HHmm");
				String detail = study.getProcedureName() + " performed on " + sdf.format(study.getPerformedDate().getDatePrecise());

				try {
					PatientDTO patDto = patientProxy.getPatientByPatientId(getPhrStudyJob().getPatientId());
					activityProxy.createAccountActivityLog(ActivityHelper.createActivityDTOForSelf(patDto.getUserProfileId(), true, ActivityTypeEnumeration.DOWNLOAD, ActivityActionTypeEnumeration.VA_MEDICAL_IMAGES_AND_REPORT_REQUEST_PROCESSED, detail));
				} catch (MHVException e) {
					e.printStackTrace();
				}
			}
		};
		return Response.ok(zipstream).header("Content-Type","application/zip").header("Content-Length", job.getFileSize()).header("Content-Disposition",	"attachment; filename=" + job.getFileName()).build();
	}

	// Parameter: images are 1 based
	public Response getImageAsStream(Long patientId, String studyIdUrn, Long series, Long image) {
		StudyJob job;
		try {

			List<StudyJob> jobs = studyJobRepository.getPhrStudyJob(patientId, studyIdUrn);
			job = jobs.get(0);
			
			if (job == null) {
				//RECORD AAL ERROR?
				return Response.status(400).entity("No record found.").build();
			}
		} catch (Exception e1) {
			return Response.status(400).build();
		}
        
		ImagingOutputStream imageStream = new ImagingOutputStream(job, series, image) {
			@Override
			public void write(OutputStream os) throws IOException, WebApplicationException {
				ByteArrayOutputStream baos = new ByteArrayOutputStream();
				MHVCipher cipher = MHVCipher.createCipher(StudyJobConstants.PASSPHRASE, false);
				FileInputStream fis = new FileInputStream(FOLDER_PROP + "/" + getPhrStudyJob().getServerNode() +  "/" + getPhrStudyJob().getTempFileName() + "_preview");
				try {
					cipher.decrypt(fis, baos, new byte[4096]);
					ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
					ZipInputStream zin = new ZipInputStream(bais);
	
					ZipEntry ze = null;
					
					String filename = "Series " + (getSeries()<10?"0":"")+getSeries() + " - Image " + (getImage()<10?"0":"")+getImage();
					
			        while ((ze = zin.getNextEntry()) != null) {
			          if(log.isDebugEnabled()) {
			        	log.debug("Found File " + ze.getName() + " looking up image " + getImage());  
			          }
			          if( ze.getName().contains(filename)) {
				          for (int c = zin.read(); c != -1; c = zin.read()) {
				            os.write(c);
				          }
				          zin.closeEntry();
				          os.close();
				          break;
			          }
			        }
				} finally{
					  if(fis != null){
					         fis.close();
					      }
				}
			}
		};
		// The mime type "image/*" is to generically reference any type of image that might be in the response 
		return Response.ok(imageStream).header("Content-Type","image/*").build();
	}
	
	public List<String> getImagePreviewList(Long patientId, String studyIdUrn) throws MHVException {
		List<String> files = new ArrayList<String>();
		
		StudyJob job;
		try {

			List<StudyJob> jobs = studyJobRepository.getPhrStudyJob(patientId, studyIdUrn);
			job = jobs.get(0);
			
			if (job == null) {
				throw new  MHVException("No record found.");
			}
		} catch (Exception e1) {
			throw new MHVException(e1);
		}
        
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		MHVCipher cipher = MHVCipher.createCipher(StudyJobConstants.PASSPHRASE, false);
		FileInputStream fis = null;
		try {
			fis = new FileInputStream(FOLDER_PROP + "/" + job.getServerNode() +  "/" + job.getTempFileName() + "_preview");
			cipher.decrypt(fis, baos, new byte[4096]);			
		
			ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
			ZipInputStream zin = new ZipInputStream(bais);
	
			ZipEntry ze = null;
			
			while ((ze = zin.getNextEntry()) != null) {
				//Only include images
				if(ze.getName().contains("Image")) {
					files.add(ze.getName());
				}
			}
		} catch (FileNotFoundException e1) {
			throw new MHVException(e1);
		} catch (IOException e) {
			throw new MHVException(e);
		} finally{
			if(fis != null){
				try {
					fis.close();
				} catch (IOException e) {
				}
			}
		}	
        
        return files;
	}
	
	public Response getZipFileAsStreamNoPat(String studyIdUrn) throws MHVException {
		Session session =  CacheHandler.getInstance().getSession();
		PatientDTO patDto = patientProxy.getPatientByUserProfileId(session.getUserId());
		return getZipFileAsStream(patDto.getId(), studyIdUrn);
	}
	
	public List<StudyJobDTO> getStudyJobs(Long patientId) {
		List<StudyJob> phrStudyJobList = new ArrayList<StudyJob>();
		List<StudyJobDTO> phrStudyJobDTOList = new ArrayList<StudyJobDTO>();
		try {
			// RETRIEVE ALL THE JOBS FOR THE USER
			phrStudyJobList = studyJobRepository.getPhrStudyJobs(patientId);
			phrStudyJobDTOList = studyJobListConverter.convert(phrStudyJobList);

		} catch (Exception e) {
			throw new WebApplicationException(e, 400);
		}

		return phrStudyJobDTOList;
	}
	
	private StudyJob createStudyJob(Long patientId, String icn, String studyId) {
		StudyJob studyJob = new StudyJob();
		studyJob.setPatientId(patientId);
		studyJob.setStudyIdUrn(studyId);
		studyJob.setStatus(StudyJobConstants.NEW);
		studyJob.setStatusText("0");
		studyJob.setCreatedDate(new Date());
		studyJob.setServerNode(System.getProperty("weblogic.Name","None"));
		return studyJob;
	}	

	public Response getImageNoPat(String studyIdUrn, Long series, Long image) throws MHVException {
		Session session =  CacheHandler.getInstance().getSession();
		PatientDTO patDto = patientProxy.getPatientByUserProfileId(session.getUserId());
		
		return getImageAsStream(patDto.getId(), studyIdUrn, series, image);
	}

	private String getReportAsString(StudyJob job, Long patientId, String studyIdUrn) throws Exception {
        
		ByteArrayOutputStream baos = new ByteArrayOutputStream();
		MHVCipher cipher = MHVCipher.createCipher(StudyJobConstants.PASSPHRASE, false);
		FileInputStream fis = new FileInputStream(FOLDER_PROP + "/" + job.getServerNode() +  "/" + job.getTempFileName() + "_preview");
		
		try {
			cipher.decrypt(fis, baos, new byte[4096]);
			ByteArrayInputStream bais = new ByteArrayInputStream(baos.toByteArray());
			ZipInputStream zin = new ZipInputStream(bais);

			ByteArrayOutputStream report = new ByteArrayOutputStream();
			
			ZipEntry ze = null;
	        while ((ze = zin.getNextEntry()) != null) {
	          if( ze.getName().equals("Report.txt") ) {
		          for (int c = zin.read(); c != -1; c = zin.read()) {
		        	  report.write(c);
		          }
		          zin.closeEntry();
		          report.close();
		          break;
	          }
	        }
	        return report.toString();
		} finally{
	    	   if(fis != null){
	              	fis.close();
	              }
	   		}
		
	}
	
    private Response getReportAsStream(Long patientId, String studyIdUrn) throws Exception {
    	StudyJob job;
		try {

			List<StudyJob> jobs = studyJobRepository.getPhrStudyJob(patientId, studyIdUrn);
			job = jobs.get(0);
			
			if (job == null) {
				//RECORD AAL ERROR?
				return Response.status(400).entity("No record found.").build();
			}
		} catch (Exception e1) {
			return Response.status(400).build();
		}
		
		// The mime type "image/*" is to generically reference any type of image that might be in the response 
		return Response.ok(getReportAsString(job, patientId, studyIdUrn)).header("Content-Type","text/plain").build();
	}
	
    private Response getBBReportAsStream(Long patientId, String studyIdUrn, Long downloadReportId, Study study, ReportTypeEnum type) throws Exception {
    	StudyJob job;
		try {

			List<StudyJob> jobs = studyJobRepository.getPhrStudyJob(patientId, studyIdUrn);
			job = jobs.get(0);
			
			if (job == null) {
				//RECORD AAL ERROR?
				return Response.status(400).entity("No record found.").build();
			}
			
			if( study == null ) {
				return Response.status(400).entity("No study found.").build();
			}
		} catch (Exception e1) {
			return Response.status(400).build();
		}
		String report = getReportAsString(job, patientId, studyIdUrn);
		
		PatientDTO patDto = patientProxy.getPatientByPatientId(patientId);
		ReportTypeEnum dwnldType = null;
		boolean generateBBReport = false;
		switch(type) {
			case PDF:
			case PDFDL: 
				dwnldType = ReportTypeEnum.XHTML;
				generateBBReport = true;
				break;
			case TXT:
			case TXTDL:
				dwnldType = ReportTypeEnum.TXT;
				generateBBReport = true;
				break;
			case VIEWTXT:
				generateBBReport = false;
				break;
		}
		
	   	String bbreport = null;
	   	byte[] result = null;
	   	String contentType = null;
	   	String fileName = null;
		String name = "Unknown";
		String xhtml = null;
		PdfXHTMLWrapper pdf = new PdfXHTMLWrapper();

	   	if( generateBBReport ) {
	   		bbreport = (String)bbProxy.generateDownload(patDto.getUserProfileId(), downloadReportId, dwnldType).getEntity();
	   	}
		
	   	switch(type) {
	   		case VIEWTXT:
	   			result = convertVIEWTXT(report, study).getBytes();
	   			contentType = "text/plain";
	   			break;
	   		case TXTDL:
	   			String txtReport = convertTXT(bbreport, report, study);
	   			result = txtReport.getBytes();
	   			String fileNameRegExTXT = "File Name:                 (VA_IMG_RPT_[\\W\\w\\d\\s]+.txt)";
	   			Pattern filePtrnTXT = Pattern.compile(fileNameRegExTXT);
	   			Matcher m1 = filePtrnTXT.matcher(txtReport);
	   			if( m1.find() ) {
	   				fileName = m1.group(1);
	   			} else {
	   				fileName = "Unknown";
	   			}
	   			contentType = "text/plain";
	   			break;
	   		case TXT:
	   			result = convertTXT(bbreport, report, study).getBytes();
	   			contentType = "text/plain";
	   			break;
	   		case PDFDL:
	   			xhtml = convertXHTML(bbreport, report, study);
	   			Pattern filePtrnPDF = Pattern.compile("<td class=\"left_2col_label\">File Name:</td>\\s*<td class=\"right_2col_value\">(VA_IMG_RPT_[\\W\\w\\d\\s]+.pdf)</td>");
	   			Matcher m2 = filePtrnPDF.matcher(xhtml);
	   			if( m2.find() ) {
	   				fileName = m2.group(1);
	   				fileName = fileName.replaceAll("&amp;", "&");
	   			} else {
	   				fileName = "Unknown";
	   			}
	   			
	   			Pattern namePtrn1 = Pattern.compile("<td class=\"heading_bold\" style=\"text-align:left\"><b>Name: ([\\w\\d\\s,]+)</b></td>");
	   			Matcher m3 = namePtrn1.matcher(xhtml);
	   			if( m3.find() ) {
	   				name = m3.group(1);
	   			}
	   			
				result = pdf.makePdfFromXhtml(name, xhtml);
				contentType = "application/pdf";
				break;
	   		case PDF: 
	   			xhtml = convertXHTML(bbreport, report, study);
	   			
	   			Pattern namePtrn2 = Pattern.compile("<td class=\"heading_bold\" style=\"text-align:left\"><b>Name: ([\\w\\d\\s,]+)</b></td>");
	   			Matcher m = namePtrn2.matcher(xhtml);
	   			if( m.find() ) {
	   				name = m.group(1);
	   			}
	   			
				result = pdf.makePdfFromXhtml(name, xhtml);
				contentType = "application/pdf";
	   			break;
	   		default: 
	   			result = "Unknown Type".getBytes();
	   			contentType="text/plain";
	   	}
	   	
		if( fileName != null ) {
			return Response.ok(result).header("Content-Type",contentType).header("Content-Length", result.length).header("Content-Disposition",	"attachment; filename=" + fileName).build();
		} else {
			return Response.ok(result).header("Content-Type",contentType).build();
		}
	}
    
    private static String convertVIEWTXT(String report, Study study) {
    	//No template to manipulate
    	
    	return RadiologyReportHelper.generateTXTViewReport(report, study);
    }

    private static String convertTXT(String bbreport, String report, Study study) {
 
		String producedRegEx = "[ ]{31}+([0-9]{1,2}) ([A-Za-z]{3}) ([0-9]{4}) @ ([0-9]{4})";
		String requestedRegEx = "System Request Date/Time:  ([0-9]{1,2}) ([A-Za-z]{3}) ([0-9]{4}) @ ([0-9]{4})";
		String dateRangeRegEx = "Date Range Selected:       ([0-9]{1,2}) ([A-Za-z]{3}) ([0-9]{4}) to ([0-9]{1,2}) ([A-Za-z]{3}) ([0-9]{4})";
		String fileNameRegEx = "File Name:                 mhv_([\\w\\d\\s]+)_[0-9]{8}_[0-9]{4}.txt";
		String lastUpdatedRegEx = "Last Updated: ([0-9]{1,2}) ([A-Za-z]{3}) ([0-9]{4}) @ ([0-9]{4})";
		String noRadiologies=
				"No information was available that matched your selection.  However, if \\s+"+
				"you recently had a VA radiology test or procedure, your reports may be \\s+"+
				"available 3 calendar days after they have been completed.  Some studies \\s+"+
				"done at a non-VA facility may not be available or they may not \\s+"+
				"necessarily include an interpretation. ";
		
		Date current = new Date();
		SimpleDateFormat sdfDtTm = new SimpleDateFormat("dd MMM yyyy @ HHmm");
		String dtTm = sdfDtTm.format(current);
		SimpleDateFormat sdfDtFile = new SimpleDateFormat("ddMMMyyyy");
		String dtFile = sdfDtFile.format(study.getPerformedDate().getDatePrecise()).toUpperCase();
		
		bbreport = bbreport.replaceFirst(producedRegEx, "                               "+dtTm);
		bbreport = bbreport.replaceFirst(requestedRegEx, "System Request Date/Time:  "+dtTm);
		bbreport = bbreport.replaceFirst(dateRangeRegEx,"Date Range Selected:       --");
		bbreport = bbreport.replaceFirst(lastUpdatedRegEx,"Last Updated: --");
		
		String lastName = "Unknown";
		Pattern filePtrn = Pattern.compile(fileNameRegEx);
		Matcher m = filePtrn.matcher(bbreport);
		if( m.find() ) {
			lastName = m.group(1);
		}
		//VA_IMG_RPT_{lastname}_{studyname}_DDMMMyyyy }.txt
		bbreport = bbreport.replaceFirst(fileNameRegEx,"File Name:                 VA_IMG_RPT_"
				+lastName+"_"+ (study.getProcedureName().replaceAll(" ", "_")).replaceAll(",", "_")
				+"_"+dtFile+".txt");
		
		//Parse the report
		bbreport = bbreport.replaceFirst(noRadiologies, RadiologyReportHelper.generateTXTReport(report,study));
		
		return bbreport;
    }
    
    private static String convertXHTML(String bbreport, String report, Study study) {
    	 
		Date current = new Date();
		SimpleDateFormat sdfDtTm = new SimpleDateFormat("dd MMM yyyy @ HHmm");
		String dtTm = sdfDtTm.format(current);
		SimpleDateFormat sdfDtFile = new SimpleDateFormat("ddMMMyyyy");
		String dtFile = sdfDtFile.format(study.getPerformedDate().getDatePrecise()).toUpperCase();
		
		bbreport = bbreport.replaceFirst("<td class=\"heading_subtitle_black\">([0-9]{1,2} [A-Za-z]{3} [0-9]{4} @ [0-9]{4})</td>", "<td class=\"heading_subtitle_black\">"+dtTm+"</td>");
		bbreport = bbreport.replaceFirst("<td class=\"left_2col_label\">Date Range Selected:</td>\\s+<td class=\"right_2col_value\">([0-9]{1,2} [A-Za-z]{3} [0-9]{4} to [0-9]{1,2} [A-Za-z]{3} [0-9]{4})</td>", "<td class=\"left_2col_label\">Date Range Selected:</td><td class=\"right_2col_value\">--</td>");
		bbreport = bbreport.replaceFirst("<td class=\"left_2col_label\">Last Updated:</td>\\s+<td class=\"right_2col_value\">([0-9]{1,2} [A-Za-z]{3} [0-9]{4} @ [0-9]{4})</td>","<td class=\"left_2col_label\">Last Updated:</td><td class=\"right_2col_value\">--</td>");
		bbreport = bbreport.replaceFirst("<td class=\"left_2col_label\">System Request Date/Time:</td>\\s+<td class=\"right_2col_value\">([0-9]{1,2} [A-Za-z]{3} [0-9]{4} @ [0-9]{4})</td>","<td class=\"left_2col_label\">System Request Date/Time:</td><td class=\"right_2col_value\">"+dtTm+"</td>"); 
		bbreport = bbreport.replaceFirst("<td class=\"left_2col_label\">File Name:</td>\\s+<td class=\"right_2col_value\">mhv_([\\w\\d\\s]+)_[0-9]{8}_[0-9]{4}.pdf</td>",
				"<td class=\"left_2col_label\">File Name:</td><td class=\"right_2col_value\">VA_IMG_RPT_"
				+"$1_"+ (study.getProcedureName().replaceAll(" ", "_")).replaceAll(",", "_")+"_"+dtFile+".pdf</td>");
		
		//Parse the report
		bbreport = bbreport.replaceFirst("<tr>\\s+<td colspan=\"2\" class=\"1col_embedded_label\" style=\"padding-left: 10px;\">"
				+ "<pre>No information was available that matched your selection.  "
				+ "However, if you recently had a VA radiology test or procedure, your reports may be available 3 calendar days after they have been completed.  "
				+ "Some studies done at a non-VA facility may not be available or they may not necessarily include an interpretation. </pre></td>\\s+</tr>", RadiologyReportHelper.generateXHTMLReport(report, study));

		String escapedXhtml = bbreport.replaceAll("(?i)&(?!(amp|apos|lt|gt|quot);)", "&amp;");
		
		return escapedXhtml;
    }
    
    
	public Response getBBReportNoPat(String studyIdUrn, Long downloadReportId, String type) throws MHVException {
		Session session =  CacheHandler.getInstance().getSession();
		PatientDTO patDto = patientProxy.getPatientByUserProfileId(session.getUserId());
		try {
			
	       List<Study> studies = studyRespository.findByPatientIdAndStudyIdUrn(patDto.getId(), studyIdUrn);
			Study study = null;
			if( studies.size()>0 ) {
				study = studies.get(0);

				SimpleDateFormat sdf = new SimpleDateFormat("dd MMM yyyy @ HHmm");
				String detail = study.getProcedureName() + " performed on " + sdf.format(study.getPerformedDate().getDatePrecise()); 
				
				//Record AAL entry for viewing report
				if( ReportTypeEnum.isPDFDL(ReportTypeEnum.valueOf(type)) ) {
					activityProxy.createAccountActivityLog(ActivityHelper.createActivityDTOForSelf(session.getUserId(), true, ActivityTypeEnumeration.DOWNLOAD, ActivityActionTypeEnumeration.VA_MEDICAL_IMAGES_AND_REPORT_DOWNLOAD_PDF, detail));
				} else if( ReportTypeEnum.isTXTDL(ReportTypeEnum.valueOf(type)) ) {
					activityProxy.createAccountActivityLog(ActivityHelper.createActivityDTOForSelf(session.getUserId(), true, ActivityTypeEnumeration.DOWNLOAD, ActivityActionTypeEnumeration.VA_MEDICAL_IMAGES_AND_REPORT_DOWNLOAD_TEXT, detail));
				}
				
			}
			
			return getBBReportAsStream(patDto.getId(), studyIdUrn, downloadReportId, study, ReportTypeEnum.valueOf(type));
		} catch (Exception e) {
			throw new MHVException(e);
		}
	}
	
	public Response getReportNoPat(String studyIdUrn) throws MHVException {
		Session session =  CacheHandler.getInstance().getSession();
		PatientDTO patDto = patientProxy.getPatientByUserProfileId(session.getUserId());
		try {
			return getReportAsStream(patDto.getId(), studyIdUrn);
		} catch (Exception e) {
			throw new MHVException(e);
		}
	}

}
