package gov.va.med.mhv.sm.web.smActions;

import gov.va.med.mhv.foundation.crypto.MHVCipher;
import gov.va.med.mhv.foundation.service.response.CollectionServiceResponse;
import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.sm.enumeration.ParticipantTypeEnum;
import gov.va.med.mhv.sm.enumeration.UserStatusEnum;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.Credentials;
import gov.va.med.mhv.sm.model.MailParticipant;
import gov.va.med.mhv.sm.model.MhvAuthenticationSubject;
import gov.va.med.mhv.sm.model.MhvIntegrationSettings;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.SMTermsConditionsVerHist;
import gov.va.med.mhv.sm.model.Surrogate;
import gov.va.med.mhv.sm.model.TriageGroup;
import gov.va.med.mhv.sm.model.User;
import gov.va.med.mhv.sm.model.UserAcceptTermsConditionsHist;
import gov.va.med.mhv.sm.dao.AddresseeDao;
import gov.va.med.mhv.sm.dao.SurrogateDao;
import gov.va.med.mhv.sm.dao.UserDao;
import gov.va.med.mhv.sm.model.decorator.MhvAuthenticationSubjectDecorator;
import gov.va.med.mhv.sm.service.AdminService;
import gov.va.med.mhv.sm.service.AuthenticationService;
import gov.va.med.mhv.sm.service.MailboxService;
import gov.va.med.mhv.sm.service.TermsAndConditionsService;
import gov.va.med.mhv.sm.service.TriageGroupService;
import gov.va.med.mhv.sm.service.UserManagementService;
import gov.va.med.mhv.sm.util.MhvIntegrationUtils;


import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class Authenticate extends AbstractInboxAction {
	// TODO: This action is supporting too many different actions/tasks:
	//      (1) authenticate patients,
	//      (2) authenticate clinicians,
	//      (3)handle opt-in/out-out,
	//      (4) authenticate test users (both patients and clinicians).
	//
	//    This causes a conflict between parameters being set by
	//    the framework on the action instance for different actions.
	//    This forces the parameter to be explicitly retrieved from the
	//    request, rather than using a property that can be set by the
	//    framework
	//    E.g. both loginIntegration and loginClinicianIntegration use
	//    a parameter called 'station',
	//
	//    These actions/tasks should be separated into their own actions
	//    classes.

	private static final long serialVersionUID = 3848513453789012534L;
	private static final Log LOG = LogFactory.getLog(Authenticate.class);


	private static final SimpleDateFormat DOB_DATE_FORMAT =
		new SimpleDateFormat("MM/dd/yyyy");

	private static final String CANNOT_USE_IN_PRODUCTION_MODE =
		"An attempt was made to use a test method in production mode";

	private MailboxService mailboxService;
	private AuthenticationService authenticationService;
	private UserManagementService userManagementService;
	private TriageGroupService triageGroupService;
	private TermsAndConditionsService termsAndConditionsService;

	private AdminService	adminService;
	private SurrogateDao surrogateDao;
	private UserDao	userDao;
	private AddresseeDao addresseeDao;
	private MhvIntegrationUtils utils = null;
	private MhvAuthenticationSubject subject = new MhvAuthenticationSubject();
	private boolean userReacceptTerms;
	private int cprsAlertsCount;

	private String optIn;
	private Credentials credentials = null;

	public void prepare() throws Exception {
		super.prepare();
		mailboxService = (MailboxService) getBean("mailboxService");
		authenticationService = (AuthenticationService) getBean(
			"authenticationService");
		userManagementService = (UserManagementService) getBean(
			"userManagementService");
		triageGroupService = (TriageGroupService) getBean("triageGroupService");
		termsAndConditionsService = (TermsAndConditionsService) getBean("termsAndConditionsService");
		adminService = (AdminService)getBean("adminService");
		userDao = (UserDao)getBean("userDao");
		surrogateDao = (SurrogateDao)getBean("surrogateDao");
		utils = (MhvIntegrationUtils) getBean("mhvIntegrationUtils");
		addresseeDao = (AddresseeDao) getBean("addresseeDao");
	}

	/**
	 * Provides the authentication method for patient logging in to SM
	 * through MHV.
	 * @return One of the the following values
	 * <ul>
	 * 		<li>LOGINFAILED when the authentication of the patient failed</li>
	 * 		<li>PATIENT when the authentication succeeded and
	 *          the patient has opted-in to SM</li>
	 * 		<li>PATIENT_BLOCKED when the authentication succeeded and
	 *          the patient access to SM has been blocked</li>
	 * 		<li>OPTIN when the authentication succeeded and
	 *          the patient has opted-out or not yet opted-in to SM</li>
	 * </ul>
	 */
	public String loginIntegration() {
		ServiceResponse<Patient> response = (credentials != null)
			? authenticationService.authenticatePatient(credentials)
			: authenticationService.authenticatePatient(subject);
		if (hasError(response)) {
			return loginFailed(getErrorText(response) + " for " +
				MhvAuthenticationSubject.describe(subject));
		}
		Patient patient = response.getPayload();
		if (patient == null) {
			return loginFailed("No patient found for " +
				MhvAuthenticationSubject.describe(subject));
		}

		if (UserStatusEnum.OPT_IN.equals(patient.getStatus())) {
			//Check If T&C changed and force user to reaccept the updated T&C.
			if(getTermsAndConditionsService().getActiveTermsConditionsHistory().getPayload()!=null){
				
				if(LOG.isInfoEnabled()){
					LOG.info("Patient"+patient.getId()+"SM T&C has been changed");	
				}

				SMTermsConditionsVerHist smTermsVerHist = getTermsAndConditionsService().getActiveTermsConditionsHistory().getPayload();
				UserAcceptTermsConditionsHist userAcceptHist = getTermsAndConditionsService().getUserAcceptTermsAndConditionsHist(smTermsVerHist.getId(),patient.getId()).getPayload();
				if(userAcceptHist==null){
					if(LOG.isInfoEnabled()){
						LOG.info("Patient"+patient.getId()+"did not accept the new T&C");	
					}
					setUserReacceptTerms(true);
					setCurrentUser(patient);
					return "OPTIN";
				}else{
					if(LOG.isInfoEnabled()){
						LOG.info("Patient"+patient.getId()+"accepted the new T&C");	
					}
					openCurrentFolderInMailBox(patient);
					return "PATIENT";
				}
			}else{
				if(LOG.isInfoEnabled()){
					LOG.info("Patient: "+patient.getId()+"SM T&C is not changed");	
				}
				openCurrentFolderInMailBox(patient);
				return "PATIENT";
			}
		}

		if (UserStatusEnum.BLOCKED.equals(patient.getStatus())) {
			invalidateSession();
			return "PATIENT_BLOCKED";
		}

		setCurrentUser(patient);
		return "OPTIN";
	}


	/**
	 * Handles a clinician logging in from CPRS.
	 * This accepts the station and DUZ and authenticates a clinician
	 * based on the principal in the session.
	 */

	public String loginClinicianIntegration() {

		String userName = getRequest().getRemoteUser();

		if (userName == null) {
			String error = "Login attempt failed: Unknown user.";
			return loginFailed(error, error);
		}

		ServiceResponse<Clinician> response = authenticationService.
			authenticateClinician(getClinicianStation(), getDUZ(), userName);
		Clinician clinician = response.getPayload();



		if (handleMessages(response)) {
			return loginFailed();
		}

		assert (clinician != null);
		if(clinician.getStatus().equals(UserStatusEnum.OPT_OUT)) {
			if (getRequest() != null) {
				getSession().invalidate();
			}
			getSession().setAttribute("CLINICIANSTATUS","OPT_OUT");
		    return "OPTOUTMESSAGE";
		}

		//syncSMSUserDataWithVistaUserData(clinician);
		openCurrentFolderInMailBox(clinician);
		return "CLINICIAN";
	}

	public void syncSMSUserDataWithVistaUserData(Clinician clinician){
		User selectedVistaUser = null;
		List staffMembers= null;
		CollectionServiceResponse<gov.va.med.mhv.sm.wsclient.adminqueriessvc.User> response = adminService.findVistaClinician(getDUZ(), getClinicianStation());

		staffMembers = (List)response.getCollection();

		if(staffMembers!=null && staffMembers.size()!=0){

		gov.va.med.mhv.sm.wsclient.adminqueriessvc.User vistAUser = (gov.va.med.mhv.sm.wsclient.adminqueriessvc.User)staffMembers.get(0);

			boolean isNameChanged=false;

			if(!clinician.getFirstName().equalsIgnoreCase(vistAUser.getFirstName())){
				isNameChanged=true;
				clinician.setFirstName(vistAUser.getFirstName());
			}
			if(!clinician.getLastName().equalsIgnoreCase(vistAUser.getLastName())){
				isNameChanged=true;
				clinician.setLastName(vistAUser.getLastName());
			}
			if(isNameChanged){
				userManagementService.updateClinician(clinician);
			}
			response = null;
			vistAUser = null;
			staffMembers = null;
		}
	}

	public String getSurrogateName(User user) {
		Surrogate surrogate = surrogateDao.getCurrentSurrogateFor((Clinician)user);
		String surrogateName = null;
		if(surrogate!=null){
			if(surrogate.getSurrogateType()==ParticipantTypeEnum.CLINICIAN){
			 User surrogateClinician = userDao.findById(surrogate.getSurrogateId());
			 surrogateName = surrogateClinician.getName();
			 surrogateClinician = null;
			}
			if(surrogate.getSurrogateType()==ParticipantTypeEnum.TRIAGE_GROUP){
				TriageGroup triageGroup = (TriageGroup)triageGroupService.findTriageGroupById(surrogate.getSurrogateId()).getPayload();
				surrogateName = triageGroup.getName();
				triageGroup=null;
			}
			getRequest().getSession().setAttribute("SURROGATE_ID_TO_REMOVE",surrogate.getId());
		}
		getRequest().getSession().setAttribute("surrogateName",surrogateName);
		return surrogateName;

	}

	public String getClinicianStation() {
		return (getRequest() != null) ? getRequest().getParameter("station")
			: null;
	}

	public String getDUZ() {
		return (getRequest() != null) ? getRequest().getParameter("DUZ")
			: null;
	}

	public String getOriginalUrl() {
		return (getRequest() != null) ? getRequest().getRequestURL().toString()
			: null;
	}


	public void setUserId(String userId) {
		// Note the difference between userid (used for test methods)
		// and userId (used for patient login from MHV)
		subject.setUserName(userId);
	}


	public void setFirstName(String firstName) {
		subject.setFirstName(firstName);
	}
	public void setLastName(String lastName) {
		subject.setLastName(lastName);
	}
	public void setSource(String source) {
		subject.setSource(source);
	}
	public void setIcn(String icn) {
		subject.setIcn(icn);
	}
	public void setSsn(String ssn) {
		subject.setSsn(ssn);
	}
    public void setAuthenticatedStatus(String authenticatedStatus) {
		subject.setAuthenticated("1".equals(authenticatedStatus));
	}
	public void setEmail(String email) {
		subject.setEmail(email);
	}
	public void setDob(String dob) {
		try {
			subject.setDob(DOB_DATE_FORMAT.parse(dob));
		} catch (ParseException e) {
			if(LOG.isErrorEnabled()){
				LOG.error("Unable to parse DOB (" + dob + ") for " +
						// Must use getParameter, because parameter may not
						// yet be set on this Authenticate instance
						getRequest().getParameter("userId"));
			}
			subject.setDob(null);
		}
	}
	public void setChecksum(String checksum) {
		subject.setChecksum(checksum);
	}
	public void setTimestamp(String timestamp) {
		try {
			subject.setTimestamp(Long.parseLong(timestamp));
		} catch (NumberFormatException e) {
			if(LOG.isErrorEnabled()){
				LOG.error("Unable to parse timestamp (" + timestamp +
						")", e);
			}
			subject.setTimestamp(null);
		}
	}
	public void setVisn(String[] visn) {
		subject.setVisns(visn);
	}
	public void setStation(String[] station) {
		subject.setFacilities(station);
	}
	public void setKey(String key) {
		credentials = new Credentials();
		credentials.setKey(key);
	}

	@Override
	public Log getLog() {
		return LOG;
	}

	/**
	 * Retrieves the 'userid' parameter from the request as a Long value
	 * @return The userid, provided it can be parsed as Long; null otherwise.
	 */
	private Long getUserid() {
		// Note the difference between userid (used for test methods)
		// and userId (used for patient login from MHV)
		String value = getRequest().getParameter("userid");
		try {
			return Long.valueOf(value);
		} catch(NumberFormatException e) {
			if(LOG.isErrorEnabled()){
				LOG.error("Unable to get userId from '" + value + "'", e);
			}
			return null;
		}
	}

	private String loginFailed() {
		invalidateSession();
		return "LOGINFAILED";
	}
	private String loginFailed(String logMessage) {
		return loginFailed(logMessage, null);
	}
	private String loginFailed(String logMessage, String userMessage) {
		if (StringUtils.isBlank(userMessage)) {
			userMessage = "Login attempt failed, due to invalid credentials.";
		}
		addActionError(userMessage);
		if (!StringUtils.isBlank(logMessage)) {
			if(LOG.isErrorEnabled()){
				LOG.error("login failed, because " + logMessage);
			}
		} else {
			if(LOG.isDebugEnabled()){
				LOG.debug("login failed");
			}
		}
		return loginFailed();
	}


	private void invalidateSession() {
		if (getRequest() != null) {
			// The user successfully authenticated against the container,
			// but the credentials provided by the user conflicted with
			// the original request.
			// Therefore, the session must invalidated, such that the user
			// will be forced to re-authenticate and can provide other,
			// correct credentials.
			getSession().invalidate();
			// When upon failure, typically the browser should be redirected
			// to the original. In that case the session is the only way in
			// which to pass error information, such that it can be shown on
			// the authentication form.
			getSession().setAttribute(
				SESSION_ATTRIBUTE_LOGIN_ERRORS, getActionErrors());
		}
	}

	private void openCurrentFolderInMailBox(User user) {
		final boolean isPatient = isPatient(user);
		assert isPatient || isClinician(user);
		setCurrentUser(user);
		if (LOG.isDebugEnabled()) {
			LOG.debug("Fetching mailbox for '" + user.getUsername() + "'");
		}
		if (isPatient) {
			mailboxService.getMailbox((Patient) user);
		} else {
			int cprsCount = mailboxService.getCPRSAlertsCount(user.getId());
			if(cprsCount!=0){
				setCprsAlertsCount(cprsCount);
			}
			mailboxService.getMailbox((Clinician) user);
			String currentSurrogate = getSurrogateName((Clinician) user);
			getRequest().getSession().setAttribute("currentSurrogate", currentSurrogate);
			int reminderCount = addresseeDao.getMessageReminderToday(user.getId());
			if(reminderCount >=1){
				getRequest().getSession().setAttribute("REMINDER_FLAG","YES");
			}

		}
		setMessageFilterId(user.getMessageFilter().getId());	// set the default message filter Id to display as 'selected' <option> for Bug# 6268.
		setCurrentFolder(getCurrentUserInbox());

	}

	protected MailParticipant getSurrogate(Clinician clinician) {
		return null;
	}

	//****************************************************************
	//Begin Opt In - Opt Out methods
	//****************************************************************

	public String optInOptOut(){
		
		User user = getCurrentUser();
		if (user == null) {
			return loginFailed("Authentication failed: user not found",
				"Login attempt failed: invalid username/password combination.");
		}
		
		if(getRequest().getParameter("opt_cancel")!=null && getRequest().getParameter("opt_cancel").equals("Cancel")){
			setUserReacceptTerms(true);
			return "OPTIN";
		}
		
		if (hasCancelledOptIn()) {
			return (user.getStatus().equals(UserStatusEnum.OPT_IN))? "CANCEL" : "OPTIN";
		}

		UserStatusEnum newUserStatus = getNewUserStatus();
		if (UserStatusEnum.OPT_OUT.equals(newUserStatus)) {
			if (hasSubmittedOptOut()) {
				userManagementService.optOutUser(user);
				addActionError("You have successfully opted out of " +
					"'Secure Messaging'.");
				return "OPTOUTCONFIRM";
			}
			return "OPTOUT";
		}

		if (UserStatusEnum.OPT_IN.equals(newUserStatus)) {
			userManagementService.optInUser(user);
			openCurrentFolderInMailBox(user);
			return isPatient(user) ? "PATIENT" : "CLINICIAN";
		}

		addActionError("Authentication Failed: Opt in/Opt out failed.");
		return "OPTIN";
	}

	public void setOptIn(String optIn) {
			this.optIn = optIn;
	}

	public String getOptIn() {
			return optIn;
	}

	private String getOptOutSubmit() {
		return getRequest().getParameter("optOut_submit");
	}

	private boolean hasCancelledOptIn() {
		return "Cancel".equals(getRequest().getParameter("btn_cancel"))
			|| "Cancel".equals(getOptOutSubmit());
	}

	private boolean hasSubmittedOptOut() {
		return "Confirm".equalsIgnoreCase(getOptOutSubmit());
	}

	private UserStatusEnum getNewUserStatus() {
		try {
			Long value = Long.valueOf(getOptIn());
			return UserStatusEnum.valueOf(value);
		} catch (NumberFormatException e) {
			if(LOG.isErrorEnabled()){
				LOG.error("Unable to parse long value from '" + getOptIn() + "'");
			}
			return null;
		}
	}

	//********************************************************************
	//End Opt In - Opt Out methods
	//********************************************************************

	//********************************************************************
	// Start Test related methods
	//********************************************************************

	public String authenticatePatient() {
		if (isProductionMode()) {
			return loginFailed(CANNOT_USE_IN_PRODUCTION_MODE);
		}
		subject = createTestPatientSubject(getUserid());
		return loginIntegration();
	}

	public String authenticateClinician() {
		if (isProductionMode()) {
			return loginFailed(CANNOT_USE_IN_PRODUCTION_MODE);
		}
		Long userId = getUserid();
		ServiceResponse<Clinician> r = authenticationService.
			authenticateClinicianById(userId);
		Clinician clinician = r.getPayload();
		if (clinician == null) {
			return loginFailed("Clinician " + userId + " not found");
		}
		else{
			if(clinician.getStatus().equals(UserStatusEnum.OPT_OUT))
			{
				String error = "Login attempt failed, due to invalid credentials.";
				return loginFailed(error);
			}
		}


		openCurrentFolderInMailBox(clinician);
		return SUCCESS;
	}

	private boolean isProductionMode() {
		return utils.getSettings().isProductionMode();
	}

	private MhvAuthenticationSubject createTestPatientSubject(Long userId) {
		if (isProductionMode() || (userId == null)) {
			return null;
		}
		switch (userId.intValue()) {

		    case 148:
			return createPatientSubject("potter", "Harry", "Potter",
				"123456777	", "456789132456789", new String[] { "111" });

		    case 179:
				return createPatientSubject("potter", "Harry", "Potter",
					"123456777	", "456789132456789", new String[] { "111" });
			case 180:
				return createPatientSubject("rweasely", "Ron", "Weasley",
					"444444442", "11111111111112", new String[] { "111" });
			case 181:
				return createPatientSubject("granger", "Hermione", "Granger",
					"111119234", "11111119234", new String[] { "111" });
			case 5179:
				return createPatientSubject("smsmoke1", "MHVSM", "SMOKEPATIENT",
					"710988777", "1011218157V958875", new String[] { "994" });
			case 6034:
				return createPatientSubject("test11", "Test", "One",
					"623456789", "6580051760V837067", new String[] { "658"});
			case 6035:
				return createPatientSubject("test12", "Null", "Two",
					"223456789", "6580051335V383903", new String[] { "658"});
			case 6036:
				return createPatientSubject("test13", "Test", "Three",
					"323456789", "1003521967V296296", new String[] {"658"});
			case 6037:
				return createPatientSubject("test14", "Test", "Four",
					"423456789", "1011215535V949791", new String[] { "658"});
			case 6038:
				return createPatientSubject("test15", "Test", "Five",
					"523456789", "1008371394V017525", new String[] {"658"});
			case 7005:
				return createPatientSubject("plashu", "PLASHU", "CRUIS",
					"101146335", "1011195583V017461", new String[] {"658"});
			case 7006:
				return createPatientSubject("irulyi", "IRULYI", "MJKUDIH",
					"101012555", "1011195583V017462", new String[] {"658"});
			case 7007:
				return createPatientSubject("smsmoke1", "MHVSM", "SMOKEPATIENT",
					"710988777", "1011218157V958875", new String[] {"994"});
			default:
				return null;
		}
	}

	private MhvAuthenticationSubject createPatientSubject(
		String userName, String firstName, String lastName, String ssn,
		String icn, String[] facilities)
	{
		return createPatientSubject(userName, firstName, lastName, ssn,
				new Date(), icn, facilities, "PII                   ");
	}
	private MhvAuthenticationSubject createPatientSubject(
		String userName, String firstName, String lastName, String ssn,
		Date dob, String icn, String[] facilities, String emailAddress)
	{
		MhvAuthenticationSubject subject = new MhvAuthenticationSubject();
		MHVCipher cipher = utils.createCipher();
		MhvIntegrationSettings settings = utils.getSettings();
		Long timestamp = System.currentTimeMillis() + 5000;
		subject.setTimestamp(timestamp);
		subject.setUserName(userName);
		if(firstName == ""){
			subject.setFirstName("Null");
		}
		else{
			subject.setFirstName(firstName);
		}
		subject.setLastName(lastName);
		subject.setSsn((cipher != null) ? cipher.encrypt(ssn) : ssn);
		subject.setDob(dob);
		subject.setIcn((cipher != null) ? cipher.encrypt(icn) : icn);
		subject.setFacilities(facilities);
		subject.setEmail(emailAddress);
		subject.setSource((settings != null) ? settings.getPatientSource()
			: "MHV");
		subject.setAuthenticated(true);
		subject.setChecksum(new MhvAuthenticationSubjectDecorator(subject,
			utils).createChecksum());
		return subject;
	}

	public TermsAndConditionsService getTermsAndConditionsService() {
		return termsAndConditionsService;
	}

	public void setTermsAndConditionsService(
			TermsAndConditionsService termsAndConditionsService) {
		this.termsAndConditionsService = termsAndConditionsService;
	}
	
	public boolean isUserReacceptTerms() {
		return userReacceptTerms;
	}

	public void setUserReacceptTerms(boolean userReacceptTerms) {
		this.userReacceptTerms = userReacceptTerms;
	}
	
	public int getCprsAlertsCount() {
		return cprsAlertsCount;
	}

	public void setCprsAlertsCount(int cprsAlertsCount) {
		this.cprsAlertsCount = cprsAlertsCount;
	}
}

