package gov.va.med.mhv.sm.util;

import gov.va.med.mhv.foundation.util.Precondition;
import gov.va.med.mhv.persistence.dao.hibernate.BaseEntityDaoHibernate;
import gov.va.med.mhv.sm.dao.FacilityDao;
import gov.va.med.mhv.sm.dao.SignatureDao;
import gov.va.med.mhv.sm.dao.UserDao;
import gov.va.med.mhv.sm.enumeration.PerformerTypeEnum;
import gov.va.med.mhv.sm.enumeration.UserStatusEnum;
import gov.va.med.mhv.sm.enumeration.UserTypeEnum;
import gov.va.med.mhv.sm.enumeration.EmailNotificationEnum;
import gov.va.med.mhv.sm.model.MHVPatient;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.PatientFacility;
import gov.va.med.mhv.sm.model.Signature;
import gov.va.med.mhv.sm.model.User;
import gov.va.med.mhv.sm.model.decorator.MhvAuthenticationSubjectDecorator;
import gov.va.med.mhv.sm.service.LoggingService;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.Locale;

import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.validator.EmailValidator;

public final class UserUtils {
	
	private static final Log LOG = LogFactory.getLog(UserUtils.class);
	
	
	private UserUtils() {
		Locale.setDefault(Locale.ENGLISH);
		// hide constructor because utility class
	}

	/**
	 * Masks a given SSN returning a string that only show the last 4 digits.
	 * @param ssn The given SSN. The SSN may contain dashes.
	 * @return The masked SSN with the first 5 digits of the SSN replaced 
	 *    with '*'. When blank string is given the identical string is 
	 *    returned. 
	 */
	public static String maskSsn(String ssn) {
		if (StringUtils.isBlank(ssn)) {
			return ssn;
		}
		int dashesOffset = (ssn.contains("-")) ? 1 : 0;
		return StringUtils.overlay(StringUtils.overlay(ssn, "***", 0, 2),
			"**", 3 + dashesOffset, 4 + dashesOffset);
	}
	
	/*
	 *  method getUserSignature return the String of User Signature Block to attach with outgoing and reply email.
	 *  @param userId The given userId to find the signature record.
	 *  @signatureDao object reference to access the Data.
	 */
	public static String getUserSignature(Long userId,SignatureDao signatureDao){
		
		List signatureList = signatureDao.getSignatureForUser(userId);
		Signature signature = null;
		String signatureBlock = "";;
		
		if(signatureList!=null && signatureList.size() >0){
			 signature  = (Signature)signatureList.get(0);
			 if(signature.isActive()){
				 signatureBlock =  "\n\n\n\n\n" + (signature.getName()!=null?signature.getName():"")+"\n"+(signature.getTitle()!=null?signature.getTitle():"");
			 }
		}
		signatureList = null;
		signature=null;
		return signatureBlock;
	}
	
	/**
	 * Masks a given ICN returning a string that only show the last 4 digits.
	 * @param icn The given ICN.
	 * @return The masked ICN with the first 5 digits of the ICN replaced 
	 *    with '*'. When blank string is given the identical string is 
	 *    returned. 
	 */
	public static String maskIcn(String icn) {
		if (StringUtils.isBlank(icn)) {
			return icn;
		}
		return StringUtils.overlay(icn, "*****", 0, 4);
	}
	
	public static boolean isPatient(User user) {
		return (user != null) && UserTypeEnum.PATIENT.equals(user.
			getUserType());
	}

	public static boolean isClinician(User user) {
		return (user != null) && UserTypeEnum.CLINICIAN.equals(user.
			getUserType());
	}
	
	public static boolean isAdministrator(User user) {
		return (user != null) && UserTypeEnum.ADMINISTRATOR.equals(user.
			getUserType());
	}
	
	public static Patient createOrUpdatePatientFrom(
			MhvAuthenticationSubjectDecorator subject, UserDao userDao, FacilityDao facilityDao, 
			BaseEntityDaoHibernate<PatientFacility, Long> patientFacilityDao, LoggingService loggingService) 
		{
			assert subject != null;
			String username = StringUtils.trim(subject.getUserName());
			String ssn = StringUtils.replaceChars(subject.getSsn(), "-", "");
			Patient patient = userDao.findPatientByUsername(username);
			boolean mhvOptIn=false;
			if ((patient != null)) { // If Patient Exists.
				if (StringUtils.isBlank(patient.getEmail()) && EmailValidator.
						getInstance().isValid(subject.getEmail()))
				{
					// Only update the email address if it was not yet set
					patient.setEmail(subject.getEmail());
				}
				
				//CR1160 Changes
				if(subject.isTermsStatus())
				{
					if(patient.getStatus()==(UserStatusEnum.OPT_OUT) || patient.getStatus()==(UserStatusEnum.MANUALLY_ASSOCIATED_PATIENT) 
																	 || patient.getStatus()==(UserStatusEnum.NO_OPTION_CHOSEN)) 
							
					{
						patient.setStatus(UserStatusEnum.OPT_IN);
						mhvOptIn=true;
					}
				}
				else if(patient.getStatus()==UserStatusEnum.MANUALLY_ASSOCIATED_PATIENT){ 
						patient.setStatus(UserStatusEnum.NO_OPTION_CHOSEN);	
				}
			
			} else { // if Patient Not Exist
				patient = new Patient();
				patient.setUsername(subject.getUserName());
				// CR1160 Changes
				if(LOG.isInfoEnabled()){
					LOG.info("UserUtils.subject.isTermsStatus...."+subject.isTermsStatus());
					LOG.info("UserUtils.subject.getTermsStatusDateTime...."+subject.getTermsStatusDateTime());
				}
				if(subject.isTermsStatus()){
					patient.setStatus(UserStatusEnum.OPT_IN);
					mhvOptIn=true;
				}
				else{
					patient.setStatus(UserStatusEnum.NO_OPTION_CHOSEN);	
				}
				patient.setEmailNotification(EmailNotificationEnum.ONE_DAILY);  //As per CR 6764 email notification for patient id Once Daily
				
			}
			patient.setLastName(subject.getLastName());
			patient.setFirstName(subject.getFirstName());
			patient.setDob(subject.getDob());
			patient.setSsn(ssn);
			patient.setIcn(subject.getIcn());
			patient.setNssn(StringUtils.substring(subject.getLastName(), 0, 1) + 
				StringUtils.substring(ssn, 5, 9));
			
			mergeFacilities(patient, toSet(subject.getFacilities()),patientFacilityDao);
			
			if(patient.getCurrentLogin()!=null){
				patient.setLastLogin(patient.getCurrentLogin());
			}
			patient.setCurrentLogin(new Date());
			Patient updatedPatient = (Patient)userDao.save(patient);
			
			if(updatedPatient.getStatus()==UserStatusEnum.NO_OPTION_CHOSEN){
				loggingService.actionPending(updatedPatient,PerformerTypeEnum.SELF,"Patietn Not Opted In",true);
			}else if(mhvOptIn){
				if(LOG.isInfoEnabled()){
					LOG.info("mhvOptIn Status...."+mhvOptIn+"  :getTermsStatusDateTime..."+subject.getTermsStatusDateTime());
				}
				loggingService.optInFromMHV(updatedPatient, PerformerTypeEnum.SELF, "Patient Accepted SM T&C in MHV", true, subject.getTermsStatusDateTime());
			}
			return updatedPatient;
		}
		
	public static Patient createOrUpdatePatientFrom(
			MHVPatient mhvPatient, UserDao userDao, FacilityDao facilityDao, 
			BaseEntityDaoHibernate<PatientFacility, Long> patientFacilityDao) 
		{
			assert mhvPatient != null;
			String username = StringUtils.trim(mhvPatient.getUserName());
			String ssn = StringUtils.replaceChars(mhvPatient.getSsn(), "-", "");
			Patient patient = userDao.findPatientByUsername(username);
			if ((patient != null)) {
				if (StringUtils.isBlank(patient.getEmail()) && EmailValidator.
						getInstance().isValid(mhvPatient.getEmail()))
				{
					// Only update the email address if it was not yet set
					patient.setEmail(mhvPatient.getEmail());
				}
			} else {
				patient = new Patient();
				patient.setUsername(mhvPatient.getUserName());
				if(mhvPatient.getAcceptSMTerms()){
					patient.setStatus(UserStatusEnum.OPT_IN);
				}else{
					patient.setStatus(UserStatusEnum.MANUALLY_ASSOCIATED_PATIENT);	
				}
				
				if (StringUtils.isNotBlank(mhvPatient.getEmail()) && EmailValidator.getInstance().isValid(mhvPatient.getEmail()))
				{
					// Only update the email address if it was not yet set
					patient.setEmail(mhvPatient.getEmail());
				}
				
				patient.setEmailNotification(EmailNotificationEnum.ONE_DAILY);  //As per CR 6764 email notification for patient id Once Daily
			}
			String firstName = mhvPatient.getFirstName();
			if(firstName==null)firstName = "Null";
			patient.setLastName(mhvPatient.getLastName());
			patient.setFirstName(firstName);
			patient.setDob(mhvPatient.getDob());
			patient.setSsn(ssn);
			patient.setIcn(mhvPatient.getIcn());
			patient.setNssn(StringUtils.substring(mhvPatient.getLastName(), 0, 1) + 
				StringUtils.substring(ssn, 5, 9));
			mergeFacilities(patient, toSet(new String[]{mhvPatient.getFacility()}),patientFacilityDao);
			
			return (Patient) userDao.save(patient);
		}
	
	private static void mergeFacilities(Patient patient, Set<String> facilities, 
										BaseEntityDaoHibernate<PatientFacility, Long> patientFacilityDao){
		// Create a map of the facilities. Map each facility to false, 
		// indicating it is (possibly) no longer current
		Map<String, Boolean> facilitiesMap = createFacilitiesMap(patient);
		
		if (facilities != null) {
			for(String facility : facilities) {
				String vistAStation = extractVistAStation(facility);
				// This (vistA) facility is (still) current, 
				// indicated by mapping the facility to true
				if (facilitiesMap.put(vistAStation, Boolean.TRUE) == null) {
					PatientFacility patientFacility = new PatientFacility();
					patientFacility.setUser(patient);
					patientFacility.setStationNo(vistAStation);
					patient.getFacilities().add(patientFacility);
				}
			}
		}
		
		for (String stationNumber: facilitiesMap.keySet()){
			boolean patientNoLongerHasFacility = BooleanUtils.isFalse(
				facilitiesMap.get(stationNumber));
			if (patientNoLongerHasFacility) {
				// Remove the facilities that are mapped to false,
				// i.e. are no longer current
				PatientFacility patientFacilityToRemove = null;
				for(PatientFacility patientFacility: patient.getFacilities()){
					if(patientFacility.getStationNo().equals(stationNumber)){
						patientFacilityToRemove = patientFacility;
					}
				}
				if(patientFacilityToRemove != null){
					if (LOG.isDebugEnabled()) {
						LOG.debug("Remove facility '" + patientFacilityToRemove.
							getStationNo() + "' from '" + patient.getUsername() 
							+ "'");
					}
					patient.getFacilities().remove(patientFacilityToRemove);
					patientFacilityDao.delete(patientFacilityToRemove.getId());
				}
			}
		}
	}
	
	
	private static Map<String,Boolean> createFacilitiesMap(Patient patient){
		Map<String,Boolean> facilitiesMap = new Hashtable<String, Boolean>();
		if (patient.getFacilities() == null) {
			// NOTE: Is this side-effect intentional?
			patient.setFacilities(new ArrayList<PatientFacility>());
		}
		for(PatientFacility patientFacility: patient.getFacilities()){
			facilitiesMap.put(patientFacility.getStationNo(), Boolean.FALSE);
		}
		return facilitiesMap;
	}
	
	public static Set<String> toSet(String[] values) {
		if (values == null) {
			return null;
		}
		Set<String> valueSet = new HashSet<String>();
		for (String value: values) {
			valueSet.add(value);
		}
		return valueSet;
	}
	
	/**
	 * return a 3 digit numeric facility from a potentially longer string
	 * @param stationNumber
	 * @return
	 */
	private static String extractVistAStation(String stationNumber) {
		Precondition.assertMinLength("stationNumber", stationNumber, 3);
		stationNumber = stationNumber.substring(0, 3);
		try {
			// NOTE: Note sure if this test is valid. 
			// Why do we need to know whether the station number consists 
			// of 3 digits?
			Integer.parseInt(stationNumber);
		} catch (NumberFormatException e) {
			Precondition.fail("Station number not in correct format: " + 
				stationNumber);
		}
		return stationNumber;
	}

	public static TreeMap sortedResultMap(Map<String,String> resultMap) 
	{
		TreeMap sortedResultMap = new TreeMap(LIST_STRING_COMPARATOR);
		List mapKeys = new ArrayList(resultMap.keySet());
		List mapValues = new ArrayList(resultMap.values());
		resultMap.clear();
		TreeSet sortedSet = new TreeSet(mapValues);
		Object[] sortedArray = sortedSet.toArray();
		for (int i=0; i<sortedArray.length; i++){
			sortedResultMap.put((String)mapKeys.get(mapValues.indexOf(sortedArray[i])), (String)sortedArray[i]);
 		}
	
		return sortedResultMap;
	}

	private static final Comparator<Object> LIST_STRING_COMPARATOR = new Comparator<Object>() 
	{
		public int compare ( Object o1, Object o2 )   {  
		     String s1 =  ( String ) o1; 
		     String s2 =  ( String ) o2; 
		     return s1.toUpperCase (  ) .compareTo ( s2.toUpperCase (  )  ) ; 
		    }                                      
		   public boolean equals ( Object o )   {  
		     String s =  ( String ) o; 
		     return compare ( this, o ) ==0; 
		    }  
	};
	
	
}
