package gov.va.med.mhv.sm.service.impl;

import gov.va.med.mhv.foundation.service.response.CollectionServiceResponse;
import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.foundation.util.Precondition;
import gov.va.med.mhv.persistence.dao.hibernate.BaseEntityDaoHibernate;
import gov.va.med.mhv.sm.dao.FacilityDao;
import gov.va.med.mhv.sm.dao.SignatureDao;
import gov.va.med.mhv.sm.dao.StationTimezoneDao;
import gov.va.med.mhv.sm.dao.SurrogateDao;
import gov.va.med.mhv.sm.dao.TriageGroupDao;
import gov.va.med.mhv.sm.dao.UserDao;
import gov.va.med.mhv.sm.dto.UserSurrogateDTO;
import gov.va.med.mhv.sm.enumeration.EmailNotificationEnum;
import gov.va.med.mhv.sm.enumeration.MessageFilterEnum;
import gov.va.med.mhv.sm.enumeration.ParticipantTypeEnum;
import gov.va.med.mhv.sm.enumeration.PerformerTypeEnum;
import gov.va.med.mhv.sm.enumeration.UserStatusEnum;
import gov.va.med.mhv.sm.enumeration.UserTypeEnum;
import gov.va.med.mhv.sm.model.Administrator;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.MHVPatient;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.PatientFacility;
import gov.va.med.mhv.sm.model.SMTermsConditionsVerHist;
import gov.va.med.mhv.sm.model.Signature;
import gov.va.med.mhv.sm.model.StationTimezone;
import gov.va.med.mhv.sm.model.Surrogate;
import gov.va.med.mhv.sm.model.TriageGroup;
import gov.va.med.mhv.sm.model.User;
import gov.va.med.mhv.sm.model.UserAcceptTermsConditionsHist;
import gov.va.med.mhv.sm.model.UserProfileVW;
import gov.va.med.mhv.sm.service.LoggingService;
import gov.va.med.mhv.sm.service.TermsAndConditionsService;
import gov.va.med.mhv.sm.service.UserManagementService;
import gov.va.med.mhv.sm.service.response.messages.SmsServiceMessages;
import gov.va.med.mhv.sm.util.DateUtils;
import gov.va.med.mhv.sm.util.UserUtils;
import gov.va.med.mhv.sm.wsclient.IntegrationServiceDelegate;
import gov.va.med.mhv.sm.wsclient.queriessvc.PatientDemographics;
import gov.va.med.mhv.sm.wsclient.queriessvc.PatientDemographicsResponse;
import gov.va.med.mhv.sm.wsclient.queriessvc.UserLookupResponse;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.TimeZone;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.validator.EmailValidator;

public class UserManagementServiceImpl implements UserManagementService {

	@SuppressWarnings("unused")
	private static final Log log = LogFactory.getLog(UserManagementServiceImpl.class);

	private UserDao userDao;
	private SignatureDao signatureDao;
	private FacilityDao facilityDao = null;
	private BaseEntityDaoHibernate<PatientFacility, Long> patientFacilityDao = null;
	private TriageGroupDao triageGroupDao;
	private SurrogateDao surrogateDao;
	private IntegrationServiceDelegate delegate;
	private LoggingService loggingService;
	private TermsAndConditionsService termsAndConditionsService;
	


	private StationTimezoneDao stationTimezoneDao;


	public ServiceResponse<Boolean> optInUser(User u,PerformerTypeEnum performerType) {
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		boolean result = changeOptStatus(u, UserStatusEnum.OPT_IN);
		loggingService.optIn(u, performerType, null, true);
		response.setPayload(Boolean.valueOf(result));
		return response;
	}
	
	private void updateUserAcceptTermsConditions(User u){
		if(getTermsAndConditionsService().getActiveTermsConditionsHistory().getPayload()!=null){
			SMTermsConditionsVerHist smTermsVerHist = getTermsAndConditionsService().getActiveTermsConditionsHistory().getPayload();
			if(log.isInfoEnabled()){
				log.info("&&&&&&&&&&&&&&&&SMTERMSVERHIST......."+smTermsVerHist.getId()+"User Id:"+u.getId());	
			}
			
			UserAcceptTermsConditionsHist userAcceptHist = getTermsAndConditionsService().getUserAcceptTermsAndConditionsHist(smTermsVerHist.getId(),u.getId()).getPayload();
			if(userAcceptHist==null){
				userAcceptHist = new UserAcceptTermsConditionsHist();
				userAcceptHist.setSmTermsConditionsVerHist(smTermsVerHist);
				userAcceptHist.setAccepted(Boolean.TRUE);
				userAcceptHist.setPatientId(u.getId());
				userAcceptHist.setCreatedDate(new Date());
				getTermsAndConditionsService().createUserAcceptTermsConditionsHist(userAcceptHist);
				if(log.isInfoEnabled()){
					log.info("User "+u.getId()+" Accepted New T&C ");
				}
			}
		}
	}

	public ServiceResponse<Boolean> optInUser(User u) {
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		boolean result = changeOptStatus(u, UserStatusEnum.OPT_IN);
		updateUserAcceptTermsConditions(u);
		int updatedOptStatus = updateSMOptStatusByUserName(u.getUsername(),Boolean.TRUE);
		if(log.isInfoEnabled()){
			log.info("Updated Opt-In User Profile VW for the User "+u.getId()+" Updated record:"+updatedOptStatus);
		}
		loggingService.optIn(u, PerformerTypeEnum.SELF, null, true);
		response.setPayload(Boolean.valueOf(result));
		return response;
	}

	public ServiceResponse<Boolean> optOutUser(User u,PerformerTypeEnum performerType) {
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		boolean result = changeOptStatus(u, UserStatusEnum.OPT_OUT);
		loggingService.optOut(u, performerType, null, true);
		response.setPayload(Boolean.valueOf(result));
		return response;
	}


	public ServiceResponse<Boolean> optOutUser(User u) {
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		boolean result = changeOptStatus(u, UserStatusEnum.OPT_OUT);
		int updatedOptStatus = updateSMOptStatusByUserName(u.getUsername(),Boolean.FALSE);
		if(log.isInfoEnabled()){
			log.info("Updated Opt-Out User Profile VW for the User "+u.getId()+" Updated record:"+updatedOptStatus);
		}
		
		loggingService.optOut(u, PerformerTypeEnum.SELF, null, true);
		response.setPayload(Boolean.valueOf(result));
		return response;
	}


	public ServiceResponse<Patient> getPatientById(Long userId){
		ServiceResponse<Patient> response = new ServiceResponse<Patient>();
		response.setPayload(userDao.findPatientById(userId));
		return response;
	}

	public List<Object[]> getPatientFacilitiesByUserId(Long userId){
		return userDao.getTreatmentFacilitiesByPatient(userId);
	}

	/**
	 * Fetch the user from the database to make sure that
	 * no inadvertent changes are persisted.
	 * @param u       User to update
	 * @param status  Status to persist
	 * @return        true/false
	 */
	private boolean changeOptStatus(User u, UserStatusEnum status){
		try {
			User x = userDao.findById(u.getId());
			x.setStatus(status);
			userDao.save(x);
			// propagate change to caller, but only if successfully persisted
			u.setStatus(status);
		} catch (Exception e) {
			return false;
		}
		return true;
	}


    public CollectionServiceResponse<Patient> searchForPatients(String fn, String ln, String nssn, String station){

    	Precondition.assertNotBlank("station", station);

    	if(StringUtils.isBlank(ln) && StringUtils.isBlank(nssn)){
    		throw new IllegalArgumentException("The last name or the nssn must be filled in");
    	}

    	Collection<Patient> l = userDao.searchPatient(fn, ln, nssn, station);

    	CollectionServiceResponse<Patient> response = new CollectionServiceResponse<Patient>();
    	response.setCollection(l);


        return response;
    }

    public CollectionServiceResponse<MHVPatient> searchForMHVPatients(String fn, String ln, String nssn, String station){

    	Precondition.assertNotBlank("station", station);

    	if(StringUtils.isBlank(ln) && StringUtils.isBlank(nssn)){
    		throw new IllegalArgumentException("The last name or the nssn must be filled in");
    	}

    	List<MHVPatient> mhvPatients = userDao.searchMHVPatient(fn, ln, nssn, station);


    	CollectionServiceResponse<MHVPatient> response = new CollectionServiceResponse<MHVPatient>();
    	response.setCollection(mhvPatients);

        return response;
    }

    public ServiceResponse<Patient> createOrUpdatePatientFrom(MHVPatient mhvPatient){

    	Patient patient = UserUtils.createOrUpdatePatientFrom(mhvPatient, userDao, facilityDao, patientFacilityDao);

    	ServiceResponse<Patient> response = new ServiceResponse<Patient>();
    	response.setPayload(patient);


        return response;
    }


    /**
	 * Return a list of clinicians includes active/inactive matching the passed in criteria.
	 * The last name must be specified.  The first name
	 * is not required.
	 *
	 * @param fn    first name, full or partial
	 * @param ln    last name, full or partial
	 * @param station a must
	 * @return      List<Clinician>
	 */

    public CollectionServiceResponse<Clinician> searchForClinicians(String fn, String ln, String station){
    	Precondition.assertNotBlank("station", station);

    	if(StringUtils.isBlank(ln)){
    		throw new IllegalArgumentException("The last name must be filled in");
    	}

    	List<Clinician> l = userDao.searchClinician(fn, ln, station);

    	CollectionServiceResponse<Clinician> response = new CollectionServiceResponse<Clinician>();
    	response.setCollection(l);
        return response;
    }

    public CollectionServiceResponse<Clinician> searchForClinicians(String fn, String ln, String station, boolean active){
    	Precondition.assertNotBlank("station", station);

    	if(StringUtils.isBlank(ln)){
    		throw new IllegalArgumentException("The last name must be filled in");
    	}

    	Collection<Clinician> l = userDao.searchClinician(fn, ln, station, active);

    	CollectionServiceResponse<Clinician> response = new CollectionServiceResponse<Clinician>();
    	response.setCollection(l);
        return response;
    }


    /**
     * Adjust the times of start and end dates based on facility timezone set during creation of the
     * surrogate records.
     */
    public CollectionServiceResponse<Surrogate> getSurrogatesFor(Clinician c){
    	CollectionServiceResponse<Surrogate> response = new CollectionServiceResponse<Surrogate>();
    	List<Surrogate> ls = surrogateDao.getSurrogatesFor(c);
    	for( Surrogate s: ls) {
    		s.setSurrogateStartDate(new Date(s.getSurrogateStartDate().getTime()+s.getTimeZone()));
    		s.setSurrogateEndDate(new Date(s.getSurrogateEndDate().getTime()+s.getTimeZone()));
    	}
    	response.setCollection(ls);
    	return response;
    }

    public ServiceResponse<Clinician> fetchClinician(Long userId){

    	ServiceResponse<Clinician> response = new ServiceResponse<Clinician>();
    	response.setPayload(fetchClinicianX(userId));
    	return response;

    }

    /**
     * Internal method for fetching clinicians.  Sets the appropriate
     * surrogate in the user object
     *
     * @param userId
     * @return
     */
    private Clinician fetchClinicianX(Long userId){

    	Clinician c = userDao.findClinicianById(userId);
    	if(c == null) return c;
    	return c;

    }


    public CollectionServiceResponse<Clinician> getCliniciansForStation(String stationNo){
    	Precondition.assertNotBlank("stationNo", stationNo);
    	CollectionServiceResponse<Clinician> response = new CollectionServiceResponse<Clinician>();
    	response.setCollection(userDao.getCliniciansForStation(stationNo));
    	return response;
    }


    public CollectionServiceResponse<Clinician> getCliniciansForStationAll(String stationNo){
    	Precondition.assertNotBlank("stationNo", stationNo);
    	CollectionServiceResponse<Clinician> response = new CollectionServiceResponse<Clinician>();
    	response.setCollection(userDao.getCliniciansForStationAll(stationNo));
    	return response;
    }

	public ServiceResponse<Clinician> getClinicianForStationAndDuz(String stationNo, String duz, boolean active) {
    	Precondition.assertNotBlank("stationNo", stationNo);
    	Precondition.assertNotBlank("duz", duz);

    	ServiceResponse<Clinician> response = new ServiceResponse<Clinician>();
    	response.setPayload(userDao.findClinicianByStationAndDuz(stationNo, duz, active));
    	return response;
	}


    public ServiceResponse<User> savePreferences(User u){

    	Precondition.assertNotNull("user", u);
    	Precondition.assertPositive("user id", u.getId());

    	ServiceResponse<User> response = new ServiceResponse<User>();

    	EmailValidator emailV = EmailValidator.getInstance();

    	if(u.getEmail() != null){
    		if(!emailV.isValid(u.getEmail())){
    			response.setPayload(u);
    			response.addError(SmsServiceMessages.EMAIL_IS_NOT_VALID);
    			return response;
    		}
    	}

    	if(u.getEmailNotification() == null){
    		u.setEmailNotification(EmailNotificationEnum.NONE);
    	}else if(u.getEmailNotification() == EmailNotificationEnum.ON_ASSIGNEMNT &&
    			 u.getUserType() == UserTypeEnum.PATIENT){
    		u.setEmailNotification(EmailNotificationEnum.NONE);
    	}

    	/**
    	 * we could default this to ?? OPT_IN ?? but I am not sure
    	 * if that presents a security risk.  It is unlikely to happen
    	 * very often so for now I think it best to just bomb.
    	 */
    	if(u.getStatus() == null){
    		response.setPayload(u);
    		response.addError(SmsServiceMessages.USER_STATUS_IS_NOT_VALID);
    		return response;
    	}

    	// The only valid filter for a patient is ALL
    	if(u.getMessageFilter() == null ||
    	   u.getUserType() == UserTypeEnum.PATIENT){
    		u.setMessageFilter(MessageFilterEnum.ALL);
    	}

    	User n = userDao.savePreferences(u);

    	response.setPayload(n);
    	return response;

	}


    public ServiceResponse<Boolean> blockPatient(Patient p, Administrator admin, String reason){

    	Precondition.assertNotNull("patient", p);
    	Precondition.assertNotNull("administrator", admin);
    	Precondition.assertNotBlank("reason", reason);
    	loggingService.blockedPatient(p, "admin^" + admin.getId() + "^" + reason, true);
    	ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		boolean result = changeOptStatus(p, UserStatusEnum.BLOCKED);
		response.setPayload(Boolean.valueOf(result));
		return response;

    }

    public ServiceResponse<Boolean> unblockPatient(Patient p, Administrator admin, String reason){

    	Precondition.assertNotNull("patient", p);
    	Precondition.assertNotNull("administrator", admin);
    	Precondition.assertNotBlank("reason", reason);
    	loggingService.unblockPatient(p, "admin^" + admin.getId() + "^" + reason, true);
    	ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		boolean result = changeOptStatus(p, UserStatusEnum.OPT_OUT);
		response.setPayload(Boolean.valueOf(result));
		return response;

    }


    public ServiceResponse<PatientDemographics> getPatientDemographics(String ssn, String stationNo){

    	Precondition.assertNotBlank("ssn", ssn);
    	Precondition.assertNotBlank("stationNo", stationNo);

    	ServiceResponse<PatientDemographics> response = new ServiceResponse<PatientDemographics>();

    	try {
			PatientDemographicsResponse demo = delegate
					.getPatientDemographics(ssn, stationNo);

			if(!demo.getStatus().equalsIgnoreCase("Ok") || demo.getPatient() == null){
				/* FORTIFY SCAN REMOVE:
				 *  log.warn("unable to get demographics for : " + stationNo + "^" + ssn);
				 *  log.warn("status: " + demo.getStatus() + "   reason: " + demo.getReason());
				 */
				response.addError(SmsServiceMessages.PATIENT_DEMOGRAPHICS_NOT_FOUND);
			}else{
				response.setPayload(demo.getPatient());
			}
		} catch (Exception e) {
			/* FORTIFY SCAN REMOVE:
			 *  log.warn("unable to get demographics for : " + stationNo + "^" + ssn, e);
			 */
			response.addError(SmsServiceMessages.UNABLE_TO_FETCH_DEMOGRAPHICS);
		}
		return response;

    }

    public ServiceResponse<Patient> fleshPatientFacilities(Patient p){

    	Precondition.assertNotNull("patient", p);

    	ServiceResponse<Patient> response = new ServiceResponse<Patient>();
    	userDao.fleshFacilities(p);
    	response.setPayload(p);
    	return response;
    }

    public ServiceResponse<Clinician> inactivateReactivateClinician(Clinician c, boolean active){

    	Precondition.assertNotNull("clinician", c);

    	ServiceResponse<Clinician> response = new ServiceResponse<Clinician>();
    	c.setActive(active);
    	User u = userDao.save(c);
    	c = (Clinician) u;
		response.setPayload(c);
    	return response;
	}

    public ServiceResponse<Clinician> updateClinician(User u, Clinician clinician){
    	ServiceResponse<Clinician> response = new ServiceResponse<Clinician>();
    	userDao.save(clinician);
    	response.setPayload(clinician);

    	//Log event to audit logger
    	String comment = "For id " + clinician.getId() + " and DUZ " + clinician.getDuz() +
    	                 ";Update: "+clinician.getClinicalUserType().getName()+", "+clinician.getUsername();

    	loggingService.optOut(u, PerformerTypeEnum.SELF, comment, true);


    	return response;
    }

    public ServiceResponse<Clinician> updateClinician(Clinician clinician){
    	ServiceResponse<Clinician> response = new ServiceResponse<Clinician>();
    	userDao.save(clinician);
    	response.setPayload(clinician);
    	return response;
    }

    public ServiceResponse<Boolean> addUserSignature(Signature s){

        ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		try {
			signatureDao.save(s);
		} catch (Exception e) {
			response.setPayload(Boolean.FALSE);
			return response;
		}
        response.setPayload(Boolean.TRUE);
        return response;
    }

    public ServiceResponse<Boolean> setSurrogatesFor(Clinician c, List<Surrogate> surrogateList){

    	String stationNo = c.getStationNo();
    	StationTimezone st = stationTimezoneDao.getTimezoneForStation(stationNo);
    	long diffTz = 0;

    	if(st == null || StringUtils.isBlank(st.getTimezone())){
    		if(log.isWarnEnabled()){
    			log.warn("Station: " + stationNo + " does not have a timezone mapping.  " +
    				"Setting local date to appserver date.");
    		}
    	}else{
    		TimeZone tz = TimeZone.getTimeZone(st.getTimezone());
    		diffTz = DateUtils.getTimeOffset(tz);
    	}

        ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
        try
        {
	        for(Surrogate surrogate:surrogateList){
	        	surrogate.setSmsUser(c);
	        	surrogate.setTimeZone(diffTz);
	        	surrogate.setSurrogateStartDate(new Date(surrogate.getSurrogateStartDate().getTime()-diffTz));
	        	surrogate.setSurrogateEndDate(new Date(surrogate.getSurrogateEndDate().getTime()-diffTz));
				surrogateDao.save(surrogate);
			}
	        response.setPayload(Boolean.TRUE);
        }catch(Exception e){
        	if(log.isErrorEnabled()){
        		log.error("Error while update the surrogate..."+e);
        	}
	    	response.setPayload(Boolean.FALSE);
	    }
        return response;
    }

    public ServiceResponse<Boolean> clearSurrogatesFor(Clinician c){

        ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
        try
        {
	        surrogateDao.clearSurrogatesFor(c);
	        response.setPayload(Boolean.TRUE);
        }catch(Exception e){
        	if(log.isErrorEnabled()){
        		log.error("Error while clear the surrogate..."+e);
        	}
	    	response.setPayload(Boolean.FALSE);
	    }
        return response;
    }

    public ServiceResponse<Boolean> removeSurrogate(Long surrogateId){
        ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		try {
			surrogateDao.delete(surrogateId);
		} catch (Exception e) {
			response.setPayload(Boolean.FALSE);
			return response;
		}
        response.setPayload(Boolean.TRUE);
        return response;
    }

    public CollectionServiceResponse<UserSurrogateDTO> getTriageMembersWithCurrentSurrogate(Long triageGroupId){
    	CollectionServiceResponse<UserSurrogateDTO> response = new CollectionServiceResponse<UserSurrogateDTO>();
		List<Object[]> triageGroupMembers = getTriageGroupDao().getUsersAndSurrogatesForTriageGroup(triageGroupId);
		List userSurrogateList = new ArrayList();
		if(triageGroupMembers!=null)
		{
			for(Object[] object : triageGroupMembers)
			{
				UserSurrogateDTO userSurrogateDTO = new UserSurrogateDTO();

				if(object[0]!=null){
					 userSurrogateDTO.setUserId(new Long(object[0].toString()));
				}

				if(object[1]!=null){
					userSurrogateDTO.setName(object[1].toString());
				}

				if(object[2]!=null){
					userSurrogateDTO.setSurrogateName(object[2].toString());
				}

				if(object[3]!=null){
					userSurrogateDTO.setSurrogateName(object[3].toString());
				}
				userSurrogateList.add(userSurrogateDTO);
				response.setCollection(userSurrogateList);
			}
		}
		return response;
	}

    public User findById(Long Id) {
    	User userSearched =	userDao.findById(Id);
    	return userSearched;
    }

    public Patient findPatientById(Long id) {
    	Patient pat =	userDao.findPatientById(id);
    	return pat;
    }
    public Surrogate getCurrentSurrogateFor(Clinician clinician) {
    	Surrogate surrogate = surrogateDao.getCurrentSurrogateFor(clinician);
    	return surrogate;
    }

    public MHVPatient findMHVPatientByUserName(String userName) {
    	
    return 	getUserDao().findMHVPatientByUserName(userName);
    
    }

    public String getActiveSurrogateByUser(Clinician provider){
		Surrogate currentSurrogate = getCurrentSurrogateFor(provider);
        String surrogateName=null;
        if(currentSurrogate!=null){
				if(currentSurrogate.getSurrogateType()!=null && currentSurrogate.getSurrogateType()==ParticipantTypeEnum.CLINICIAN){
						User surrogateClinician = userDao.findById(currentSurrogate.getSurrogateId());
						surrogateName = surrogateClinician.getName();
						surrogateClinician = null;
						if(log.isInfoEnabled()){
							log.info("Provider: "+provider.getId()+"^"+provider.getName()+" Surrogate is Clinician "+surrogateName);
						}
				}
				else if(currentSurrogate.getSurrogateType()!=null && currentSurrogate.getSurrogateType()==ParticipantTypeEnum.TRIAGE_GROUP){
						TriageGroup triageGroup = (TriageGroup)triageGroupDao.findById(currentSurrogate.getSurrogateId());
						surrogateName = triageGroup.getName();
						triageGroup=null;
						if(log.isInfoEnabled()){
							log.info("Provider: "+provider.getId()+"^"+provider.getName()+" Surrogacy is TriageGroup:......"+surrogateName);
						}
				}

       }
       return surrogateName;
	 }

    public ServiceResponse<List<Object[]>> getUserDetailsByIds(Long messageId, Long userId){
    	ServiceResponse<List<Object[]>> serviceResponse = new ServiceResponse<List<Object[]>>();
    	List<Object[]> listObj = userDao.getUserDetailsByIds(messageId, userId);
    	serviceResponse.setPayload(listObj);
    	return serviceResponse;
	}
    
	public UserDao getUserDao() {
		return userDao;
	}
	public void setUserDao(UserDao userDao) {
		this.userDao = userDao;
	}
	public TriageGroupDao getTriageGroupDao() {
		return triageGroupDao;
	}
	public void setTriageGroupDao(TriageGroupDao triageGroupDao) {
		this.triageGroupDao = triageGroupDao;
	}
	public IntegrationServiceDelegate getDelegate() {
		return delegate;
	}
	public void setDelegate(IntegrationServiceDelegate delegate) {
		this.delegate = delegate;
	}
	public LoggingService getLoggingService() {
		return loggingService;
	}
	public void setLoggingService(LoggingService loggingService) {
		this.loggingService = loggingService;
	}
	public FacilityDao getFacilityDao() {
		return facilityDao;
	}
	public void setFacilityDao(FacilityDao facilityDao) {
		this.facilityDao = facilityDao;
	}
	public BaseEntityDaoHibernate<PatientFacility, Long> getPatientFacilityDao() {
		return patientFacilityDao;
	}
	public void setPatientFacilityDao(BaseEntityDaoHibernate<PatientFacility, Long> patientFacilityDao) {
		this.patientFacilityDao = patientFacilityDao;
	}
	public SignatureDao getSignatureDao() {
		return signatureDao;
	}
	public void setSignatureDao(SignatureDao signatureDao) {
		this.signatureDao = signatureDao;
	}
	public SurrogateDao getSurrogateDao() {
		return surrogateDao;
	}
	public void setSurrogateDao(SurrogateDao surrogateDao) {
		this.surrogateDao = surrogateDao;
	}
	public StationTimezoneDao getStationTimezoneDao() {
		return stationTimezoneDao;
	}
	public void setStationTimezoneDao(StationTimezoneDao stationTimezoneDao) {
		this.stationTimezoneDao = stationTimezoneDao;
	}


	/**
	 * To support API's
	 */
	public ServiceResponse<gov.va.med.mhv.sm.wsclient.queriessvc.User> getClinicianDemographics(Clinician c) {
		ServiceResponse<gov.va.med.mhv.sm.wsclient.queriessvc.User> response = new ServiceResponse<gov.va.med.mhv.sm.wsclient.queriessvc.User>();
		UserLookupResponse ulr = null;
		
		if(log.isInfoEnabled()){
			log.debug("About to get user demographics for " + c.getFirstName() + " " + c.getLastName() + ", using DUZ=" + c.getDuz() + " on station " + c.getStationNo());
		}

		try {
			boolean userFound = false;
			ulr = delegate.getUserDemographics(null, null, c.getDuz(), c.getStationNo());
			if("Ok".equalsIgnoreCase(ulr.getStatus())) {
				if (ulr.getUsers()!= null && ulr.getUsers().length > 0) {
					// SUCCESS
					if(log.isDebugEnabled()){
						log.debug("Found clinician demographics for " + ulr.getUsers().length + " users");
					}
					gov.va.med.mhv.sm.wsclient.queriessvc.User[] users = ulr.getUsers();
					if (users.length != 1) {
						if(log.isErrorEnabled()){
							log.error("User demographics query retrieved " + users.length + " results when only 1 was expected: conversationID = " + ulr.getConversationID());
						}
					} else if (users[0] != null) {
						userFound = true;
						if(log.isErrorEnabled()){
							log.debug("User demographics found: " + users[0].getFirstName() + " " + users[0].getLastName() + ": department = " + users[0].getDepartment() + ", phone = " + users[0].getPhone());
						}
						response.setPayload(users[0]);
					}
				}
			} else {
				if(log.isErrorEnabled()){
					log.error("User demographics query for " + c.getFirstName() + " " + c.getLastName() + ", DUZ=" + c.getDuz() + " on station " + c.getStationNo() + " returned " + ulr.getStatus() + ": conversationID = " + ulr.getConversationID());
				}
			}

			if (!userFound) {
				if(log.isErrorEnabled()){
					log.error("No users found matching " + c.getFirstName() + " " + c.getLastName() + ", DUZ=" + c.getDuz() + " on station " + c.getStationNo());
				}
				response.addError(SmsServiceMessages.UNABLE_TO_FETCH_DEMOGRAPHICS);
			}
		}catch(RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("Remote Exception: " + e.getMessage());
			}
			response.addError(SmsServiceMessages.UNABLE_TO_FETCH_DEMOGRAPHICS);
		}

		return response;
	}
    public Patient findApiPatientById(Long id) {
    	Patient pat =	userDao.findPatientById(id);
	if( pat != null ) {
    		userDao.fleshFacilitiesExcludeLock(pat);
	}
    	return pat;
    }
	public int updateSMOptStatusByUserName(String userName, Boolean optStatus){
		try{
			return userDao.updateSMOptStatusByUserName(userName, optStatus);	
		}catch(Exception error){
			return 0;
		}
		
	}
	
	public TermsAndConditionsService getTermsAndConditionsService() {
		return termsAndConditionsService;
	}


	public void setTermsAndConditionsService(
			TermsAndConditionsService termsAndConditionsService) {
		this.termsAndConditionsService = termsAndConditionsService;
	}
	
	public UserProfileVW getUserProfileByUserName(String userName){
		return userDao.getUserProfileByUserName(userName);
	}
	

}
