package gov.va.med.mhv.sm.service.impl;

import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.foundation.util.Precondition;
import gov.va.med.mhv.sm.dao.FacilityDao;
import gov.va.med.mhv.sm.dao.MessageAttachmentDao;
import gov.va.med.mhv.sm.dao.SMWorkloadHistoricalDao;
import gov.va.med.mhv.sm.dao.ThreadDao;
import gov.va.med.mhv.sm.dao.TiuNoteRecordDao;
import gov.va.med.mhv.sm.dao.UserDao;
import gov.va.med.mhv.sm.dao.properties.SMWorkloadProperties;
import gov.va.med.mhv.sm.enumeration.CPRSNotesStatusEnum;
import gov.va.med.mhv.sm.enumeration.ParticipantTypeEnum;
import gov.va.med.mhv.sm.enumeration.WorkloadCaptureTypeEnum;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.Facility;
import gov.va.med.mhv.sm.model.Message;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.PatientFacility;
import gov.va.med.mhv.sm.model.SMWorkloadHistorical;
import gov.va.med.mhv.sm.model.Thread;
import gov.va.med.mhv.sm.model.TiuNotePreview;
import gov.va.med.mhv.sm.model.TiuNoteRecord;
import gov.va.med.mhv.sm.service.LoggingService;
import gov.va.med.mhv.sm.service.MessageService;
import gov.va.med.mhv.sm.service.TiuNoteService;
import gov.va.med.mhv.sm.service.response.messages.SmsServiceMessages;
import gov.va.med.mhv.sm.util.AttachmentUtils;
import gov.va.med.mhv.sm.util.CPRSMessageUtils;
import gov.va.med.mhv.sm.util.CharacterUtils;
import gov.va.med.mhv.sm.util.DateUtils;
import gov.va.med.mhv.sm.util.WorkloadUtil;
import gov.va.med.mhv.sm.wsclient.IntegrationServiceDelegate;
import gov.va.med.mhv.sm.wsclient.tiusvc.AckType;
import gov.va.med.mhv.sm.wsclient.tiusvc.Clinic;
import gov.va.med.mhv.sm.wsclient.tiusvc.DocumentVLinked;
import gov.va.med.mhv.sm.wsclient.tiusvc.PatientClassifications;
import gov.va.med.mhv.sm.wsclient.tiusvc.PatientEClassQueryResponse;
import gov.va.med.mhv.sm.wsclient.tiusvc.PostWLCResponse;
import gov.va.med.mhv.sm.wsclient.tiusvc.Staff;

import java.math.BigDecimal;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class TiuNoteServiceImpl implements TiuNoteService {

	@SuppressWarnings("unused")
	private static final Log log = LogFactory.getLog(TiuNoteServiceImpl.class);


	private static final String INDENT = "";

	private IntegrationServiceDelegate delegate;
	private TiuNoteRecordDao tiuNoteRecordDao;
	private FacilityDao facilityDao;
	private ThreadDao threadDao;
	private UserDao userDao;
	private String noteTitle;
	private LoggingService loggingService;
	private SMWorkloadHistoricalDao smWorkloadHistoricalDao;
	private SMWorkloadProperties smWorkloadProperties;
	private MessageAttachmentDao messageAttachmentDao;
	private MessageService messageService;

	public TiuNoteServiceImpl(){
		// default the note title
		//noteTitle = "SECURE MESSAGING";
	}


	public static final Comparator<Message> MESSAGE_SORTER = new Comparator<Message>() {
		public int compare(Message a, Message b) {
			if(a.getSentDate() == null || b.getSentDate() == null) return 0;
			return a.getSentDate().compareTo(b.getSentDate());
		}
	};



	public ServiceResponse<Boolean> createProgressNoteWithWLCVisit(Long patientId,Long userId, Thread thread, TiuNotePreview notePreview) {


		Precondition.assertNotNull("thread", thread);
		Clinician c = userDao.findClinicianById(userId);
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();

		TiuNoteRecord r = tiuNoteRecordDao.getRecord(thread.getId(), c.getStationNo());
		Thread t = threadDao.getThreadComplete(thread.getId());
		Collections.sort(t.getMessages(), MESSAGE_SORTER);
		Patient p = findPatient(t);
		userDao.fleshFacilities(p);

		if(log.isInfoEnabled()){
			log.info("creating progress note for patient [" + p + "] on behalf of [" + c + "]");
		}

		if(p == null){
			response.addError(SmsServiceMessages.TIU_NO_PATIENT_ASSOCIATED);
			response.setPayload(Boolean.FALSE);
			return response;
		}

		if(!matchFacility(p, c)){
			response.addError(SmsServiceMessages.TIU_NO_FACILITY_IN_COMMON);
			response.setPayload(Boolean.FALSE);
			return response;
		}

		if(r == null){
			// create a new record; simplifies nested if's
			r = new TiuNoteRecord();
			//r.setNoteCreationDate(new Date());

			Calendar noteCreationDate = Calendar.getInstance();
			noteCreationDate.setTime(notePreview.getProcedureDateTime());
			noteCreationDate.set(Calendar.SECOND,0);
			noteCreationDate.set(Calendar.MILLISECOND, 0);
			r.setNoteCreationDate(noteCreationDate.getTime());
		}

		// separate the messages
		List<Message> proposed = new ArrayList<Message>();
		List<Message> existing = new ArrayList<Message>();

		if(r.getLastMessageId() > 0L){
			separateMessages(t, r.getLastMessageId(), existing, proposed);
		}else{
			proposed = t.getMessages();
		}

		if(proposed.size() == 0){
			response.addError(SmsServiceMessages.TIU_ALL_MESSAGES_SAVED);
			//???? set payload to empty preview  ?? or should it be null
			response.setPayload(Boolean.FALSE);
			return response;
		}

		String note = createNoteTextForTiu(proposed,notePreview.getSelectedMessages());

		String conversationId = pushNoteWithWLCVisit(patientId, c, note,notePreview);

		TiuNoteRecord noteRecord = new TiuNoteRecord();

		if(conversationId != null){
			// persist a new record to the database
			noteRecord.setThreadId(t.getId());
			noteRecord.setConversationId(conversationId);
			noteRecord.setVistaDiv(c.getStationNo());
			noteRecord.setLastMessageId(proposed.get(proposed.size()-1).getId());
			noteRecord.setLockedDate(new Date());
			noteRecord.setUserId(c.getId());
			noteRecord.setMessageId(notePreview.getCurrentMessageId());
			// always use the date from the last record
			noteRecord.setNoteCreationDate(r.getNoteCreationDate());

			tiuNoteRecordDao.save(noteRecord);
			response.setPayload(Boolean.TRUE);

		}else{
			response.addError(SmsServiceMessages.TIU_NOTE_SEND_ERROR);
			response.setPayload(Boolean.FALSE);
		}

		// log all of the message being saved as progress notes.....
		// note that although we are marking this as successful
		// we cannot be sure because we get an asynchronous response
		// back from VistA verifying success, unfortunately this makes
		// accurate logging very difficult
		conversationId=conversationId+"^NoteTitle:"+notePreview.getCprsTitle();

		for(Message pm : proposed){
			loggingService.progressNote(c, pm, noteRecord, conversationId,
				!response.getMessages().hasErrorMessages());
		}

		return response;
	}



	public ServiceResponse<Boolean> createProgressNote(Thread thread, Clinician c,String[] selectedMessages,TiuNotePreview notePreview) {
		//public ServiceResponse<Boolean> createProgressNote(Thread thread, Clinician c,String[] selectedMessages,String tiuNoteTitle)
			Precondition.assertNotNull("thread", thread);
			Precondition.assertNotNull("clinician", c);

			ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();

			TiuNoteRecord r = tiuNoteRecordDao.getRecord(thread.getId(), c.getStationNo());
			Thread t = threadDao.getThreadComplete(thread.getId());
			Collections.sort(t.getMessages(), MESSAGE_SORTER);
			Patient p = findPatient(t);
			userDao.fleshFacilities(p);

			if(log.isInfoEnabled()){
				log.info("creating progress note for patient [" + p + "] on behalf of [" + c + "]");
			}
			// make sure that the clinician has signature capabilities

			/* As per CR 4863, All the Triage member should be able to save as CPRS
			 * if(!c.isProvider()){
				response.addError(SmsServiceMessages.TIU_NO_PROVIDER_KEY);
				response.setPayload(Boolean.FALSE);
				return response;
			}*/

			if(p == null){
				response.addError(SmsServiceMessages.TIU_NO_PATIENT_ASSOCIATED);
				response.setPayload(Boolean.FALSE);
				return response;
			}

			if(!matchFacility(p, c)){
				response.addError(SmsServiceMessages.TIU_NO_FACILITY_IN_COMMON);
				response.setPayload(Boolean.FALSE);
				return response;
			}

			if(r == null){
				// create a new record; simplifies nested if's
				r = new TiuNoteRecord();
				r.setNoteCreationDate(new Date());
			}

			// see if the thread is locked
			if(r.getLockedDate() != null){
				response.addError(SmsServiceMessages.TIU_THREAD_LOCKED);
				response.setPayload(Boolean.FALSE);
				return response;
			}

			// separate the messages
			List<Message> proposed = new ArrayList<Message>();
			List<Message> existing = new ArrayList<Message>();

			if(r.getLastMessageId() > 0L){
				separateMessages(t, r.getLastMessageId(), existing, proposed);
			}else{
				proposed = t.getMessages();
			}

			if(proposed.size() == 0){
				response.addError(SmsServiceMessages.TIU_ALL_MESSAGES_SAVED);
				//???? set payload to empty preview  ?? or should it be null
				response.setPayload(Boolean.FALSE);
				return response;
			}

			// we have passed all of the requirements.
			// now we just need to send it off and update the
			// database
			//String note = createNoteText(proposed);
			String note = createNoteTextForTiu(proposed,selectedMessages);


			String conversationId = pushNote(p, c, note, r.getNoteCreationDate(),notePreview);
			//String conversationId = null;

			TiuNoteRecord noteRecord = new TiuNoteRecord();

			if(conversationId != null){
				// persist a new record to the database
				noteRecord.setThreadId(t.getId());
				noteRecord.setConversationId(conversationId);
				noteRecord.setVistaDiv(c.getStationNo());
				noteRecord.setLastMessageId(proposed.get(proposed.size()-1).getId());
				noteRecord.setLockedDate(new Date());
				// always use the date from the last record
				noteRecord.setNoteCreationDate(r.getNoteCreationDate());

				tiuNoteRecordDao.save(noteRecord);
				response.setPayload(Boolean.TRUE);

			}else{
				response.addError(SmsServiceMessages.TIU_NOTE_SEND_ERROR);
				response.setPayload(Boolean.FALSE);
			}

			// log all of the message being saved as progress notes.....
			// note that although we are marking this as successful
			// we cannot be sure because we get an asynchronous response
			// back from VistA verifying success, unfortunately this makes
			// accurate logging very difficult
			conversationId=conversationId+"^NoteTitle:"+notePreview.getCprsTitle();

			for(Message pm : proposed){
				loggingService.progressNote(c, pm, noteRecord, conversationId,
					!response.getMessages().hasErrorMessages());
			}

			return response;
		}



	public ServiceResponse<SMWorkloadHistorical> createWorkload(SMWorkloadHistorical workloadHistorical) {
			ServiceResponse<SMWorkloadHistorical> response = new ServiceResponse<SMWorkloadHistorical>();

			String stationNumber = workloadHistorical.getStationNumber().toString();
			String location = WorkloadUtil.getIen(workloadHistorical.getLocation());

 	 	    //System.out.println(">>>>>>>>>> workloadHistorical.getPatientId():: "+workloadHistorical.getPatientId());

			Patient patient = userDao.findPatientById(workloadHistorical.getPatientId());

			//System.out.println(">>>>>>>>>> patient:: "+patient.getId());

			String patientIcn = patient.getIcn();

			//System.out.println(">>>>>>>>>> patientIcn:: "+patientIcn);

			String eligibility = null;
			String dssUnit =  WorkloadUtil.getIen(workloadHistorical.getDssUnit());
			String procedure = WorkloadUtil.getIen(workloadHistorical.getProcedureName());
			Clinician clinician  = userDao.findClinicianById(workloadHistorical.getUserId());
			String duz = clinician.getDuz();
			String providers = workloadHistorical.getProviders();
			String diagnosis = null;
			String clinic = WorkloadUtil.getIen(workloadHistorical.getSmClinic());
			PatientClassifications patClass = null;

			if(workloadHistorical.getCaptureType().equals(WorkloadCaptureTypeEnum.WORKLOAD_CREDIT)){
				eligibility = WorkloadUtil.getIen(workloadHistorical.getEligibilityCode());
				patClass = workloadHistorical.getPatientClass();
				diagnosis = workloadHistorical.getDiagnosis();
				workloadHistorical.setPatientClassification(getPatientClassification(workloadHistorical.getPatientClass()));
			}else{
				patClass = getPatientClassForHistorical(stationNumber,dssUnit,patient.getIcn());
				workloadHistorical.setPatientClassification(patClass.getInpatientOutpatientStatus());
				workloadHistorical.setDiagnosis(null);
				workloadHistorical.setEligibilityCode(null);
			}
			Calendar procDateTimeToEnsemble = Calendar.getInstance();
			procDateTimeToEnsemble.setTime(workloadHistorical.getProcedureDate());
			procDateTimeToEnsemble.add(Calendar.HOUR, DateUtils.getHoursOffsetFromUTC(procDateTimeToEnsemble));
			procDateTimeToEnsemble.set(Calendar.SECOND, 0);
			procDateTimeToEnsemble.set(Calendar.MILLISECOND, 0);

			PostWLCResponse wlcResponse=null;
			try{
				if(log.isInfoEnabled()){
					log.info("Before Posting WLC to Ensemble &&&&&&&&&&&&&");
					log.info("Station->"+stationNumber+"^Patient->"+patientIcn+"^Dss->"+dssUnit+"^ProcDate->"+procDateTimeToEnsemble.getTime());
					log.info("^Duz->"+duz+"^providers->"+providers+"^Clinic->"+clinic+"^Diagnosis->"+diagnosis+"^Thread->"+workloadHistorical.getThreadId());
				}
				wlcResponse = delegate.postWLC(stationNumber,location ,patientIcn,eligibility,patClass, dssUnit, procDateTimeToEnsemble, procedure, duz, providers, clinic , diagnosis);

				if(log.isInfoEnabled()){
					log.info("After Posting WLC to Ensemble->Resposne.....***********"+wlcResponse);
					log.info("Conv->"+wlcResponse.getConversationID()+"^ Status->"+wlcResponse.getStatus()+"^ Reason->"+wlcResponse.getReason());
					log.info("Visit Ien->"+wlcResponse.getVisitIEN()+" ^ECS Ien->"+wlcResponse.getECSrecordIEN());
				}



				if(wlcResponse!=null && wlcResponse.getConversationID()!=null){
					workloadHistorical.setConversationId(wlcResponse.getConversationID());
					workloadHistorical.setStatus(wlcResponse.getStatus());
					workloadHistorical.setEcsError(wlcResponse.getReason());

					if(!wlcResponse.getStatus().equalsIgnoreCase("OK")){
						workloadHistorical.setStatus(wlcResponse.getStatus()+"^Reason:"+wlcResponse.getReason());
					}
					if(wlcResponse.getVisitIEN()!=null && wlcResponse.getVisitIEN()!="")
							workloadHistorical.setVisitIen(wlcResponse.getVisitIEN());

					if(wlcResponse.getECSrecordIEN()!=null && wlcResponse.getECSrecordIEN()!="")
							workloadHistorical.setEcsIen(wlcResponse.getECSrecordIEN());
				}
				else{
					response.setPayload(null);
				}
			}catch(Exception e1){
				if(log.isErrorEnabled()){
					log.error("&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&Error while post WLC to ensemble&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&"+e1);
				}
				response.setPayload(null);
			}
			try{
				if(wlcResponse!=null && wlcResponse.getConversationID()!=null){

					Calendar procDateTimeToSM = Calendar.getInstance();
					procDateTimeToSM.setTime(workloadHistorical.getProcedureDate());
					procDateTimeToSM.set(Calendar.SECOND,0);
					procDateTimeToSM.set(Calendar.MILLISECOND, 0);
					workloadHistorical.setProcedureDate(procDateTimeToSM.getTime());

					SMWorkloadHistorical worloadCredit = getSmWorkloadHistoricalDao().save(workloadHistorical);
					response.setPayload(worloadCredit);
				}
			}catch(Exception e2){
				if(log.isErrorEnabled()){
					log.error("*************************Error while Save Worload Credit to DB********************************"+e2);
				}
				response.setPayload(null);
			}

		return response;
	}

	// Fetch Patient I/O status for Historical Form filing
	private PatientClassifications getPatientClassForHistorical(String stationNumber,String dssUnit,String patientIcn){
		PatientClassifications patClass = new PatientClassifications();
		try{
			PatientEClassQueryResponse patQueryResponse = delegate.getPatientEClassifications(stationNumber, dssUnit, patientIcn);
			if(patQueryResponse!=null){
				if(patQueryResponse.getPatientClassifications()!=null && patQueryResponse.getPatientClassifications().getInpatientOutpatientStatus()!=null){
					patClass.setInpatientOutpatientStatus(patQueryResponse.getPatientClassifications().getInpatientOutpatientStatus());
				}
			}
		}catch(Exception e2){
			if(log.isInfoEnabled()){
				log.info("Error while fetching Patient EClassifications..."+e2);
			}
		}

		return patClass;
	}

	private String getPatientClassification(PatientClassifications patClass){
		StringBuilder strPatClass = new StringBuilder();
		if(patClass.getSCCondition()!=null) strPatClass.append("SC:").append(patClass.getSCCondition());
		if(patClass.getAgentOrange()!=null) strPatClass.append("^").append("AO:").append(patClass.getAgentOrange());
		if(patClass.getCombatVet()!=null) strPatClass.append("^").append("CV:").append(patClass.getCombatVet());
		if(patClass.getIonizingRadiation()!=null) strPatClass.append("^").append("IO:").append(patClass.getIonizingRadiation());
		if(patClass.getEnvironmentalContaminants()!=null) strPatClass.append("^").append("EC:").append(patClass.getEnvironmentalContaminants());
		if(patClass.getMilitarySexualTrauma()!=null) strPatClass.append("^").append("MST:").append(patClass.getMilitarySexualTrauma());
		if(patClass.getHeadNeckCancer()!=null) strPatClass.append("^").append("HNC:").append(patClass.getHeadNeckCancer());
		if(patClass.getProject112SHAD()!=null) strPatClass.append("^").append("PS:").append(patClass.getProject112SHAD());
		return strPatClass.toString();
	}

	public ServiceResponse<TiuNotePreview> getNotePreview(Thread thread, Clinician c) {

		ServiceResponse<TiuNotePreview> response = new ServiceResponse<TiuNotePreview>();

		TiuNoteRecord r = tiuNoteRecordDao.getRecord(thread.getId(), c.getStationNo());
		Thread t = threadDao.getThreadComplete(thread.getId());
		Collections.sort(t.getMessages(), MESSAGE_SORTER);

		Patient p = findPatient(t);

		if(p == null){
			response.addError(SmsServiceMessages.TIU_NO_PATIENT_ASSOCIATED);
			response.setPayload(null);
			return response;
		}

		userDao.fleshFacilities(p);

		// make sure that the clinician has signature capabilities
		/* As per the CR 4863, All the Triage members can be able to save as cprs
		 * if(!c.isProvider()){
			response.addError(SmsServiceMessages.TIU_NO_PROVIDER_KEY);
			response.setPayload(null);
			return response;
		}*/

		if(!matchFacility(p, c)){
			response.addError(SmsServiceMessages.TIU_NO_FACILITY_IN_COMMON);
			response.setPayload(null);
			return response;
		}

		// see if the thread is locked
		if(r != null && r.getLockedDate() != null){

			response.addError(SmsServiceMessages.TIU_THREAD_LOCKED);

			// set the previewed note to the last one sent so that it can
			// be shown greyed out
			List<TiuNoteRecord> rs = tiuNoteRecordDao.getRecords(t.getId(), c.getStationNo());

			TiuNotePreview preview = null;
			// if there is only one record then use all messages
			// otherwise only preview the last message draft
			if(rs.size() == 1){
				preview = createNotePreviewObj(t, c, p, null);
			}else{
				TiuNoteRecord nextLast = rs.get(rs.size()-2);
				preview = createNotePreviewObj(t, c, p, nextLast);
			}


			preview.setLocked(true);
			response.setPayload(preview);
			return response;
		}

		TiuNotePreview preview = createNotePreviewObj(t, c, p, r);
		response.setPayload(preview);

		if(preview.getProposedMessages().size() == 0){
			response.addError(SmsServiceMessages.TIU_ALL_MESSAGES_SAVED);
		}

		response.setPayload(preview);
		return response;
	}

	private TiuNotePreview createNotePreviewObjForNewSM(Message message, Clinician c, Patient p, TiuNoteRecord r){
		TiuNotePreview preview = new TiuNotePreview();
		System.out.println("Create Note Preview Obj for New SM");
		try{
			preview.setPatient(p);
			preview.setCreatedDate(new Date());
			try {
				Facility f = facilityDao.getFacilityByStationNumber(c.getStationNo());
				if (f != null) {
					preview.setFacilityName(f.getName());
				}
			} catch (Exception e) {
				if(log.isErrorEnabled()){
					log.error("******************************************************");
					log.error("******************************************************");
					log.error("The facilities table may not be created!!!!!");
					log.error("******************************************************");
					log.error("******************************************************");
				}
			}
			List<Message> messageList = new ArrayList<Message>();
			messageList.add(message);
			preview.setProposedMessages(messageList);
			preview.setProposedNote(createNoteText(messageList));
		}catch(Exception e){
			if(log.isErrorEnabled()){
				log.error("TiuNoteServiceImpl->500 Error Occured"+e);
				e.printStackTrace();
			}
		}
		return preview;
	}
	
	private TiuNotePreview createNotePreviewObj(Thread t, Clinician c, Patient p, TiuNoteRecord r){

		TiuNotePreview preview = new TiuNotePreview();

		preview.setPatient(p);
		preview.setCreatedDate(new Date());
		try {
			Facility f = facilityDao.getFacilityByStationNumber(c.getStationNo());
			if (f != null) {
				preview.setFacilityName(f.getName());
			}
		} catch (Exception e) {
			if(log.isErrorEnabled()){
				log.error("******************************************************");
				log.error("******************************************************");
				log.error("The facilities table may not be created!!!!!");
				log.error("******************************************************");
				log.error("******************************************************");
			}
		}

		if(r == null){
			// there is no prior notes created
			// so fill out the preview and return
			preview.setProposedMessages(t.getMessages());
			preview.setProposedNote(createNoteText(t.getMessages()));
		}else{
			// there was a previous note created on this thread
			// so we are creating an addendum
			preview.setAddendum(true);

			SMWorkloadHistorical workloadHistorical = getSmWorkloadHistoricalDao().getSMWorkloadByThreadId(t.getId());

			//log.info("<<<<<<<<<<<workloadHistorical>>>>>>>>>>>>>>>>"+workloadHistorical);
			if(workloadHistorical!=null){
				//log.info("Visit Ien...."+workloadHistorical.getVisitIen());
				if(workloadHistorical.getVisitIen()!=null) preview.setVisitIen(workloadHistorical.getVisitIen());
			}

			separateMessages(t, r.getLastMessageId(), preview.getExistingMessages(),
					preview.getProposedMessages());

		}
		
		preview.setExistingNote(createNoteText(preview.getExistingMessages()));
		preview.setProposedNote(createNoteText(preview.getProposedMessages()));

		return preview;
	}


	public ServiceResponse<Boolean> unlockTiuRecord(String conversationId, boolean successful,String reason) {

		if(log.isInfoEnabled()){
			log.info("TiuNoteService unlockTiuRecord " + conversationId + " was reported " + (successful ? "successful" : "failed"));
		}

		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		TiuNoteRecord r = tiuNoteRecordDao.getRecord(conversationId);

		if(r == null){
			if(log.isInfoEnabled()){
				log.info("TiuNoteService unlockTiuRecord-> "+conversationId+" NOT FOUND in TIU_CREATION_RECORD");
			}
			response.addError(SmsServiceMessages.TIU_RECORD_NOT_FOUND);
			response.setPayload(Boolean.FALSE);
			return response;
		}
		try
		{
			if(successful){
				r.setLockedDate(null);
				r.setModifiedDate(Calendar.getInstance().getTime());
				tiuNoteRecordDao.save(r);
				if(log.isInfoEnabled()){
					log.info("TiuNoteService->Updated Success Status "+r.getLastMessageId()+"^Conversation: "+conversationId);
				}
			}else{
				r.setActive(false);
				r.setComments(reason);
				r.setModifiedDate(Calendar.getInstance().getTime());
				r.setLockedDate(null);
				//r.setComments("failed: " + DateUtils.getEnglishDateTime(new Date()));
				tiuNoteRecordDao.save(r);
				Message updatedMessage = messageService.updateMessageWithCPRSNotesStatus(r.getMessageId(),CPRSNotesStatusEnum.FAILED.getDescription(),Calendar.getInstance().getTime()).getPayload();
				if(log.isInfoEnabled()){
					if(updatedMessage!=null){
						log.info("TiuNoteService->Updated CPRS Failed Status with Message"+r.getLastMessageId()+"^Conversation: "+conversationId);
					}
				}
			}
		}catch(Exception e){
			if(log.isErrorEnabled()){
				log.error("An Error Occured while updating-TIU_CREATION_RECORD-Conversation Id"+conversationId+e);
			}
		}
		response.setPayload(Boolean.TRUE);
		return response;
	}

	private String pushNoteWithWLCVisit(Long patientId, Clinician clinician, String note,TiuNotePreview notePreview){

		AckType ack = null;

		gov.va.med.mhv.sm.wsclient.tiusvc.Patient patient;
		DocumentVLinked document;

    	try {
    		Patient patObj = userDao.findPatientById(patientId);

			patient = new gov.va.med.mhv.sm.wsclient.tiusvc.Patient();
			patient.setLastName(patObj.getLastName());
			patient.setFirstName(patObj.getFirstName());
			patient.setDateOfBirth(patObj.getDob());
			patient.setICN(patObj.getIcn());
			try {
				patient.setSSN(BigDecimal.valueOf(Long.parseLong(patObj.getSsn())));
			} catch (NumberFormatException e) {
				if(log.isWarnEnabled()){
					log.warn("Unable to format SSN into number format for patient " + patObj.getId() + "^" + patObj.getName());
				}
				patient.setSSN(null);
			}
			Staff author = new Staff();
			try{
				author.setDUZ(BigDecimal.valueOf(Long.parseLong(clinician.getDuz())));
			} catch (NumberFormatException e) {
				if(log.isWarnEnabled()){
					log.warn("Unable to format SSN into number format for patient " + clinician.getId() + "^" + clinician.getName());
				}
				author.setDUZ(null);
			}

			try{
				author.setSSN(BigDecimal.valueOf(Long.parseLong(clinician.getSsn())));
    		} catch (NumberFormatException e) {
    			if(log.isWarnEnabled()){
    				log.warn("Unable to format SSN into number format for clinician " + clinician.getId() + "^" + clinician.getName());
    			}
    			author.setSSN(null);
    		}

			author.setLastName(clinician.getLastName());
			author.setFirstName(clinician.getFirstName());

			document = new DocumentVLinked();

			document.setNoteTitle(notePreview.getCprsTitle());
			document.setAuthor(author);


			Calendar docDateTime = Calendar.getInstance();
			Calendar docVisitDateTime = Calendar.getInstance();

			//docDateTime.setTime(notePreview.getProcedureDateTime());
			//docDateTime.add(Calendar.HOUR, DateUtils.getHoursOffsetFromUTC(docDateTime));
			//document.setDateTime(docDateTime);

			docDateTime.setTime(notePreview.getProcedureDateTime());
			docDateTime.add(Calendar.HOUR, DateUtils.getHoursOffsetFromUTC(docVisitDateTime));
			docDateTime.set(Calendar.SECOND, 0);
			docDateTime.set(Calendar.MILLISECOND, 0);
			document.setDateTime(docDateTime);


			docVisitDateTime.setTime(notePreview.getProcedureDateTime());
			docVisitDateTime.add(Calendar.HOUR, DateUtils.getHoursOffsetFromUTC(docVisitDateTime));
			docVisitDateTime.set(Calendar.SECOND, 0);
			docVisitDateTime.set(Calendar.MILLISECOND, 0);


			document.setVisitDateTime(docVisitDateTime);


			//document.setText(note);
			String regularText = StringEscapeUtils.unescapeHtml(note);
			long startTime = System.currentTimeMillis();
			String convertedStr = CharacterUtils.getConvertedString(regularText);
			long endTime = System.currentTimeMillis();

			if(log.isInfoEnabled()){
				log.info(">>>>>>>Provider->Posting WLC CPRS Notes"+clinician.getId()+"^"+clinician.getName());
				log.info(">>>>>>>Provider->WLC CPRS Note to converting quotes - time taken#########"+((endTime - startTime)/ 1000)+" seconds");
			}
			document.setText(convertedStr);

			Clinic smClinic = new Clinic();
			smClinic.setIEN(notePreview.getClinicIen());
			smClinic.setName(notePreview.getClinicName());
			document.setClinic(smClinic);

			if(notePreview.getVisitIen()!=null && notePreview.getVisitIen().trim().length()!=0)
			{
				document.setVisitIEN(notePreview.getVisitIen());
			}

		} catch (Exception e) {
			if(log.isErrorEnabled()){
				log.error(">>>>>>>ERROR OCCURED WHILE SETTING UP POST NOTE WITH WLC DOCUMENT OBJ TO CPRS>>>>>",e);
			}
			return null;
		}

		try {
		   String messageId=notePreview.getCurrentMessageId()!=null?notePreview.getCurrentMessageId().toString():"";
		   ack=delegate.postNoteafterWLCwithSMmsgID(patient, document, clinician.getStationNo(), messageId);
		   //ack=delegate.postNoteafterWLC(patient, document, clinician.getStationNo());
    	} catch (RemoteException e) {
    		if(log.isErrorEnabled()){
    			log.error(">>>>>>>ERROR OCCURED WHILE PUSHING POST NOTE AFTER WLC TO ENSEMBLE WEB SERVICE>>>>>",e);
    		}
			return null;
		}

    	if(ack.getStatus().equalsIgnoreCase("Ok")){
    		if(log.isInfoEnabled()){
    			log.info("postNoteafterWLCwithSMmsgID Posted to Ensemble "  + ack.getConversationID());
    		}
			return ack.getConversationID();
		}else{
			if(log.isErrorEnabled()){
				log.error("Error pushing note to ensemble web service: postNoteafterWLCwithSMmsgID" +
					ack.getStatus() + "  " + ack.getReason());
			}
			return null;
		}

	}


	private String pushNote(Patient p, Clinician c, String note, Date noteCreationDate,TiuNotePreview notePreview){

		AckType ack = null;

		gov.va.med.mhv.sm.wsclient.tiusvc.Patient patient;
        //Document document;
        DocumentVLinked document;
    	try {
			patient = new gov.va.med.mhv.sm.wsclient.tiusvc.Patient();
			patient.setLastName(p.getLastName());
			patient.setFirstName(p.getFirstName());
			patient.setDateOfBirth(p.getDob());
			patient.setICN(p.getIcn());
			try {
				patient.setSSN(BigDecimal.valueOf(Long.parseLong(p.getSsn())));
			} catch (NumberFormatException e) {
				if(log.isWarnEnabled()){
					log.warn("Unable to format SSN into number format for patient " + p.getId() + "^" + p.getName());
				}
				patient.setSSN(null);
			}
			Staff author = new Staff();
			try{
				author.setDUZ(BigDecimal.valueOf(Long.parseLong(c.getDuz())));
			} catch (NumberFormatException e) {
				if(log.isWarnEnabled()){
					log.warn("Unable to format SSN into number format for patient " + c.getId() + "^" + c.getName());
				}
				author.setDUZ(null);
			}

			try{
				author.setSSN(BigDecimal.valueOf(Long.parseLong(c.getSsn())));
    		} catch (NumberFormatException e) {
    			if(log.isWarnEnabled()){
    				log.warn("Unable to format SSN into number format for clinician " + c.getId() + "^" + c.getName());
    			}
    			author.setSSN(null);
    		}

			author.setLastName(c.getLastName());
			author.setFirstName(c.getFirstName());
			//document = new Document();
			document = new DocumentVLinked();
			document.setNoteTitle(notePreview.getCprsTitle());
			document.setAuthor(author);
			Calendar docDateTime = Calendar.getInstance();
			Calendar docVisitDateTime = Calendar.getInstance();

			docDateTime.add(Calendar.HOUR, DateUtils.getHoursOffsetFromUTC(docDateTime));
			document.setDateTime(docDateTime);

			docVisitDateTime.setTime(noteCreationDate);
			docVisitDateTime.add(Calendar.HOUR, DateUtils.getHoursOffsetFromUTC(docVisitDateTime));
			document.setVisitDateTime(docVisitDateTime);

			Clinic clinic = new Clinic();
			clinic.setIEN(notePreview.getClinicIen());
			clinic.setName(notePreview.getClinicName());
			document.setClinic(clinic);

			document.setLocation(null);
			if(notePreview.getVisitIen()!=null) {

				document.setVisitIEN(notePreview.getVisitIen());
			}
			String regularText = StringEscapeUtils.unescapeHtml(note);
			String convertedStr = CharacterUtils.getConvertedString(regularText);
			document.setText(convertedStr);
		} catch (Exception e) {
			if(log.isErrorEnabled()){
				log.error(">>>>>>>ERROR OCCURED WHILE SETTING UP DOCUMENT OBJ TO CPRS>>>>>",e);
			}
			return null;
		}

		try {
			  // 12.13 method-> postNoteafterWLC
			  // ack=delegate.postNoteafterWLC(patient, document, c.getStationNo());

			  // 12.14 ensemble service method includes the secure message Id
			  String messageId=notePreview.getCurrentMessageId()!=null?notePreview.getCurrentMessageId().toString():"";
			  ack=delegate.postNoteafterWLCwithSMmsgID(patient, document, c.getStationNo(),messageId);
			  if(log.isInfoEnabled()){
				  log.info("Save to Progress notes ack"+ack.getStatus()+"^Conversation"+ack.getConversationID()+"Provider^"+c.getId());
			  }
    	} catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error(">>>>>>>ERROR OCCURED WHILE PUSHING NOTE TO ENSEMBLE WEB SERVICE>>>>>",e);
			}
			return null;
		}

    	if(ack.getStatus().equalsIgnoreCase("Ok")){
    		if(log.isDebugEnabled()){
    			log.debug("TIU note pushed: "  + ack.getConversationID());
    		}
			return ack.getConversationID();
		}else{
			if(log.isErrorEnabled()){
				log.error("Error pushing note to ensemble web service: " + 	ack.getStatus() + "  " + ack.getReason());
			}
			return null;
		}
	}



	/**
	 * return true if the patient and the clinician share
	 * a common vista facility
	 * @return
	 */
	private boolean matchFacility(Patient p, Clinician c){

		// verify that the patient and clinician have a common facility
		boolean matchFacility = false;
		for(PatientFacility pf : p.getFacilities()){
			if(pf.getStationNo().equals(c.getStationNo())){
				matchFacility = true;
				break;
			}
		}
		return matchFacility;
	}


	/**
	 * There *should* only ever be one patient associated with a
	 * message.
	 * @param t
	 * @return
	 */
	private Patient findPatient(Thread t){

		for(Message m : t.getMessages()){

			if(m.getSenderType() == ParticipantTypeEnum.PATIENT){
				return userDao.findPatientById(m.getSenderId());
			}
			if(m.getRecipientType() == ParticipantTypeEnum.PATIENT){
				return userDao.findPatientById(m.getRecipientId());
			}

		}
		// if we are here then there was not patient found
		return null;
	}


	/**
	 * some of the messages have already been persisted.
	 * separate out the ones already sent to vista and
	 * ones that have not.
	 *
	 * @param t
	 * @param lastMessageId
	 * @param existing List to place persisted messages
	 * @param proposed List to place proposed messages
	 */
	private void separateMessages(Thread t, Long lastMessageId, List<Message> existing,
			List<Message> proposed){

		boolean found = false;
		for(Message m : t.getMessages()){
			if(!found){
				if(m.getId().equals(lastMessageId)){
					found = true;
					existing.add(m);
					continue;
				}
			}

			if(found){
				proposed.add(m);
			}else{
				existing.add(m);
			}

		}

	}


	private String createNoteText(List<Message> messages){

		StringBuffer note = new StringBuffer();
		String indent = "";
		String selected="";
		String inputType="checkbox";
		if(messages!=null && messages.size()==1){
			selected="checked";
			inputType="hidden";
		}
		for(Message m : messages){
			if(m.getSentDate()!=null){
				note.append("<input type='"+inputType+"' name='selectedMessages' id='selectedMessages' "+selected+" value='"+m.getId()+"'/>");
				note.append(indent + "------Original Message------------------------\r\n");
				note.append(indent + "Sent:  " + DateUtils.getEnglishDateTime(m.getSentDate()) + "\r\n");
				note.append(indent + "From:  " +  m.getSenderName() + "\r\n");
				note.append(indent + "To:  " +  m.getRecipientName() + "\r\n");
				note.append(indent + "Subject:  " + m.getThread().getSubject() + "\r\n");
				if(m.isAttachment()){
					note.append(indent + "Attachments:  " + getAttachmentNames(m) + "\r\n");
				}
				note.append(indent + "\r\n");
				note.append(indent + StringEscapeUtils.unescapeHtml(m.getBody()) + "\r\n");			// Removed the <table><tr>, because it will appear as it is in CPRS message(Refer Bug # 5832)
				note.append("\r\n\r\n");
				indent += INDENT;
				}
		}
		return note.toString();
	}

	public String getAttachmentNames(Message message){
		List attachmentIds = new ArrayList();
		StringBuffer tiuAttachmentNames = new StringBuffer();
		if(message.getAttachmentId()!=null) attachmentIds.add(message.getAttachmentId());
		if(message.getAttachmentId2()!=null) attachmentIds.add(message.getAttachmentId2());
		if(message.getAttachmentId3()!=null) attachmentIds.add(message.getAttachmentId3());
		if(message.getAttachmentId4()!=null) attachmentIds.add(message.getAttachmentId4());
		List<Object[]> attachmentNames = messageAttachmentDao.getAttachmentNamesByAttachmentIds(attachmentIds);

		if(attachmentNames!=null && attachmentNames.size()!=0){
			int count=1;
			for(Object[] attachmentArr : attachmentNames)
			{
				String attachmentName = (String)attachmentArr[1]+" ("+AttachmentUtils.displayBytesSize(((BigDecimal)attachmentArr[2]).longValue())+")";
				tiuAttachmentNames.append(attachmentName);
				if(count!=attachmentNames.size()) tiuAttachmentNames.append(", ");
				count++;
			}
		}
		return tiuAttachmentNames.toString();
	}

	private String createNoteTextForTiu(List<Message> messages, String[] getSelectedMessages){

		StringBuffer note = new StringBuffer();
		String indent = "";



		if(getSelectedMessages !=null && getSelectedMessages.length >0)
		{
			List<String> selectedMsgList = null;
			try{
				selectedMsgList = Arrays.asList(getSelectedMessages);
			}catch(Exception e1)
			{
				if(log.isErrorEnabled()){
					log.error("Error in selectedMsgList..."+e1.getStackTrace().toString());
				}
			}

			for(Message m : messages)
			{
				try{
				if(selectedMsgList.contains(m.getId().toString()))
				{
					note.append(indent + "------Original Message------------------------\r\n");
					note.append(indent + "Sent:  " + DateUtils.getEnglishDateTime(m.getSentDate()) + "\r\n");
					note.append(indent + "From:  " +  m.getSenderName() + "\r\n");
					note.append(indent + "To:  " +  m.getRecipientName() + "\r\n");
					note.append(indent + "Subject:  " + m.getThread().getSubject() + "\r\n");
					if(m.isAttachment()){
						note.append(indent + "Attachments:  " + getAttachmentNames(m) + "\r\n");
					}
					note.append(indent + "\r\n");
					note.append(indent + StringEscapeUtils.unescapeHtml(m.getBody()) + "\r\n");				// Removed the <table><tr>, because it will appear as it is in CPRS message(Refer Bug # 5832)
					note.append("\r\n\r\n");
					indent += INDENT;
				}
				}catch(Exception e3){
					if(log.isErrorEnabled()){
						log.error("Error in e3......"+e3.getStackTrace().toString());
					}
				}
			}
		}
		else
		{
			try{
			for(Message m : messages){
				note.append(indent + "------Original Message------------------------\r\n");
				note.append(indent + "Sent:  " + DateUtils.getEnglishDateTime(m.getSentDate()) + "\r\n");
				note.append(indent + "From:  " +  m.getSenderName() + "\r\n");
				note.append(indent + "To:  " +  m.getRecipientName() + "\r\n");
				note.append(indent + "Subject:  " + m.getThread().getSubject() + "\r\n");
				if(m.isAttachment()){
					note.append(indent + "Attachments:  " + getAttachmentNames(m) + "\r\n");
				}
				note.append(indent + "\r\n");
				note.append(indent + StringEscapeUtils.unescapeHtml(m.getBody()) + "\r\n");// Removed the <table><tr>, because it will appear as it is in CPRS message(Refer Bug # 5832)
				note.append("\r\n\r\n");

				indent += INDENT;
			}
			}catch(Exception e2){
				if(log.isErrorEnabled()){
					log.error("Exception e2..."+e2.getStackTrace().toString());
				}
			}
		}
		return note.toString();
	}

	public ServiceResponse<TiuNotePreview> getNotePreviewSMWeb(Thread thread, Clinician c) {

		ServiceResponse<TiuNotePreview> response = new ServiceResponse<TiuNotePreview>();
		
		TiuNoteRecord r = tiuNoteRecordDao.getRecord(thread.getId(), c.getStationNo());
		
		if(r!=null){
			if(log.isInfoEnabled()){
				log.info("GetNotePreview->TIU RECORD ID........."+r.getId());
			}
		}
		Thread t = threadDao.getThreadComplete(thread.getId());
		
		Collections.sort(t.getMessages(), MESSAGE_SORTER);
		
		
		Patient p = findPatient(t);
		
		if(p == null){
			response.addError(SmsServiceMessages.TIU_NO_PATIENT_ASSOCIATED);
			response.setPayload(null);
			return response;
		}

		userDao.fleshFacilities(p);

		// make sure that the clinician has signature capabilities
		/* As per the CR 4863, All the Triage members can be able to save as cprs
		 * if(!c.isProvider()){
			response.addError(SmsServiceMessages.TIU_NO_PROVIDER_KEY);
			response.setPayload(null);
			return response;
		}*/
		
		if(!matchFacility(p, c)){
			response.addError(SmsServiceMessages.TIU_NO_FACILITY_IN_COMMON);
			response.setPayload(null);
			return response;
		}
		
		//If the Previous progress notes status is "Failed" then get the Comments and set it with TIuNotePreview and return to GUI
		//and set proposed = t.getMessages(); which will show all messages in the thread.
		if(r!=null && !r.isActive()){
			if(log.isInfoEnabled()){
				log.info("Inside tiu note record"+r.getId()+"!=null && not active");
				log.info("Reason"+r.getComments());
				log.info("Display the actual error reason to SM GUI");
			}
			// Display the Previous Failed Reason to GUI
			Object[] errorParam ={ CPRSMessageUtils.getErrorMessage(r.getComments())};
			response.addError(SmsServiceMessages.TIU_PROGRESS_NOTES_FAILED, errorParam);
			TiuNotePreview preview = createNotePreviewObj(t, c, p, null);

			List<TiuNoteRecord> rs = tiuNoteRecordDao.getRecords(t.getId(), c.getStationNo());
			if(rs.size() == 1){
				preview.setAddendum(false);
			}else{
				preview.setAddendum(true);
			}
			response.setPayload(preview);
			preview.setFailedProgressNote(true);
			return response;
		}

		// see if the thread is locked
		if(r != null && r.getLockedDate() != null){

			// 12.14 comment -> If the record is locked or previous message is failed, as per new requirement allow the user to retry again.
			//response.addError(SmsServiceMessages.TIU_THREAD_LOCKED);

			// set the previewed note to the last one sent so that it can
			// be shown greyed out
			List<TiuNoteRecord> rs = tiuNoteRecordDao.getRecords(t.getId(), c.getStationNo());

			TiuNotePreview preview = null;
			// if there is only one record then use all messages
			// otherwise only preview the last message draft
			if(rs.size() == 1){
				preview = createNotePreviewObj(t, c, p, null);
			}else{
				TiuNoteRecord nextLast = rs.get(rs.size()-2);
				preview = createNotePreviewObj(t, c, p, nextLast);
			}

			if(preview.getProposedMessages().size() == 0){
				response.addError(SmsServiceMessages.TIU_ALL_MESSAGES_SAVED);
			}

			preview.setLocked(true);
			response.setPayload(preview);
			return response;
		}
		TiuNotePreview preview = createNotePreviewObj(t, c, p, r);
		response.setPayload(preview);

		if(preview.getProposedMessages().size() == 0){
			response.addError(SmsServiceMessages.TIU_ALL_MESSAGES_SAVED);
		}
		response.setPayload(preview);
		return response;
	}

	public ServiceResponse<Boolean> createProgressNoteSMWeb(Thread thread, Clinician c,String[] selectedMessages,TiuNotePreview notePreview) {
		if(log.isInfoEnabled()){
			log.info("TIUNoteService->CreateProgressNoteSMWeb->Current Message Id..."+notePreview.getCurrentMessageId());
		}
		Precondition.assertNotNull("thread", thread);
		Precondition.assertNotNull("clinician", c);

		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();

		try{
			messageService.resetCPRSNotesStatusByThread(thread.getId());
		}catch(Exception e5){
			if(log.isErrorEnabled()){
				log.error("An Error Occured while resetting CPRS notes status by thread "+e5);
			}
		}

		TiuNoteRecord r = tiuNoteRecordDao.getRecord(thread.getId(), c.getStationNo());
		Thread t = threadDao.getThreadComplete(thread.getId());
		Collections.sort(t.getMessages(), MESSAGE_SORTER);
		Patient p = findPatient(t);
		userDao.fleshFacilities(p);

		if(log.isInfoEnabled()){
			log.info("creating progress note for patient [" + p + "] on behalf of [" + c + "]");
		}

		if(p == null){
			response.addError(SmsServiceMessages.TIU_NO_PATIENT_ASSOCIATED);
			response.setPayload(Boolean.FALSE);
			return response;
		}

		if(!matchFacility(p, c)){
			response.addError(SmsServiceMessages.TIU_NO_FACILITY_IN_COMMON);
			response.setPayload(Boolean.FALSE);
			return response;
		}

		if(r == null){
			// create a new record; simplifies nested if's
			r = new TiuNoteRecord();
			r.setNoteCreationDate(new Date());
		}

		// see if the thread is locked
		if(r.getLockedDate() != null){
			//12.14 comment - Even if the previous transaction is failed/locked, as per business requirement system should all the user to retry progress notes again
			/*	response.addError(SmsServiceMessages.TIU_THREAD_LOCKED);
				response.setPayload(Boolean.FALSE);
				return response;
			*/
		}

		// separate the messages
		List<Message> proposed = new ArrayList<Message>();
		List<Message> existing = new ArrayList<Message>();

		if(r!=null && !r.isActive()){
			proposed = t.getMessages();
		}else{
			if(r.getLastMessageId() > 0L){
				separateMessages(t, r.getLastMessageId(), existing, proposed);
			}else{
				proposed = t.getMessages();
			}
		}

		if(proposed.size() == 0){
			response.addError(SmsServiceMessages.TIU_ALL_MESSAGES_SAVED);
			response.setPayload(Boolean.FALSE);
			return response;
		}

		try{
			changePreviousTiuNoteRecordStatusByThreadAndMessageId(thread.getId(), notePreview.getCurrentMessageId());
		}catch(Exception e2){
			log.error("An Error occured while reset Previously saved notes by thread id"+thread.getId()+" and  messageId "+notePreview.getCurrentMessageId());
		}
		// we have passed all of the requirements.
		// now we just need to send it off and update the
		// database
		//String note = createNoteText(proposed);
		String note = createNoteTextForTiu(proposed,selectedMessages);

		String conversationId = pushNote(p, c, note, r.getNoteCreationDate(),notePreview);

		TiuNoteRecord noteRecord = new TiuNoteRecord();

		if(conversationId != null){
			// persist a new record to the database
			noteRecord.setThreadId(t.getId());
			noteRecord.setConversationId(conversationId);
			noteRecord.setVistaDiv(c.getStationNo());
			noteRecord.setLastMessageId(proposed.get(proposed.size()-1).getId());
			noteRecord.setUserId(c.getId());
			noteRecord.setMessageId(notePreview.getCurrentMessageId());
			noteRecord.setLockedDate(new Date());
			// always use the date from the last record
			noteRecord.setNoteCreationDate(r.getNoteCreationDate());

			tiuNoteRecordDao.save(noteRecord);
			response.setPayload(Boolean.TRUE);

		}else{
			response.addError(SmsServiceMessages.TIU_NOTE_SEND_ERROR);
			response.setPayload(Boolean.FALSE);
		}

		// log all of the message being saved as progress notes.....
		// note that although we are marking this as successful
		// we cannot be sure because we get an asynchronous response
		// back from VistA verifying success, unfortunately this makes
		// accurate logging very difficult
		conversationId=conversationId+"^NoteTitle:"+notePreview.getCprsTitle();

		for(Message pm : proposed){
			loggingService.progressNote(c, pm, noteRecord, conversationId,
				!response.getMessages().hasErrorMessages());
		}

		return response;
	}


	public List<Object[]> getProgressNotesStatusByThreadId(Long threadId){
		return tiuNoteRecordDao.getProgressNotesStatusByThreadId(threadId);
	}

	public ServiceResponse<TiuNoteRecord> getTiuNoteRecordByThreadAndMessageId(Long threadId,Long messageId){
		ServiceResponse<TiuNoteRecord> response = new ServiceResponse<TiuNoteRecord>();
		response.setPayload(tiuNoteRecordDao.getTiuNoteRecordByThreadAndMessageId(threadId, messageId));
		return response;
	}

	public List<Object[]> getFailedCPRSProgressNotesByStation(String stationNumber){
		return tiuNoteRecordDao.getFailedCPRSProgressNotesByStation(stationNumber);
	}

	private ServiceResponse<Boolean> changePreviousTiuNoteRecordStatusByThreadAndMessageId(Long threadId, Long messageId){
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		response.setPayload(tiuNoteRecordDao.changePreviousTiuNoteRecordStatusByThreadAndMessageId(threadId, messageId));
		return response;
	}
	
	public ServiceResponse<TiuNotePreview> getNotePreviewForNewMessage(Message message, Clinician clinician, Long patientId){
		ServiceResponse<TiuNotePreview> response = new ServiceResponse<TiuNotePreview>();
		Patient patient = userDao.findPatientById(patientId);
		TiuNotePreview preview = createNotePreviewObjForNewSM(message, clinician, patient, null);
		response.setPayload(preview);
		return response;
	}


	public String getVistAPatch11Fields() {
		return getSmWorkloadProperties().getVistAPatch11Status();
	}
	public String getVistAPatch11Status() {
		return getSmWorkloadProperties().getVistAPatch11Status();
	}

	public IntegrationServiceDelegate getDelegate() {
		return delegate;
	}
	public void setDelegate(IntegrationServiceDelegate delegate) {
		this.delegate = delegate;
	}
	public TiuNoteRecordDao getTiuNoteRecordDao() {
		return tiuNoteRecordDao;
	}
	public void setTiuNoteRecordDao(TiuNoteRecordDao tiuNoteRecordDao) {
		this.tiuNoteRecordDao = tiuNoteRecordDao;
	}
	public ThreadDao getThreadDao() {
		return threadDao;
	}
	public void setThreadDao(ThreadDao threadDao) {
		this.threadDao = threadDao;
	}
	public UserDao getUserDao() {
		return userDao;
	}
	public void setUserDao(UserDao userDao) {
		this.userDao = userDao;
	}
	public String getNoteTitle() {
		return noteTitle;
	}
	public void setNoteTitle(String noteTitle) {
		this.noteTitle = noteTitle;
	}
	public FacilityDao getFacilityDao() {
		return facilityDao;
	}
	public void setFacilityDao(FacilityDao facilityDao) {
		this.facilityDao = facilityDao;
	}
	public LoggingService getLoggingService() {
		return loggingService;
	}
	public void setLoggingService(LoggingService loggingService) {
		this.loggingService = loggingService;
	}

	public SMWorkloadHistoricalDao getSmWorkloadHistoricalDao() {
		return smWorkloadHistoricalDao;
	}

	public void setSmWorkloadHistoricalDao(
			SMWorkloadHistoricalDao smWorkloadHistoricalDao) {
		this.smWorkloadHistoricalDao = smWorkloadHistoricalDao;
	}



	public SMWorkloadProperties getSmWorkloadProperties() {
		return smWorkloadProperties;
	}



	public void setSmWorkloadProperties(SMWorkloadProperties smWorkloadProperties) {
		this.smWorkloadProperties = smWorkloadProperties;
	}

	public MessageAttachmentDao getMessageAttachmentDao() {
		return messageAttachmentDao;
	}



	public void setMessageAttachmentDao(MessageAttachmentDao messageAttachmentDao) {
		this.messageAttachmentDao = messageAttachmentDao;
	}

	public MessageService getMessageService() {
		return messageService;
	}

	public void setMessageService(MessageService messageService) {
		this.messageService = messageService;
	}

}
