package gov.va.med.mhv.sm.service.impl;

import gov.va.med.mhv.foundation.service.response.CollectionServiceResponse;
import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.foundation.util.Precondition;
import gov.va.med.mhv.sm.dao.AddresseeDao;
import gov.va.med.mhv.sm.dao.DistributionGroupDao;
import gov.va.med.mhv.sm.dao.MessageAttachmentDao;
import gov.va.med.mhv.sm.dao.MessageDao;
import gov.va.med.mhv.sm.dao.StationTimezoneDao;
import gov.va.med.mhv.sm.dao.SurrogateDao;
import gov.va.med.mhv.sm.dao.ThreadDao;
import gov.va.med.mhv.sm.dao.TriageGroupDao;
import gov.va.med.mhv.sm.dao.UserDao;
import gov.va.med.mhv.sm.enumeration.AddresseeRoleEnum;
import gov.va.med.mhv.sm.enumeration.ClinicianStatusEnum;
import gov.va.med.mhv.sm.enumeration.EmailNotificationEnum;
import gov.va.med.mhv.sm.enumeration.MessageCategoryTypeEnum;
import gov.va.med.mhv.sm.enumeration.ParticipantTypeEnum;
import gov.va.med.mhv.sm.enumeration.SystemFolderEnum;
import gov.va.med.mhv.sm.enumeration.UserStatusEnum;
import gov.va.med.mhv.sm.enumeration.UserTypeEnum;
import gov.va.med.mhv.sm.model.Addressee;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.DistributionGroup;
import gov.va.med.mhv.sm.model.Folder;
import gov.va.med.mhv.sm.model.MailParticipant;
import gov.va.med.mhv.sm.model.Message;
import gov.va.med.mhv.sm.model.NewMessage;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.PatientBlockedTriageGroup;
import gov.va.med.mhv.sm.model.PatientBlockedFacility;
import gov.va.med.mhv.sm.model.StationTimezone;
import gov.va.med.mhv.sm.model.Surrogate;
import gov.va.med.mhv.sm.model.Thread;
import gov.va.med.mhv.sm.model.TriageGroup;
import gov.va.med.mhv.sm.model.User;
import gov.va.med.mhv.sm.service.EmailService;
import gov.va.med.mhv.sm.service.LoggingService;
import gov.va.med.mhv.sm.service.PatientBlockedService;
import gov.va.med.mhv.sm.service.SendMessageService;
import gov.va.med.mhv.sm.service.TriageGroupService;
import gov.va.med.mhv.sm.service.exception.SendMessageException;
import gov.va.med.mhv.sm.service.response.messages.SmsServiceMessages;
import gov.va.med.mhv.sm.util.ChecksumUtils;
import gov.va.med.mhv.sm.util.DateUtils;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.StringEscapeUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.validator.EmailValidator;

public class SendMessageServiceImpl implements SendMessageService {

	@SuppressWarnings("unused")
	private static final Log log = LogFactory.getLog(SendMessageServiceImpl.class);

    private ThreadDao threadDao;
    private AddresseeDao addresseeDao;
    private MessageDao messageDao;
    private UserDao userDao;
    private TriageGroupDao triageGroupDao;
    private SurrogateDao surrogateDao;
    private DistributionGroupDao distributionGroupDao;
    private EmailService emailService;
    private TriageGroupService triageGroupService;
    private StationTimezoneDao stationTimezoneDao;
    private LoggingService loggingService;
    private MessageAttachmentDao messageAttachmentDao;
    private PatientBlockedService patientBlockedService;


	public static final Comparator<Message> MESSAGE_SORTER = new Comparator<Message>() {
		public int compare(Message a, Message b) {
			if(a.getSentDate() == null || b.getSentDate() == null) return 0;
			return a.getSentDate().compareTo(b.getSentDate());
		}
	};

    public ServiceResponse<Message> send(NewMessage nm){

    	Precondition.assertNotNull("from", nm.getFrom());
        Precondition.assertNotNull("to", nm.getTo());
        Precondition.assertNotBlank("subject", nm.getSubject());
        Precondition.assertNotBlank("body", nm.getBody());
        Message m = null;
        ServiceResponse<Message> response = new ServiceResponse<Message>();

        switch(nm.getFrom().getParticipantType()){

	        case PATIENT:{
	        	if (nm.getTo().getParticipantType() != ParticipantTypeEnum.TRIAGE_GROUP) {
	                response.addError(SmsServiceMessages.PATIENT_SEND_TO_TRIAGEGROUP_ONLY);
	        		return response;
	            }
	        	m = patientSendToTriage(nm, null);

	        	break;
	        }
	        case CLINICIAN:{
	        	try {
					//m = sendFromClinician(nm);
	        	 	 m = messageSendFromClinician(nm);	 // Fixes for Bug#5075
				} catch (SendMessageException e) {
					response.addError(e.getMsgkey());
	        		return response;
				}
				break;
	        }
	        default:{
	        	response.addError(SmsServiceMessages.MESSAGE_SENDER_NOT_PAITENT_OR_CLINICIAN);
        		return response;
	        }
        }
        loggingService.sendMessage((User)nm.getFrom(), m,
        		"Recipient:" + nm.getTo().getParticipantType().getId() + "^" + nm.getTo().getId() + "^" + nm.getTo().getName(),
        		!response.getMessages().hasErrorMessages());
        response.setPayload(m);
        return response;

    }

    /* New Method has been Implemented for message send from clinician to Triage Group.... Bug#5075*/
     private Message messageSendFromClinician(NewMessage nm) throws SendMessageException{

    	Message m = null;

    	switch(nm.getTo().getParticipantType()){

	    	case PATIENT:{
	    		if (nm.getTriageGroup() == null) {
	    			throw new SendMessageException(SmsServiceMessages.TRIAGE_GROUP_NOT_DEFINED);
	    	    }
	    		m = sendToUser(nm, null);
	    		break;
	    	}
	    	case CLINICIAN:{
	    		Clinician f = (Clinician)nm.getFrom();
	    		Clinician t = (Clinician)nm.getTo();
	    		if(t.equals(f)){
	    			throw new SendMessageException(SmsServiceMessages.CANNOT_SEND_MESSAGE_TO_YOURSELF);
	    		}
	    		m = sendToUser(nm, null);
	    		break;
	    	}
	    	case DISTRIBUTION_GROUP:{
					m = sendToDistributionGroup(nm);
				break;
	    	}
	    	case TRIAGE_GROUP:{
	    		//m = clinicianSendToTriage(nm,orig);
	    		m = clinicianSendMessageToTriage(nm);
	    		break;
	    	}
	    }

    	return m;

       }

     /* New Method has been Implemented for message send from clinician to Triage Group.... Bug#5075*/

     private Message clinicianSendMessageToTriage(NewMessage nm) {

     	Message m = null;
    		nm.setTriageGroup((TriageGroup)nm.getTo());
    		m = createMessage(nm, null);
    		Clinician from = (Clinician)nm.getFrom();
         // address the message to the sender
         addressMessage(m, from, SystemFolderEnum.SENT, true,false);

         // address the message to the group
         TriageGroup tr = (TriageGroup)nm.getTo();
         triageGroupDao.getCliniciansForTriageGroup(tr);


         for (Clinician c : tr.getClinicians()) {
            	if(from.equals(c)) continue;
             addressMessage(m, c, SystemFolderEnum.INBOX, true,false);
         }

         return m;
     }

     /**
      * Only address the message at this point since the triagegroup could have changed since the
      * message was created.
      *
      * The message was already created prior to this but we need to make the Addressee's up to date.
      *
      *
      * @param orig
      * @param nm
      * @return
      */
     public ServiceResponse<Message> sendReplyDraft(Message orig, NewMessage nm) {

      	Precondition.assertNotNull("Message", orig);
      	Precondition.assertPositive("Message Id", orig.getId());
      	Precondition.assertNotNull("from", nm.getFrom());
         Precondition.assertNotBlank("body", nm.getBody());

          ServiceResponse<Message> response = new ServiceResponse<Message>();
          Message message = null;

          if(orig.getThread() == null){
          	messageDao.fillThread(orig);
          }

          switch(nm.getFrom().getParticipantType()){
 	         case PATIENT:{
 		        	if(orig.getThread().getMailGroup() == null){
 		        		throw new IllegalStateException("Triage Group not defined for thread.");
 		        	}
 		        	nm.setTo(orig.getThread().getMailGroup());
 		        	message = patientSendToTriage(nm, orig);
 					break;
 		        }
 		        case CLINICIAN:{
 		        	  try {
 	 	        		if(orig.getRecipientType().getId().equals(ParticipantTypeEnum.TRIAGE_GROUP.getId())){
 	 	        			nm.setTo(orig.getThread().getMailGroup());
 	 	        			message = clinicianReplyToTriage(nm,orig);
 	 	        			if(log.isInfoEnabled()){
 	 	        				log.info("Send Reply Draft Message to Triage Group"+message.getId());
 	 	        			}
 	 	        			break;
 	 	        		}
 	 	        		if(orig.getRecipientType().getId().equals(ParticipantTypeEnum.DISTRIBUTION_GROUP.getId()))
 	 	        		{
 	 	        			DistributionGroup dg = distributionGroupDao.findById(orig.getRecipientId());
 	 	        			nm.setTo(dg);
 	 	        			message = clinicianReplyToDistributionGroup(nm,orig);
 	 	        			if(log.isInfoEnabled()){
 	 	        				log.info("Send Reply Draft Message to DG Group"+message.getId());
 	 	        			}
 	 	        			break;
 	 	            	}
 	 	        		else
 	 	        		{
 	 	        			User user = userDao.findById(orig.getRecipientId());
 	 	        			nm.setTo(user);
 	 	        			message = sendToUser(nm, orig);
 	 	        		}
 	 				}catch (SendMessageException e) {
 	 				}
 		        }
          }
          Addressee a = orig.getAddressees().get(0);
          if(a!=null){
        	  addresseeDao.delete(a.getId());
        	  messageDao.delete(orig.getId());
        	  response.setPayload(message);
          }
          return response;
      }

     /**
      * Only address the message at this point since the triagegroup could have changed since the
      * message was created.
      *
      * The message was already created prior to this but we need to make the Addressee's up to date.
      *
      *
      * @param orig
      * @param nm
      * @return
      */
     public ServiceResponse<Message> sendReplyDraftForAPI(Message orig, NewMessage nm) {

     	Precondition.assertNotNull("Message", orig);
     	Precondition.assertPositive("Message Id", orig.getId());
     	Precondition.assertNotNull("from", nm.getFrom());
        Precondition.assertNotBlank("body", nm.getBody());

         ServiceResponse<Message> response = new ServiceResponse<Message>();
         Message message = null;

         if(orig.getThread() == null){
         	messageDao.fillThread(orig);
         }

         switch(nm.getFrom().getParticipantType()){
	         case PATIENT:{
		        	if(orig.getThread().getMailGroup() == null){
		        		throw new IllegalStateException("Triage Group not defined for thread.");
		        	}
		        	// Ignore and setting it at API Level to avoid Hibernate initialization error
		        	//nm.setTo(orig.getThread().getMailGroup());
		        	message = patientSendToTriage(nm, orig);
					break;
		        }
		        case CLINICIAN:{
		        	// original message is already in the secure_messaging table withe sent_date=null.
		        	User user = userDao.findById(orig.getRecipientId());
		        	nm.setTo(user);
		        	message = sendToUser(nm, orig);
		        	break;
		        }
		        default:{
		        	response.addError(SmsServiceMessages.MESSAGE_SENDER_NOT_PAITENT_OR_CLINICIAN);
		        	return response;
		        }
         }

         Addressee a = orig.getAddressees().get(0);
     	 addresseeDao.delete(a.getId());
     	 messageDao.delete(orig.getId());

         response.setPayload(message);
         return response;
     }

    public ServiceResponse<Message> reply(Message orig, NewMessage nm) {

    	Precondition.assertNotNull("Message", orig);
    	Precondition.assertNotNull("from", nm.getFrom());
        Precondition.assertNotBlank("body", nm.getBody());
    	Precondition.assertPositive("Message Id", orig.getId());

    	Message m = null;
        ServiceResponse<Message> response = new ServiceResponse<Message>();

        // fetch the complete message from the database
        orig = messageDao.getMessageComplete(orig.getId());

        // a clinician cannot reply to a message that has already been completed
        if(nm.getFrom().getParticipantType()== ParticipantTypeEnum.CLINICIAN &&
           orig.getStatus() == ClinicianStatusEnum.COMPLETE){
        	response.addError(SmsServiceMessages.MESSAGE_ALREADY_COMPLETE);
        	response.setPayload(null);
        	return response;
        }


        // the thread should already be populated but just to be sure
        if(orig.getThread() == null){
        	messageDao.fillThread(orig);
        }

        switch(nm.getFrom().getParticipantType()){
            case PATIENT:{
	        	if(orig.getThread().getMailGroup() == null){
	        		throw new IllegalStateException("Triage Group not defined for thread.");
	        	}
	        	// set the triage group into the new message object
	        	nm.setTo(orig.getThread().getMailGroup());
	        	m = patientSendToTriage(nm, orig);
				break;
	        }
	        case CLINICIAN:{

	        	User user = userDao.findById(orig.getSenderId());
	        	nm.setTo(user);

	        	/*// set the recipient from the original message
	        	MailParticipant r = devineRecipient((Clinician)nm.getFrom(), orig);
	        	if(r == null){
	        		response.addError(SmsServiceMessages.UNABLE_TO_DETERMINE_RECIPIENT_OF_REPLY);
	        		return response;
	        		nm.setTo(r);
	        	}*/

	        	try {
	        		m = sendToUser(nm, orig);

	        		// set the "assignedTo" on the message being replied to
	        		// if the assignedTo is not set already
	        		Message reply = messageDao.findById(orig.getId());
	        		if(reply.getAssignedTo() == null){
	        			reply.setAssignedTo((Clinician)nm.getFrom());
	        			reply.setModifiedDate(new Date());
	        			messageDao.save(reply);
	        		}
				} catch (SendMessageException e) {
					response.addError(e.getMsgkey());
	        		return response;
				}

	        	break;
	        }
	        default:{
	        	response.addError(SmsServiceMessages.MESSAGE_SENDER_NOT_PAITENT_OR_CLINICIAN);
        		return response;
	        }
        }

        loggingService.sendMessage((User)nm.getFrom(), m,
        		"Recipient:" + nm.getTo().getParticipantType().getId() + "^" + nm.getTo().getId() + "^" + nm.getTo().getName(),
        		!response.getMessages().hasErrorMessages());
        response.setPayload(m);
        return response;

    }

    
    public ServiceResponse<Message> saveReplyDraftForWeb(Message orig, NewMessage nm,Folder folder) {

   		Precondition.assertNotNull("Message", orig);
   		Precondition.assertNotNull("from", nm.getFrom());
   		Precondition.assertNotBlank("body", nm.getBody());
   		Precondition.assertPositive("Message Id", orig.getId());

   		ServiceResponse<Message> response = new ServiceResponse<Message>();

   		// fetch the complete message from the database
   		orig = messageDao.getMessageComplete(orig.getId());

   		// a clinician cannot reply to a message that has already been completed
   		if(nm.getFrom().getParticipantType()== ParticipantTypeEnum.CLINICIAN &&
   				orig.getStatus() == ClinicianStatusEnum.COMPLETE){
   			response.addError(SmsServiceMessages.MESSAGE_ALREADY_COMPLETE);
   			response.setPayload(null);
   			return response;
   		}

   		// the thread should already be populated but just to be sure
   		if(orig.getThread() == null){
   			messageDao.fillThread(orig);
   		}

   		Addressee a;
   		if( orig.getSentDate() != null ) {
   			//This is a new Reply Draft Message
   			// (create new message and save it)
   			Message m = new Message();
   			m.setThread(orig.getThread());
	        m.setBody(nm.getBody());

			m.setSenderId(nm.getFrom().getId());
			m.setSenderName(nm.getFrom().getName());
			m.setSenderType(nm.getFrom().getParticipantType());
			switch(nm.getFrom().getParticipantType()){
	            case PATIENT:{
		        	if(orig.getThread().getMailGroup() == null){
		        		throw new IllegalStateException("Triage Group not defined for thread.");
		        	}
		        	// set the triage group into the new message object
		        	nm.setTo(orig.getThread().getMailGroup());
					break;
		        }
		        case CLINICIAN:{
		        	//User user = userDao.findById(orig.getSenderId());
		        	//nm.setTo(user);
		        	nm.setTo(getRecipient(orig,folder));
		        }
			}
			m.setRecipientId(nm.getTo().getId());
			m.setRecipientType(nm.getTo().getParticipantType());
			if(nm.getFrom().getParticipantType().equals(ParticipantTypeEnum.PATIENT)){
				m.setRecipientName(orig.getSenderName()+" ("+nm.getTo().getName()+")");
			}else{
				m.setRecipientName(nm.getTo().getName());	
			}
			
			m.setModifiedDate(new Date());
			if(!m.getSenderId().equals(m.getRecipientId())){
				messageDao.save(m);
				if(log.isDebugEnabled()){
					log.debug("Draft message id: " + m.getId());
				}
			

				//a = orig.getAddressees().get(0);
				a = new Addressee();
				a.setMessage(m);
				a.setFolderId(SystemFolderEnum.DRAFTS.getId());
				a.setOwner((User)nm.getFrom());
				a.setReadDate(new Date());
				a.setRole(AddresseeRoleEnum.SENDER);
				addresseeDao.save(a);
	
				//Make sure to associate the Addressee in the Message itself
				List<Addressee> addr = new ArrayList<Addressee>();
				addr.add(a);
				m.setAddressees(addr);
			}
			response.setPayload(m);
			return response;
   		} else {
   			// This is an existing reply draft message so just update the body and save
   			orig.setBody(nm.getBody());
   			orig.setModifiedDate(new Date());
   			messageDao.save(orig);
   			if(log.isDebugEnabled()){
   				log.debug("Draft message id: " + orig.getId());
   			}
   		}
   		response.setPayload(orig);
   		return response;
   	}
    

    private MailParticipant getRecipient(Message orig,Folder folder){
    	ParticipantTypeEnum type=orig.getRecipientType();
    	Long recipientId = orig.getSenderId();
    	if(folder.getId().equals(SystemFolderEnum.SENT.getId())){
    		type = orig.getRecipientType();
    		recipientId = orig.getRecipientId();
    	}else{
    		type = orig.getSenderType();
    		recipientId = orig.getSenderId();
    	}
    	switch(type){
    		case PATIENT: {
    			User user = userDao.findById(recipientId);
    			return user;
    		}
    		
    		case CLINICIAN:{
    			User user = userDao.findById(recipientId);
    			return user;
    		}
    			
    		case TRIAGE_GROUP:{
    			TriageGroup triageGroup = triageGroupDao.findById(recipientId);
    			return triageGroup;
    		}
    			
    		case DISTRIBUTION_GROUP:{
    			DistributionGroup distGroup = distributionGroupDao.findById(recipientId);
    			return distGroup;
    		}
    		
    		default:{
    			User user = userDao.findById(recipientId);
    			return user;
    		}
    	}
    }

    /**
     *
     *  Do the same as reply except push the message to the DRAFTS folder
     *
     *	Don't have to do all the addressing, etc. here because delay until the
     *  actual message is sent.  This prevents stale Addressee's.
     *
     * @param orig
     * @param nm
     * @return
     */
public ServiceResponse<Message> saveReplyDraft(Message orig, NewMessage nm) {

    	Precondition.assertNotNull("Message", orig);
    	Precondition.assertNotNull("from", nm.getFrom());
        Precondition.assertNotBlank("body", nm.getBody());
    	Precondition.assertPositive("Message Id", orig.getId());

        ServiceResponse<Message> response = new ServiceResponse<Message>();

        // fetch the complete message from the database
        orig = messageDao.getMessageComplete(orig.getId());

        // a clinician cannot reply to a message that has already been completed
        if(nm.getFrom().getParticipantType()== ParticipantTypeEnum.CLINICIAN &&
           orig.getStatus() == ClinicianStatusEnum.COMPLETE){
        	response.addError(SmsServiceMessages.MESSAGE_ALREADY_COMPLETE);
        	response.setPayload(null);
        	return response;
        }

        // the thread should already be populated but just to be sure
        if(orig.getThread() == null){
        	messageDao.fillThread(orig);
        }

        Addressee a;
        if( orig.getSentDate() != null ) {
        	//This is a new Reply Draft Message
        	// (create new message and save it)
        	Message m = new Message();
        	m.setThread(orig.getThread());
	        m.setBody(nm.getBody());
			//6598 KJG
			m.setSenderId(nm.getFrom().getId());
			m.setSenderName(nm.getFrom().getName());
			m.setSenderType(nm.getFrom().getParticipantType());
			switch(nm.getFrom().getParticipantType()){
	            case PATIENT:{
		        	if(orig.getThread().getMailGroup() == null){
		        		throw new IllegalStateException("Triage Group not defined for thread.");
		        	}
		        	// set the triage group into the new message object
		        	nm.setTo(orig.getThread().getMailGroup());
					break;
		        }
		        case CLINICIAN:{
		        	User user = userDao.findById(orig.getSenderId());
		        	nm.setTo(user);
		        	
		        }
			}
			m.setRecipientId(nm.getTo().getId());
			m.setRecipientType(nm.getTo().getParticipantType());
			m.setRecipientName(nm.getTo().getName());
            m.setModifiedDate(new Date());
			messageDao.save(m);
			if(log.isDebugEnabled()){
				log.debug("Draft message id: " + m.getId());
			}

			//a = orig.getAddressees().get(0);
			a = new Addressee();
			a.setMessage(m);
			a.setFolderId(SystemFolderEnum.DRAFTS.getId());
			a.setOwner((User)nm.getFrom());
			a.setReadDate(new Date());
			a.setRole(AddresseeRoleEnum.SENDER);
			addresseeDao.save(a);

			//Make sure to associate the Addressee in the Message itself
			List<Addressee> addr = new ArrayList<Addressee>();
			addr.add(a);
			m.setAddressees(addr);

			response.setPayload(m);
			return response;
        } else {
        	// This is an existing reply draft message so just update the body and save
        	orig.setBody(nm.getBody());
        	orig.setModifiedDate(new Date());
        	messageDao.save(orig);
        	if(log.isDebugEnabled()){
        		log.debug("Draft message id: " + orig.getId());
        	}

        	// And update the read date by addressee
//        	a = orig.getAddressees().get(0);
//			a.setReadDate(new Date());
//			addresseeDao.save(a);
        }

		response.setPayload(orig);
		return response;
    }

       /**
     * To handle reply-to save.
     *
     * Only expect "from" and "body" be populated.
     * Then check from is matching record.  And replace current body with body
     * from NewMessage.  If thread exists in the message then just leave it alone.
     *
     */
    public ServiceResponse<Message> saveDraft (NewMessage nm, Long messageId){

    	Precondition.assertNotNull("New Message", nm);
    	Precondition.assertNotNull("from", nm.getFrom());
        Precondition.assertNotBlank("subject", nm.getSubject());
        Precondition.assertNotBlank("body", nm.getBody());

        ServiceResponse<Message> response = new ServiceResponse<Message>();

        // can't save a draft if same person is sender and recipient
        if(nm.getTo() != null){
	        if((nm.getFrom().getId().equals(nm.getTo().getId())) &&
	        	(nm.getFrom().getParticipantType().equals(nm.getTo().getParticipantType()))){
	        	response.addError(SmsServiceMessages.CANNOT_SEND_MESSAGE_TO_YOURSELF);
	        	return response;
	        }
        }

        Thread t = null;
    	Message m = null;
    	Addressee a = null;

    	if(messageId == null || messageId.equals(0L)){
    		// this is a new draft message
    		// create a new  thread, message and addressee
    		t = new Thread();
    		m = new Message();
    		a = new Addressee();

    	}else{
    		// this is an existing draft with possible changes
    		// fetch the existing draft message, thread, addressee
    		m = messageDao.findById(messageId);

    		//make sure the message exists
    		if(m == null){
    			response.addError(SmsServiceMessages.MESSAGE_NOT_FOUND);
    			response.setPayload(null);
    			if(log.isWarnEnabled()){
    				log.warn("message not found: " + messageId);
    				log.warn("attempted editing of message: " + messageId);
    			}
    			return response;
    		}

    		// make sure the senders are the same
    		if(!m.getSenderId().equals(nm.getFrom().getId())){
    			response.addError(SmsServiceMessages.MESSAGE_NOT_OWNED_BY_USER);
    			response.setPayload(null);
    			if(log.isWarnEnabled()){
    				log.warn("attempted editing of message " + messageId + " by user " +
    					nm.getFrom().getId() + " who is not the owner of the message");
    			}
    			return response;
    		}

    		// make sure this message has not been sent
    		if (m.getSentDate() != null){
    			response.addError(SmsServiceMessages.MESSAGE_NOT_EDITABLE);
    			response.setPayload(null);
    			if(log.isWarnEnabled()){
    				log.warn("attempted editing of sent message: " + messageId);
    			}
    			return response;
    		}
    		// there should only be a single addressee
    		if(m.getAddressees().size() > 1){
    			response.addError(SmsServiceMessages.DATA_INTEGRITY_ERROR);
    			if(log.isErrorEnabled()){
    				log.error("draft message: " + messageId + " has more than one addressee");
    			}
    			response.setPayload(null);
    			return response;
    		}

    		a = m.getAddressees().get(0);
    		t = m.getThread();
    	}

    	t.setMessageCategoryType(MessageCategoryTypeEnum.valueOf(nm.getMessageCategoryTypeId()));//6598
    	t.setSubject(StringEscapeUtils.escapeHtml(nm.getSubject()));
		t.setMailGroup(nm.getTriageGroup());
		threadDao.save(t);

		//!!!! IMPORTANT do not set - SENT_DATE
		m.setThread(t);
		m.setBody(nm.getBody());

		m.setSenderId(nm.getFrom().getId());
		m.setSenderName(nm.getFrom().getName());
		m.setSenderType(nm.getFrom().getParticipantType());
		if(nm.getTo() != null){
			m.setRecipientId(nm.getTo().getId());
			m.setRecipientName(nm.getTo().getName());
			m.setRecipientType(nm.getTo().getParticipantType());
		}else{
			m.setRecipientId(null);
			m.setRecipientName(null);
			m.setRecipientType(null);
		}
		m.setModifiedDate(new Date());
		messageDao.save(m);
		if(log.isDebugEnabled()){
			log.debug("Draft message id: " + m.getId());
		}
		a.setMessage(m);
		a.setFolderId(SystemFolderEnum.DRAFTS.getId());
		a.setOwner((User)nm.getFrom());
		if(a.getReadDate()==null){
			a.setReadDate(null);
		}
		a.setRole(AddresseeRoleEnum.SENDER);
		addresseeDao.save(a);

		response.setPayload(m);
		return response;

	}


    /**
     * The simplest way to send a draft is to pass the NewMessage
     * into the existing sendMessage function. If it returns successfully
     * then delete the draft message,thread,addressee.
     *
     * This approach prevents the duplication of all of the business
     * rules for sending messages just to accommodate drafts.
     *
     * @param nm        NewMessage that contains the contents of the message to send
     * @param messageId Id of the existing draft message.  It will be deleted
     *                  if the message is successfully sent
     * @return
     */
    public ServiceResponse<Message> sendDraft(NewMessage nm, Long messageId) {

    	ServiceResponse<Message> response = new ServiceResponse<Message>();
    	Message m = messageDao.findById(messageId);
    	Message replyTo = null;
    	boolean deleteThread = false;

    	//make sure the message exists
		if(m == null){
			response.addError(SmsServiceMessages.MESSAGE_NOT_FOUND);
			response.setPayload(null);
			if(log.isWarnEnabled()){
				log.warn("message not found: " + messageId);
				log.warn("attempted editing of message: " + messageId);
			}
			return response;
		}

		// make sure the senders are the same
		if(!m.getSenderId().equals(nm.getFrom().getId())){
			response.addError(SmsServiceMessages.MESSAGE_NOT_OWNED_BY_USER);
			response.setPayload(null);
			if(log.isWarnEnabled()){
				log.warn("attempted editing of message " + messageId + " by user " +
					nm.getFrom().getId() + " who is not the owner of the message");
			}
			return response;
		}

		// make sure this message has not been sent
		if (m.getSentDate() != null){
			response.addError(SmsServiceMessages.MESSAGE_NOT_EDITABLE);
			response.setPayload(null);
			if(log.isWarnEnabled()){
				log.warn("attempted editing of sent message: " + messageId);
			}
			return response;
		}

		// see if this is the only message on the thread
		// if it is the thread will be deleted when the message is
		// sent as a new message.  if not then the thread will
		// be kept, the message will be sent as a reply
		if(m.getThread().getMessages().size() > 1){
			deleteThread = false;

			// sort the messages by date and use the last message
			// --that is not a draft-- as the reply
			Collections.sort(m.getThread().getMessages(), MESSAGE_SORTER);
			for(int i = m.getThread().getMessages().size() -1; i > -1; i--){
				replyTo = m.getThread().getMessages().get(i);
				if(replyTo.getSentDate() != null) break;
			}

			if(replyTo == null){
				if(log.isErrorEnabled()){
					log.error("Houston we have a problem.  This was supposed to be a reply but can't find a message to reply to!!!");
					log.error("Attempting to send the message as a new message");
				}
			}

		}else{
			deleteThread = true;
		}

		nm.setBody(StringEscapeUtils.unescapeHtml(nm.getBody()));
		nm.setSubject(StringEscapeUtils.unescapeHtml(nm.getSubject()));
		if(replyTo == null)
			response = send(nm);
		else
			response = reply(replyTo, nm);

    	if(response.getPayload() == null){
    		return response;
    	}

    	Thread t = m.getThread();
    	Addressee a = m.getAddressees().get(0);

    	addresseeDao.delete(a.getId());
    	messageDao.delete(m.getId());

    	if(deleteThread){
    		threadDao.delete(t.getId());
    	}

    	return response;
    }


    public ServiceResponse<NewMessage>draftToNewMessage(User u, Long messageId){

    	Precondition.assertNotNull("user", u);
    	Precondition.assertPositive("messageId", messageId);

    	ServiceResponse<NewMessage> response = new ServiceResponse<NewMessage>();
    	NewMessage nm = new NewMessage();
    	nm.setDraft(true);

    	Message m = messageDao.findById(messageId);

    	if(m == null){
    		response.setPayload(new NewMessage());
    		response.addError(SmsServiceMessages.MESSAGE_NOT_FOUND);
    		return response;
    	}

    	// make sure the senders are the same
		if(!m.getSenderId().equals(u.getId())){
			response.addError(SmsServiceMessages.MESSAGE_NOT_OWNED_BY_USER);
			response.setPayload(nm);
			if(log.isWarnEnabled()){
				log.warn("attempted editing of message " + messageId + " by user " +
					 u.getId() + " who is not the owner of the message");
			}
			return response;
		}

		// make sure this message has not been sent
		if (m.getSentDate() != null){
			response.addError(SmsServiceMessages.MESSAGE_NOT_EDITABLE);
			response.setPayload(nm);
			if(log.isWarnEnabled()){
				log.warn("attempted editing of sent message: " + messageId);
			}
			return response;
		}
		nm.setMessageCategoryTypeId(m.getThread().getMessageCategoryType().getId());
    	nm.setSubject(m.getThread().getSubject());
    	//log.info("The body: " + m.getBody());
    	nm.setBody(m.getBody());
    	nm.setFrom(u);
//    	6598 KJG
    	// drafts may not have a recipient defined yet
    	if(m.getRecipientType() != null){
	    	switch(m.getRecipientType()){
	    	  case PATIENT:
	    	  case CLINICIAN:
	    		  nm.setTo(userDao.findById(m.getRecipientId()));
	    		  break;
	    	  case TRIAGE_GROUP:
	    		  nm.setTo(triageGroupDao.findById(m.getRecipientId()));
	    		  break;
	    	  case DISTRIBUTION_GROUP:
	    		  nm.setTo(distributionGroupDao.findById(m.getRecipientId()));
	    		  break;
	    	}
    	}
    	nm.setTriageGroup(m.getThread().getMailGroup());
    	response.setPayload(nm);
    	return response;

    }


    public CollectionServiceResponse<TriageGroup> getTriageGroupsForStation(String station){

    	Precondition.assertNotBlank("station", station);

    	CollectionServiceResponse<TriageGroup> response = new
    		CollectionServiceResponse<TriageGroup>();

    	response.setCollection(triageGroupDao.getTriageGroupsForStation(station));
    	return response;

    }


    public CollectionServiceResponse<TriageGroup> getTriageGroupsAllActive(){
    	CollectionServiceResponse<TriageGroup> response = new
		CollectionServiceResponse<TriageGroup>();

    	response.setCollection(triageGroupDao.getTriageGroupsAllActive());
    	return response;
    }

    private Message sendFromClinician(NewMessage nm) throws SendMessageException{

    	Message m = null;

    	switch(nm.getTo().getParticipantType()){

	    	case PATIENT:{
	    		if (nm.getTriageGroup() == null) {
	    			throw new SendMessageException(SmsServiceMessages.TRIAGE_GROUP_NOT_DEFINED);
	    	    }
	    		m = sendToUser(nm, null);
	    		break;
	    	}
	    	case CLINICIAN:{
	    		Clinician f = (Clinician)nm.getFrom();
	    		Clinician t = (Clinician)nm.getTo();
	    		if(t.equals(f)){
	    			throw new SendMessageException(SmsServiceMessages.CANNOT_SEND_MESSAGE_TO_YOURSELF);
	    		}
	    		m = sendToUser(nm, null);
	    		break;
	    	}
	    	case DISTRIBUTION_GROUP:{
					m = sendToDistributionGroup(nm);
				break;
	    	}
	    	case TRIAGE_GROUP:{
	    		m = clinicianSendToTriage(nm);
	    		break;
	    	}
	    }

    	return m;

    }


    /**
     * Send a message to an individual.
     *
     * @param nm   the NewMessage being sent
     * @param orig the Message being replied to.  May be
     *             <code>null</code> if this is a new message.
     * @return
     */
    private Message sendToUser(NewMessage nm, Message orig){
    	Message m = null;
    	if(orig == null){
    		m = createMessage(nm, null);
    	}else{
    		m = createMessage(nm, orig.getThread());
    	}

        // address the message to the sender
        addressMessage(m, (User)nm.getFrom(), SystemFolderEnum.SENT, true,false);

        // address the message to the recipient
        addressMessage(m, (User)nm.getTo(), SystemFolderEnum.INBOX, true,false);

        if(nm.getCc()!=null){
        	addressMessage(m, (User)nm.getCc(), SystemFolderEnum.INBOX, true,true);
         }
        return m;
    }


    /**
     * Address a message from a patient to a TriageGroup.
     * Only one message thread is created.  All members of
     * the TriageGroup get the message.
     *
     * @param nm   the NewMessage being sent
     * @param orig the Message being replied to.  May be
     *             <code>null</code> if this is a new message.
     * @return
     */

    private Message patientSendToTriage(NewMessage nm, Message orig) {

    	Message m = null;

    	try
    	{
	    	if(orig == null){
	    		nm.setTriageGroup((TriageGroup)nm.getTo());
	    		m = createMessage(nm, null);
	    	}else{
	    		m = createMessage(nm, orig.getThread());
	    	}
	    	if(m!=null){
	    		if(log.isInfoEnabled()){
	    			log.info("patientSendToTriage====>Message Created====>"+m.getId());
	    		}
	    	}
    	}catch(Exception e2){
    		if(log.isErrorEnabled()){
    			log.error("Error While Create New Message...."+e2.getMessage());
    		}
    	}

        // address the message to the sender
        addressMessage(m, (Patient)nm.getFrom(), SystemFolderEnum.SENT, true,false);
        if(log.isInfoEnabled()){
        	log.info("SendMessageServiceImpl===Addressee Created for SENDER==============>");
        }


        // address the message to the group
        TriageGroup tr = (TriageGroup)nm.getTo();
        triageGroupDao.getCliniciansForTriageGroup(tr);
        if(log.isInfoEnabled()){
        	log.info("SendMessageServiceImpl===Creating Addressee for TRIAGE MEMBER==============>");
        }

        for (Clinician c : tr.getClinicians()) {
            addressMessage(m, c, SystemFolderEnum.INBOX, true,false);
        }

        return m;
    }


    /**
     * Send a message to the Clinicians of a TriageGroup.  This can
     * only be used for <b>new</b> messages.  In this
     * method a separate thread is created for each member of the
     * TriageGroup.  Essentially creating a "mass mailing" affect.
     * All replies will be clinician->clinician only.
     *
     * @param nm the NewMessage to send
     * @return
     */
    private Message clinicianSendToTriage(NewMessage nm) {

    	Message m = null;
    	TriageGroup tr = (TriageGroup)nm.getTo();
        triageGroupDao.getCliniciansForTriageGroup(tr);

        nm.setTriageGroup(null);
        Clinician from = (Clinician)nm.getFrom();

        for (Clinician c : tr.getClinicians()) {
        	// don't send it to the sender; causes constraint errors
        	if(from.equals(c)) continue;
        	m = createMessage(nm, null);
        	// address the message to the sender
            addressMessage(m, from, SystemFolderEnum.SENT, true,false);
            // address the message to the group member
            addressMessage(m, c, SystemFolderEnum.INBOX, true,false);
        }

        return m;
    }



    /**
     * Creates a new message thread for every member of the
     * DistributionGroup.  This can only be used for new messages.
     *
     * The TriageGroup must be defined if any member of the
     * DistributionGroup is a Patient.
     *
     * @param nm the NewMessage to send
     * @return
     */
    private Message sendToDistributionGroup(NewMessage nm) throws SendMessageException {

    	Message m = null;
    	DistributionGroup to = (DistributionGroup)nm.getTo();


    	DistributionGroup distGroup = distributionGroupDao.findById(to.getId());
    	distributionGroupDao.getMembersForDistributionGroup(distGroup);

    	// if there are no members in the group then fail now
    	if(distGroup.getMembers() == null || distGroup.getMembers().size() == 0){
    		throw new SendMessageException(SmsServiceMessages.DISTRIBUTION_GROUP_HAS_NO_MEMBERS);
    	}

    	// if the TriageGroup is null then the message
    	// cannot be sent to any patients

    	if (nm.getTriageGroup() == null) {
    		for(User u : distGroup.getMembers()){
    			if(u.getUserType() == UserTypeEnum.PATIENT){
    				throw new SendMessageException(SmsServiceMessages.TRIAGE_GROUP_NOT_DEFINED);
    			}
    		}
        }


    	CollectionServiceResponse blockedPatientResponse = patientBlockedService.getBlockedPatientsByTriageGroupId(nm.getTriageGroupId());
    	Collection<PatientBlockedTriageGroup> blockedPatientsList =  blockedPatientResponse.getCollection();
    	List blockedPatientIdsList = new ArrayList();
    	for(PatientBlockedTriageGroup patBlockedGroup:blockedPatientsList){
    		blockedPatientIdsList.add(patBlockedGroup.getPatientId());
    	}

    	// Now patients can be blocked from facility - we need to add patients blocked by facility -
    	if (nm.getTriageGroupId() != null) {
			TriageGroup tGroup = triageGroupDao.findById(nm.getTriageGroupId());
			if(tGroup!=null) {
				Long stationNumber = Long.valueOf(tGroup.getVistaDiv()).longValue();

				CollectionServiceResponse<PatientBlockedFacility> collectionResponse = patientBlockedService.getBlockedFacilityPatientsByStation(stationNumber);
				Collection<PatientBlockedFacility> patBlockedFacilities = collectionResponse.getCollection();
				for(PatientBlockedFacility patBlockedFacility:patBlockedFacilities){
					blockedPatientIdsList.add(patBlockedFacility.getPatientId());
				}
			}
		}


    	if(log.isInfoEnabled()){
    		log.info(">>>>>>>>SENT TO DISTRIBUTION GROUP->List of Patients Blocked from Selected Tg or Facility:"+nm.getTriageGroupId()+" Patients->"+blockedPatientIdsList);
    	}
    	Long timeInMills = Calendar.getInstance().getTimeInMillis();

        for(User u : distGroup.getMembers()){
        	boolean isActive=false;

        	if(u.getUserType().equals(UserTypeEnum.CLINICIAN)){
        		isActive=true;
        	}else{
        		if(!blockedPatientIdsList.contains(u.getId())){
        			isActive=true;
        		}
        	}
        	nm.setChecksumDateInMills(timeInMills);
        	if(isActive){
        		m = createMessage(nm, null);
            	// address the message to the sender
                // address to the group member
            	addressMessage(m, u, SystemFolderEnum.INBOX, true,false);

        	}
        }
		Message senderMessage = createMessage(nm, null);
        addressMessage(senderMessage, (Clinician)nm.getFrom(), SystemFolderEnum.SENT, true,false);
    	return senderMessage;
    }


    /**
     * Send a message to a particular user.  This may be called numerous
     * times (at least twice) for each message, once for the sender and
     * at least once for the recipient.  Notifications are sent as
     * protocol and user preferences dictate.
     *
     * Also manages surrogates.  If a user has a surrogate set then the
     * message will also be sent to the surrogate.
     *
     * Most external callers of this method should set sendToSurrogate=TRUE.
     * Internally we will only send to one generation of surrogates and not follow
     * the chain any further.
     *
     * @param m               the Message that is being addressed
     * @param u               the User that is being addressed
     * @param sendToSurrogate if true, message will be addressed to surrogate if specified in user object
     *                        if false, surrogate will be ignored.
     * @param folder SystemFolderEnum identifying message's location
     */
    public void addressMessage(Message m, User u, SystemFolderEnum folder, boolean sendToSurrogate,boolean ccMessage) {

    	if(u.getStatus() == UserStatusEnum.OPT_OUT){
    		if(log.isWarnEnabled()){
    			log.warn("Attempted to address a message to an OPT_OUT user: " + u.getName());
    		}
    		return;
    	}

    	if(log.isInfoEnabled()){
    		log.info("addressMessage to " + u.getName() + " into folder " + folder.getName());
    	}
    	boolean addresseeFlag=false;
    	// checks to make sure the user doesn't already have this message
    	if(addresseeDao.getAddresseeForUser(u.getId(), m.getId()) == null){
	        Addressee add = new Addressee();
	        try{


	        add.setMessage(m);
	        add.setOwner(u);
	        add.setFolderId(folder.getId());
	        if(folder == SystemFolderEnum.INBOX){
	        	if(ccMessage){
	        		add.setRole(AddresseeRoleEnum.CC);
	        	}else{
	        		add.setRole(AddresseeRoleEnum.RECIPIENT);
	        	}
	        }else{
	        	add.setReadDate(new Date());
	        	add.setRole(AddresseeRoleEnum.SENDER);
	        }
	        Addressee adse = addresseeDao.save(add);
	        addresseeFlag=true;
	        if(log.isInfoEnabled()){
	        	log.info("SendMessageServiceImpl====>Addressee Created Successfully for messageId-->"+m.getId()+" addresseeId:"+add.getId());
	        }
	        }catch(Exception addEx){
	        	loggingService.messageSentError(m.getSenderId(),m.getId(),"Error While Create Addressee|RecipientId"+m.getRecipientId()+"Error",true);
	        	if(log.isErrorEnabled()){
	        		log.error("SendMessageServiceImpl====>Error Occured While Creating Addressee for message-->"+m.getId()+"|"+addEx.getMessage());
	        	}
	        }
    	}
    	
        if (folder == SystemFolderEnum.INBOX) {
        	if(m.getId()!=null && addresseeFlag){
        		sendNotification(u,m.getThread().getMailGroup(),m,!sendToSurrogate);
        	}
        	else{
        		if(log.isInfoEnabled()){
        			log.info("SendMessageServiceImpl=====>Message or Addressee is null");
        		}

        	}
        }

        // only process surrogates for INBOX messages (or messages sent to the user)
        if(folder == SystemFolderEnum.INBOX){
        	// check to see if we are following surrogates and that the user is a clinician
	        if(sendToSurrogate == true && u.getUserType() == UserTypeEnum.CLINICIAN){
	          // TODO If there are any surrogate for this clinician in SURROGATE table, get the surrogate user Id and
	        	// send to addressee.
	        	Clinician c = userDao.findClinicianById(u.getId());
	        	Surrogate surrogate = surrogateDao.getCurrentSurrogateFor(c);
	        	if(surrogate!=null){
	        		if(surrogate.getSurrogateType()==ParticipantTypeEnum.CLINICIAN){
	        			Clinician x = userDao.findClinicianById(surrogate.getSurrogateId());
	        			// Forward/Send an email to surrogate once Clinician receives the message
	        			if(ccMessage){
	        				surrogateCcMessage(m, x);
	        			}else{
	        				addressMessage(m, x, SystemFolderEnum.INBOX, false,false);
	        			}
	        		}else if(surrogate.getSurrogateType()==ParticipantTypeEnum.TRIAGE_GROUP){
	        			TriageGroup tg = triageGroupDao.findById(surrogate.getSurrogateId());
	        			triageGroupDao.getCliniciansForTriageGroup(tg);
	        			for(Clinician x : tg.getClinicians()){
		        			addressMessage(m, x, SystemFolderEnum.INBOX, false,false);
		        		}
	        		}
	        	}
	         }
	    }
    }
    
    public void surrogateCcMessage(Message m, User u) {

    	if(u.getStatus() == UserStatusEnum.OPT_OUT){
    		if(log.isWarnEnabled()){
    			log.warn("Attempted to address a message to an OPT_OUT user: " + u.getName());
    		}
    		return;
    	}

    	boolean addresseeFlag=false;
    	// checks to make sure the user doesn't already have this message
    	if(addresseeDao.getAddresseeForUser(u.getId(), m.getId()) == null){
	        Addressee add = new Addressee();
	        try{
		        add.setMessage(m);
		        add.setOwner(u);
		        add.setFolderId(SystemFolderEnum.INBOX.getId());
		        add.setRole(AddresseeRoleEnum.CC_SURROGATE);
		        Addressee adse = addresseeDao.save(add);
		        addresseeFlag=true;
		     }catch(Exception addEx){
		        	if(log.isErrorEnabled()){
		        		log.error("SendMessageServiceImpl====>Error Occured While Creating CC Surrogate message-->"+m.getId()+"|"+addEx.getMessage());
		        	}
		    }
    	}
    	if(m.getId()!=null && addresseeFlag){
    		sendNotification(u,m.getThread().getMailGroup(),m,false);
    	}
    }

    private void sendNotification(User u, TriageGroup tg, Message message,boolean sendToSurrogate){

    	if (StringUtils.isBlank(u.getEmail())) {
    		if(log.isWarnEnabled()){
    			log.warn(u.getId() + " does not have an email address.");
    		}
            return;
        }

    	EmailValidator emailV = EmailValidator.getInstance();
    	if(!emailV.isValid(u.getEmail())){
    		if(log.isWarnEnabled()){
    			log.warn(u.getId() + " does not have a valid email address.");
    		}
       		return;
    	}

		/*
		*  Story 151054: Send email notification to Surrogate irrespective of main SM Clinicians preferences (Once daily, Each message, On_Assignment)
		*/
    	/*
    	if(sendToSurrogate){
    		emailService.sendNewMessageEmailNotification(u,tg,message);

			if(log.isInfoEnabled()){
				log.info("Send email notification for SM Clinician delegate: "+u.getId());
			}

    	}
    	else{
*/

    	//If the Recipient is Clinician apply the differnt business rule(Each Message, One_Daily, On_Assignment)
    	if((u.getParticipantType().equals(ParticipantTypeEnum.CLINICIAN)) || (sendToSurrogate))
    	{
    		 if(log.isInfoEnabled()){
    			 log.info("Before triggering Email to Clinician:  "+"SenderId:"+message.getSenderId()+" RecipientId: "+u.getId()+"  email: "+u.getEmail()+"  Message Id: "+message.getId());
    		 }
    		if(u.getEmailNotification().equals(EmailNotificationEnum.EACH_MESSAGE)){
    			if(log.isInfoEnabled()){
    				log.info("user:"+u.getId()+"-->Preferences------> EACH_MESSAGE");
    			}
    			emailService.sendNewMessageEmailNotification(u,tg,message);
    			return;
    		}
    		else if(u.getEmailNotification().equals(EmailNotificationEnum.ONE_DAILY))
    		{
    			if(log.isInfoEnabled()){
    				log.info("user:"+u.getId()+"-->Preferences------> ONE_DAILY--->Last Email Notification-------->"+u.getLastNotification());
    			}
    			//If users already got notification for today then don't send it
    			if (!DateUtils.isToday(u.getLastNotification()))
                {
        			emailService.sendNewMessageEmailNotification(u,tg,message);
    				return;
                }

    		}
    		else if(u.getEmailNotification().equals(EmailNotificationEnum.ON_ASSIGNEMNT))
    		{
    			if(log.isInfoEnabled()){
    				log.info("user:"+u.getId()+"-->Preferences------> ON_ASSIGNMENT");
    			}

    			if(message.getSenderType().equals(ParticipantTypeEnum.CLINICIAN) && message.getRecipientType().equals(ParticipantTypeEnum.CLINICIAN))
    			{
    				emailService.sendNewMessageEmailNotification(u,tg,message);
    				return;
    			}
    		}
    	}
    	// If the Recipient is Patient apply the business rules for (One Daily,
    	else if(u.getParticipantType().equals(ParticipantTypeEnum.PATIENT))
    	{
    		 if(log.isInfoEnabled()){
    			 log.info("Before triggering Email to Patient:  "+"SenderId:"+message.getSenderId()+" RecipientId: "+u.getId()+"  email: "+u.getEmail()+"  Message Id: "+message.getId());
    		 }

   		    if(u.getEmailNotification().equals(EmailNotificationEnum.ONE_DAILY))
	   		{
	   			//If users already got notification for today then don't send it
	   			if (!DateUtils.isToday(u.getLastNotification()))
	            {
	       			emailService.sendNewMessageEmailNotification(u,tg,message);
	   				return;
	            }
	   		}
	   		else if(u.getEmailNotification().equals(EmailNotificationEnum.EACH_MESSAGE))
	   		{
	   				emailService.sendNewMessageEmailNotification(u,tg,message);
	   				return;
	   		}
	   	}
    	//}
   }

    /**
     * Create a new Message
     *
     * @param nm the NewMessage to create
     * @param t  existing Thread to which this Message is being added.
     *           If Thread is <code>null</code> a new thread will be
     *           created.
     * @return   a new Message
     */
    private Message createMessage(NewMessage nm, Thread t) throws SendMessageException{

    	if (t==null)
    	{
    		try{

    		if (StringUtils.isBlank(nm.getSubject()))
    		{
        		throw new SendMessageException(SmsServiceMessages.SUBJECT_MUST_BE_DEFINED);
            }
            t = new Thread();
            t.setMessageCategoryType(MessageCategoryTypeEnum.valueOf(nm.getMessageCategoryTypeId()));//6598
            t.setSubject(StringEscapeUtils.escapeHtml(nm.getSubject()));
            t.setMailGroup(nm.getTriageGroup());
            threadDao.save(t);
            if(t!=null)
            {
            	if(log.isInfoEnabled()){
            		log.info("SendMessageServiceImpl==>New Thread Created for Sender==>"+nm.getFrom().getId()+" Thread Id:"+t.getId()+" Time:"+new Date());
            	}
            }
    		}catch(Exception e2){
    			StringBuffer logError = new StringBuffer();
    			logError.append("Error While Creating Thread: Subject:").append(nm.getSubject());
    			logError.append(" Category:").append(nm.getMessageCategoryTypeId()).append("Error");
    			loggingService.messageSentError(nm.getFrom().getId(),nm.getFrom().getId(),logError.toString(),true);
    			if(log.isErrorEnabled()){
    				log.error("SendMessageServiceImpl===>Error While Creating New Thread....."+e2.getMessage());
    			}

    		}

    	}
    	Message m = new Message();
    	try
    	{
	        m.setThread(t);
	        m.setBody(StringEscapeUtils.escapeHtml(nm.getBody()));
        	m.setChecksum(ChecksumUtils.calculateCRC32(nm.getBody())+(nm.getChecksumDateInMills()!=null?"-"+nm.getChecksumDateInMills():""));
	        m.setSenderId(nm.getFrom().getId());
	        m.setSenderName(nm.getFrom().getName());
	        m.setSenderType(nm.getFrom().getParticipantType());
	        m.setRecipientId(nm.getTo().getId());
	        m.setRecipientName(nm.getTo().getName());
	        m.setRecipientType(nm.getTo().getParticipantType());
	        m.setSentDate(new Date());
	        if(nm.getAttachments()!=null && nm.getAttachments().size()!=0){
	        	m.setAttachment(Boolean.TRUE);
	        	try{
		        	Map<Long, String> map = nm.getAttachments();
		        	int i=1;
		        	for(Map.Entry<Long,String> entry : map.entrySet()){
							Long attachmentKey =entry.getKey();
							if(i==1) m.setAttachmentId(attachmentKey);
							if(i==2) m.setAttachmentId2(attachmentKey);
							if(i==3) m.setAttachmentId3(attachmentKey);
							if(i==4) m.setAttachmentId4(attachmentKey);
							i++;
		        	}
	        	}catch(Exception e1){
	        		if(log.isErrorEnabled()){
	        			log.error("Error while setting attachments reference with Message..."+e1);
	        		}
	        	}
	        }

	        // if clinician, store the local time
	        if(nm.getFrom().getParticipantType() == ParticipantTypeEnum.CLINICIAN){
	        	String stationNo = ((Clinician)nm.getFrom()).getStationNo();
	        	StationTimezone st = stationTimezoneDao.getTimezoneForStation(stationNo);

	        	if(st == null || StringUtils.isBlank(st.getTimezone())){
	        		if(log.isWarnEnabled()){
	        			log.warn("Station: " + stationNo + " does not have a timezone mapping.  " +
	        				"Setting local date to appserver date.");
	        		}
	        		m.setSentDateLocal(new Date());
	        	}else{
	        		try{
	        			m.setSentDateLocal(DateUtils.convertDateToTimeZone(new Date(), st.getTimezone()));
	        		}catch(Exception e){
	        			log.warn("error converting date for station: " + stationNo + ".  Timezone: " +
	        					st.getTimezone() + ".  Setting local date to appserver date.");
	        			m.setSentDateLocal(new Date());
	        		}
	        	}


	        }else{
	        	m.setSentDateLocal(new Date());
	        }
	        m.setModifiedDate(new Date());
	        messageDao.save(m);
	        if(m.getSenderId().equals(m.getRecipientId())){
	    		if(log.isInfoEnabled()){
	    			log.error("Send=700:SenderAndRecipient are SAME"+m.getId()+"^"+m.getSenderId()+"^"+m.getRecipientId());
	    		}
	        }
        	StringBuffer logMessage = new StringBuffer();
        	logMessage.append("SendMessageServiceImpl====>New Message Created=====for Sender:").append(m.getSenderId());
        	logMessage.append(" Recipient:").append(m.getRecipientId()).append(" MessageId:").append(m.getId());
        	logMessage.append(" Time:"+new Date());
        	if(log.isInfoEnabled()){
        		log.info(logMessage.toString());
        	}

    	}catch(Exception me)
    	{
    		loggingService.messageSentError(m.getSenderId(),m.getRecipientId(),"Error While New Message|Group:"+m.getRecipientName()+"Error",true);
    		if(log.isErrorEnabled()){
    			log.error("SendMessageServiceImpl==>Error While Creating New Message......"+me.getMessage());
    		}


    	}
    		return m;
    }

    /**
     * Try to devine who the new message should be sent to.
     *
     * @param sender User sending the new message
     * @param orig   the Message being replied to
     * @return       the MailParticipant that should receive the message
     */
    private MailParticipant devineRecipient(Clinician sender, Message orig){

    	// populate the Addressees
    	//messageDao.fillAddressees(orig);

    	// who sent the original message?
    	User u = null;
    	for(Addressee a : orig.getAddressees()){
    		if(a.getRole() == AddresseeRoleEnum.SENDER){
    			u = a.getOwner();
    			break;
    		}
    	}

    	if(u.getUserType() == UserTypeEnum.PATIENT){
    		return u;
    	}

    	// what do we do if the user is replying to a
    	// message he sent
    	if(sender == u){
    		// OK the current sender sent the orig.
    		// Assume that he wants to just send more
    		// information to the recipient of the original

    		// check to see if there is more than one recipient
    		// of the original message.  If so PUNT
    		if(orig.getAddressees().size() > 2){
    			// there were more than one recipient
    			return null;
    		}

    		for(Addressee a : orig.getAddressees()){
        		if(a.getRole() == AddresseeRoleEnum.RECIPIENT){
        			u = a.getOwner();
        			break;
        		}
        	}

    	}

    	return u;
    }

    public ServiceResponse<Message> replyFromSentFolder(Message orig, NewMessage nm) {

    	Message m = null;
         ServiceResponse<Message> response = new ServiceResponse<Message>();

         // fetch the complete message from the database
         orig = messageDao.getMessageComplete(orig.getId());

         // a clinician cannot reply to a message that has already been completed
         if(nm.getFrom().getParticipantType()== ParticipantTypeEnum.CLINICIAN &&
            orig.getStatus() == ClinicianStatusEnum.COMPLETE){
         	response.addError(SmsServiceMessages.MESSAGE_ALREADY_COMPLETE);
         	response.setPayload(null);
         	return response;
         }


         // the thread should already be populated but just to be sure
         if(orig.getThread() == null){
         	messageDao.fillThread(orig);
         }
         
         
         switch(nm.getFrom().getParticipantType()){
             case PATIENT:{
 	        	if(orig.getThread().getMailGroup() == null){
 	        		throw new IllegalStateException("Triage Group not defined for thread.");
 	        	}
 	        	// set the triage group into the new message object
 	        	nm.setTo(orig.getThread().getMailGroup());
 	        	m = patientSendToTriage(nm, orig);
 				break;
 	        }
 	        case CLINICIAN:{
 	        	User user = userDao.findById(orig.getRecipientId());
 	        	try {
 	        		if(orig.getRecipientType().getId().equals(ParticipantTypeEnum.TRIAGE_GROUP.getId())){
 	        			nm.setTo(orig.getThread().getMailGroup());
 	        			m = clinicianReplyToTriage(nm,orig);
 	        			if(log.isInfoEnabled()){
 	        				log.info("Reply message Sent to Triage Group"+m.getId());
 	        			}
 	        			break;
 	        		}
 	        		if(orig.getRecipientType().getId().equals(ParticipantTypeEnum.DISTRIBUTION_GROUP.getId()))
 	        		{
 	        			DistributionGroup dg = distributionGroupDao.findById(orig.getRecipientId());
 	        			nm.setTo(dg);
 	        			m = clinicianReplyToDistributionGroup(nm,orig);
 	        			if(log.isInfoEnabled()){
 	        				log.info("Reply message Sent to Dist. Group"+m.getId());
 	        			}
 	        			break;
 	            	}
 	        		else
 	        		{
 	        			nm.setTo(user);
 	        			m = sendToUser(nm, orig);
 	        		}
 	        		// set the "assignedTo" on the message being replied to
 	        		// if the assignedTo is not set already
 	        		Message reply = messageDao.findById(orig.getId());
 	        		if(reply.getAssignedTo() == null){
 	        			reply.setAssignedTo((Clinician)nm.getFrom());
 	        			reply.setModifiedDate(new Date());
 	        			messageDao.save(reply);
 	        		}
 	        		
 				} catch (SendMessageException e) {
 					response.addError(e.getMsgkey());
 	        		return response;
 				}
 				break;
 	        }
 	        default:{
 	        	response.addError(SmsServiceMessages.MESSAGE_SENDER_NOT_PAITENT_OR_CLINICIAN);
         		return response;
 	        }
         }
         try{
	         Addressee a = orig.getAddressees().get(0);
	         if(a!=null && a.getFolderId().equals(SystemFolderEnum.DRAFTS.getId())){
	        	 addresseeDao.delete(a.getId());
	        	 messageDao.delete(orig.getId());
	         }
         }catch(Exception e2){
        	 if(log.isInfoEnabled()){
        		 log.info("An Error Occured while deleting the message!!!"+orig.getId());
        	 }
         }
         
         loggingService.sendMessage((User)nm.getFrom(), m,
         		"Recipient:" + nm.getTo().getParticipantType().getId() + "^" + nm.getTo().getId() + "^" + nm.getTo().getName(),
         		!response.getMessages().hasErrorMessages());
         response.setPayload(m);
         return response;

     }

 /* New Method has been Implemented for message send from clinician to Triage Group from Sent Folder */

    private Message clinicianReplyToTriage(NewMessage nm, Message orig) {

    	Message m = null;

    	if(orig == null)
    	{
    		m = createMessage(nm, null);
    	}
    	else
    	{
    		m = createMessage(nm, orig.getThread());
    	}
   		nm.setTriageGroup((TriageGroup)nm.getTo());


   		Clinician from = (Clinician)nm.getFrom();
        // address the message to the sender
        addressMessage(m, from, SystemFolderEnum.SENT, true,false);

        // address the message to the group
        TriageGroup tr = (TriageGroup)nm.getTo();
        triageGroupDao.getCliniciansForTriageGroup(tr);

        for (Clinician c : tr.getClinicians()) {
           	if(from.equals(c)) continue;
            addressMessage(m, c, SystemFolderEnum.INBOX, true,false);
        }
        return m;
    }


  private Message clinicianReplyToDistributionGroup(NewMessage nm, Message orig) throws SendMessageException {

    	Message m = null;
    	if(orig == null){
    		m = createMessage(nm,null);
    	}
    	else{
    		m = createMessage(nm,orig.getThread());
    	}

    	DistributionGroup to = (DistributionGroup)nm.getTo();
    	distributionGroupDao.getMembersForDistributionGroup(to);

    	// if there are no members in the group then fail now
    	if(to.getMembers() == null || to.getMembers().size() == 0){
    		throw new SendMessageException(SmsServiceMessages.DISTRIBUTION_GROUP_HAS_NO_MEMBERS);
    	}

    	// if the TriageGroup is null then the message
    	// cannot be sent to any patients
    	if (nm.getTriageGroup() == null) {
    		for(User u : to.getMembers()){
    			if(u.getUserType() == UserTypeEnum.PATIENT){
    				throw new SendMessageException(SmsServiceMessages.TRIAGE_GROUP_NOT_DEFINED);
    			}
    		}
        }
        for(User u : to.getMembers()){
        	m = createMessage(nm, null);
        	// address the message to the sender
            addressMessage(m, (Clinician)nm.getFrom(), SystemFolderEnum.SENT, true,false);
            // address to the group member
        	addressMessage(m, u, SystemFolderEnum.INBOX, true,false);
        }
        if(log.isInfoEnabled()){
        	log.info("Message Sent successfully.......");
        }

    	return m;
    }



	public ThreadDao getThreadDao() {
		return threadDao;
	}
	public void setThreadDao(ThreadDao threadDao) {
		this.threadDao = threadDao;
	}
	public AddresseeDao getAddresseeDao() {
		return addresseeDao;
	}
	public void setAddresseeDao(AddresseeDao addresseeDao) {
		this.addresseeDao = addresseeDao;
	}
	public MessageDao getMessageDao() {
		return messageDao;
	}
	public void setMessageDao(MessageDao messageDao) {
		this.messageDao = messageDao;
	}
	public TriageGroupDao getTriageGroupDao() {
		return triageGroupDao;
	}
	public void setTriageGroupDao(TriageGroupDao triageGroupDao) {
		this.triageGroupDao = triageGroupDao;
	}
	public DistributionGroupDao getDistributionGroupDao() {
		return distributionGroupDao;
	}
	public void setDistributionGroupDao(DistributionGroupDao distributionGroupDao) {
		this.distributionGroupDao = distributionGroupDao;
	}
	public EmailService getEmailService() {
		return emailService;
	}
	public void setEmailService(EmailService emailService) {
		this.emailService = emailService;
	}
	public UserDao getUserDao() {
		return userDao;
	}
	public void setUserDao(UserDao userDao) {
		this.userDao = userDao;
	}
	public StationTimezoneDao getStationTimezoneDao() {
		return stationTimezoneDao;
	}
	public void setStationTimezoneDao(StationTimezoneDao stationTimezoneDao) {
		this.stationTimezoneDao = stationTimezoneDao;
	}
	public LoggingService getLoggingService() {
		return loggingService;
	}
	public void setLoggingService(LoggingService loggingService) {
		this.loggingService = loggingService;
	}

	public MessageAttachmentDao getMessageAttachmentDao() {
		return messageAttachmentDao;
	}

	public void setMessageAttachmentDao(MessageAttachmentDao messageAttachmentDao) {
		this.messageAttachmentDao = messageAttachmentDao;
	}

	public SurrogateDao getSurrogateDao() {
		return surrogateDao;
	}

	public void setSurrogateDao(SurrogateDao surrogateDao) {
		this.surrogateDao = surrogateDao;
	}

	public PatientBlockedService getPatientBlockedService() {
		return patientBlockedService;
	}

	public void setPatientBlockedService(PatientBlockedService patientBlockedService) {
		this.patientBlockedService = patientBlockedService;
	}

}
