/**
 *
 */
package gov.va.med.mhv.sm.service.impl;

import gov.va.med.mhv.foundation.service.response.CollectionServiceResponse;

import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.foundation.util.Precondition;
import gov.va.med.mhv.sm.dao.ReportsDao;
import gov.va.med.mhv.sm.enumeration.ClinicalUserTypeEnum;
import gov.va.med.mhv.sm.enumeration.CountsSubCategoryEnum;
import gov.va.med.mhv.sm.enumeration.ListsFieldEnum;
import gov.va.med.mhv.sm.enumeration.ListsSubCategoryEnum;
import gov.va.med.mhv.sm.enumeration.ReportingFilterTypeEnum;
import gov.va.med.mhv.sm.enumeration.ReportingPeriodEnum;
import gov.va.med.mhv.sm.model.Facility;
import gov.va.med.mhv.sm.model.TriageGroup;
import gov.va.med.mhv.sm.model.report.CountsResults;
import gov.va.med.mhv.sm.model.report.ListsResults;
import gov.va.med.mhv.sm.model.report.ResultKey;
import gov.va.med.mhv.sm.service.FacilityService;
import gov.va.med.mhv.sm.service.ReportsService;
import gov.va.med.mhv.sm.service.TriageGroupService;
import gov.va.med.mhv.sm.util.DateUtils;
import gov.va.med.mhv.sm.util.PeriodsHelper;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class ReportsServiceImpl implements ReportsService {
	private static final Log log = LogFactory.getLog(ReportsServiceImpl.class);

	private ReportsDao reportsDao;
	private FacilityService facilityService;
	private TriageGroupService triageGroupService;

	public ReportsDao getReportsDao() {
		return reportsDao;
	}

	public void setReportsDao(ReportsDao reportsDao) {
		this.reportsDao = reportsDao;
	}

	public FacilityService getFacilityService() {
		return facilityService;
	}

	public void setFacilityService(FacilityService facilityService) {
		this.facilityService = facilityService;
	}

	public TriageGroupService getTriageGroupService() {
		return triageGroupService;
	}

	public void setTriageGroupService(TriageGroupService triageGroupService) {
		this.triageGroupService = triageGroupService;
	}

	public ServiceResponse<CountsResults> runCountsReport(Date start,
			Date end, ReportingPeriodEnum period,
			List<CountsSubCategoryEnum> subCategories,
			ReportingFilterTypeEnum filter, ResultKey mainId,
			List<ResultKey> keys) {
		Precondition.assertNotNull("start", start);
		Precondition.assertNotNull("end", end);
		Precondition.assertNotNull("period", period);
		Precondition.assertAfter("end date", end, start);

		ServiceResponse<CountsResults> response = new ServiceResponse<CountsResults>();

		CountsResults result = new CountsResults();
		result.setPeriod(period);
		result.setStart(start);
		result.setEnd(end);
		result.setSubCategories(subCategories);
		result.setPeriods(PeriodsHelper.create(start, end, period));

		if(log.isInfoEnabled()) {
			log.info("mainId is " + mainId);
		}

		// Iterate through all the teams for the teams for that facility
		// (For a single team selection it will only put one element)
		for (ResultKey key : keys) {
			if (log.isInfoEnabled()) {
				log.info("Adding unique Id of type "+ filter.getName()+ " "+ key.getKey() + "(" + key.getDescription() + ")");
			}
			String visnName = "";
			String facilityName = "";
			switch(filter) {
				case VISN:
				case ALL_VISNS:
					visnName = key.getDescription();
					break;
				case FACILITY:
				case ALL_FACILITIES:
					ServiceResponse<Facility> sr = facilityService.getFacilityByStationNumber(key.getKey());
					visnName = "VISN " + sr.getPayload().getVisn();
					facilityName=sr.getPayload().getStationNumber();
					break;
				case TEAM:
				case ALL_TEAMS:
					ServiceResponse<TriageGroup> tr = triageGroupService.findTriageGroupById(keys.get(0).getKeyLong());
					facilityName=tr.getPayload().getVistaDiv();
					ServiceResponse<Facility> fr = facilityService.getFacilityByStationNumber(tr.getPayload().getVistaDiv());
					visnName = "VISN " + fr.getPayload().getVisn();
					break;
			}


			result.addCountsResultsDetails(key.getKey(), key.getDescription(), visnName, facilityName);
		}

		boolean runUniquePatientSenders = false;
		boolean runUniqueTeamMemberSenders = false;
		boolean runOptedInPatients = false;
		boolean runBlockedPatients = false;
		boolean runOptedOutPatients = false;
		boolean runActionPendingPatients = false;
		boolean runVHLEducationCategory = false;

		List<Long> subCatsIntList = new ArrayList<Long>();
		for (CountsSubCategoryEnum subCategory : subCategories) {
			if(CountsSubCategoryEnum.UNIQUE_PATIENT_SENDERS.equals(subCategory)) {
				runUniquePatientSenders = true;
				//REMOVE the live data search from the snapshot set
				continue;
			}
			else if(CountsSubCategoryEnum.UNIQUE_TEAM_MEMBERS_SENDERS.equals(subCategory)) {
				runUniqueTeamMemberSenders = true;
				//REMOVE the live data search from the snapshot set
				continue;
			}
			else if(CountsSubCategoryEnum.OPTED_IN_PATIENTS.equals(subCategory)) {
				runOptedInPatients = true;
				continue;
			}
			else if(CountsSubCategoryEnum.BLOCKED_PATIENTS.equals(subCategory)) {
				runBlockedPatients = true;
				continue;
			}
			else if(CountsSubCategoryEnum.OPTED_OUT_PATIENTS.equals(subCategory)) {
				runOptedOutPatients = true;
				continue;
			}
			else if(CountsSubCategoryEnum.ACTION_PENDING_PATIENTS.equals(subCategory)) {
				runActionPendingPatients = true;
				continue;
			}
			else if(CountsSubCategoryEnum.MESSAGE_TYPES_EDUCATION.equals(subCategory)) {
				runVHLEducationCategory = true;
				continue;
			}
			subCatsIntList.add(new Long(subCategory.getId()));
		}

		List<Object[]> rawResults = new ArrayList<Object[]>();

		//If the only ones selected are live data don't run the snapshot
		if( subCatsIntList.size() > 0 ) {
			//Fetch all the snapshot data here:
			switch(filter) {
				case VISN:
					rawResults = getReportsDao().getCountReportsDataForOneVisn(start, end, mainId.getKeyLong(), period.getDbPeriod(),subCatsIntList);
					break;
				case ALL_VISNS:
					rawResults = getReportsDao().getCountReportsDataForAllVisns(start, end, period.getDbPeriod(),subCatsIntList);
					break;
				case FACILITY:
					rawResults = getReportsDao().getCountReportsDataForOneFacilityAndOneVisn(start, end, keys.get(0).getKeyLong(), mainId.getKeyLong(),period.getDbPeriod(),subCatsIntList);
					break;
				case ALL_FACILITIES:
					rawResults = getReportsDao().getCountReportsDataForAllFacilitiesByOneVisn(start, end, mainId.getKeyLong(),period.getDbPeriod(),subCatsIntList);
					break;
				case TEAM:
					rawResults = getReportsDao().getCountReportsDataForOneTeamAndOneFacility(start, end, mainId.getKeyLong(), keys.get(0).getKeyLong(),period.getDbPeriod(),subCatsIntList);
					break;
				case ALL_TEAMS:
					rawResults = getReportsDao().getCountReportsDataForAllTeamsByOneFacility(start, end, mainId.getKeyLong(),period.getDbPeriod(),subCatsIntList);
					break;
			}
		}

		//Fetch and merge all the "live" data searches here:
		//(We still pass in the list of subcategories as a placeholder if more are required.)
		if(runUniquePatientSenders) {
			switch(filter) {
				case VISN:
					rawResults.addAll(getReportsDao().getCountReportsDataUniquePatientSendersForOneVisn(start, end, mainId.getKeyLong(), period.getDbPeriod()));
					break;
				case ALL_VISNS:
					rawResults.addAll(getReportsDao().getCountReportsDataUniquePatientSendersForAllVisns(start, end, period.getDbPeriod()));
					break;
				case FACILITY:
					rawResults.addAll(getReportsDao().getCountReportsDataUniquePatientSendersForOneFacilityAndOneVisn(start, end, keys.get(0).getKeyLong(), mainId.getKeyLong(),period.getDbPeriod()));
					break;
				case ALL_FACILITIES:
					rawResults.addAll(getReportsDao().getCountReportsDataUniquePatientSendersForAllFacilitiesByOneVisn(start, end, mainId.getKeyLong(),period.getDbPeriod()));
					break;
				case TEAM:
					rawResults.addAll(getReportsDao().getCountReportsDataUniquePatientSendersForOneTeamAndOneFacility(start, end, mainId.getKeyLong(), keys.get(0).getKeyLong(),period.getDbPeriod()));
					break;
				case ALL_TEAMS:
					rawResults.addAll(getReportsDao().getCountReportsDataUniquePatientSendersForAllTeamsByOneFacility(start, end, mainId.getKeyLong(),period.getDbPeriod()));
					break;
			}
		}
		if(runUniqueTeamMemberSenders) {
			switch(filter) {
				case VISN:
					rawResults.addAll(getReportsDao().getCountReportsDataUniqueMemberSendersForOneVisn(start, end, mainId.getKeyLong(), period.getDbPeriod()));
					break;
				case ALL_VISNS:
					rawResults.addAll(getReportsDao().getCountReportsDataUniqueMemberSendersForAllVisns(start, end, period.getDbPeriod()));
					break;
				case FACILITY:
					rawResults.addAll(getReportsDao().getCountReportsDataUniqueMemberSendersForOneFacilityAndOneVisn(start, end, keys.get(0).getKeyLong(), mainId.getKeyLong(),period.getDbPeriod()));
					break;
				case ALL_FACILITIES:
					rawResults.addAll(getReportsDao().getCountReportsDataUniqueMemberSendersForAllFacilitiesByOneVisn(start, end, mainId.getKeyLong(),period.getDbPeriod()));
					break;
				case TEAM:
					rawResults.addAll(getReportsDao().getCountReportsDataUniqueMemberSendersForOneTeamAndOneFacility(start, end, mainId.getKeyLong(), keys.get(0).getKeyLong(),period.getDbPeriod()));
					break;
				case ALL_TEAMS:
					rawResults.addAll(getReportsDao().getCountReportsDataUniqueMemberSendersForAllTeamsByOneFacility(start, end, mainId.getKeyLong(),period.getDbPeriod()));
					break;
			}
		}
		if(runOptedInPatients) {
			switch(filter) {
				case VISN:
					rawResults.addAll(getReportsDao().getCountReportsDataOptedInPatientsForOneVisn(start, end, mainId.getKeyLong(), period.getDbPeriod()));
					break;
				case ALL_VISNS:
					rawResults.addAll(getReportsDao().getCountReportsDataOptedInPatientsForAllVisns(start, end, period.getDbPeriod()));
					break;
				case FACILITY:
					rawResults.addAll(getReportsDao().getCountReportsDataOptedInPatientsForOneFacilityAndOneVisn(start, end, keys.get(0).getKeyLong(), mainId.getKeyLong(),period.getDbPeriod()));
					break;
				case ALL_FACILITIES:
					rawResults.addAll(getReportsDao().getCountReportsDataOptedInPatientsForAllFacilitiesByOneVisn(start, end, mainId.getKeyLong(),period.getDbPeriod()));
					break;
				case TEAM:
					rawResults.addAll(getReportsDao().getCountReportsDataOptedInPatientsForOneTeamAndOneFacility(start, end, mainId.getKeyLong(), keys.get(0).getKeyLong(),period.getDbPeriod()));
					break;
				case ALL_TEAMS:
					rawResults.addAll(getReportsDao().getCountReportsDataOptedInPatientsForAllTeamsByOneFacility(start, end, mainId.getKeyLong(),period.getDbPeriod()));
					break;
			}
		}
		if(runBlockedPatients) {
			switch(filter) {
				case VISN:
					rawResults.addAll(getReportsDao().getCountReportsDataBlockedPatientsForOneVisn(start, end, mainId.getKeyLong(), period.getDbPeriod()));
					break;
				case ALL_VISNS:
					rawResults.addAll(getReportsDao().getCountReportsDataBlockedPatientsForAllVisns(start, end, period.getDbPeriod()));
					break;
				case FACILITY:
					rawResults.addAll(getReportsDao().getCountReportsDataBlockedPatientsForOneFacilityAndOneVisn(start, end, keys.get(0).getKeyLong(), mainId.getKeyLong(),period.getDbPeriod()));
					break;
				case ALL_FACILITIES:
					rawResults.addAll(getReportsDao().getCountReportsDataBlockedPatientsForAllFacilitiesByOneVisn(start, end, mainId.getKeyLong(),period.getDbPeriod()));
					break;
				case TEAM:
					rawResults.addAll(getReportsDao().getCountReportsDataBlockedPatientsForOneTeamAndOneFacility(start, end, mainId.getKeyLong(), keys.get(0).getKeyLong(),period.getDbPeriod()));
					break;
				case ALL_TEAMS:
					rawResults.addAll(getReportsDao().getCountReportsDataBlockedPatientsForAllTeamsByOneFacility(start, end, mainId.getKeyLong(),period.getDbPeriod()));
					break;
			}
		}

		if(runOptedOutPatients) {
			switch(filter) {
				case VISN:
					rawResults.addAll(getReportsDao().getCountReportsDataOptedOutPatientsForOneVisn(start, end, mainId.getKeyLong(), period.getDbPeriod()));
					break;
				case ALL_VISNS:
					rawResults.addAll(getReportsDao().getCountReportsDataOptedOutPatientsForAllVisns(start, end, period.getDbPeriod()));
					break;
				case FACILITY:
					rawResults.addAll(getReportsDao().getCountReportsDataOptedOutPatientsForOneFacilityAndOneVisn(start, end, keys.get(0).getKeyLong(), mainId.getKeyLong(),period.getDbPeriod()));
					break;
				case ALL_FACILITIES:
					rawResults.addAll(getReportsDao().getCountReportsDataOptedOutPatientsForAllFacilitiesByOneVisn(start, end, mainId.getKeyLong(),period.getDbPeriod()));
					break;
				case TEAM:
					rawResults.addAll(getReportsDao().getCountReportsDataOptedOutPatientsForOneTeamAndOneFacility(start, end, mainId.getKeyLong(), keys.get(0).getKeyLong(),period.getDbPeriod()));
					break;
				case ALL_TEAMS:
					rawResults.addAll(getReportsDao().getCountReportsDataOptedOutPatientsForAllTeamsByOneFacility(start, end, mainId.getKeyLong(),period.getDbPeriod()));
					break;
			}
		}

		if(runActionPendingPatients) {
			switch(filter) {
				case VISN:
					rawResults.addAll(getReportsDao().getCountReportsDataActionPendingPatientsForOneVisn(start, end, mainId.getKeyLong(), period.getDbPeriod()));
					break;
				case ALL_VISNS:
					rawResults.addAll(getReportsDao().getCountReportsDataActionPendingPatientsForAllVisns(start, end, period.getDbPeriod()));
					break;
				case FACILITY:
					rawResults.addAll(getReportsDao().getCountReportsDataActionPendingPatientsForOneFacilityAndOneVisn(start, end, keys.get(0).getKeyLong(), mainId.getKeyLong(),period.getDbPeriod()));
					break;
				case ALL_FACILITIES:
					rawResults.addAll(getReportsDao().getCountReportsDataActionPendingPatientsForAllFacilitiesByOneVisn(start, end, mainId.getKeyLong(),period.getDbPeriod()));
					break;
				case TEAM:
					rawResults.addAll(getReportsDao().getCountReportsDataActionPendingPatientsForOneTeamAndOneFacility(start, end, mainId.getKeyLong(), keys.get(0).getKeyLong(),period.getDbPeriod()));
					break;
				case ALL_TEAMS:
					rawResults.addAll(getReportsDao().getCountReportsDataActionPendingPatientsForAllTeamsByOneFacility(start, end, mainId.getKeyLong(),period.getDbPeriod()));
					break;
			}
		}

		if(runVHLEducationCategory) {
			switch(filter) {
				case VISN:
					rawResults.addAll(getReportsDao().getCountReportsDataEducationCategoryForOneVisn(start, end, mainId.getKeyLong(), period.getDbPeriod()));
					break;
				case ALL_VISNS:
					rawResults.addAll(getReportsDao().getCountReportsDataEducationCategoryForAllVisns(start, end, period.getDbPeriod()));
					break;
				case FACILITY:
					rawResults.addAll(getReportsDao().getCountReportsDataEducationCategoryForOneFacilityAndOneVisn(start, end, keys.get(0).getKeyLong(), mainId.getKeyLong(),period.getDbPeriod()));
					break;
				case ALL_FACILITIES:
					rawResults.addAll(getReportsDao().getCountReportsDataEducationCategoryForAllFacilitiesByOneVisn(start, end, mainId.getKeyLong(),period.getDbPeriod()));
					break;
				case TEAM:
					rawResults.addAll(getReportsDao().getCountReportsDataEducationCategoryForOneTeamAndOneFacility(start, end, mainId.getKeyLong(), keys.get(0).getKeyLong(),period.getDbPeriod()));
					break;
				case ALL_TEAMS:
					rawResults.addAll(getReportsDao().getCountReportsDataEducationCategoryForAllTeamsByOneFacility(start, end, mainId.getKeyLong(),period.getDbPeriod()));
					break;
			}
		}

		if( result.getFilterTitle().length()==0 ) {
			result.setFilterTitle(filter.getName());
		}

		//PERIOD, STATION_NUMBER,TRIAGE_GROUP_ID, VISN_ID,SUB_CATEGORY_ID,SUM(val),MIN(val),MAX(val),AVG(val)
		for(Object[] row : rawResults) {
			String dateVal = DateUtils.getEnglishDate((Date)row[0]);
			String stationNumber = toStringObject(row[1]);
			String tgId = toStringObject(row[2]);
			String visnId = toStringObject(row[3]);
			int subCategoryId = Integer.parseInt(toStringObject(row[4]));
			String value = "0";

			CountsSubCategoryEnum currentSubCat = CountsSubCategoryEnum.valueOf(subCategoryId);
			switch(currentSubCat.getType()) {
				case SUM:
					value = toStringObject(row[5]);
					break;
				case MINIMUM:
					value = toStringObject(row[6]);
					break;
				case MAXIMUM:
					value = toStringObject(row[7]);
					break;
				case AVERAGE:
					value = toStringObject(row[8]);
					break;
			}

			//Modify the tgId to be the correct Id for uniqueness:
			//  For visn it would be visnId
			//  For Facility it would be facilityId
			String uid = "";
			switch(filter) {
				case VISN:
				case ALL_VISNS:
					uid=visnId;
					break;
				case FACILITY:
				case ALL_FACILITIES:
					uid=stationNumber;
					break;
				case TEAM:
				case ALL_TEAMS:
					uid=tgId;
					break;
			}

			result.setValue(uid,currentSubCat,dateVal,value);
		}

		result.setLastRefreshDate(getLastRefreshDate());

		response.setPayload(result);
		return response;
	}

	private Date getLastRefreshDate() {
		List l = getReportsDao().getLastReportDataRefreshDate();
		if( l.size() > 0) {
			return(Date)l.get(0);
		}

		return null;
	}

	public CollectionServiceResponse<ListsResults> runListsReport(
			List<ListsSubCategoryEnum> subCategories,
			ReportingFilterTypeEnum filter, ResultKey mainId,
			List<ResultKey> keys) {

		CollectionServiceResponse<ListsResults> response = new CollectionServiceResponse<ListsResults>();

		//Run each list report and store the results for each one.
		//create place holders for the results.
		Map<String, ListsResults> results = new HashMap<String, ListsResults>();
		String stationNumber="",tgId="",visnName="";

		for (ResultKey key : keys) {
			ListsResults result = new ListsResults();
			if (log.isInfoEnabled()) {
				log.info("Adding unique Id of type "+ filter.getName()+ " "+ key.getKey() + "(" + key.getDescription() + ")");
			}
			for( ListsSubCategoryEnum subCategory: subCategories ) {
				List<String> fields = new ArrayList<String>();
				//Add the field names
				switch(subCategory) {
					case ACTION_PENDING_PATIENTS:
					case ASSOCIATED_PATIENTS:
					case BLOCKED_PATIENTS:
					case OPTED_IN_PATIENTS:
					case OPTED_OUT_PATIENTS:
						fields.add(ListsFieldEnum.LAST_NAME.getName());
						fields.add(ListsFieldEnum.FIRST_NAME.getName());
						fields.add(ListsFieldEnum.NSSN.getName());
						fields.add(ListsFieldEnum.DOB_MONTH_DAY.getName());
						fields.add(ListsFieldEnum.SIGNATURE.getName());
						break;
					case SM_HEALTHCARE_TEAM_MEMBERS:
						fields.add(ListsFieldEnum.LAST_NAME.getName());
						fields.add(ListsFieldEnum.FIRST_NAME.getName());
						fields.add(ListsFieldEnum.USER_TYPE.getName());
						fields.add(ListsFieldEnum.NSSN.getName());
						fields.add(ListsFieldEnum.DOB_MONTH_DAY.getName());
						fields.add(ListsFieldEnum.SIGNATURE.getName());
						break;
					case SM_HEALTHCARE_TEAMS:
						if(filter.equals(ReportingFilterTypeEnum.ALL_FACILITIES)|| filter.equals(ReportingFilterTypeEnum.FACILITY)){
							fields.add(ListsFieldEnum.TEAM_NAME.getName());
						}
						fields.add(ListsFieldEnum.SM_CLINIC_NAME.getName());
						fields.add(ListsFieldEnum.CPRS_TITLE_NAME.getName());
						break;
					case CPRS_TIU_PROGRESS_NOTE_FAILURE:
						fields.add(ListsFieldEnum.LAST_MESSAGE_ID.getName());
						fields.add(ListsFieldEnum.CREATED_DATE.getName());
						break;
				}
				result.addListsResultsDetails(subCategory, key.getKey(), key.getDescription(), fields);
			}
			result.setFilterTitle(filter.getName());
			result.setEnd(new Date());
			result.setSubCategories(subCategories);

			if( (result.getVisnTitle() == null || result.getVisnTitle().length() == 0 ) &&
			    (result.getFacilityTitle() == null || result.getFacilityTitle().length() == 0) ) {
				switch(filter) {
					case FACILITY:
					case ALL_FACILITIES:
						ServiceResponse<Facility> sr = facilityService.getFacilityByStationNumber(key.getKey());
						visnName = "VISN " + sr.getPayload().getVisn();
						stationNumber=sr.getPayload().getStationNumber();
						break;
					case TEAM:
					case ALL_TEAMS:
						ServiceResponse<TriageGroup> tr = triageGroupService.findTriageGroupById(keys.get(0).getKeyLong());
						stationNumber=tr.getPayload().getVistaDiv();
						ServiceResponse<Facility> fr = facilityService.getFacilityByStationNumber(tr.getPayload().getVistaDiv());
						visnName = "VISN " + fr.getPayload().getVisn();
						break;
				}
				result.setVisnTitle(visnName);
				result.setFacilityTitle(stationNumber);
			}

			if(log.isInfoEnabled()) {
				log.info("Adding results container for " + key.getKey());
			}
			results.put(key.getKey(),result);
		}

		for( ListsSubCategoryEnum subCategory: subCategories ) {
			if(log.isInfoEnabled()) {
				log.info("Running report for " + subCategory);
			}

			List<Object[]> rawResults = runQueryForLists(filter,subCategory,mainId,keys);

			for(Object[] row : rawResults) {
				List<ResultKey> resValues = new ArrayList<ResultKey>();
				int rec = 0;

				String uid = "";
				switch(filter) {
					case FACILITY:
					case ALL_FACILITIES:
						stationNumber = row[rec].toString();
						uid=stationNumber;
						break;
					case TEAM:
					case ALL_TEAMS:
						tgId = row[rec].toString();
						uid=tgId;
						break;
				}

				switch(subCategory) {
					case ACTION_PENDING_PATIENTS:
					case ASSOCIATED_PATIENTS:
					case BLOCKED_PATIENTS:
					case OPTED_IN_PATIENTS:
					case OPTED_OUT_PATIENTS:
						resValues.add(new ResultKey(ListsFieldEnum.FIRST_NAME.getName(), toStringObject(row[++rec])));
						resValues.add(new ResultKey(ListsFieldEnum.LAST_NAME.getName(), toStringObject(row[++rec])));
						resValues.add(new ResultKey(ListsFieldEnum.NSSN.getName(), toStringObject(row[++rec])));
						resValues.add(new ResultKey(ListsFieldEnum.DOB_MONTH_DAY.getName(), DateUtils.getEnglishDateMonthDayOnly((Date)row[++rec])));
						resValues.add(new ResultKey(ListsFieldEnum.SIGNATURE.getName(), toStringObject(row[++rec])));
						break;
					case SM_HEALTHCARE_TEAM_MEMBERS:
						resValues.add(new ResultKey(ListsFieldEnum.FIRST_NAME.getName(), toStringObject(row[++rec])));
						resValues.add(new ResultKey(ListsFieldEnum.LAST_NAME.getName(), toStringObject(row[++rec])));
						resValues.add(new ResultKey(ListsFieldEnum.NSSN.getName(), toStringObject(row[++rec])));
						resValues.add(new ResultKey(ListsFieldEnum.USER_TYPE.getName(), getClinicalUserType(row[++rec])));
						resValues.add(new ResultKey(ListsFieldEnum.DOB_MONTH_DAY.getName(), DateUtils.getEnglishDateMonthDayOnly((Date)row[++rec])));
						resValues.add(new ResultKey(ListsFieldEnum.SIGNATURE.getName(), toStringObject(row[++rec])));
						break;
					case SM_HEALTHCARE_TEAMS:
						if(filter.equals(ReportingFilterTypeEnum.FACILITY)|| filter.equals(ReportingFilterTypeEnum.ALL_FACILITIES)){
							resValues.add(new ResultKey(ListsFieldEnum.TEAM_NAME.getName(),  toStringObject(row[++rec])));
						}

						resValues.add(new ResultKey(ListsFieldEnum.SM_CLINIC_NAME.getName(),  toStringObject(row[++rec])));
						resValues.add(new ResultKey(ListsFieldEnum.CPRS_TITLE_NAME.getName(),  toStringObject(row[++rec])));
						break;
					case CPRS_TIU_PROGRESS_NOTE_FAILURE:
						resValues.add(new ResultKey(ListsFieldEnum.LAST_MESSAGE_ID.getName(), toStringObject(row[++rec])));
						resValues.add(new ResultKey(ListsFieldEnum.CREATED_DATE.getName(), DateUtils.getEnglishDate((Date)row[++rec])));
						break;
				}

				if(results.get(uid) == null) {
					if(log.isWarnEnabled()){
						log.warn("Unexpected uid returned in the results: ");
					}
					continue;  //Skip the results
				}

				results.get(uid).addRow(subCategory,resValues);
			}
		}

		//RETURN AN ALPHABETICAL LIST OF RESULTS
		List<String> resultkeys = new ArrayList<String>();
		for(String key: results.keySet()){
			resultkeys.add(key);
		}
		Collections.sort(resultkeys);

		List<ListsResults> sortedlist = new ArrayList<ListsResults>();
		for(String key: resultkeys) {
			sortedlist.add(results.get(key));
		}
		response.setCollection(sortedlist);
		return response;
	}

	private String getClinicalUserType( Object value ) {
		String result = "";
		try {
			result = ClinicalUserTypeEnum.valueOf(Long.parseLong(toStringObject(value))).getDescription();
		} catch (NumberFormatException e) {
			//return default ""
		} catch(RuntimeException e1) {
			//return default ""
		}

		return result;
	}

	private List<Object[]> runQueryForLists(ReportingFilterTypeEnum filter, ListsSubCategoryEnum subCategory, ResultKey mainId, List<ResultKey> keys) {
		List<Object[]> rawResults = new ArrayList<Object[]>();
		switch(filter) {
		case FACILITY:
			switch(subCategory) {
			case SM_HEALTHCARE_TEAMS:
				rawResults = getReportsDao().getListsHealthcareTeamsDataForOneFacility(keys.get(0).getKeyLong());
				break;
			case BLOCKED_PATIENTS:
				rawResults = getReportsDao().getListsBlockedPatientsDataForOneFacility(keys.get(0).getKeyLong());
				break;
			}
			break;
		case ALL_FACILITIES:
			switch(subCategory) {
			case SM_HEALTHCARE_TEAMS:
				rawResults = getReportsDao().getListsHealthcareTeamsDataForAllFacilitiesByOneVisn(mainId.getKeyLong());
				break;
			case BLOCKED_PATIENTS:
				rawResults = getReportsDao().getListsBlockedPatientsDataForAllFacilitiesByOneVisn(mainId.getKeyLong());
				break;
			}
			break;
		case TEAM:
			switch(subCategory) {
			case SM_HEALTHCARE_TEAM_MEMBERS:
				rawResults = getReportsDao().getListsHealthcareTeamMembersDataForOneTeam(keys.get(0).getKeyLong());
			break;
			case SM_HEALTHCARE_TEAMS:
				rawResults = getReportsDao().getListsHealthcareTeamsDataForOneTeam(keys.get(0).getKeyLong());
			break;
			case BLOCKED_PATIENTS:
				rawResults = getReportsDao().getListsBlockedPatientsDataForOneTeam(keys.get(0).getKeyLong());
			break;
			case ASSOCIATED_PATIENTS:
				rawResults = getReportsDao().getListsAssociatedPatientsDataForOneTeam(keys.get(0).getKeyLong());
			break;
			case OPTED_IN_PATIENTS:
				rawResults = getReportsDao().getListsOptedInPatientsDataForOneTeam(keys.get(0).getKeyLong());
			break;
			case OPTED_OUT_PATIENTS:
				rawResults = getReportsDao().getListsOptedOutPatientsDataForOneTeam(keys.get(0).getKeyLong());
			break;
			case ACTION_PENDING_PATIENTS:
				rawResults = getReportsDao().getListsActionPendingPatientsDataForOneTeam(keys.get(0).getKeyLong());
			break;
			case CPRS_TIU_PROGRESS_NOTE_FAILURE:
				rawResults = getReportsDao().getListsCPRSTiuFailureDataForOneTeam(keys.get(0).getKeyLong());
			break;
			}
		break;
		case ALL_TEAMS:
			switch(subCategory) {
			case SM_HEALTHCARE_TEAM_MEMBERS:
				rawResults = getReportsDao().getListsHealthcareTeamMembersDataForAllTeamsByOneFacility(mainId.getKeyLong());
			break;
			case SM_HEALTHCARE_TEAMS:
				rawResults = getReportsDao().getListsHealthcareTeamsDataForAllTeamsByOneFacility(mainId.getKeyLong());
			break;
			case BLOCKED_PATIENTS:
				rawResults = getReportsDao().getListsBlockedPatientsDataForAllTeamsByOneFacility(mainId.getKeyLong());
			break;
			case ASSOCIATED_PATIENTS:
				rawResults = getReportsDao().getListsAssociatedPatientsDataForAllTeamsByOneFacility(mainId.getKeyLong());
			break;
			case OPTED_IN_PATIENTS:
				rawResults = getReportsDao().getListsOptedInPatientsDataForAllTeamsByOneFacility(mainId.getKeyLong());
			break;
			case OPTED_OUT_PATIENTS:
				rawResults = getReportsDao().getListsOptedOutPatientsDataForAllTeamsByOneFacility(mainId.getKeyLong());
			break;
			case ACTION_PENDING_PATIENTS:
				rawResults = getReportsDao().getListsActionPendingPatientsDataForAllTeamsByOneFacility(mainId.getKeyLong());
			break;
			case CPRS_TIU_PROGRESS_NOTE_FAILURE:
				rawResults = getReportsDao().getListsCPRSTiuFailureDataForAllTeamsByOneFacility(mainId.getKeyLong());
			break;
			}
		break;
		}

		if(log.isInfoEnabled()) {
			log.info("Results contains " + rawResults.size() + " rows");
		}

		return rawResults;
	}

	private String toStringObject( Object value ) {
		return (value!=null?value.toString():"");
	}

}
