package gov.va.med.mhv.sm.service.impl;

import gov.va.med.mhv.foundation.service.response.CollectionServiceResponse;
import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.foundation.util.Precondition;
import gov.va.med.mhv.sm.dao.PatientPCPAssignmentDao;
import gov.va.med.mhv.sm.dao.PatientTriageMapDao;
import gov.va.med.mhv.sm.dao.TriageRelationDao;
import gov.va.med.mhv.sm.dao.UserDao;
import gov.va.med.mhv.sm.dao.properties.SMPcmmProperties;
import gov.va.med.mhv.sm.dto.PatientPCPAssignmentDTO;
import gov.va.med.mhv.sm.dto.PatientPCPAssignmentDTO.ProviderSummary;
import gov.va.med.mhv.sm.enumeration.RelationActionEnum;
import gov.va.med.mhv.sm.enumeration.RelationTypeEnum;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.PatientFacility;
import gov.va.med.mhv.sm.model.PatientPCPAssignment;
import gov.va.med.mhv.sm.model.PatientTriageMap;
import gov.va.med.mhv.sm.model.RelationAction;
import gov.va.med.mhv.sm.model.TriageGroup;
import gov.va.med.mhv.sm.model.TriageRelation;
import gov.va.med.mhv.sm.model.report.RelationReport;
import gov.va.med.mhv.sm.service.PcmmService;
import gov.va.med.mhv.sm.service.RelationshipManagementService;
import gov.va.med.mhv.sm.util.PcmmUtil;
import gov.va.med.mhv.sm.wsclient.IntegrationServiceDelegate;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.Clinic;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.ClinicPatientsResponse;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.PCMMProviderPatientsResponse;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.PatientRelationshipsResponse;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.Provider;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.Team;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.TeamPatientsResponse;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class RelationshipManagementServiceImpl implements RelationshipManagementService{

	@SuppressWarnings("unused")
	private static final Log log = LogFactory.getLog(RelationshipManagementServiceImpl.class);
	
	private UserDao userDao;
	private TriageRelationDao triageRelationDao;
	private IntegrationServiceDelegate integrationServiceDelegate;
	private PatientTriageMapDao patientTriageMapDao;
	private PatientPCPAssignmentDao patientPCPAssignmentDao;
	private SMPcmmProperties smPcmmProperties;
	private final static String PCMM_WEB_NATIONAL="National";
	private final static String PCMM_WEB_FIELD="Field";

	private PcmmService pcmmService;
	
	// date range for clinic appointments before and after today
	private int clinicDaysBefore;
	private int clinicDaysAfter;
	private int hoursToCache;
	
	
	
	/**
	 * type used when syncing "by relationship" as opposed to "by patient"
	 * Indicates (by the "touched" property) whether the patient relationship
	 * needs to be inserted/updated/deleted
	 */
	private class RPatient{
		public Patient patient;
		public RelationActionEnum touched;
	};
	

	/**
	 * type used in the sync by patient algorithms
	 */
	private class MRelation{
		public List<RelationAction> relations;
	}
	
	/**
	 * use throughout as a way to map relationships so
	 * they can be found in collections by key
	 */
	private class UniqueRelation{
		public String ien;
		public String stationNo;
		public RelationTypeEnum relationType;
		
		public boolean equals(Object a){
			if(!(a instanceof UniqueRelation)) return false;
			UniqueRelation other = (UniqueRelation)a;
			if(other == null) return false;
			if(this == other) return true;
			
			boolean _equals;
			_equals = true &&
			  (this.ien.equals(other.ien) &&
			   this.stationNo.equals(other.stationNo) &&
			   this.relationType.equals(other.relationType));
				
			return _equals;
		}
		
		public int hashCode(){
			int _hashcode = 1;
			if(ien != null) _hashcode += ien.hashCode();
			if(stationNo != null) _hashcode += stationNo.hashCode();
			if(relationType != null) _hashcode += relationType.hashCode();
			return _hashcode;
		}
	}
	
	
	
	/**
	 * Given a populated relationMap and a UniqueRelation, see if any existing
	 * TriageRelations exist in the map.  If a TriageRelation is found, if it is 
	 * active then change touched to UPDATE, if it is inactive then leave the status
	 * to DELETE.
	 * 
	 * Then check the database to see if any new TriageRelations have been defined for
	 * this UniqueRelation that the patient might not already be mapped to.  If any
	 * are found insert them into the map with an INSERT action.  
	 * 
	 * No updates to the database are actually performed here.  The map is returned with
	 * actions that <em>might</em> need to be taken.  Assuming that all of the results from
	 * VistA have been processed.
	 * 
	 * @param relationMap
	 * @param ur
	 */
	private void updatePatientRelationMap(Map<UniqueRelation, MRelation> relationMap, UniqueRelation ur){
		
		log.debug("updatePatientRelationMap with relation - " + ur.ien + "^" + ur.stationNo + "^" + ur.relationType);
		// update the records that are already in the map
		if(relationMap.containsKey(ur)){
			log.debug("relationMap contains relation");
			MRelation mr = relationMap.get(ur);
			for(RelationAction rr : mr.relations){
				log.debug("checking RelationAction rr " + rr.getTriageRelation().getId() + "^" + rr.getAction());
				if(rr.getTriageRelation().isActive()){
					log.debug("rr is active - set to update");
					rr.setAction(RelationActionEnum.UPDATE);
				}
			}
		}
		
		// this is the more difficult part.  fetch the relationships from 
		// the database matching the UniqueRelation.  If the relationship
		// already exists in the map then ignore it.  Otherwise create 
		// a new object and add it to the list
		
		List<TriageRelation> ltr = triageRelationDao.find(ur.ien, ur.stationNo, ur.relationType);
		
		for(TriageRelation tr : ltr){
			log.debug("fetched triage relation " + tr.getId());
			// we are only interested in active Relationships
			if(!tr.isActive()) continue;
			log.debug("triage relation " + tr.getId() + " is active");
			
			boolean found = false;
			// see if we already have it
			if(relationMap.containsKey(ur)){
				log.debug("relationMap contains " + ur.ien + "^" + ur.stationNo + "^" + ur.relationType);
				MRelation mr = relationMap.get(ur);
				for(RelationAction rr : mr.relations){
					if(rr.getTriageRelation().getId().equals(tr.getId())){
						found = true;
						break;
					}	
				}
				if(found == true) {
					log.debug("ignoring relation " + tr.getId());
					continue;
				}
				// didn't find it so add it to the list
				RelationAction x = new RelationAction();
				x.setTriageRelation(tr);
				x.setAction(RelationActionEnum.INSERT);
				mr.relations.add(x);
				log.debug("insert relation " + tr.getId());
			}else{
				log.debug("relationMap does not contain " + ur.ien + "^" + ur.stationNo + "^" + ur.relationType);
				// it was not in the map to begin with so add it
				MRelation nmr = new MRelation();
				nmr.relations = new ArrayList<RelationAction>();
				RelationAction x = new RelationAction();
				x.setTriageRelation(tr);
				x.setAction(RelationActionEnum.INSERT);
				nmr.relations.add(x);
				relationMap.put(ur, nmr);
			}
		}
	}
	
	
	/**
	 * create a map of the patient's existing relationships.
	 * 
	 * Set stationNo to <code>null</code> for a map of all
	 * relationships.  Otherwise the map will contain only
	 * the relationships for the specified station.
	 * 
	 * @param p
	 * @param stationNo
	 * @return
	 */
	private Map<UniqueRelation, MRelation> createPatientRelationMap(Patient p, String stationNo){
	
		log.debug("Creating patient relation map for patient " + p.getId());
		// get all of the relationships for the patient
		List<TriageRelation> trs = null;
		
		if(stationNo == null){
			trs = triageRelationDao.getTriageGroupRelationsForPatientAll(p);
		}else{
			trs = triageRelationDao.getTriageGroupRelationsForPatientStationAll(p, stationNo);
		}
		
		Map<UniqueRelation, MRelation> relationMap = new Hashtable<UniqueRelation, MRelation>(); 
		
		// make a map of the relationships so that we can identify ones
		// that need to be inserted/updated/deleted
		for(TriageRelation tr : trs){
			
			UniqueRelation ur = new UniqueRelation();
			ur.ien = tr.getVistaIen();
			ur.stationNo = tr.getStationNumber();
			ur.relationType = tr.getRelationType();
			log.debug("creating unique relation " + ur.ien + "^" + ur.stationNo + "^" + ur.relationType);
			
			RelationAction rr = new RelationAction();
			rr.setTriageRelation(tr);
			// the PATIENT relationship type is a manually kept list
			// of patients.  it is not dynamically driven by VistA
			if(ur.relationType == RelationTypeEnum.PATIENT){
				log.debug("Set relation action to update for patient relation type");
				rr.setAction(RelationActionEnum.UPDATE);
			}else{
				rr.setAction(RelationActionEnum.DELETE);
			}
			
			if(relationMap.containsKey(ur)){
				MRelation r = relationMap.get(ur);
				r.relations.add(rr);
			}else{
				MRelation r = new MRelation();
				r.relations = new ArrayList<RelationAction>();
				r.relations.add(rr);
				relationMap.put(ur, r);
			}
		}
		
		return relationMap;
	}
	
	
	/**
	 * convenience function for looping through a map and setting the action
	 * on any relations associated with the station to IGNORE.
	 * Typically this is done when an error from VistA has been returned.
	 * 
	 * @param relationMap
	 * @param station
	 */
	private void ignoreStation(Map<UniqueRelation, MRelation> relationMap, String station){
		
		// set all of this stations relations to IGNORE, otherwise they will be deleted.  Not sure which is right.
		for(UniqueRelation ur : relationMap.keySet()){
			if(!ur.stationNo.equals(station)) continue;
			MRelation mr = relationMap.get(ur);
			for(RelationAction rr : mr.relations){
				rr.setAction(RelationActionEnum.IGNORE);
			}
		}
	}
	
	
	/**
	 * Loop through the PatientRelationshipsResponse and update the relationMap as appropriate.
	 * 
	 * @param relationMap
	 * @param res
	 */
	private void consumeRelationshipResponseObj(Map<UniqueRelation, MRelation> relationMap, PatientRelationshipsResponse res,Long patientId){
		if(!getPCMMWebFlag(res.getStation())){
			if(log.isInfoEnabled()){
				log.info("Patient "+patientId+" Station# "+res.getStation()+" PCMM LEGACY Call");
			}
			if(res.getProviders() != null){
				for(int i = 0; i < res.getProviders().length; i++){
					Provider provider = res.getProviders()[i];
					UniqueRelation ur = new UniqueRelation();
					ur.stationNo = res.getStation();
					ur.ien = provider.getIEN().toString();
					ur.relationType = RelationTypeEnum.PRIMARY_PROVIDER;
					if(log.isInfoEnabled()){
						log.info("Patient PCMM Legacy->"+patientId+" Station# "+res.getStation()+"Provider->"+provider.getIEN()+"^"+provider.getLastName()+" "+provider.getFirstName());
					}
					updatePatientRelationMap(relationMap, ur);
				}
			}
		}		
		
		// update the clinic relationships
		if(res.getClinics() != null){
			for(int i = 0; i < res.getClinics().length; i++){
				Clinic clinic = res.getClinics()[i];
				UniqueRelation ur = new UniqueRelation();
				ur.stationNo = res.getStation();
				ur.ien = clinic.getIEN().toString();
				ur.relationType = RelationTypeEnum.CLINIC;
			
				updatePatientRelationMap(relationMap, ur);
			
			}
		}
		
		// update the team relationships
		if(res.getTeams() != null){
			for(int i = 0; i < res.getTeams().length; i++){
				Team team = res.getTeams()[i];
				UniqueRelation ur = new UniqueRelation();
				ur.stationNo = res.getStation();
				ur.ien = team.getIEN().toString();
				ur.relationType = RelationTypeEnum.TEAM;
			
				updatePatientRelationMap(relationMap, ur);
			}
		}
	}
	
	/**
	 * Update the database per the relationMap. 
	 * 
	 * @param p
	 * @param relationMap
	 */
	private void syncRelationsWithDatabase(Patient p, Map<UniqueRelation, MRelation> relationMap){
		
		// sync the database with the map
		for(UniqueRelation ur : relationMap.keySet()){
			
			MRelation mr = relationMap.get(ur);
			
			for (RelationAction rr : mr.relations){
				
				RelationActionEnum t = rr.getAction();
				TriageRelation tr = rr.getTriageRelation();
				
				if(t == RelationActionEnum.INSERT){
					PatientTriageMap ptm = new PatientTriageMap();
					ptm.setTriageRelation(tr);
					ptm.setPatient(p);
					log.info("insert relation-map  --patient: " + p + "  --relation: " + tr);
					patientTriageMapDao.save(ptm);
				}else if(t == RelationActionEnum.UPDATE){
					// probably not necessary but lets bump the modified date
					// for existing relations
					PatientTriageMap ptm = patientTriageMapDao.find(tr.getId(), p.getId());
					ptm.setModifiedDate(new Date());
					log.info("update relation-map  --patient: " + p + "  --relation: " + tr);
					patientTriageMapDao.save(ptm);
				}else if(t == RelationActionEnum.DELETE){
					if(!ur.relationType.equals(RelationTypeEnum.PRIMARY_PROVIDER)){
						log.info("delete relation-map  --patient: " + p + "  --relation: " + tr);
						patientTriageMapDao.delete(tr.getId(), p.getId());
					}
				}
			}
		}
	}
	
	/**
	 * Query all of the associated facilities for updates to the relationships.
	 * Returns a relationMap that identifies all of the actions to be taken.
	 * @param p
	 * @return
	 */
	private Map<UniqueRelation, MRelation> getModifiedRelations(Patient p){
		
		
		Date startDate = new Date();
		Date endDate = new Date();
	
		startDate = DateUtils.addDays(startDate, clinicDaysBefore > 0 ? clinicDaysBefore * -1 : clinicDaysBefore);
		endDate = DateUtils.addDays(endDate, clinicDaysAfter);
		
		// make sure we have a list of the patient's facilities
		userDao.fleshFacilities(p);
		
		Map<UniqueRelation, MRelation> relationMap = createPatientRelationMap(p, null); 
		
		for(PatientFacility pf : p.getFacilities()){
			
			try {
				
				log.debug("query vista relationships  station: " + pf.getStationNo() + "   patient: " + p.getIcn() + 
						"   startDate: " + startDate + "   endDate:" + endDate);
				PatientRelationshipsResponse res = 
					integrationServiceDelegate.getRelationshipsByPatientAndDate(p.getIcn(), startDate, endDate, pf.getStationNo());
				
				if(res.getStatus().equalsIgnoreCase("Reject")){
					log.info("Reject getting relationships  --status: " + res.getStatus() + "    --reason: " + res.getReason() 
							+ "  for patient (id^station^icn):" + p.getId() + "^" + pf.getStationNo() + "^" + p.getIcn());
					// Reject means that the station passed has no endpoint in Ensemble.
					// This is expected and this station should be ignored
					ignoreStation(relationMap, res.getStation());
					continue;
				}

				if(!res.getStatus().equalsIgnoreCase("Ok")){
					log.error("Error getting relationships  --status: " + res.getStatus() + "    --reason: " + res.getReason() 
							+ "  for patient (id^station^icn):" + p.getId() + "^" + pf.getStationNo() + "^" + p.getIcn());
					// assuming that Vista didn't find the user we would delete the existing relations.  Otherwise
					// uncomment the next line to keep the existing relations
					//ignoreStation(relationMap, res.getStation());
					continue;
				}
				
				consumeRelationshipResponseObj(relationMap, res,p.getId());
				
			} catch (RemoteException e) {
				log.error("Error getting relationships  --ensemble through remote exception: " + e.getMessage() 
						+ "  for patient (id^station^icn):" + p.getId() + "^" + pf.getStationNo() + "^" + p.getIcn(), e);
				// set all of this stations relations to IGNORE, otherwise they will be deleted.  Not sure which is right.
				ignoreStation(relationMap, pf.getStationNo());
				continue;
			}

		}
		
		return relationMap;
		
	}
	
	
	
	public ServiceResponse<Boolean> updatePatientRelationsByStation(String icn, PatientRelationshipsResponse relations){
		
		Precondition.assertNotBlank("icn", icn);
		Precondition.assertNotNull("relations", relations);
		Precondition.assertNotBlank("relations.station", relations.getStation());
		
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		
		if(!relations.getStatus().equalsIgnoreCase("Ok")){
			response.setPayload(Boolean.FALSE);
			return response;
		}
		
		Patient p = userDao.findPatientByIcn(icn);
		
		if(p == null){
			// we didn't find the patient by the ICN so something has gone wrong
			log.fatal("Unable to update relationships becuase patient not found.  ICN: " + icn);
			response.setPayload(Boolean.FALSE);
			return response;
		}
		
		Map<UniqueRelation, MRelation> relationMap = createPatientRelationMap(p, relations.getStation()); 
		consumeRelationshipResponseObj(relationMap, relations,p.getId());
		syncRelationsWithDatabase(p, relationMap);
		
		return response;
	}
	
	
	private boolean checkCacheContinue(Patient p){
		
		
		// if hoursToCache==-1 then the 
		// caching has been turned off and we run the sync every time
		if(hoursToCache == -1){
			log.warn("relationship sync is set to run with every login");
			return true;
		}
		
		// if hoursToCache==0 then the 
		// sync has effectively been turned off by an administrator
		if(hoursToCache == 0){
			// sync is turned off
			log.warn("relationship sync is turned off for the system");
			return false;
		}
		
		Date today = new Date();
		Date last = p.getRelationshipUpdate();
		
		if(last == null){
			// sync has never been run for this patient
			return true;
		}
		
		long diff = today.getTime() - last.getTime();
		long hours = diff / (60 * 60 * 1000);
		if (hours < hoursToCache){
			// the sync has happened within the specified time-frame so do not do it again
			log.info("using cached relationships for patient: " + p.getId() + "^" + p.getName());
			return false;
		}
		
		return true;
		
	}
	
	
	
	public ServiceResponse<Boolean> syncPatientA(Patient p){
		
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		
		
		if(!checkCacheContinue(p)){
			response.setPayload(Boolean.TRUE);
			return response;
		}
		
		log.info("syncing patient: " + p.getId() + "^" + p.getName());
		
		// update the relationship date so that we don't overdo this
		p.setRelationshipUpdate(new Date());
		userDao.updateRelationshipSyncDate(p);
		
		
		// call an asynchronous web service with each of the facilities identified.
		//TODO:
		
		response.setPayload(Boolean.TRUE);
		return response;
		
	}
	
	public ServiceResponse<Boolean> syncPatient(Patient p){
		
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		Boolean updateFlag=false;
		if(!checkCacheContinue(p)){
			response.setPayload(Boolean.TRUE);
			//return response;
		}else{
			log.info("syncing patient: " + p.getId() + "^" + p.getName());
			Map<UniqueRelation, MRelation> relationMap = getModifiedRelations(p);
			if(relationMap != null){
				syncRelationsWithDatabase(p, relationMap);
				updateFlag = true;
			}
			
			try{
				if(getPcmmWebDeploymentStatus()!=null){
					updateFlag = pcmmRelationshipUpdates(p);
				}
			}catch(Exception pcmmExp){
				if(log.isErrorEnabled()){
					log.error("RelationshipManagementService->ERROR Occured pcmmRelationUpdaets"+pcmmExp);
				}
			}
		}

		if(updateFlag){
			p.setRelationshipUpdate(new Date());
			userDao.updateRelationshipSyncDate(p);
		}
		response.setPayload(Boolean.TRUE);
		return response;
	}
	
	private Boolean pcmmRelationshipUpdates(Patient patient){
		try{
			List<RelationAction> pcpTriageRelationActions = getPCPRelationships(patient);
			for(RelationAction relationAction:pcpTriageRelationActions){
				if(relationAction.getAction().equals(RelationActionEnum.INSERT)){
					PatientTriageMap ptm = patientTriageMapDao.find(relationAction.getTriageRelation().getId(), patient.getId());
					if(ptm==null){
						PatientTriageMap ptMap = new PatientTriageMap();
						ptMap.setTriageRelation(relationAction.getTriageRelation());
						ptMap.setPatient(patient);
						patientTriageMapDao.save(ptMap);
					}	
				}
			}
		}catch(Exception exp){
			if(log.isErrorEnabled()){
				log.error("Exception Occured in PCP-Patient Relationshipupdate patient"+patient.getId());
				exp.printStackTrace();
				return Boolean.FALSE;
			}
		}
		return Boolean.TRUE;
	}
	
	private boolean getPCMMWebFlag(String stationNo){
		if(getPcmmWebDeploymentStatus()==null){
			return false;
		}
		if(getPcmmWebDeploymentStatus().equalsIgnoreCase(PCMM_WEB_NATIONAL)){
			return true;	
		}else if(getPcmmWebDeploymentStatus().equalsIgnoreCase(PCMM_WEB_FIELD)) {
			if(PcmmUtil.isPcmmWebFieldTestingSite(stationNo, getPcmmWebFieldSites())){
				return true;
			}			
		}
		return false;
	}
	
	
	public List<RelationAction> getPCPRelationships(Patient patient){
		List<RelationAction> relationActions = new ArrayList<RelationAction>();
		try{
			String pcmmWebFieldSites = getPcmmWebFieldSites();
			for(PatientFacility pf : patient.getFacilities()){

				if(getPCMMWebFlag(pf.getStationNo())){
					if(log.isInfoEnabled()){
						log.info("Patient "+patient.getId()+" Station# "+pf.getStationNo()+" PCMM WEB Call");
					}
					ServiceResponse<PatientPCPAssignmentDTO> patientPCPAssignmentResponse = pcmmService.getPatientPCPSummary(patient.getIcn(),pf.getStationNo());
					if(patientPCPAssignmentResponse.getPayload()!=null){
						PatientPCPAssignmentDTO patientPCPAssignmentdto = patientPCPAssignmentResponse.getPayload();
						for(ProviderSummary providerSummary:patientPCPAssignmentdto.getProviderSummarys()){
							try{
								PatientPCPAssignment patientPCPAssignment = new PatientPCPAssignment();
								patientPCPAssignment.setPatientId(patient.getId());
								patientPCPAssignment.setStationNumber(new Long(patientPCPAssignmentdto.getStationNumber()));
								patientPCPAssignment.setProviderIen(new Long(providerSummary.getProviderIen()));
								patientPCPAssignment.setProviderName(providerSummary.getProviderName());
								patientPCPAssignment.setAssignmentStatus(providerSummary.getAssignmentStatus());
								getPatientPCPAssignmentDao().save(patientPCPAssignment);
							}catch(Exception exp3){
								if(log.isErrorEnabled()){
									log.error("Error Occured on-RelationshipServiceImpl-100-getPCPRelations-Save PatientPCPAssignment");
									exp3.printStackTrace();
								}
							}
							List<TriageRelation> triageRelations = triageRelationDao.find(providerSummary.getProviderIen(),pf.getStationNo(), RelationTypeEnum.PRIMARY_PROVIDER);
							for(TriageRelation triageRelation:triageRelations){
								RelationAction relationAction = new RelationAction();
								relationAction.setAction(RelationActionEnum.INSERT);
								relationAction.setTriageRelation(triageRelation);
								relationActions.add(relationAction);
							}
						}
					}
				}
		   }
		}catch(Exception exp){
			if(log.isErrorEnabled()){
				log.error("Exception Occured getPCPRelationships for patient"+patient.getId());
				exp.printStackTrace();
			}
		}
		return relationActions;
	}
	
	/**
	 * Grab all of the active triage relations.  And since it is likely
	 * that many are duplicated across various triage groups, merge them
	 * into a unique set.  This will allow us to optimize network transmissions
	 * by only asking for information once.
	 */
	public ServiceResponse<Boolean> syncAll(){
		
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		List<TriageRelation> allRelations;
		Map<UniqueRelation, TriageRelation> uniqueMap = new Hashtable<UniqueRelation, TriageRelation>();
		
		allRelations = triageRelationDao.findAll();
		
		// loop through and create a unique set
		for(TriageRelation tr : allRelations){
			UniqueRelation u = new UniqueRelation();
			u.ien = tr.getVistaIen();
			u.stationNo = tr.getStationNumber();
			u.relationType = tr.getRelationType();
			
			if(!uniqueMap.containsKey(u)){
				uniqueMap.put(u, tr);
			}
		}
		
		Collection<TriageRelation> unique = uniqueMap.values();
		
		for (TriageRelation tr : unique) {
			try {
				log.debug("syncing triage relation: " + tr);
				buildRelationships(tr, true);
			} catch (Exception e) {
				log.error("Error syncing triage relation: " + tr, e);
				response.setPayload(Boolean.FALSE);
				return response;
			}
		}
		response.setPayload(Boolean.TRUE);
		return response;
	}
	
	
	
	public ServiceResponse<Boolean> syncTriageGroup(TriageGroup tg){
		
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		
		log.info("building relationships for triage group: " + tg);
		try {
			for (TriageRelation tr : tg.getRelations()) {
				log.debug("syncing triage relation: " + tr);
				buildRelationships(tr, false);
			}
			response.setPayload(Boolean.TRUE);
			
		} catch (Exception e) {
			log.error("Error syncing triage group: " + tg, e);
			response.setPayload(Boolean.FALSE);
		}
		
		return response;
	}
	
	
	
	
	

	private void buildRelationships(TriageRelation relation, boolean allLikeRelations){
		
		
		List<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient> listVistaPatients = null;
		
		// call the correct WS depending on the RelationType and convert the returned array into
		// a collection
		switch(relation.getRelationType()){
		case PRIMARY_PROVIDER:
			listVistaPatients = getPatientsByProvider(relation);
			break;
		case CLINIC:
			listVistaPatients = getPatientsByClinic(relation);
			break;
		case TEAM:
			listVistaPatients = getPatientsByTeam(relation);
			break;
		}
		
		// if the listVistaPatients is null then an error occured.
		// abort the mission now
		if(listVistaPatients == null){
			return;
		}
		
		List<TriageRelation> listRelations = null;
		
		// if allLikeRelations == true then we need to fetch all of the TriageRelations with 
		// the same IEN & StationNo.  We will then use the freshly fetched list of patients to
		// update them all
		
		if(allLikeRelations){
			listRelations = triageRelationDao.find(relation.getVistaIen(),
					relation.getStationNumber(), relation.getRelationType());
		}else{
			listRelations = new ArrayList<TriageRelation>();
			listRelations.add(relation);
		}
			
		// update the patient list for each relationship
		for(TriageRelation r : listRelations){
			
			// create a map of pre-existing related patients
			Map<String, RPatient> relationships = createPatientMap(r);
			
			// Loop through the vista patients and check to see if they exist in SMS.
			// If they do exist and they are already in the map, change value to UPDATE 
			// otherwise change value to INSERT.  If they don't exist in SMS then ignore
			// them completely
			
			for(gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient vp : listVistaPatients){
			
				// is he already in the map?
				if(relationships.containsKey(vp.getICN())){
					RPatient rp = relationships.get(vp.getICN());
					rp.touched = RelationActionEnum.UPDATE;
				}else{
					// ok, so is he in the SMS USER table
					Patient p = userDao.findPatientByIcn(vp.getICN());
					if(p == null) continue;
					// he is, so put him in the map
					RPatient rp = new RPatient();
					rp.patient = p;
					rp.touched = RelationActionEnum.INSERT;
					relationships.put(vp.getICN(), rp);
				}
			}
			
			// Now the map contains all of the patient relationships.  Ones with UPDATE will be ignored,
			// because they are already in the database.  Ones with INSERT will be added.  And ones
			// with DELETE will be removed because they were not found in the existing dataset.
			
			for(RPatient rp : relationships.values()){
				if(rp.touched == RelationActionEnum.DELETE){
					patientTriageMapDao.delete(r.getId(), rp.patient.getId());
				}else if(rp.touched == RelationActionEnum.INSERT){
					PatientTriageMap ptm = new PatientTriageMap();
					ptm.setPatient(rp.patient);
					ptm.setTriageRelation(r);
					patientTriageMapDao.save(ptm);
				}else{
					// probably not necessary but lets bump the modified date
					// for exisitng relations
					PatientTriageMap ptm = patientTriageMapDao.find(r.getId(), rp.patient.getId());
					ptm.setModifiedDate(new Date());
					patientTriageMapDao.save(ptm);
				} //if
			} // for patients
		} // for TriageRelations
		
	}
	
	
	private List<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient> getPatientsByProvider(TriageRelation relation){
		
		List<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient> listVistaPatients = 
			new ArrayList<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient>();
		
		try {
			PCMMProviderPatientsResponse response = integrationServiceDelegate.getPatientsByProvider(
					relation.getVistaIen(), relation.getStationNumber());
			if(!response.getStatus().equalsIgnoreCase("Ok")){
				log.error("Error fetching patients for triage relation: " + relation + 
						".  ACK=" + response.getStatus() + ".  Reason: " + response.getReason());
				return null;
			}
			
			for(int i = 0; i < response.getPatients().length; i++){
				listVistaPatients.add(response.getPatients()[i]);
			}
			
		} catch (RemoteException e) {
			log.error("Error fetching patients for triage relation: " + relation, e);
			return null;
		}
		
		return listVistaPatients;
		
	}
	
	private List<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient> getPatientsByClinic(TriageRelation relation){
		
		List<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient> listVistaPatients = 
			new ArrayList<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient>();
		
		try {
			Date start = DateUtils.addDays(new Date(), clinicDaysBefore);
			Date end = DateUtils.addDays(new Date(), clinicDaysAfter);
			
			ClinicPatientsResponse response = integrationServiceDelegate.getPatientsByClinicAndDate(
					relation.getVistaIen(), start, end, relation.getStationNumber());
			if(!response.getStatus().equalsIgnoreCase("Ok")){
				log.error("Error fetching patients for triage relation: " + relation + 
						".  ACK=" + response.getStatus() + ".  Reason: " + response.getReason());
				return null;
			}
			
			for(int i = 0; i < response.getPatients().length; i++){
				listVistaPatients.add(response.getPatients()[i]);
			}
			
		} catch (RemoteException e) {
			log.error("Error fetching patients for triage relation: " + relation, e);
			return null;
		}
		
		return listVistaPatients;
		
	}
	
	
	private List<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient> getPatientsByTeam(TriageRelation relation){
		
		List<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient> listVistaPatients = 
			new ArrayList<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient>();
		
		try {
			
			TeamPatientsResponse response = integrationServiceDelegate.getPatientsByTeam(
					relation.getVistaIen(), relation.getStationNumber());
			if(!response.getStatus().equalsIgnoreCase("Ok")){
				log.error("Error fetching patients for triage relation: " + relation + 
						".  ACK=" + response.getStatus() + ".  Reason: " + response.getReason());
				return null;
			}
			
			for(int i = 0; i < response.getPatients().length; i++){
				listVistaPatients.add(response.getPatients()[i]);
			}
			
		} catch (RemoteException e) {
			log.error("Error fetching patients for triage relation: " + relation, e);
			return null;
		}
		
		return listVistaPatients;
		
	}
	
	
	
	/**
	 * Create a map of existing patients in a relationship with the 
	 * key being the patients ICN
	 * The default action will be RelationActionEnum.DELETE.  As the algorithm
	 * progresses if same patient is returned from the vista
	 * query then the action will be change to RelationActionEnum.UPDATE
	 * 
	 * @param relation
	 * @return
	 */
	private Map<String, RPatient> createPatientMap(TriageRelation relation){
		
		Map<String, RPatient> map = new Hashtable<String, RPatient>();
		
		for(Patient p : relation.getPatients()){
			RPatient rp = new RPatient();
			rp.patient = p;
			rp.touched = RelationActionEnum.DELETE;
			map.put(p.getIcn(), rp);	
		}
		
		return map;
	}
	/**
	 * This method is used by AbstractTriageGroupAction. Implmented for CR#4826. Manually Associate Patient to TriageMap. 
	 * Method will add/update the entry into PatientTriageMap table based onthe relation selected by the admin. If the relationship
	 * already exist in PatientTriageMap. It will update the status according to the TriageRelation Patient selection in the UI.
	 */
	public ServiceResponse<Boolean> updatePatientTriageMap(Patient patient, TriageRelation triageRelation){
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		PatientTriageMap patientTriageMap = patientTriageMapDao.find(triageRelation.getId(), patient.getId());
		if(patientTriageMap == null){
			PatientTriageMap newPatientTriageMap = new PatientTriageMap();
			newPatientTriageMap.setPatient(patient);
			newPatientTriageMap.setTriageRelation(triageRelation);
			patientTriageMapDao.save(newPatientTriageMap);
		}
		else
		{
			patientTriageMap.setActive(triageRelation.isActive());
			patientTriageMapDao.save(patientTriageMap);
		}
		response.setPayload(Boolean.TRUE);
		return response;
	}
	
	
	public ServiceResponse<Boolean> addPatientToRelation(Patient p, TriageRelation tr){
		
		Precondition.assertNotNull("patient", p);
		Precondition.assertNotNull("relationship", tr);
		
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		
		PatientTriageMap x = patientTriageMapDao.find(tr.getId(), p.getId());
		if(x != null){
			response.setPayload(Boolean.TRUE);
			return response;
		}
		
		PatientTriageMap ptm = new PatientTriageMap();
		ptm.setPatient(p);
		ptm.setTriageRelation(tr);
		patientTriageMapDao.save(ptm);
		
		response.setPayload(Boolean.TRUE);
		return response;
		
	}


	/**
	 * create a method that will iterate over existing users and print out their 
	 * primary care provider in any associated vista systems and any triage groups
	 * to which they are related by primary care provider 
	 * @return
	 */
	
	public CollectionServiceResponse<RelationReport> relationshipReport(){

		CollectionServiceResponse<RelationReport> response = new CollectionServiceResponse<RelationReport>();
		
		List<RelationReport> report = new ArrayList<RelationReport>();
		
		List<Patient> patients = userDao.findAllPatients();
		
		
		for(Patient p : patients){
			
			// create the report;
			//report.add(p.getName() + U + p.getIcn() + U + p.getId());
			RelationReport rr = new RelationReport();
			List<RelationAction> ras = new ArrayList<RelationAction>();
			rr.setPatient(p);
			rr.setActions(ras);
			
			if(p.getIcn() == null){
				rr.setComments("ICN null, nothing updated");
				report.add(rr);
				continue;
			}
			
			Map<UniqueRelation, MRelation> relationMap = getModifiedRelations(p);
			
			if(relationMap == null){
				rr.setComments("an error occurred with this patient");
				report.add(rr);
				continue;
			}
			
			for(MRelation mr : relationMap.values()){
				ras.addAll(mr.relations);
			}
			
			report.add(rr);
	
		}  // for Patient
		
		response.setCollection(report);
		return response;
	}
		
		
	
	
	
	
	
	
	public UserDao getUserDao() {
		return userDao;
	}
	public void setUserDao(UserDao userDao) {
		this.userDao = userDao;
	}
	public TriageRelationDao getTriageRelationDao() {
		return triageRelationDao;
	}
	public void setTriageRelationDao(TriageRelationDao triageRelationDao) {
		this.triageRelationDao = triageRelationDao;
	}
	public IntegrationServiceDelegate getIntegrationServiceDelegate() {
		return integrationServiceDelegate;
	}
	public void setIntegrationServiceDelegate(
			IntegrationServiceDelegate integrationServiceDelegate) {
		this.integrationServiceDelegate = integrationServiceDelegate;
	}
	public PatientTriageMapDao getPatientTriageMapDao() {
		return patientTriageMapDao;
	}
	public void setPatientTriageMapDao(PatientTriageMapDao patientTriageMapDao) {
		this.patientTriageMapDao = patientTriageMapDao;
	}
	public int getClinicDaysBefore() {
		return clinicDaysBefore;
	}
	public void setClinicDaysBefore(int clinicDaysBefore) {
		this.clinicDaysBefore = clinicDaysBefore;
	}
	public int getClinicDaysAfter() {
		return clinicDaysAfter;
	}
	public void setClinicDaysAfter(int clinicDaysAfter) {
		this.clinicDaysAfter = clinicDaysAfter;
	}
	
	public int getHoursToCache() {
		return hoursToCache;
	}
	
	public void setHoursToCache(int hoursToCache) {
		this.hoursToCache = hoursToCache;
	}
	
	public PcmmService getPcmmService() {
		return pcmmService;
	}


	public void setPcmmService(PcmmService pcmmService) {
		this.pcmmService = pcmmService;
	}
	
	public PatientPCPAssignmentDao getPatientPCPAssignmentDao() {
		return patientPCPAssignmentDao;
	}


	public void setPatientPCPAssignmentDao(
			PatientPCPAssignmentDao patientPCPAssignmentDao) {
		this.patientPCPAssignmentDao = patientPCPAssignmentDao;
	}
	
	public SMPcmmProperties getSmPcmmProperties() {
		return smPcmmProperties;
	}


	public void setSmPcmmProperties(SMPcmmProperties smPcmmProperties) {
		this.smPcmmProperties = smPcmmProperties;
	}
	
	private String getPcmmWebDeploymentStatus(){
		return (getSmPcmmProperties()!=null && getSmPcmmProperties().getPcmmWebDeploymentStatus()!=null && !getSmPcmmProperties().getPcmmWebDeploymentStatus().trim().equals(""))?getSmPcmmProperties().getPcmmWebDeploymentStatus():null;
	}
	
	private String getPcmmWebFieldSites(){
		return (getSmPcmmProperties()!=null && getSmPcmmProperties().getPcmmWebFieldTestingSites()!=null)?getSmPcmmProperties().getPcmmWebFieldTestingSites():null;
	}
	
	

}
