package gov.va.med.mhv.sm.service.impl;

import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.foundation.util.Precondition;
import gov.va.med.mhv.sm.dao.AddresseeDao;
import gov.va.med.mhv.sm.dao.AnnotationDao;
import gov.va.med.mhv.sm.dao.DistGroupRecallMessageDao;
import gov.va.med.mhv.sm.dao.FacilityDao;
import gov.va.med.mhv.sm.dao.MessageAttachmentDao;
import gov.va.med.mhv.sm.dao.MessageDao;
import gov.va.med.mhv.sm.dao.ReAssignFacilityDao;
import gov.va.med.mhv.sm.dao.SurrogateDao;
import gov.va.med.mhv.sm.dao.ThreadDao;
import gov.va.med.mhv.sm.dao.TriageGroupDao;
import gov.va.med.mhv.sm.dao.UserDao;
import gov.va.med.mhv.sm.dao.exception.DataIntegrityException;
import gov.va.med.mhv.sm.enumeration.AddresseeRoleEnum;
import gov.va.med.mhv.sm.enumeration.ClinicianStatusEnum;
import gov.va.med.mhv.sm.enumeration.EmailNotificationEnum;
import gov.va.med.mhv.sm.enumeration.ParticipantTypeEnum;
import gov.va.med.mhv.sm.enumeration.PerformerTypeEnum;
import gov.va.med.mhv.sm.enumeration.ReAssignMessageActionEnum;
import gov.va.med.mhv.sm.enumeration.SystemFolderEnum;
import gov.va.med.mhv.sm.enumeration.UserStatusEnum;
import gov.va.med.mhv.sm.enumeration.UserTypeEnum;
import gov.va.med.mhv.sm.model.Addressee;
import gov.va.med.mhv.sm.model.Annotation;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.DistGroupRecallMessage;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.Folder;
import gov.va.med.mhv.sm.model.Message;
import gov.va.med.mhv.sm.model.MessageAttachment;
import gov.va.med.mhv.sm.model.PatientReassignFacility;
import gov.va.med.mhv.sm.model.ReAssignMessageHistory;
import gov.va.med.mhv.sm.model.Surrogate;
import gov.va.med.mhv.sm.model.Thread;
import gov.va.med.mhv.sm.model.TriageGroup;
import gov.va.med.mhv.sm.model.User;
import gov.va.med.mhv.sm.service.EmailService;
import gov.va.med.mhv.sm.service.LoggingService;
import gov.va.med.mhv.sm.service.MessageService;
import gov.va.med.mhv.sm.service.ReAssignMessageService;
import gov.va.med.mhv.sm.service.TriageGroupService;
import gov.va.med.mhv.sm.service.UserManagementService;
import gov.va.med.mhv.sm.service.response.messages.SmsServiceMessages;
import gov.va.med.mhv.sm.util.DateUtils;
import gov.va.med.mhv.sm.util.SystemFolderFactory;

import java.sql.Blob;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Hibernate;

public class MessageServiceImpl implements MessageService {

	@SuppressWarnings("unused")
	private static final Log log = LogFactory.getLog(MessageServiceImpl.class);

	private AddresseeDao addresseeDao;

	private AnnotationDao annotationDao;

	private MessageDao messageDao;

	private EmailService emailService;

	private UserDao userDao;

	private ThreadDao threadDao;

	private SurrogateDao surrogateDao;

	private MessageAttachmentDao messageAttachmentDao;

	private ReAssignMessageService reAssignMessageService;

    private TriageGroupDao triageGroupDao;

	private FacilityDao facilityDao;

    private TriageGroupService triageGroupService;

	private LoggingService loggingService;

	private ReAssignFacilityDao reAssignFacilityDao;

	private UserManagementService userManagementService;

	private DistGroupRecallMessageDao distGroupRecallMessageDao;






	/**
	 * Find a MessageBy Id.
	 * @param id  Message id
	 * @return
	 */
	public Message findMessageById(Long id){

		return messageDao.findById(id);
	}



	public ServiceResponse<Boolean> annotateThread(Annotation a) {

		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();

		Precondition.assertNotNull("annotation", a);
		Precondition.assertNotNull("annotation.thread", a.getThread());
		Precondition.assertNotNull("annotation.author", a.getAuthor());
		Precondition.assertNotBlank("annotation.annotation", a.getAnnotation());

		try {
			annotationDao.save(a);
		} catch (Exception e) {
			response.setPayload(Boolean.FALSE);
			return response;
		}

		response.setPayload(Boolean.TRUE);
		return response;

	}

	public ServiceResponse<Boolean> assignMessageAndUpdateHistory(Message message, Clinician assignedTo, Clinician assignedBy, String assignedToWithSurrogate)
	{
		 ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		 assignMessage(message,assignedTo,assignedBy);
		 if(message.getSenderType()==ParticipantTypeEnum.PATIENT){
			 reAssignMessageService.createReAssignMessageWithInTeamHistory(message.getId(), message.getRecipientId(), assignedBy.getName(),assignedBy.getId(), assignedToWithSurrogate);
		 }
		 response.setPayload(Boolean.TRUE);
		return response;
	}

	public void sendAssignmentNotificationForSurrogate(User user,TriageGroup tg, Message message) {
		Surrogate surrogate = userManagementService.getCurrentSurrogateFor((Clinician)user);
		if(surrogate!=null){
			if(surrogate.getSurrogateType()==ParticipantTypeEnum.CLINICIAN){
			   User surrogateClinician = userManagementService.findById(surrogate.getSurrogateId());
			   sendAssignmentNotification((Clinician)surrogateClinician, tg, message);
			}
		}
	}
	
	public ServiceResponse<Boolean> assignMessage(Message m,
			Clinician assignTo, Clinician assignedBy) {

		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();

		Precondition.assertNotNull("message", m);
		Precondition.assertPositive("message.id", m.getId());
		Precondition.assertNotNull("assignTo", assignTo);

		// fetch the message from the database to make sure nothing gets
		// changed except the assignTo
		Message x = messageDao.findById(m.getId());
		Clinician c = userDao.findClinicianById(assignTo.getId());

		if (!c.isActive() || c.getStatus() != UserStatusEnum.OPT_IN) {
			response.setPayload(Boolean.FALSE);
			response.addError(SmsServiceMessages.ASSIGNEE_IS_NOT_ACTIVE);
			return response;
		}

		if (x == null) {
			response.setPayload(Boolean.FALSE);
			response.addError(SmsServiceMessages.MESSAGE_NOT_FOUND);
			return response;
		}

		// if the message is complete then it is frozen and cannot
		// be modified
		if (x.getStatus() == ClinicianStatusEnum.COMPLETE) {
			response.setPayload(Boolean.FALSE);
			response.addError(SmsServiceMessages.MESSAGE_ALREADY_COMPLETE);
			return response;
		}

		x.setAssignedTo(assignTo);
		x.setStatus(ClinicianStatusEnum.INPROCESS);
		x.setStatusSetBy(assignedBy);
		x.setModifiedDate(new Date());
		Message savedMessage = messageDao.save(x);


		// make sure the assignee has actually been sent the message
		// it is possible that a message sent by a patient on date X
		// was sent to all available triage members. But a new member
		// was added to the group at date X+1. If the new member is
		// assigned the message it is possible that they do not have it
		// in their mailbox.

		Addressee a = addresseeDao.getAddresseeForUser(assignTo.getId(), x
				.getId());
		if (a == null) {
			// the message needs to be addressed to the assignee
			a = new Addressee();
			a.setFolderId(SystemFolderEnum.INBOX.getId());
			a.setMessage(x);
			a.setOwner(assignTo);
			a.setRole(AddresseeRoleEnum.RECIPIENT);
			addresseeDao.save(a);
		}

		// send out an assignment notification if the
		// user preferences are set this way.
		// As per CR 6767, If the triage member assinged the message to themselve, we need not required to notify them.
		if (c.getEmailNotification() == EmailNotificationEnum.ON_ASSIGNEMNT ) {
			if(!assignTo.getId().equals(assignedBy.getId())){
				emailService.sendAssigmentNotification(c,m.getThread().getMailGroup(),m);
			}
		}else if(c.getEmailNotification() == EmailNotificationEnum.EACH_MESSAGE){
			if(!assignTo.getId().equals(assignedBy.getId())){
				emailService.sendAssigmentNotification(c,m.getThread().getMailGroup(),m);
			}
		}else if(c.getEmailNotification() == EmailNotificationEnum.ONE_DAILY) {
			if (!DateUtils.isToday(c.getLastNotification())) {
				if(!assignTo.getId().equals(assignedBy.getId())){
					emailService.sendAssigmentNotification(c,m.getThread().getMailGroup(),m);
				}
			}
		}
		
		sendAssignmentNotificationForSurrogate(c,m.getThread().getMailGroup(), m);
		loggingService.assignMessage(assignTo, x, "assigned by: "
				+ assignedBy.getName(), true);
		response.setPayload(Boolean.TRUE);
		return response;
	}

	public ServiceResponse<Boolean> deleteMessage(Message m, User u) {
		Folder f = (Folder) SystemFolderFactory.createDeletedFolder();
		return moveMessagePriv(m, u, f);
	}

	public ServiceResponse<Boolean> deleteDraft(User u, Long messageId) {

		Precondition.assertNotNull("user", u);
		Precondition.assertPositive("messageId", messageId);

		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		Message m = messageDao.findById(messageId);

		// make sure this message has not been sent
		if (m.getSentDate() != null) {
			response.addError(SmsServiceMessages.MESSAGE_NOT_EDITABLE);
			response.setPayload(Boolean.FALSE);
			if(log.isWarnEnabled()){
				log.warn("attempted deletion of sent message: " + messageId);
			}
			return response;
		}

		// make sure the user is the owner of the message
		if (!m.getSenderId().equals(m.getSenderId())) {
			response.addError(SmsServiceMessages.MESSAGE_NOT_OWNED_BY_USER);
			response.setPayload(Boolean.FALSE);
			if(log.isWarnEnabled()){
				log.warn("attempted deletion of message " + messageId + " by user "
					+ u.getId() + " who is not the owner of the message");
			}
			return response;
		}

		Thread t = m.getThread();

		Addressee a = m.getAddressees().get(0);

		addresseeDao.delete(a.getId());
		messageDao.delete(m.getId());
		List lt = t.getMessages();
		if (lt.size() < 1 )
			threadDao.delete(t.getId());

		response.setPayload(Boolean.TRUE);
		return response;

	}

	public ServiceResponse<Boolean> moveMessage(Message m, User u, Folder f) {

		Precondition.assertNotNull("message", m);
		Precondition.assertNotNull("folder", f);
		Precondition.assertNotNull("user", u);

		return moveMessagePriv(m, u, f);

	}

	private ServiceResponse<Boolean> moveMessagePriv(Message m, User u, Folder f) {

		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();

		// fetch the right Addressee from the database
		Addressee a = null;
		try {
			a = addresseeDao.getAddresseeForUser(u.getId(), m.getId());
		} catch (DataIntegrityException e) {
			response.addError(SmsServiceMessages.DATA_INTEGRITY_ERROR);
			response.setPayload(null);
			return response;
		}
		if (a == null) {
			response.addError(SmsServiceMessages.USER_NOT_A_RECIPIENT);
			response.setPayload(null);
			return response;
		}

		// If the folder is not SystemFolder verify that the folder
		// is owned by the owner of the Addressee
		if (!f.isSystemFolder()) {
			if (!f.getOwner().getId().equals(a.getOwner().getId())) {
				if(log.isInfoEnabled()){
					log.info("Could not move message: Folder not owned by user.  f.getOwner()=" + f.getOwner()+ ", a.getOwner()=" + a.getOwner());
				}
				// the owners don't match so abort
				response.addError(SmsServiceMessages.FOLDER_NOT_OWNED_BY_USER);
				response.setPayload(Boolean.FALSE);
				return response;
			}
		}

		a.setFolderId(f.getId());
		addresseeDao.save(a);

		response.setPayload(Boolean.TRUE);
		return response;

	}

	public ServiceResponse<Boolean> moveMessage(Addressee a) {

		Precondition.assertNotNull("addressee", a);
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();

		if (a == null) {
			response.addError(SmsServiceMessages.USER_NOT_A_RECIPIENT);
			response.setPayload(null);
			return response;
		}

		addresseeDao.save(a);

		response.setPayload(Boolean.TRUE);
		return response;
	}

	public ServiceResponse<Message> readMessage(Message m, User u) {

		ServiceResponse<Message> response = new ServiceResponse<Message>();

		Precondition.assertNotNull("message", m);
		Precondition.assertNotNull("user", u);

		// fetch the right Addressee from the database
		Addressee a = null;
		try {
			a = addresseeDao.getAddresseeForUser(u.getId(), m.getId());
		} catch (DataIntegrityException e) {
			response.addError(SmsServiceMessages.DATA_INTEGRITY_ERROR);
			response.setPayload(null);
			return response;
		}
		if (a == null) {
			response.addError(SmsServiceMessages.USER_NOT_A_RECIPIENT);
			response.setPayload(null);
			return response;
		}

		// set the Addressee read date
		a.setReadDate(new Date());
		addresseeDao.save(a);

		// fetch the message from the database
		Message x = messageDao.getMessageComplete(a.getMessage().getId());

		if (x == null) {
			// the message wasn't found
			response.addError(SmsServiceMessages.MESSAGE_NOT_FOUND);
			;
		}

		loggingService.readMessage(u, x, null, true);
		response.setPayload(x);
		return response;

	}

	public ServiceResponse<Message> fetchMessage(Long id) {

		ServiceResponse<Message> response = new ServiceResponse<Message>();

		Precondition.assertNotNull("id", id);

		// fetch the message from the database
		Message x = messageDao.getMessageComplete(id);

		if (x == null) {
			// the message wasn't found
			response.addError(SmsServiceMessages.MESSAGE_NOT_FOUND);
			;
		}

		response.setPayload(x);
		return response;

	}



	public ServiceResponse<Boolean> setMessageStatus(Message m,ClinicianStatusEnum status, Clinician setBy, Date reminderDate) {

		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();


		// fetch the message from the database to make sure nothing gets
		// changed except the status
		Message x = messageDao.findById(m.getId());

		// the status cannot be changed for messages that have
		// been already marked complete
		if (x.getStatus() == ClinicianStatusEnum.COMPLETE) {
			response.addError(SmsServiceMessages.MESSAGE_ALREADY_COMPLETE);
			response.setPayload(Boolean.FALSE);
			return response;
		}

		// the message can only be set by the clinician currently assigned the
		// message
		// except for the status of IN_PROCESS, which should be done
		// automatically when
		// assigning a message
		if (x.getAssignedTo() == null || !x.getAssignedTo().equals(setBy)) {
			response.addError(SmsServiceMessages.MESSAGE_NOT_ASSIGNED_TO_USER);
			response.setPayload(Boolean.FALSE);
			return response;
		}

		// move all of the messages into the completed folder
		if (status == ClinicianStatusEnum.COMPLETE) {
			completedMessageProc(m,setBy,reminderDate);
			x.setCompletedDate(new Date());
			loggingService.completeMessage(setBy, x, null, true);
			//completeAllMessagesInThread(m, setBy); // Newly added for CR 5198
			completePreviousMessagesInThread(m,setBy);

			// if the message is not assigned, assign it to the current user
			if (m.getAssignedTo() == null) {
				assignMessage(m, setBy, setBy);
			}
		}


		x.setStatus(status);
		x.setStatusSetBy(setBy);
        x.setModifiedDate(new Date());
		messageDao.save(x);
		// 12.5 Changes
		if(x.getSenderType()==ParticipantTypeEnum.PATIENT && x.getStatus()==ClinicianStatusEnum.COMPLETE){
			reAssignMessageService.createCompletedMessageHistory(x.getId(), x.getRecipientId(),setBy.getId() ,setBy.getName());
		}
		response.setPayload(Boolean.TRUE);
		return response;
	}

    public ServiceResponse<Boolean> setReminderDate(Message message,Clinician user, Date reminderDate){
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
    	Addressee addressee = addresseeDao.getAddresseeForUser(user.getId(),message.getId());
    	if(addressee.getFolderId().equals(SystemFolderEnum.COMPLETED.getId())){
    		addressee.setReminderDate(reminderDate);
    		addresseeDao.save(addressee);
       		response.setPayload(Boolean.TRUE);
	   	} else {
	   		response.setPayload(Boolean.FALSE);
	   	}

		return response;
    }

	public ServiceResponse<Boolean> setAllPreviousMessageStatus(
			Message m, ClinicianStatusEnum status, Clinician setBy,Long newReplyMessageId,Date reminderDate){
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();

		// fetch the message from the database to make sure nothing gets
		// changed except the status
		Message x = messageDao.findById(m.getId());

		// the status cannot be changed for messages that have
		// been already marked complete
		if (x.getStatus() == ClinicianStatusEnum.COMPLETE) {
			response.addError(SmsServiceMessages.MESSAGE_ALREADY_COMPLETE);
			response.setPayload(Boolean.FALSE);
			return response;
		}

		// the message can only be set by the clinician currently assigned the
		// message
		// except for the status of IN_PROCESS, which should be done
		// automatically when
		// assigning a message

		if (x.getAssignedTo() == null || !x.getAssignedTo().equals(setBy)) {
			response.addError(SmsServiceMessages.MESSAGE_NOT_ASSIGNED_TO_USER);
			response.setPayload(Boolean.FALSE);
			return response;
		}

		// move all of the messages into the completed folder
		if (status == ClinicianStatusEnum.COMPLETE) {
			completedMessageProc(m,setBy,reminderDate);
			x.setCompletedDate(new Date());
			loggingService.completeMessage(setBy, x, null, true);
			completePreviousMessagesInThread(m,setBy);

			/* 12.5 Changes Start -- No need to complete the reply messages.
			Message replyMessage = messageDao.findById(newReplyMessageId);
			// Complete the reply Message Also.
			messageComplete(replyMessage,setBy);
			12.5 Changes End */

			// if the message is not assigned, assign it to the current user
			if (m.getAssignedTo() == null) {
				assignMessage(m, setBy, setBy);
			}
		}

		x.setStatus(status);
		x.setStatusSetBy(setBy);
        x.setModifiedDate(new Date());
		messageDao.save(x);

		// 12.5 Changes
		if(x.getSenderType()==ParticipantTypeEnum.PATIENT && x.getStatus()==ClinicianStatusEnum.COMPLETE){
			reAssignMessageService.createCompletedMessageHistory(x.getId(), x.getRecipientId(),setBy.getId() ,setBy.getName());
		}
		response.setPayload(Boolean.TRUE);
		return response;
	}



	/**
	 * If a Message is marked "Complete" then we are supposed move the message
	 * to the "Completed" folder. With the following 2 exceptions.
	 * <ul>
	 * <li>The senders message shouldn't be moved</li>
	 * <li>If the message has already been deliberately moved to a new folder
	 * by its owner</li>
	 * </ul>
	 *
	 * @param m
	 *            Message to move to the Completed folder
	 */
	private void completedMessageProc(Message m,Clinician clinician, Date reminderDate) {

		m = messageDao.getMessageComplete(m.getId());
		Long inboxId = SystemFolderEnum.INBOX.getId();

		for (Addressee a : m.getAddressees()) {

			if (a.getFolderId().equals(inboxId)) {
				Addressee x = addresseeDao.findById(a.getId());
				if(x.getOwner().getUserType().equals(UserTypeEnum.CLINICIAN)){
					x.setFolderId(SystemFolderEnum.COMPLETED.getId());
					if(x.getOwner().getId().equals(clinician.getId()) && reminderDate !=null){
						x.setReminderDate(reminderDate);
					}
					addresseeDao.save(x);
				}else{
					if(log.isInfoEnabled()){
						log.info("**** Not a clinician");
					}
				}
			}
		}
	}

	public ServiceResponse<Boolean> completedMessageReassignment(Message m,Clinician assignedBy, Long reAssignee,String annotation,String surrogateEnabled) {
		Message message = messageDao.getMessageComplete(m.getId());

		Clinician clinician = userDao.findClinicianById(reAssignee);
		message.setStatus(ClinicianStatusEnum.INPROCESS);
		message.setEscalatedDate(null);
		message.setEscalationNotificationDate(null);
		message.setCompletedDate(null);
		message.setStatusSetBy(assignedBy);
		message.setAssignedTo(clinician);
		message.setModifiedDate(new Date());
		messageDao.save(message);

		for (Addressee addressee : message.getAddressees()) {
			if (addressee.getFolderId().equals(SystemFolderEnum.COMPLETED.getId())) {
					addressee.setFolderId(SystemFolderEnum.INBOX.getId());
					addressee.setModifiedDate(new Date());
					addresseeDao.save(addressee);
				}
			}
		Annotation annotate = new Annotation();
		annotate.setAnnotation(annotation);
		annotate.setAuthor(assignedBy);
		annotate.setThread(m.getThread());
		annotateThread(annotate);

		String logDetails = "Completed Message Reassignment-> Previous Assignee: "+m.getAssignedTo()+" Previous Completed Date: "+m.getCompletedDate()+" " +
											"Current Assignee: "+reAssignee+" Current Assigner: "+assignedBy;
		loggingService.completedMessageReAssignment(assignedBy, PerformerTypeEnum.SELF, logDetails,true);

		//12.5 Changes
		if(message.getSenderType()==ParticipantTypeEnum.PATIENT){
			reAssignMessageService.createReAssignMessageWithInTeamHistory(message.getId(), message.getRecipientId(), assignedBy.getName(), assignedBy.getId(),clinician.getName()+" "+surrogateEnabled);
		}
		ServiceResponse<Boolean> serviceResponse = new ServiceResponse<Boolean>();
		serviceResponse.setPayload(Boolean.TRUE);
		return serviceResponse;
		}


	/*
	 * Method completePreviousMessagesInThread will set the status to
	 * "Completed" for All Previous Messages in Thread.(Refer CR# 5198 - As per Notes)
	 */
	private void completePreviousMessagesInThread(Message m, Clinician user) {

		for (Message msg : messageDao
				.getPreviousMessagesInThread(m.getId(),m.getThread().getId())) {
			if(log.isInfoEnabled()){
				log.info("message Id....."+msg.getId());
			}
			try{

			// If the message is unassigned, the below logic will move the Inbox message to Completed Folder, and set the status to 'Complete' for the message.
			if(msg.getAssignedTo() == null){
				if(log.isInfoEnabled()){
					log.info(msg.getId()+"...Message is not assigned to anyone...first");
				}
				// Move the Inbox messages to Completed Folder.
				updateAddresseeFolderStatus(msg);
				messageComplete(msg,user);
				continue;
			}

			// If the previous message is Assigned to the current logged in user.
			if(msg.getAssignedTo().getId().equals(user.getId())){
				if(log.isInfoEnabled()){
					log.info(msg.getId()+"...Message is assigned to..."+msg.getAssignedTo().getId()+"...second");
				}
				updateAddresseeFolderStatus(msg);
				messageComplete(msg,user);
				continue;
			}


			/* 12.5 Changes
			// message.getAssignedToUser has the logged in user has surrogate then he can complete it.
			List<Surrogate>  surrogateList = surrogateDao.getSurrogatesFor(msg.getAssignedTo());
			Surrogate surrogate = surrogateList.get(0);
			if(surrogate!=null){
				if(surrogate.getSurrogateType()==ParticipantTypeEnum.CLINICIAN)
				{
					if(surrogate.getSurrogateId().equals(user.getId())){
						updateAddresseeFolderStatus(msg);
						messageComplete(msg,user);
						continue;
					}
				}
				if(surrogate.getSurrogateType()==ParticipantTypeEnum.TRIAGE_GROUP){
					updateAddresseeFolderStatus(msg);
					messageComplete(msg,user);
					continue;
				}
			}*/

		 }catch(Exception e){
			e.printStackTrace();
			log.info("Error...."+e.toString());
		 }
		}
	  }


	/**
	 * messageComplete(Message,Clinician).
	 * method update with teh CompletedDate,status and status SetBy
	 * @param m Message to move to the Completed folder
	 */
	private void messageComplete(Message message,Clinician user){
		message.setAssignedTo(user);
		message.setCompletedDate(new Date());
		message.setStatus(ClinicianStatusEnum.COMPLETE);
		message.setStatusSetBy(user);
		message.setModifiedDate(new Date());
		messageDao.save(message);
		if(message.getSenderType()==ParticipantTypeEnum.PATIENT){ // Audit History only for Patient messages completed
			reAssignMessageService.createCompletedMessageHistory(message.getId(), message.getRecipientId(), user.getId(), user.getName());
		}
	    loggingService.completeMessage(user,message,null,true);
	}

	public ServiceResponse<Boolean> completeIndividualMessage(Message message,Clinician user){
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		try{
			// 12.5 Changes
			if(!message.getAssignedTo().getId().equals(user.getId()) && message.getSenderType()==ParticipantTypeEnum.PATIENT){
				reAssignMessageService.createReAssignMessageWithInTeamHistory(message.getId(), message.getRecipientId(), user.getName(),user.getId(),user.getName());
			}
			messageComplete(message,user);
			completedMessageProc(message,user,null);
			response.setPayload(Boolean.TRUE);

		}catch(Exception e){
			response.addError("Error while completing message and move messages to Completed folder");
			response.setPayload(Boolean.FALSE);
		}
		return response;
	}

	/**
	 * updateAddresseeFolderStatus(Message).
	 * method move all the Inbox and Escalated messages to Completed Folder.
	 * @param m Message to move to the Completed folder
	 */
	private void updateAddresseeFolderStatus(Message message)
	{
		for (Addressee a : message.getAddressees()) {
			Long inboxId = SystemFolderEnum.INBOX.getId();
			Long escalatedFolderId = SystemFolderEnum.ESCALATED.getId();
			if (a.getFolderId().equals(inboxId) || a.getFolderId().equals(escalatedFolderId)) {
				Addressee addressee = addresseeDao.findById(a.getId());
				if(addressee.getOwner().getUserType().equals(UserTypeEnum.CLINICIAN)){
					addressee.setFolderId(SystemFolderEnum.COMPLETED.getId());
					addresseeDao.save(addressee);
				}
			}
		}
	}

	/**
	 * updateAddresseeReAssignedActive(Message).
	 * method set flag of Reassigned=Yes for all the Clinician messages.
	 */
	public void updateAddresseeReAssignedActive(Message message)
	{
		for (Addressee a : message.getAddressees()) {
				Addressee addressee = addresseeDao.findById(a.getId());
				if(addressee.getOwner().getUserType().equals(UserTypeEnum.CLINICIAN) && addressee.isReAssigned() !=null && !addressee.isReAssigned()){
					addressee.setReAssigned(Boolean.TRUE);
					addresseeDao.save(addressee);
				}
		}
	}

	/**
	 * updateMessage with CPRS Notes Status(Message).
	 *
	 */
	public ServiceResponse<Message> updateMessageWithCPRSNotesStatus(Long messageId,String cprsNotesStatus,Date updatedDate)
	{
		ServiceResponse<Message> response = new ServiceResponse<Message>();
		try{
			Message message = findMessageById(messageId);
		message.setCprsNotesStatus(cprsNotesStatus);
		message.setCprsNotesFailedDate(updatedDate);
		message.setModifiedDate(new Date());
		message = messageDao.save(message);
		response.setPayload(message);
		return response;
		}catch(Exception e1){
			System.out.println("An Error occured while updating message with CPRS notes status..."+e1);

		}
		return response;
	}

	/**
	 * resetCPRSNotes Status while retry save as progress notes
	 *
	 */
	public ServiceResponse<Boolean> resetCPRSNotesStatusByThread(Long threadId)
	{
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		response.setPayload(messageDao.resetCPRSNotesStatusByThread(threadId));
		return response;
	}


	public ServiceResponse<Message> updateReadReceipt(Long messageId){
		ServiceResponse<Message> response = new ServiceResponse<Message>();
		Message message = messageDao.findById(messageId);
		message.setReadReceipt("READ");
		message.setModifiedDate(new Date());
		message = messageDao.save(message);
		response.setPayload(message);
		return response;
	}

	public ServiceResponse<Message> updateAssignmentStatus(Long messageId, Clinician clinician) {
		ServiceResponse<Message> response = new ServiceResponse<Message>();
		Message message = messageDao.findById(messageId);
		message.setStatus(ClinicianStatusEnum.INPROCESS);
		message.setAssignedTo(clinician);
		message.setModifiedDate(new Date());
		message = messageDao.save(message);
		response.setPayload(message);
		return response;
	}

	public ServiceResponse<Message> recall(Long messageId, Clinician clinician) {
		ServiceResponse<Message> response = new ServiceResponse<Message>();
		Message message = messageDao.findById(messageId);
		message.setActive(false);
		message.setModifiedDate(new Date());
		message = messageDao.save(message);
		loggingService.recallMessage(clinician, PerformerTypeEnum.SELF, "Recalled message: " + messageId + "; ReadReceipt Status: " + message.getReadReceipt(), true);

		response.setPayload(message);
		return response;
	}



	public void deleteAddresseeReminders(Long addresseeId) {
		Addressee addressee = addresseeDao.findById(addresseeId);
		addressee.setReminderDate(null);
		addresseeDao.save(addressee);
	}

	public ServiceResponse<MessageAttachment> saveAttachment(byte[] fileBytes, String fileName, String mimeType){
		Precondition.assertNotNull("fileBytes", fileBytes);
		return saveAttachment(Hibernate.createBlob(fileBytes), fileName, mimeType,null);
	}


	public ServiceResponse<MessageAttachment> saveAttachment(Blob attachment, String fileName, String mimeType, Long userId){

		ServiceResponse<MessageAttachment> response = new ServiceResponse<MessageAttachment>();
		try{
		MessageAttachment messageAttachment = new MessageAttachment();
		messageAttachment.setAttachment(attachment);
		messageAttachment.setActive(Boolean.TRUE);
		messageAttachment.setCreatedDate(new Date());
		messageAttachment.setAttachmentName(fileName);
		messageAttachment.setMimeType(mimeType);
		messageAttachment.setUserId(userId);
		messageAttachmentDao.save(messageAttachment);
		response.setPayload(messageAttachment);
		}catch(Exception error){
			if(log.isErrorEnabled()){
				log.error("Error while saving attachment l3......"+error.getStackTrace());
			}
		}
		return response;
	}

	private String getSurrogateEnabled(Clinician provider){
        String surrogateEnabled="";
     	String userSurrogateName = userManagementService.getActiveSurrogateByUser(provider);
		if(userSurrogateName!=null)
			return "Surrogate Enabled: "+userSurrogateName;
		else
			return surrogateEnabled;
	 }

	public ServiceResponse<Message> reAssignMessage(Long messageId,Long origTriageGroupId, Long reAssignedTriageGroupId, Long messageAssignedTo, Boolean manuallyAssociate, User user,ReAssignMessageActionEnum action){
		Precondition.assertNotNull("messageId", messageId);
		Precondition.assertNotNull("reAssignedTriageGroupId", reAssignedTriageGroupId);
		Precondition.assertNotNull("manuallyAssociate", manuallyAssociate);
		Precondition.assertNotNull("user", user);
		Precondition.assertNotNull("action", action);

		if(action.equals(ReAssignMessageActionEnum.REASSIGN_ANOTHER_STAFF_WITH_IN_FACILITY) || action.equals(ReAssignMessageActionEnum.MESSAGE_TRANSFERRED_STAFF_OUTSIDE_FACILITY)){
			Precondition.assertNotNull("messageAssignedTo",messageAssignedTo);
		}

		if(log.isInfoEnabled()){
			log.info("[<<<<<202>>>>ReAssign msgId...."+messageId+" Original TG Id..."+origTriageGroupId+" AssignedTo..."+messageAssignedTo+" ManuallyAssociate..."+manuallyAssociate+" Assigner.."+user.getId()+"^"+user.getName()+" Reassign action "+action.getId()+"]");
		}

		ServiceResponse<Message> response = new ServiceResponse<Message>();

		try{
			Message message = messageDao.findById(messageId);
			TriageGroup assignedTriageGroup = triageGroupDao.findById(reAssignedTriageGroupId);
			Clinician reAssignedProvider = null;
            if(messageAssignedTo!=null){
            	reAssignedProvider = (Clinician)userDao.findById(messageAssignedTo);
            }

			Patient patient = (Patient) userDao.findById(message.getSenderId());
            TriageGroup origTriageGroup = triageGroupDao.findById(origTriageGroupId);


            if(manuallyAssociate){
            	if(log.isInfoEnabled()){
            		log.info("[<<<<<202-A>>>>Alternate flows stared.....>>>>>>>>");
            	}
            	ServiceResponse<Boolean> patientResponse = triageGroupService.savePatientManualAssociation(reAssignedTriageGroupId,patient);
        	    if(patientResponse.getPayload()){
        	    	reAssignMessageService.createAutoAssociatedPatientTGHistory(message.getId(),origTriageGroupId,reAssignedTriageGroupId,user.getId());
        	    	if(log.isInfoEnabled()){
        	    		log.info("[<<<<<202-A>>>>Alternate flows ended.....>>>>>>>>");
        	    	}
        	    }else{
        	    	response.addError("An Error occured while associate Patient to.....>"+action.getName());
        	    	if(log.isInfoEnabled()){
        	    		log.info("ReAssign Message to TG->AssociateTriageGroup Error-------------------->");
        	    		log.info("Patient.."+patient.getId());
        	    		log.info("MessageId..."+message.getId());
        	    		log.info("ReAssign TriageGroup..."+assignedTriageGroup.getId());
        	    		log.info("<------------------------------------------------------------------");
        	    	}
        			response.setPayload(null);
        			return response;
        	    }
            }


			Thread messageThread = threadDao.findById(message.getThread().getId());
			messageThread.setMailGroup(assignedTriageGroup);
			threadDao.save(messageThread);
			if(log.isInfoEnabled()){
				log.info("[<<<<<203>>>>Reassign Message Thread Updated>>>>>>>>");
			}
			if(action.equals(ReAssignMessageActionEnum.REASSIGN_ANOTHER_STAFF_WITH_IN_FACILITY) || action.equals(ReAssignMessageActionEnum.MESSAGE_TRANSFERRED_STAFF_OUTSIDE_FACILITY)){
				  	message.setAssignedTo(reAssignedProvider);
				  	message.setStatus(ClinicianStatusEnum.INPROCESS);
				  	message.setStatusSetBy((Clinician)user);

            }else if(action.equals(ReAssignMessageActionEnum.REASSIGN_ANOTHER_TG_WITH_IN_FACILITY) || action.equals(ReAssignMessageActionEnum.MESSAGE_TRANSFERRED_TG_OUTSIDE_FACILITY)){
            	message.setAssignedTo(null);
            	message.setStatus(ClinicianStatusEnum.INCOMPLETE);
            	message.setStatusSetBy(null);

            }

			// Set the Message status to 'Unassigned' and Save it
			message.setModifiedDate(new Date());
			message.setRecipientId(assignedTriageGroup.getId());
			message.setRecipientName(assignedTriageGroup.getName());
			//message.setThread(messageThread);  // Test and uncomment it.
			Message updatedMessage = messageDao.save(message);
			if(log.isInfoEnabled()){
				log.info("[<<<<<204>>>>Reassign Message Updated>>>>>>>>"+updatedMessage.getId());
			}

			// Fetch current triage group clinicians addressee and deactivate the addressee.
			List<Addressee> messageAddressees = updatedMessage.getAddressees();
			for(Addressee addressee:messageAddressees){
				// Make sure it's clinician recorde before make Inactive.
				if(addressee.getRole()==AddresseeRoleEnum.RECIPIENT)
				{
					 addressee.setActive(Boolean.FALSE);
					 addresseeDao.save(addressee);
				}
			}


			// Create Addressees for the reassigned triage group members.
			triageGroupDao.getCliniciansForTriageGroup(assignedTriageGroup);
			List<Clinician> triageMembers = assignedTriageGroup.getClinicians();
			for(Clinician clinician:triageMembers){
				createAddressee(clinician,updatedMessage);
			}
			if(log.isInfoEnabled()){
				log.info("[<<<<<205>>>>Reassign Message Addresees's Created>>>>>>>>");
			}

			if(action.equals(ReAssignMessageActionEnum.REASSIGN_ANOTHER_STAFF_WITH_IN_FACILITY)){
					   if(log.isInfoEnabled()){
						   log.info("[<<<<<206>>>>Reassign Message to Staff with in facility....Started");
					   }
					   reAssignMessageService.createReAssignMessageToAnotherStaffWithInFacility(message.getId(), origTriageGroupId, reAssignedTriageGroupId, user.getName(), reAssignedProvider.getName()+" "+getSurrogateEnabled(reAssignedProvider), user.getId());
					   sendAssignmentNotification(reAssignedProvider,assignedTriageGroup ,updatedMessage );
			           if(log.isInfoEnabled()){
			        	   log.info("[<<<<<206>>>>Reassign Message to Staff(with in facility) "+reAssignedProvider.getId()+"^"+reAssignedProvider.getName());
			           }

			}else if(action.equals(ReAssignMessageActionEnum.REASSIGN_ANOTHER_TG_WITH_IN_FACILITY)){
				    if(log.isInfoEnabled()){
				    	log.info("[<<<<<207>>>>Reassign Message to Triage Group with in facility....Started");
				    }
					reAssignMessageService.createReAssignMessageToAnotherTeamHistory(message.getId(), origTriageGroupId, reAssignedTriageGroupId, user.getName(),user.getId());
					// Notify email notification to the members of the reassigned triage group.
					for(User clinician:triageMembers){
						sendEmailNotification(clinician, assignedTriageGroup, updatedMessage);
					}
					if(log.isInfoEnabled()){
						log.info("[<<<<<207>>>>Reassign Message to Triage Group(with in facility) "+assignedTriageGroup.getId()+"^"+assignedTriageGroup.getName());
					}

			}else if(action.equals(ReAssignMessageActionEnum.MESSAGE_TRANSFERRED_STAFF_OUTSIDE_FACILITY)){
				    if(log.isInfoEnabled()){
				    	log.info("[<<<<<208>>>>Message Transferred to Staff outside facility....Started");
				    }
					reAssignMessageService.createReAssignMessageToAnotherStaffOutsideFacility(message.getId(), origTriageGroupId, reAssignedTriageGroupId, user.getName(), reAssignedProvider.getName()+" "+getSurrogateEnabled(reAssignedProvider), user.getId());
					sendAssignmentNotification(reAssignedProvider,assignedTriageGroup ,updatedMessage );
					if(log.isInfoEnabled()){
						log.info("[<<<<<208>>>>Message Transferred to Staff(outside facility) "+reAssignedProvider.getId()+"^"+reAssignedProvider.getName());
					}

			}else if(action.equals(ReAssignMessageActionEnum.MESSAGE_TRANSFERRED_TG_OUTSIDE_FACILITY)){
					if(log.isInfoEnabled()){
						log.info("[<<<<<209>>>>Message Transferred to Triage Group outside facility....Started");
					}
					reAssignMessageService.createReAssignMessageToAnotherTGOutsideFacility(message.getId(), origTriageGroupId, reAssignedTriageGroupId, user.getName(),user.getId());
					// Notify email notification to the members of the reassigned triage group.
					for(User clinician:triageMembers){
						sendEmailNotification(clinician, assignedTriageGroup, updatedMessage);
					}
					if(log.isInfoEnabled()){
						log.info("[<<<<<209>>>>Message Transferred to Triage Group(outside facility) "+assignedTriageGroup.getId()+"^"+assignedTriageGroup.getName());
					}
			}
			response.setPayload(updatedMessage);
			//response.setPayload(message);
		}catch(Exception e){
			if(log.isErrorEnabled()){
				log.error("Exception occured @MessageServiceImpl.reAssignMessage"+action.getName()+" "+e.getStackTrace());
			}
			response.addError("Exception occured @ >>>>>>>>>"+action.getName());
			response.setPayload(null);
		}
		return response;
	}

	public ServiceResponse<Boolean> nofityFacilityAdminsAddPatientToFacility(Long messageId, Long receivingStation,User user){
		Precondition.assertNotNull("messageId", messageId);
		Precondition.assertNotNull("reassignFacility", receivingStation);
		Precondition.assertNotNull("user", user);
		ServiceResponse<Boolean> response=new ServiceResponse<Boolean>();
		Message message = messageDao.findById(messageId);
		if(log.isInfoEnabled()){
			log.info("[<<<<<301>>>>Notify to Facility Admins to Add PatientToFacility>>>>>>>> Message Id"+messageId);
			log.info("[<<<<<302>>>>Station>>>>>>>>"+receivingStation);
			log.info("[<<<<<303>>>>Requested By"+user.getId()+"^"+user.getName());
		}
		PatientReassignFacility patFacility = new PatientReassignFacility();
		try{

			patFacility.setSecureMessageId(messageId);
			patFacility.setTriageGroupId(message.getRecipientId());
			patFacility.setTriageGroupName(reAssignMessageService.getVisnAndFacilityByTriageGroup(message.getRecipientId()));
			patFacility.setUserId(user.getId());
			patFacility.setReAssignByStation(new Long(((Clinician)user).getStationNo()));
			patFacility.setReAssignToStation(receivingStation);
			patFacility.setPatientId(message.getSenderId());
			PatientReassignFacility reAssignFacility = reAssignFacilityDao.save(patFacility);
			response.setPayload(Boolean.TRUE);
			if(reAssignFacility.getId()!=null){
				String triageGroupWithFaciltiy=reAssignMessageService.getVisnAndFacilityByTriageGroup(message.getRecipientId());
				String visnName=null;
				if(triageGroupWithFaciltiy!=null){
					visnName=triageGroupWithFaciltiy.substring(0,triageGroupWithFaciltiy.indexOf("/"));
				}
				ReAssignMessageHistory reAssignMessageHistory =  reAssignMessageService.createEmailNotificationSentHistory(messageId,message.getRecipientId(),user.getId()).getPayload();
				if(reAssignMessageHistory!=null){
					sendEmailNotificationToFacilityAdmins(user.getId(),message.getId(),message.getRecipientName(), receivingStation, new Long(((Clinician)user).getStationNo()), visnName);
				}
			}

		}catch(Exception e1){
			if(log.isErrorEnabled()){
				log.error("Error occured while saving Patient Reassign Facility....."+e1.getStackTrace());
			}
			response.setPayload(Boolean.FALSE);
		}
		return response;
	}



	private void sendEmailNotification(User user,TriageGroup tg, Message message){

		if(user.getEmailNotification() == EmailNotificationEnum.EACH_MESSAGE){
				emailService.sendNewMessageEmailNotification(user, tg, message);

		}else if(user.getEmailNotification() == EmailNotificationEnum.ONE_DAILY) {
			if (!DateUtils.isToday(user.getLastNotification())) {
				emailService.sendNewMessageEmailNotification(user, tg, message);
			}
		}

	}

	public void sendAssignmentNotification(Clinician clinician,TriageGroup tg,Message message ){
		if (clinician.getEmailNotification() == EmailNotificationEnum.ON_ASSIGNEMNT ) {
				emailService.sendAssigmentNotification(clinician,tg,message);
			}
		else if(clinician.getEmailNotification() == EmailNotificationEnum.EACH_MESSAGE){
				emailService.sendAssigmentNotification(clinician,tg,message);
		}
		else if(clinician.getEmailNotification() == EmailNotificationEnum.ONE_DAILY) {
			if (!DateUtils.isToday(clinician.getLastNotification())) {
				emailService.sendAssigmentNotification(clinician,tg,message);
			}
		}
	}

	private void sendEmailNotificationToFacilityAdmins(Long userId,Long messageId, String triageGroup, Long assignedToStation, Long assignedByStation, String visnName){
		Precondition.assertNotNull("triageGroup", triageGroup);
		Precondition.assertNotNull("assignedToStation", assignedToStation);
		Precondition.assertNotNull("assignedByStation", assignedByStation);
		Precondition.assertNotNull("visnName", visnName);

		List<Object[]> emailAddresses = triageGroupDao.getFacilityAdminsEmailsByStation(assignedToStation.toString());

		if(emailAddresses!=null && emailAddresses.size()!=0){
			String[] adminsEmail = new String[emailAddresses.size()];
			emailAddresses.toArray(adminsEmail);
			if(log.isInfoEnabled()){
				log.info("<<<<<<<<<<<<<Facility Administrators of station "+assignedToStation+" >>>>>>>>>>>>>"+Arrays.toString(adminsEmail));
			}
			emailService.sendEmailNotificationToFacilityAdmins(userId,messageId,triageGroup, assignedToStation,assignedByStation, visnName, adminsEmail);
			if(log.isInfoEnabled()){
				log.info("%%%%%%%%%%%%%%%%%%%%%%Facility Email Notification Successful...........");
			}
		}else{
			if(log.isInfoEnabled()){
				log.info("Facility admins email is null");
			}

		}
	}


	private void createAddressee(User user, Message message){
		Addressee userAddressee = addresseeDao.getAddresseeForUser(user.getId(), message.getId());
		if (userAddressee == null) {
			// the message needs to be addressed to the assignee
			userAddressee = new Addressee();
			userAddressee.setFolderId(SystemFolderEnum.INBOX.getId());
			userAddressee.setMessage(message);
			userAddressee.setOwner(user);
			userAddressee.setRole(AddresseeRoleEnum.RECIPIENT);
			userAddressee.setReAssigned(Boolean.TRUE);
			addresseeDao.save(userAddressee);
		}
		else{
			userAddressee.setActive(Boolean.TRUE);
			addresseeDao.save(userAddressee);
		}
	}


	public void removeAttachment(Long attachmentId){
		messageAttachmentDao.delete(attachmentId);
	}



	public void markReadAsUnread(Addressee ad){
		Addressee addressee = ad;
		addressee.setReadDate(null);
		addresseeDao.save(addressee);
	}

   	public ServiceResponse<Date> fetchReminderDate(Message message, Clinician user){
    	ServiceResponse<Date> response = new ServiceResponse<Date>();
    	Addressee addressee = addresseeDao.getAddresseeForUser(user.getId(),message.getId());
    	response.setPayload(addressee.getReminderDate());
    	return response;
    }

	public AddresseeDao getAddresseeDao() {
		return addresseeDao;
	}

	public void setAddresseeDao(AddresseeDao addresseeDao) {
		this.addresseeDao = addresseeDao;
	}

	public AnnotationDao getAnnotationDao() {
		return annotationDao;
	}

	public void setAnnotationDao(AnnotationDao annotationDao) {
		this.annotationDao = annotationDao;
	}

	public MessageDao getMessageDao() {
		return messageDao;
	}

	public void setMessageDao(MessageDao messageDao) {
		this.messageDao = messageDao;
	}

	public EmailService getEmailService() {
		return emailService;
	}

	public void setEmailService(EmailService emailService) {
		this.emailService = emailService;
	}

	public UserDao getUserDao() {
		return userDao;
	}

	public void setUserDao(UserDao userDao) {
		this.userDao = userDao;
	}

	public ThreadDao getThreadDao() {
		return threadDao;
	}

	public void setThreadDao(ThreadDao threadDao) {
		this.threadDao = threadDao;
	}

	public LoggingService getLoggingService() {
		return loggingService;
	}

	public void setLoggingService(LoggingService loggingService) {
		this.loggingService = loggingService;
	}

	public ServiceResponse<User> getSender(Long Id) {
		ServiceResponse<User> response = new ServiceResponse<User>();
		Precondition.assertNotNull("long", Id);
		User x = userDao.findById(Id);
		if (x == null) {
			// the user wasn't found
			response.addError(SmsServiceMessages.USER_NOT_FOUND);
			;
		}
		response.setPayload(x);
		return response;
	}

	public MessageAttachmentDao getMessageAttachmentDao() {
		return messageAttachmentDao;
	}

	public void setMessageAttachmentDao(MessageAttachmentDao messageAttachmentDao) {
		this.messageAttachmentDao = messageAttachmentDao;
	}



	public ReAssignMessageService getReAssignMessageService() {
		return reAssignMessageService;
	}


	public TriageGroupDao getTriageGroupDao() {
		return triageGroupDao;
	}



	public void setTriageGroupDao(TriageGroupDao triageGroupDao) {
		this.triageGroupDao = triageGroupDao;
	}

	public void setReAssignMessageService(
			ReAssignMessageService reAssignMessageService) {
		this.reAssignMessageService = reAssignMessageService;
	}

	public FacilityDao getFacilityDao() {
			return facilityDao;
		}

	public void setFacilityDao(FacilityDao facilityDao) {
			this.facilityDao = facilityDao;
	}

	public ServiceResponse<MessageAttachment> fetchAttachment(Long attachmentId) {
		ServiceResponse<MessageAttachment> response = new ServiceResponse<MessageAttachment>();

		MessageAttachment ma = messageAttachmentDao.findById(attachmentId);

		response.setPayload(ma);
		return response;
	}

	public ServiceResponse<List<Object[]>> getAttachmentNamesByAttachmentIds(List attachmentIds) {
		ServiceResponse<List<Object[]>> response = new ServiceResponse<List<Object[]>>();

		List<Object[]> attachmentNames =  messageAttachmentDao.getAttachmentNamesByAttachmentIds(attachmentIds);
		response.setPayload(attachmentNames);
		return response;
	}

	public ServiceResponse<List<Object[]>> getDistributionGroupMessagesByIds(Long senderId,Long recipientId,String checksum){
		ServiceResponse<List<Object[]>> response = new ServiceResponse<List<Object[]>>();
		List<Object[]> distGroupMessages = messageDao.getDistributionGroupMessagesByIds(senderId, recipientId, checksum);
		response.setPayload(distGroupMessages);
		return response;
	}

	/**
	 * recall set the active value to false of the distribution group messages.  Making it recalled.
	 *
	 * @param Id    - secureMessageId needs to be set to "inactive" value.
	 *
	 */
	public ServiceResponse<Boolean> recallDistributionGroupMessages(Long secureMessageId){
		Precondition.assertNotNull("secureMessageId", secureMessageId);
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		try{
			Message message = messageDao.findById(secureMessageId);
			Boolean recallUpdatedStatus=messageDao.recallDistGroupMessages(secureMessageId);

			if(recallUpdatedStatus){
				if(log.isInfoEnabled()){
					log.info("###9005-Recall DistGroup Messages Successful"+message.getId());
				}
				response.setPayload(recallUpdatedStatus);
			}else{
				if(log.isInfoEnabled()){
					log.info("###9005-Recall DistGroup Messages Failed "+message.getId());
				}
				response.setPayload(recallUpdatedStatus);
			}
		}catch(Exception e2){
			response.setPayload(Boolean.FALSE);
			if(log.isErrorEnabled()){
				log.error("###9005-E2 Error in recallDistributionGroupMessage()"+e2);
			}
		}

		return response;
	}

	public DistGroupRecallMessage  getDistGroupRecallMessageById(Long secureMessageId){
		return distGroupRecallMessageDao.getDistGroupRecallMessageById(secureMessageId);
	}

	/**
	 * update distribution group recall messages
	 *
	 * @param Id    - secureMessageId
	 *
	 */
	public ServiceResponse<Boolean> updateDgRecallMessages(Long secureMessageId, Long userId,String recallStatus){
		Precondition.assertNotNull("secureMessageId", secureMessageId);
		Precondition.assertNotNull("userId", userId);
		Precondition.assertNotNull("recallStatus", recallStatus);
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		try{
			Message message = messageDao.findById(secureMessageId);
			DistGroupRecallMessage dgRecallMessage = new DistGroupRecallMessage();
			dgRecallMessage.setSecureMessageId(message.getId());
			dgRecallMessage.setRecallReference(message.getChecksum());
			dgRecallMessage.setUserId(userId);
			dgRecallMessage.setRecallStatus(recallStatus);
			distGroupRecallMessageDao.save(dgRecallMessage);
			if(log.isInfoEnabled()){
				log.info("###9006###Distribution Group Recall Message status updated successfully");
			}
			response.setPayload(Boolean.TRUE);

		}catch(Exception e1){
			response.setPayload(Boolean.FALSE);
			if(log.isErrorEnabled()){
				log.error("###9006-E1-Distribution Group Recall Message status update failed"+e1);
			}
		}
		return response;
	}

	public TriageGroupService getTriageGroupService() {
		return triageGroupService;
	}

	public void setTriageGroupService(TriageGroupService triageGroupService) {
		this.triageGroupService = triageGroupService;
	}

	public SurrogateDao getSurrogateDao() {
			return surrogateDao;
	}

	public void setSurrogateDao(SurrogateDao surrogateDao) {
			this.surrogateDao = surrogateDao;
	}



	public ReAssignFacilityDao getReAssignFacilityDao() {
		return reAssignFacilityDao;
	}



	public void setReAssignFacilityDao(ReAssignFacilityDao reAssignFacilityDao) {
		this.reAssignFacilityDao = reAssignFacilityDao;
	}

	public UserManagementService getUserManagementService() {
		return userManagementService;
	}



	public void setUserManagementService(UserManagementService userManagementService) {
		this.userManagementService = userManagementService;
	}

	public DistGroupRecallMessageDao getDistGroupRecallMessageDao() {
		return distGroupRecallMessageDao;
	}



	public void setDistGroupRecallMessageDao(
			DistGroupRecallMessageDao distGroupRecallMessageDao) {
		this.distGroupRecallMessageDao = distGroupRecallMessageDao;
	}
}