package gov.va.med.mhv.sm.service.impl;

import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.sm.dao.AddresseeDao;
import gov.va.med.mhv.sm.dao.FolderDao;
import gov.va.med.mhv.sm.dto.SearchFormDTO;
import gov.va.med.mhv.sm.enumeration.MessageCategoryTypeEnum;
import gov.va.med.mhv.sm.enumeration.MessageFilterEnum;
import gov.va.med.mhv.sm.enumeration.MessagesOrderByEnum;
import gov.va.med.mhv.sm.enumeration.MessagesPageEnum;
import gov.va.med.mhv.sm.enumeration.SystemFolderEnum;
import gov.va.med.mhv.sm.enumeration.UserTypeEnum;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.Folder;
import gov.va.med.mhv.sm.model.Mailbox;
import gov.va.med.mhv.sm.model.MessagesPage;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.SubFolder;
import gov.va.med.mhv.sm.model.SystemFolder;
import gov.va.med.mhv.sm.model.User;
import gov.va.med.mhv.sm.model.priv.MessageCount;
import gov.va.med.mhv.sm.service.MailboxService;
import gov.va.med.mhv.sm.util.SystemFolderFactory;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;


public class MailboxServiceImpl implements MailboxService {

	private FolderDao folderDao;
	private AddresseeDao addresseeDao;

	public ServiceResponse<Patient> getMailbox(Patient p){
		getMailboxX(p, true);
		ServiceResponse<Patient> response = new ServiceResponse<Patient>();
		response.setPayload(p);
		return response;
	}

	public ServiceResponse<Clinician> getMailbox(Clinician c){
		return getMailbox(c, true);
	}

	public ServiceResponse<Clinician> getMailbox(Clinician c, boolean preloadInbox){
		return getMailbox(c, preloadInbox, false);
	}

	public ServiceResponse<Clinician> getMailbox(Clinician c, boolean preloadInbox, boolean loadAllCounts){
		getMailboxX(c, preloadInbox);

		// populate escalated counts
		populateEscalatedCounts(c);
		populateCPRSAlertsCounts(c);
		if( loadAllCounts ) {
			//Load the counts for Reassign and Reminders
			populateReminderCounts(c);
			populateReassignedCounts(c);
		}

		ServiceResponse<Clinician> response = new ServiceResponse<Clinician>();
		response.setPayload(c);
		return response;
	}

	public ServiceResponse<Patient> getMailbox(Patient p, boolean preloadInbox){
		getMailboxX(p, preloadInbox);
		ServiceResponse<Patient> response = new ServiceResponse<Patient>();
		response.setPayload(p);
		return response;
	}

	private void getMailboxX(User u, boolean preloadInbox){

		Mailbox mailbox = new Mailbox();

		List<SystemFolder> lsf = createSystemFolders(u);
		Collections.sort(lsf, SystemFolder.SYSTEM_FOLDER_SORTER);
		mailbox.setSystemFolders(lsf);

		List<Folder> luf = folderDao.getFoldersForUser(u.getId());
		Collections.sort(luf, Folder.USER_FOLDER_SORTER);
		mailbox.setUserFolders(luf);

		Map<Long, Folder> lf = new Hashtable<Long,Folder>();
		for(Folder f : lsf){
			if(f.getId()==SystemFolderEnum.CPRSPROGRESSALERTS.getId()){
				f.setOrderBy(MessagesOrderByEnum.CPRSALERTS_DATE);
			}
			lf.put(f.getId(), f);
		}
		for(Folder f : luf){
			lf.put(f.getId(), f);
		}

		mailbox.setFolders(lf);
		u.setMailbox(mailbox);

		populateCounts(u);

		if( preloadInbox ) {
			// populate the inbox with the first page
			Folder inbox = lf.get(SystemFolderEnum.INBOX.getId());
			// set the filter with the default filter from user preferences
			inbox.setFilter(u.getMessageFilter());

			getMessagesInt(inbox, MessagesPageEnum.FIRST, null);
		}
	}

	public ServiceResponse<Patient> updateCounts(Patient p){
		populateCounts(p);
		ServiceResponse<Patient> response = new ServiceResponse<Patient>();
		response.setPayload(p);
		return response;
	}

	public ServiceResponse<Clinician> updateCounts(Clinician c){
		populateCounts(c);
		populateEscalatedCounts(c);
		populateCPRSAlertsCounts(c);

		ServiceResponse<Clinician> response = new ServiceResponse<Clinician>();
		response.setPayload(c);
		return response;
	}

	private void populateCounts(User u){

		List<MessageCount> counts = addresseeDao.getFolderCounts(u.getId());

		Map<Long, Folder> folders = u.getMailbox().getFolders();

		for(MessageCount c : counts){
			if(!folders.containsKey(c.getFolderId()))
				continue;
			Folder f = folders.get(c.getFolderId());
			f.setCount(c.getCount());
			f.setUnreadCount(c.getUnreadCount());
			if( f.getId() == SystemFolderEnum.INBOX.getId() && u.getUserType() == UserTypeEnum.CLINICIAN) {
				List<MessageCount> categories = addresseeDao.getCategoryCounts(u.getId());
				List<SubFolder> list = new ArrayList<SubFolder>();
				List<MessageCategoryTypeEnum> all = MessageCategoryTypeEnum.toList();
				for(MessageCategoryTypeEnum each: all) {
					SubFolder sf = new SubFolder();
					sf.setId(each.getId());
					sf.setName(each.getName());
					list.add(sf);
				}

				Map<Long,MessageCount> results = new HashMap<Long,MessageCount>();
				for(MessageCount cat : categories){
					results.put(cat.getMessageCategoryTypeId(),cat);
				}

				for(SubFolder sub: list)
				{
					if( results.containsKey(sub.getId()) ) {
						sub.setCount(results.get(sub.getId()).getCount());
						sub.setUnreadCount(results.get(sub.getId()).getUnreadCount());
					}
				}
				f.setSubfolderList(list);
			}
		}
	}

	private void populateEscalatedCounts(Clinician c){
		Folder escalated = c.getMailbox().getFolders().
				get(SystemFolderEnum.ESCALATED.getId());
		MessageCount x = addresseeDao.getEscalatedCount(c.getId());
		escalated.setCount(x.getCount());
	}

	private void populateReassignedCounts(Clinician c){
		Folder reassign = c.getMailbox().getFolders().
				get(SystemFolderEnum.REASSIGN.getId());
		MessageCount x = addresseeDao.getReassignedCount(c.getId());
		reassign.setCount(x.getCount());
	}

	private void populateReminderCounts(Clinician c){
		Folder reminder = c.getMailbox().getFolders().
				get(SystemFolderEnum.REMINDER.getId());
		MessageCount x = addresseeDao.getReminderCount(c.getId());
		reminder.setCount(x.getCount());
	}

	private void populateCPRSAlertsCounts(Clinician c){
		Folder cprsAlerts = c.getMailbox().getFolders().
				get(SystemFolderEnum.CPRSPROGRESSALERTS.getId());
		MessageCount x = addresseeDao.getCPRSAlertsCount(c.getId());
		cprsAlerts.setCount(x.getCount());
	}
	
	private List<SystemFolder> createSystemFolders(User u){

		List<SystemFolder> l = new ArrayList<SystemFolder>();
		boolean isClinician = u.getUserType() == UserTypeEnum.CLINICIAN ? true : false;

		for(SystemFolderEnum f : SystemFolderEnum.values()){

			if(f.isClinicianOnly() && !isClinician){
				continue;
			}
			SystemFolder sf = SystemFolderFactory.createFolder(f);
			sf.setOwner(u);
			l.add(sf);
		}

		return l;
	}

	public ServiceResponse<Folder> getMessages(Folder f, MessagesPageEnum page, SearchFormDTO sf){
		ServiceResponse<Folder> response = new ServiceResponse<Folder>();
		response.setPayload(getMessagesInt(f, page, sf));
		return response;
	}

	public ServiceResponse<Folder> getMessages(Folder f, SearchFormDTO sf){
		ServiceResponse<Folder> response = new ServiceResponse<Folder>();
		response.setPayload(getMessagesInt(f, MessagesPageEnum.FIRST, sf));
		return response;
	}

	public ServiceResponse<Folder> getMessages(Folder f, MessagesPageEnum page){
		ServiceResponse<Folder> response = new ServiceResponse<Folder>();
		response.setPayload(getMessagesInt(f, page, null));

		return response;
	}

	public ServiceResponse<Folder> getMessages(Folder f){
		ServiceResponse<Folder> response = new ServiceResponse<Folder>();
		response.setPayload(getMessagesInt(f, MessagesPageEnum.FIRST, null));
		return response;
	}

	private Folder getMessagesInt(Folder f, MessagesPageEnum page, SearchFormDTO search){

		// if this is not a system folder then we need to
		// fill out the data from the database to prevent hibernate errors
		if(!f.isSystemFolder()){
			folderDao.fleshFolder(f);
		}

		// if the messages collection is null then assume the
		// only valid option is the FIRST page

		if(f.getMessages() == null){
			page = MessagesPageEnum.FIRST;
		}

		MessageFilterEnum filter;

		// make sure the filter is correct based upon the userType and folder
		if(f.getOwner().getUserType() == UserTypeEnum.PATIENT){
			// if the owner is a patient the only valid filter is "ALL"
			filter = MessageFilterEnum.ALL;
		}else if(f.getId() != SystemFolderEnum.INBOX.getId()){
			// if the folder is not the inbox then the only valid filter is "ALL"
			filter = MessageFilterEnum.ALL;
		}else{
			filter = f.getFilter();
		}

		MessagesPage m = null;

		//Needs to set Folder Count for Bug#6269.
		//TODO Call the addresseeDao to update the folder counts
		//TODO Lookup all the references to this method and make sure they "SAVE" the folder to the session
		//     where appropriate.  getMessage(Folder), getMessages(Folder, Enum)
		if( f.getId() != SystemFolderEnum.DELETED.getId() && f.getId() != SystemFolderEnum.ESCALATED.getId()){
			MessageCount mc = addresseeDao.getFolderCounts(f);
			f.setCount(mc.getCount());
			f.setUnreadCount(mc.getUnreadCount());

			if( f.getId() == SystemFolderEnum.INBOX.getId() && f.getOwner().getUserType() == UserTypeEnum.CLINICIAN) {
				List<MessageCount> categories = addresseeDao.getCategoryCounts(f);
				List<SubFolder> list = new ArrayList<SubFolder>();
				List<MessageCategoryTypeEnum> all = MessageCategoryTypeEnum.toList();
				for(MessageCategoryTypeEnum each: all) {
					SubFolder sf = new SubFolder();
					sf.setId(each.getId());
					sf.setName(each.getName());
					list.add(sf);
				}

				Map<Long,MessageCount> results = new HashMap<Long,MessageCount>();
				for(MessageCount cat : categories){
					results.put(cat.getMessageCategoryTypeId(),cat);
				}

				for(SubFolder sub: list)
				{
					if( results.containsKey(sub.getId()) ) {
						sub.setCount(results.get(sub.getId()).getCount());
						sub.setUnreadCount(results.get(sub.getId()).getUnreadCount());
					}
				}
				f.setSubfolderList(list);
			}
		}
		switch(page){
			case CURRENT:{
				m = addresseeDao.getMessages(f.getOwner().getId(), f.getId(),
						f.getMessages().getPageNumber(), f.getMessages().getPageSize(),
						f.getOrderBy(), filter, f.getSortOrder(), f.getCount(), search, f.getCurrentSubFolderId());
				break;
			}

			case FIRST:{
				m = addresseeDao.getMessages(f.getOwner().getId(), f.getId(),
						0, MessagesPage.DEFAULT_PAGE_SIZE, f.getOrderBy(), filter, f.getSortOrder(),
						f.getCount(), search, f.getCurrentSubFolderId());
				break;
			}
			case NEXT:{
				if(!f.getMessages().hasNextPage()){
					throw new RuntimeException("Invalid page.");
				}

				m = addresseeDao.getMessages(f.getOwner().getId(), f.getId(),
						f.getMessages().getPageNumber() + 1, f.getMessages().getPageSize(),
						f.getOrderBy(), filter, f.getSortOrder(), f.getCount(), search, f.getCurrentSubFolderId());
				break;
			}
			case PREV:{
				if(!f.getMessages().hasPreviousPage()){
					throw new RuntimeException("Invalid page.");
				}
				m = addresseeDao.getMessages(f.getOwner().getId(), f.getId(),
						f.getMessages().getPageNumber() - 1, f.getMessages().getPageSize(),
						f.getOrderBy(), filter, f.getSortOrder(), f.getCount(), search, f.getCurrentSubFolderId());
				break;
			}
			case LAST:{
				m = addresseeDao.getMessages(f.getOwner().getId(), f.getId(),
						Integer.MAX_VALUE, f.getMessages().getPageSize(),
						f.getOrderBy(), filter, f.getSortOrder(), f.getCount(), search, f.getCurrentSubFolderId());
				break;
			}
		}

		f.setMessages(m);
		return f;
	}

    public ServiceResponse<Boolean> addUserFolder(Folder f){

	        ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
			try {
				folderDao.save(f);
			} catch (Exception e) {
				response.setPayload(Boolean.FALSE);
				return response;
			}
	        response.setPayload(Boolean.TRUE);
	        return response;
	    }


	public ServiceResponse<Boolean> renameUserFolder(Folder f){

        ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		try {
			folderDao.update(f);
		} catch (Exception e) {
			response.setPayload(Boolean.FALSE);
			return response;
		}
        response.setPayload(Boolean.TRUE);
        return response;
    }

    public ServiceResponse<Boolean> deleteUserFolder(Long folderId){

	        ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
			try {
				folderDao.delete(folderId);
			} catch (Exception e) {
				response.setPayload(Boolean.FALSE);
				return response;
			}
	        response.setPayload(Boolean.TRUE);
	        return response;
    }

    public int getMessageReminderToday(Long userId){
    	return addresseeDao.getMessageReminderToday(userId);
    }
    
    public int getCPRSAlertsCount(Long userId){
    	MessageCount msgCount = addresseeDao.getCPRSAlertsCount(userId);
    	return msgCount.getCount();
    }
    
	public FolderDao getFolderDao() {
		return folderDao;
	}

	public void setFolderDao(FolderDao folderDao) {
		this.folderDao = folderDao;
	}

	public AddresseeDao getAddresseeDao() {
		return addresseeDao;
	}

	public void setAddresseeDao(AddresseeDao addresseeDao) {
		this.addresseeDao = addresseeDao;
	}


}

