package gov.va.med.mhv.sm.service.impl;

import java.util.Calendar;
import java.util.Date;
import java.util.LinkedHashSet;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import gov.va.med.mhv.foundation.service.response.CollectionServiceResponse;
import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.foundation.util.Precondition;
import gov.va.med.mhv.sm.dao.HolidayDao;
import gov.va.med.mhv.sm.dao.MessageAttachmentDao;
import gov.va.med.mhv.sm.dao.MessageDao;
import gov.va.med.mhv.sm.dao.TriageGroupDao;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.Message;
import gov.va.med.mhv.sm.model.TriageGroup;
import gov.va.med.mhv.sm.service.EmailService;
import gov.va.med.mhv.sm.service.EscalationService;
import gov.va.med.mhv.sm.service.ReAssignMessageService;
import gov.va.med.mhv.sm.util.DateUtils;
import gov.va.med.mhv.sm.enumeration.ClinicianStatusEnum;
import gov.va.med.mhv.sm.enumeration.ParticipantTypeEnum;

public class EscalationServiceImpl implements EscalationService {

	@SuppressWarnings("unused")
	private static final Log log = LogFactory.getLog(EscalationServiceImpl.class);
	
	// populated by spring
	private int daysToEscalation;
	private int daysAfterEscalated;
	private MessageDao messageDao;
	private HolidayDao holidayDao;
	private TriageGroupDao triageGroupDao;
    private EmailService emailService;
    private MessageAttachmentDao messageAttachmentDao;
    private ReAssignMessageService reAssignMessageService;
	
	public ServiceResponse<Integer> escalate() {
		
		Precondition.assertPositive("daysToEscalation", daysToEscalation);
		Date escalationDate = getEscalationDate();
		
		log.info("EscalationDate.........."+getEscalationDate());
		log.debug("Escalation date: " + DateUtils.getEnglishDate(escalationDate));
		
		List<Message> escalateMessage = messageDao.getMessagesToEscalateByDate(escalationDate); 
		log.info("Total Escalat Message count...."+escalateMessage.size());
		int msgCount = messageDao.escalateMessages(escalationDate); // Updating Status to Escalated
		log.info("Total Updated Message count...."+msgCount);
	
		for(Message message:escalateMessage){
			
			if(!reAssignMessageService.findMsgReceivedHistoryByMessageId(message.getId())){
				reAssignMessageService.createMessageReceivedHistory(message.getId(), message.getRecipientId(), message.getSentDate(),new Long(0));
			}
			String messageStatus="In Process";
			if(message.getStatus()==ClinicianStatusEnum.INCOMPLETE) messageStatus="Unassigned";
			reAssignMessageService.createEscalatedMessageHistory(message.getId(), message.getRecipientId(), messageStatus, new Long(0));
		}
		
		ServiceResponse<Integer> response = new ServiceResponse<Integer>();
		response.setPayload(new Integer(msgCount));
		return response;
	}

	/**
	 * Get all triage groups that need to be notified that 
	 * messages have just been escalated
	 */
	public CollectionServiceResponse<TriageGroup> getTriageGroupsToNotify() {		
		CollectionServiceResponse<TriageGroup> response = new CollectionServiceResponse<TriageGroup>();
		
		// Collect all Messages where escalationNotificationNotificationDate is null
		// and escalationNotificationTries is not null
		List<Message> messages = messageDao.getEscalatedMessagesToNotify();
		
		// Return set of TriageGroups that have these messages
		// Message.getThread().getMailGroup();
		LinkedHashSet<TriageGroup> triageGroups = new LinkedHashSet<TriageGroup>();
		for (Message m : messages) {
			TriageGroup tg = m.getThread().getMailGroup();
			if(tg!=null){
				triageGroups.add(tg);
			}
		}
		
		response.addPayloads(triageGroups);
		return response;
	}
	
	
	
	
	/**
	 * Send email message notifying triage group that there are messages
	 * that have been escalated
	 * 
	 * @param triageGroup group to be notified
	 * @return VoidServiceResponse
	 */
	public ServiceResponse<Integer> notifyTriageGroupOnEscalation(TriageGroup triageGroup) {
		ServiceResponse<Integer> response = new ServiceResponse<Integer>();
		
		// Retrieve the messages that will be included in the notification
		// to the triage group members
		
		
		if(triageGroup!=null)
			log.info("Triage Group Name..."+triageGroup.getName());
		
		List<Message> messages = triageGroupDao.getMsgsToNotifyForTriageGroup(triageGroup);
		
		// Send the email to each clinician in the triage group
		// For all messages associate with this triage group:
		// - Increment the escalationNotificationTries 
		// - If any email was successful, set the escalationNotificationDate
		emailService.sendEscalationNotification(triageGroup, messages);
		
		response.setPayload(new Integer(messages.size()));
		return response;
	}
	
	
	/**
	 * Get all triage groups that need to be notified the Facility Admin
	 * that messages have been Escalated X number of days.
	 */
	public ServiceResponse<Boolean> notifyFacilityAdminAfterEscalatedXDays() {		
		ServiceResponse<Boolean> response = new ServiceResponse<Boolean>();
		
		//List<Message> messages = messageDao.getMessagesEscalatedXDays(getDateAfterEscalatedXDays());
		List<Object[]> adminEmailList = messageDao.getFacilityAdminToNotify(getDateAfterEscalatedXDays());
		if(adminEmailList!=null)
		{
			for(Object[] object : adminEmailList)
			{
				String emailAddress = null;
				if(object[0]!=null){
					 emailAddress = object[0].toString();
				}
				String triageGroupName = object[1].toString();
				if(emailAddress!=null){
					//emailService.sendNotificationToFacilityAdmin(triageGroupName,emailAddress,daysAfterEscalated);
				}
				else{
					log.info("EmailAddress not found for the facility under the triage group"+triageGroupName);
				}
			}
			response.setPayload(Boolean.TRUE);
		}
		return response;
	}
	 
	
	
	private Date getEscalationDate(){
		
		int count = 0;
		Calendar c = Calendar.getInstance();
		
		c.set(Calendar.HOUR_OF_DAY, 0);
		c.set(Calendar.MINUTE, 0);
		c.set(Calendar.SECOND, 0);
		c.set(Calendar.MILLISECOND, 0);
		
		while(count < daysToEscalation){
			c.add(Calendar.DAY_OF_MONTH, -1);
			if(isNonBusinessDay(c)){
				continue;
			}
			count++;
		}
		
		return c.getTime();
	}
	
	private Date getDateAfterEscalatedXDays(){
		
		int count = 0;
		Calendar c = Calendar.getInstance();
		
		c.set(Calendar.HOUR_OF_DAY, 0);
		c.set(Calendar.MINUTE, 0);
		c.set(Calendar.SECOND, 0);
		c.set(Calendar.MILLISECOND, 0);
		
		while(count < daysAfterEscalated){
			c.add(Calendar.DAY_OF_MONTH, -1);
			if(isNonBusinessDay(c)){
				continue;
			}
			count++;
		}
		
		return c.getTime();
	}
	

	
	private boolean isNonBusinessDay(Calendar d){
		
		if(d.get(Calendar.DAY_OF_WEEK) == Calendar.SATURDAY)
			return true;
		
		if(d.get(Calendar.DAY_OF_WEEK) == Calendar.SUNDAY){
			return true;
		}
		
		if(holidayDao.isDateHoliday(d.getTime())){
			return true;
		}
		
		return false;
	}
	
	/*public void removeUnassignedAttachments(){
		Calendar c = Calendar.getInstance();
		c.add(Calendar.DAY_OF_MONTH, -1);
		int count = messageAttachmentDao.removeMessageAttachment(c.getTime());
		log.info("Total Message Attachments Removed.."+count);
	}*/


	public int getDaysToEscalation() {
		return daysToEscalation;
	}
	public void setDaysToEscalation(int daysToEscalation) {
		this.daysToEscalation = daysToEscalation;
	}
	public MessageDao getMessageDao() {
		return messageDao;
	}
	public void setMessageDao(MessageDao messageDao) {
		this.messageDao = messageDao;
	}
	public TriageGroupDao getTriageGroupDao() {
		return triageGroupDao;
	}
	public void setTriageGroupDao(TriageGroupDao triageGroupDao) {
		this.triageGroupDao = triageGroupDao;
	}


	public HolidayDao getHolidayDao() {
		return holidayDao;
	}


	public void setHolidayDao(HolidayDao holidayDao) {
		this.holidayDao = holidayDao;
	}
	
	public EmailService getEmailService() {
		return emailService;
	}
	public void setEmailService(EmailService emailService) {
		this.emailService = emailService;
	}

	public int getDaysAfterEscalated() {
		return daysAfterEscalated;
	}

	public void setDaysAfterEscalated(int daysAfterEscalated) {
		this.daysAfterEscalated = daysAfterEscalated;
	}

	public MessageAttachmentDao getMessageAttachmentDao() {
		return messageAttachmentDao;
	}

	public ReAssignMessageService getReAssignMessageService() {
		return reAssignMessageService;
	}

	public void setReAssignMessageService(
			ReAssignMessageService reAssignMessageService) {
		this.reAssignMessageService = reAssignMessageService;
	}

	public void setMessageAttachmentDao(MessageAttachmentDao messageAttachmentDao) {
		this.messageAttachmentDao = messageAttachmentDao;
	}
	
	
}
