package gov.va.med.mhv.sm.service.impl;

import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.foundation.util.ConfigurationManager;
import gov.va.med.mhv.sm.dao.UserDao;
import gov.va.med.mhv.sm.enumeration.ParticipantTypeEnum;
import gov.va.med.mhv.sm.enumeration.EmailNotificationTypeEnum;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.Message;
import gov.va.med.mhv.sm.model.TriageGroup;
import gov.va.med.mhv.sm.model.User;
import gov.va.med.mhv.sm.service.EmailService;
import gov.va.med.mhv.sm.service.LoggingService;
import gov.va.med.mhv.sm.util.DateUtils;

import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.mail.MailException;
import org.springframework.mail.MailSender;
import org.springframework.mail.SimpleMailMessage;

public class EmailServiceImpl implements EmailService{

	
	@SuppressWarnings("unused")
	private static final Log log = LogFactory.getLog(EmailServiceImpl.class);
	private static final Properties PROPS; 
	private UserDao userDao;
	private MailSender mailSender;
	private LoggingService loggingService;
	private SimpleMailMessage templateMessage;
	private static final boolean newMessageNotificationEnabled;
	private static final boolean assignmentNotificationEnabled;
	private static final boolean escalationNotificationEnabled;
	private static final boolean facilityAdminNotificationEnabled;

	// ensure sort puts unassigned messages last
	private static String ESCALATION_NOTIFICATION_NOT_ASSIGNED = "~~~";
	
	static {
		String fileName = "smMailCfg.properties";
		Properties configFile = ConfigurationManager.getConfiguration("/" + fileName);
		if ( configFile != null ) {
			PROPS = configFile;
		} else {
			if(log.isErrorEnabled()){
				log.error("Could not find properties file: " + fileName);
			}
			PROPS = new Properties();
		}
		newMessageNotificationEnabled = Boolean.parseBoolean(PROPS.getProperty("newMessageNotification.enable", "false"));
		assignmentNotificationEnabled = Boolean.parseBoolean(PROPS.getProperty("assignmentNotification.enable", "false"));
		escalationNotificationEnabled = Boolean.parseBoolean(PROPS.getProperty("escalationNotification.enable", "false"));
		facilityAdminNotificationEnabled = Boolean.parseBoolean(PROPS.getProperty("facilityAdminNotificaton.enable", "false"));
	}
	
	// Send New Email Notification to the user
	public void sendNewMessageEmailNotification(User user,TriageGroup triageGroup, Message message)
	{
		boolean sent = false;
		if (newMessageNotificationEnabled) 
		{
				if(log.isDebugEnabled()){
					log.debug("Sending new message notification to " + user.getId() + ":" + user.getEmail());
				}
				sent = sendNewEmail(user, getNewMessageNotificationSubject(user), getNewMessageNotificationBody(user,triageGroup,message),EmailNotificationTypeEnum.NEW_MESSAGE_EMAIL_NOTIFICATION, triageGroup.getName(),message);
				if(sent)
				{
					try
					{
						//update the users last notification date
						// making sure not to accidentally persist
						// transient changes made by the client
						User tmp = userDao.findById(user.getId());
						tmp.setLastNotification(new Date());
						userDao.save(tmp);
					}catch(Exception e1){
						if(log.isErrorEnabled()){
							log.error("EmailServiceImpl Error->UpdatingLastEmailNotification Part----->"+e1.getStackTrace());
						}
					}
					
				}
		} 
		else 
		{
			if(log.isDebugEnabled()){
				log.debug("New message notification disabled - not sending email to " + user.getId() + ":" + user.getEmail());
			}	
		}
		
		
		
		
	}
	
	private boolean sendNewEmail(User u, String subject, String body,EmailNotificationTypeEnum emailMessageType,String groupName, Message message){
		
		SimpleMailMessage msg = new SimpleMailMessage(this.templateMessage);
		msg.setTo(u.getEmail());
		msg.setSubject(subject);
		msg.setText(body);
		StringBuffer logMessage = new StringBuffer("");
		
		try
		{
            this.mailSender.send(msg);
            
            logMessage.append("Success{SenderId:").append(message.getSenderId());
            logMessage.append(" RecipientId: ").append(u.getId());
            logMessage.append(" MessageId: ").append(message.getId());
            logMessage.append(" Group:").append(groupName).append(", Email: ").append(u.getEmail()).append(", Created Date:").append(new Date());
            logMessage.append(" Status:Success|mailType").append(emailMessageType.getId()).append("}");
            if(log.isInfoEnabled()){
            	log.info(logMessage.toString());
            	log.info("---------------------------------------------------------------------------------");
            }
            if(emailMessageType.equals(EmailNotificationTypeEnum.NEW_MESSAGE_EMAIL_NOTIFICATION)){
            	getLoggingService().newEmailNotification(u.getId(),message.getId(),logMessage.toString(),true,emailMessageType);	
            }else if(emailMessageType.equals(EmailNotificationTypeEnum.REASSIGN_MESSAGE_EMAIL_NOTIFICATION)){
            	getLoggingService().reAssignEmailNotification(u.getId(),message.getId(),logMessage.toString(),true,emailMessageType);	
            }
        }
        catch(MailException e){
        	logMessage.append("Failed{SenderId:").append(message.getSenderId());
            logMessage.append(" RecipientId: ").append(u.getId());
            logMessage.append(" MessageId: ").append(message.getId());
            logMessage.append(" Group:").append(groupName).append(", Email: ").append(u.getEmail()).append(", Created Date:").append(new Date());
            logMessage.append(" Status:Error|mailType:").append(emailMessageType.getName()).append(" Error: ").append("}");
            if(log.isErrorEnabled()){
            	log.error("ERROR Sending Email Notification: " + e.getMessage());
            	log.error(logMessage.toString());
            }
        	getLoggingService().messageSentError(u.getId(),message.getId(),logMessage.toString(),true);	
            
        	return false;
        }
        
        return true;
	}
	
	private String getNewMessageNotificationSubject(User u) {
		String defaultSubject= "Secure Message Notification";
		String type = null;
		if ( u.getParticipantType().equals(ParticipantTypeEnum.CLINICIAN) ){
			type = "clinician";
		} else {
			 type = "patient";
		}
		return PROPS.getProperty("notificationMessage." + type + ".subject", defaultSubject);
	}

	
	private String getNewMessageNotificationBody(User u,TriageGroup tg,Message message) {
		StringBuffer mailBody = new StringBuffer();
		mailBody.append(PROPS.getProperty("notificationMessage.body.groupNamePrefix")).append(": ").append(tg.getName()).append(", Message Id:").append(message.getId()).append("\n\n");
		// Get the stationNo and DUZ to construct url, Clinician can login by click the login URL in EmailNotification message. (CR#4646)
		if (u.getParticipantType().equals(ParticipantTypeEnum.CLINICIAN))
		{
			Clinician clinician = getUserDao().findClinicianById(u.getId());
			String clinicianLoginUrl = getClinicianLoginUrl(clinician);
			mailBody.append(PROPS.getProperty("notificationMessage.clinician.body.line1")).append(" ").append(clinicianLoginUrl).append(" ").append(PROPS.getProperty("notificationMessage.clinician.body.line2"));
		}
		else 
		{
			mailBody.append(PROPS.getProperty("notificationMessage.patient.body"));
		}
		return mailBody.append(PROPS.getProperty("notificationMessage.body.donotreply")).toString();
	}


	private String getClinicianLoginUrl(Clinician c) {
		String stationNo = c.getStationNo();
		String duz = c.getDuz();
		String clinicianLoginUrl = PROPS.getProperty("notificationMessage.clinician.body.login.url")+"station="+stationNo+"&DUZ="+duz;

		return clinicianLoginUrl;
	}
	
	
	
	public void sendAssigmentNotification(User u,TriageGroup tg,Message message){
		
		String subject= PROPS.getProperty("reAssignMessageNotification.clinician.subject");
		String body = getNewMessageNotificationBody(u,tg,message);
		
		boolean sent = false;
		if (assignmentNotificationEnabled) {
			if(log.isDebugEnabled()){
				log.debug("Sending assignment notification to " + u.getId() + ":" + u.getEmail());
			}
			sent = sendNewEmail(u, subject, body,EmailNotificationTypeEnum.REASSIGN_MESSAGE_EMAIL_NOTIFICATION, tg.getName(),message);
		} else {
			if(log.isDebugEnabled()){
				log.debug("Assignment notification disabled - not sending email to " + u.getId() + ":" + u.getEmail());
			}
		}
		
		// update the users last notification date
		// making sure not to accidentally persist
		// transient changes made by the client
		if(sent){
			User tmp = userDao.findById(u.getId());
			tmp.setLastNotification(new Date());
			userDao.save(tmp);
		}
		
	}
	
	public ServiceResponse sendMessage(String from, String subject, 
			String message, String recipient) {
		return sendMessageWithReplyTo(from, subject, message, recipient, null);
	}
		
	/**
	 * Execute the SendMessageWithReplyTo service
	 *
	 * @return ServiceResponse
	 */
	public ServiceResponse sendMessageWithReplyTo(String from, String subject, 
			String message, String recipient, String replyTo) {
		ServiceResponse result = new ServiceResponse();
		SimpleMailMessage msg = new SimpleMailMessage(this.templateMessage);
		msg.setTo(recipient);
		msg.setSubject(subject);
		msg.setText(message);
		msg.setFrom(from);
		if ( replyTo != null ) {
			msg.setReplyTo(replyTo);
		}
		try{
			getMailSender().send(msg);
		} catch(MailException e) {
			if(log.isErrorEnabled()){
				log.error("Error sending email notification: " + e.getMessage());
			}
			result.addError(EMAIL_SEND_ERROR);
		}
		return result;		
	}	


	private boolean sendMail(String to, String subject, String body){
		
		SimpleMailMessage msg = new SimpleMailMessage(this.templateMessage);
		msg.setTo(to);
		msg.setSubject(subject);
		msg.setText(body);
		
		try{
            this.mailSender.send(msg);
            // Log here the Email Send...
        }
        catch(MailException e){
        	if(log.isErrorEnabled()){
        		log.error("Error sending email notification: " + e.getMessage());
        	}
        	// Log here with Exception Email..
        	return false;
        }
        
        return true;
	}

	public void sendNotificationToFacilityAdmin(String triageGroupName, String emailAddress, int daysExceeded) {

		String mailBody = "";
		String subject = PROPS.getProperty("notificationMessage.facilityAdmin.subject");
		mailBody = PROPS.getProperty("notificationMessage.facilityAdmin.body.line1")+": "+triageGroupName+"\n\n";
		mailBody = mailBody+PROPS.getProperty("notificationMessage.facilityAdmin.body.line2")+" "+daysExceeded+" "+PROPS.getProperty("notificationMessage.facilityAdmin.body.line3");
		SimpleMailMessage message = new SimpleMailMessage(this.templateMessage);
		message.setSubject(subject);
		message.setTo(emailAddress);
		message.setText(mailBody);
	    //this.mailSender.send(message);
	}
	
	
	public void sendEscalationNotification(TriageGroup tg, List<Message> messages) {

		boolean anySent = false;
		boolean sent = false;
		Date notificationDate = null;
		String subject = PROPS.getProperty("escalationNotificationMessage.subject");
		String bodyPreUrl = getEscalationNotificationBodyPreUrl(tg, messages);
		String bodyPostUrl = PROPS.getProperty("escalationNotificationMessage.body.postMessagesPostUrl");

		int numClinicians = tg.getClinicians().size();
		
		if(log.isDebugEnabled()){
			log.debug("Escalation Notification - Triage Group \""
				 + tg.getId()
				 + ":"
				 + tg.getName()
				 + "\" has "
				 + numClinicians 
				 + " clinicians.");
		}
		if (tg.getClinicians().size() > 0) {
			for (Clinician c: tg.getClinicians()) {
				String to = c.getEmail();
				String body = bodyPreUrl + getClinicianLoginUrl(c) + bodyPostUrl;
		
				if (escalationNotificationEnabled) {
					if(log.isDebugEnabled()){
						log.debug("Sending escalation notification to " + c.getId() + ":" + to);
					}
					sent = sendMail(to, subject, body);
				} else {
					if(log.isDebugEnabled()){
						log.debug("Escalation notification is disabled - not sending email to " + c.getId() + ":" + to);
					}
				}
			
				if (!anySent && sent) {
					anySent = true;
					notificationDate = new Date();
				}
			}

			incrementEscalationNotificationTries(messages);
			if (anySent) {
				updateEscalationNotificationDate(messages, notificationDate);
			} else {
				if(log.isDebugEnabled()){
					log.debug("No notifications were successfully sent out for these escalated messages.");
				}
			}
		} else {
			if(log.isDebugEnabled()){
				log.debug("No notifications sent out for these escalated messages because there were no clinicians to notify.");
			}
		}
	}

	public void sendEmailNotificationToFacilityAdmins(Long userId,Long messageId,String triageGroup,Long assignedToStation,Long assignedByStation,String visnName,String[] adminsEmail){
		if(facilityAdminNotificationEnabled){
			String subject = PROPS.getProperty("notificationMessage.clinician.subject");
			String messageBody = PROPS.getProperty("facilityAdminsEmail.body.line1")+" "+assignedToStation+"\n\n";
			messageBody=messageBody + PROPS.getProperty("facilityAdminsEmail.body.line2")+" "+triageGroup+" ";
			messageBody=messageBody + "from "+visnName+"/"+assignedByStation+" "+PROPS.getProperty("facilityAdminsEmail.body.line3")+"\n\n";
			messageBody=messageBody + PROPS.getProperty("facilityAdminsEmail.body.line4");
			SimpleMailMessage message = new SimpleMailMessage(this.templateMessage);
			message.setTo(adminsEmail);
			message.setSubject(subject);
			message.setText(messageBody);
			try{
				this.mailSender.send(message);
				getLoggingService().facilityAdminEmailNotification(userId,messageId,"Email Successful:"+Arrays.toString(adminsEmail),true,EmailNotificationTypeEnum.FACILITY_ADMIN_EMAIL_NOTIFICATION);
				if(log.isInfoEnabled()){
					log.info(">>>>>>888Facility Email Notification Successful......"+Arrays.toString(adminsEmail)+" Facility Name"+assignedToStation);
				}
			}catch(Exception e1){
				getLoggingService().facilityAdminEmailNotification(userId,messageId,"Email Failure"+Arrays.toString(adminsEmail),true,EmailNotificationTypeEnum.FACILITY_ADMIN_EMAIL_NOTIFICATION);
				if(log.isErrorEnabled()){
					log.error(">>>>>>999Facility Email Notification Failure......"+e1.getStackTrace()+"Facility Name"+assignedToStation);
				}
			}	
		}else{
			if(log.isInfoEnabled()){
				log.info("Facility admin notification is not enabled");
			}
		}
	}
	
	private String getEscalationNotificationBodyPreUrl(TriageGroup tg, List<Message> messages) {
		
		String mailBody = PROPS.getProperty("escalationNotificationMessage.body.groupNamePrefix")+" ";
		mailBody += tg.getName()+" ";
		mailBody += PROPS.getProperty("escalationNotificationMessage.body.groupNameSuffix");
		mailBody += getAssignedMessagesBody(messages);
		mailBody += "\n";
		mailBody += PROPS.getProperty("escalationNotificationMessage.body.postMessagesPreUrl");
		return mailBody;
	}
	
	/**
	 *
	 * Generate sorted grouped list of messages as specified in the
	 * supplementary spec 2.28.c
	 *
	 * + <# of assigned messages> messages assigned to LastName, FirstName
	 * + 999 messages assigned to Smith, John
	 * + 999 messages not assigned
	 *
	 */
	private String getAssignedMessagesBody(List<Message> messages) {
		TreeMap<String, List<Message>> groupedMessages = groupAndSortMessages(messages);

		String messagesBody = "";
		int numMessages = 0;
		
		for (Map.Entry<String, List<Message>> entry : groupedMessages.entrySet()) {
			if (!ESCALATION_NOTIFICATION_NOT_ASSIGNED.equals(entry.getKey())) {
				numMessages = entry.getValue().size();
				messagesBody += "+ "
					+ numMessages
					+ (numMessages > 1 ? " messages " : " message ")
					+ PROPS.getProperty("escalationNotificationMessage.body.assignedLabel")
					+ " "
					+ entry.getKey()
					+ "\n";
			}
		}

		if (groupedMessages.containsKey(ESCALATION_NOTIFICATION_NOT_ASSIGNED)) {
			List<Message> unassignedMessages = groupedMessages.get(ESCALATION_NOTIFICATION_NOT_ASSIGNED);
			numMessages = unassignedMessages.size();
			messagesBody += "+ "
				+ numMessages
				+ (numMessages > 1 ? " messages " : " message ")
				+ PROPS.getProperty("escalationNotificationMessage.body.unassignedLabel")
				+ "\n";
		}
		
		return messagesBody;
	}

	/**
	 *
	 * Create a TreeMap with sorted keys being the assigned clinician's
	 * name and values as lists of messages sorted properly (current
	 * requirements are to sort by subject)
	 * 
	 */
	private TreeMap<String, List<Message>> groupAndSortMessages(List<Message> messages) {
		TreeMap<String, List<Message>> groupedMessages = new TreeMap<String, List<Message>>();
		List<Message> assignedMessages;
		
		// group messages by assignee
		for (Message m : messages) {
			Clinician c = m.getAssignedTo();
			
			String name = (c == null) ? ESCALATION_NOTIFICATION_NOT_ASSIGNED : c.getName();
			
			if (groupedMessages.containsKey(name)) {
				assignedMessages = groupedMessages.get(name);
			} else {
				assignedMessages = new LinkedList<Message>();
				groupedMessages.put(name, assignedMessages);
			}
			assignedMessages.add(m);
		}

		// sort messages for each assignee
		for (Map.Entry<String, List<Message>> entry : groupedMessages.entrySet()) {
			Collections.sort(entry.getValue(), MESSAGE_BY_SUBJECT_THEN_DATE_SORTER);
		}

		return groupedMessages;
	}

	private static final Comparator<Message> MESSAGE_BY_SUBJECT_THEN_DATE_SORTER = new Comparator<Message>() {
		public int compare(Message a, Message b) {
			if (a == null || b == null) return 0;
			if(a.getThread().getSubject() == null || b.getThread().getSubject() == null) return 0;
			int result = a.getThread().getSubject().compareTo(b.getThread().getSubject());
			if ( result == 0 ) {
				result = a.getSentDate().compareTo(b.getSentDate());
			}
			return result;
		}
	};

	
	private void incrementEscalationNotificationTries(List<Message> messages) {
		for (Message m : messages) {
			if(m.getEscalationNotificationTries()!=null){
				m.setEscalationNotificationTries(m.getEscalationNotificationTries() + 1);
			}
		}
	}

	private void updateEscalationNotificationDate(List<Message> messages, Date notificationDate) {
		for (Message m : messages) {
			m.setEscalationNotificationDate(notificationDate);
		}
	}
	
	public UserDao getUserDao() {
		return userDao;
	}
	public void setUserDao(UserDao userDao) {
		this.userDao = userDao;
	}
	public MailSender getMailSender() {
		return mailSender;
	}
	public void setMailSender(MailSender mailSender) {
		this.mailSender = mailSender;
	}
	public SimpleMailMessage getTemplateMessage() {
		return templateMessage;
	}
	public void setTemplateMessage(SimpleMailMessage templateMessage) {
		this.templateMessage = templateMessage;
	}


	public LoggingService getLoggingService() {
		return loggingService;
	}


	public void setLoggingService(LoggingService loggingService) {
		this.loggingService = loggingService;
	}
	
}
