package gov.va.med.mhv.sm.service.impl;

import gov.va.med.mhv.foundation.service.impl.AbstractBaseServiceImpl;
import gov.va.med.mhv.foundation.service.response.CollectionServiceResponse;
import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.foundation.util.ConfigurationManager;
import gov.va.med.mhv.foundation.util.Precondition;
import gov.va.med.mhv.persistence.dao.hibernate.BaseEntityDaoHibernate;
import gov.va.med.mhv.sm.crypto.IllegalKeyException;
import gov.va.med.mhv.sm.crypto.UserCredentialsBuilder;
import gov.va.med.mhv.sm.dao.CredentialsDao;
import gov.va.med.mhv.sm.dao.DistributionGroupDao;
import gov.va.med.mhv.sm.dao.FacilityDao;
import gov.va.med.mhv.sm.dao.TriageGroupDao;
import gov.va.med.mhv.sm.dao.UserDao;
import gov.va.med.mhv.sm.enumeration.UserTypeEnum;
import gov.va.med.mhv.sm.model.Administrator;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.Credentials;
import gov.va.med.mhv.sm.model.DistributionGroup;
import gov.va.med.mhv.sm.model.Facility;
import gov.va.med.mhv.sm.model.MHVPatient;
import gov.va.med.mhv.sm.model.MhvAuthenticationSubject;
import gov.va.med.mhv.sm.model.MhvIntegrationSettings;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.PatientBlockedTriageGroup;
import gov.va.med.mhv.sm.model.PatientFacility;
import gov.va.med.mhv.sm.model.TriageGroup;
import gov.va.med.mhv.sm.model.User;
import gov.va.med.mhv.sm.model.decorator.MhvAuthenticationSubjectDecorator;
import gov.va.med.mhv.sm.service.AdminService;
import gov.va.med.mhv.sm.service.AuthenticationService;
import gov.va.med.mhv.sm.service.LoggingService;
import gov.va.med.mhv.sm.service.PatientBlockedService;
import gov.va.med.mhv.sm.service.RelationshipManagementService;
import gov.va.med.mhv.sm.util.MhvIntegrationUtils;
import gov.va.med.mhv.sm.util.UserUtils;
import org.owasp.esapi.ESAPI;
import org.owasp.esapi.errors.IntrusionException;


import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.Locale;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.directory.Attributes;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.InitialLdapContext;
import javax.naming.ldap.LdapContext;

import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class AuthenticationServiceImpl extends AbstractBaseServiceImpl
	implements AuthenticationService
{

	private static final Log LOG = LogFactory.getLog(AuthenticationServiceImpl.
		class);

	private UserDao userDao;
	private FacilityDao facilityDao;
	private TriageGroupDao triageGroupDao;
	private LoggingService loggingService;
	private DistributionGroupDao distributionGroupDao;
	private RelationshipManagementService relationshipManagementService;
	private BaseEntityDaoHibernate<PatientFacility, Long> patientFacilityDao;
	private final MhvIntegrationUtils mhvIntegrationUtils;
	private final CredentialsDao credentialsDao;
	private final UserCredentialsBuilder credentialsBuilder;
	private PatientBlockedService patientBlockedService;
	private AdminService adminService;
	

	@Override
	public Log getLog() {
		return LOG;
	}

	public AuthenticationServiceImpl(MhvIntegrationUtils mhvIntegrationUtils,
		CredentialsDao credentialsDao,
		UserCredentialsBuilder credentialsBuilder)
	{
		Precondition.assertNotNull("mhvIntegrationUtils", mhvIntegrationUtils);
		Precondition.assertNotNull("credentialsDao", credentialsDao);
		Precondition.assertNotNull("credentialsBuilder", credentialsBuilder);
		this.mhvIntegrationUtils = mhvIntegrationUtils;
		this.credentialsDao = credentialsDao;
		this.credentialsBuilder = credentialsBuilder;
	}

	public ServiceResponse<Patient> authenticatePatient(
		MhvAuthenticationSubject subject)

	{
		LOG.info("Got into authenticatePatient method::MhvAuthenticationSubject");
		Precondition.assertNotNull("subject", subject);

		MhvAuthenticationSubjectDecorator decorator =
			new MhvAuthenticationSubjectDecorator(subject, mhvIntegrationUtils);

		ServiceResponse<Patient> response = new ServiceResponse<Patient>();
		if (!decorator.isValidPatient()) {
			if(LOG.isErrorEnabled()){
				LOG.error("Invalid authentication subject found " +	MhvAuthenticationSubjectDecorator.describe(decorator));
			}
			response.addError(AUTHENTICATION_FAILED);
			return response;
		}
		response.setPayload(updateRelationshipAndTriageGroups(
			UserUtils.createOrUpdatePatientFrom(decorator, userDao, facilityDao, patientFacilityDao,loggingService)));
		createOrUpdateCredentials(response.getPayload(), subject);
		return response;
	}

	public ServiceResponse<Patient> authenticatePatient(
		Credentials credentials)
	{
		LOG.info("Got into authenticatePatient method::Credentials");
		ServiceResponse<Patient> response = new ServiceResponse<Patient>();
		ServiceResponse<User> serviceResponse = authenticate(credentials,
			UserTypeEnum.PATIENT);
		if (hasError(serviceResponse)) {
			copyMessages(serviceResponse, response);
		} else {
			User user = serviceResponse.getPayload();
			if (user != null) {
				Patient patient = userDao.findPatientById(user.getId());
				addTriageGroups(patient);
				response.setPayload(patient);
			}
		}
	    return response;
	}

	public ServiceResponse<Clinician> authenticateClinician(String station,
			String duz, String userName)
		{
			ServiceResponse<Clinician> response = new ServiceResponse<Clinician>();

			Clinician clinician = userDao.findClinicianByStationAndDuz(station,
				duz, true);

			if (clinician == null) {
				if(LOG.isInfoEnabled()){
					LOG.info("Provider=>Authentication failed: Failed to find clinician '" +userName + "' with station='" + station + "' and DUZ='" +	duz + "'");
				}
				Object[] inserts = { station, duz, userName };
				response.addError(CLINICIAN_DOES_NOT_EXIST, inserts);
				return response;
			}

			if (!StringUtils.equalsIgnoreCase(userName, clinician.getUsername())) {		
				if(LOG.isInfoEnabled()){
					LOG.info("Provider=>Authentication failed: Clinician '" + userName +
							" does not match username ('" + clinician.getUsername() +
							"' of clinician with station='" + station + "' and DUZ='" +
							duz + "'");
				}
				Object[] inserts = { station, duz, userName }; 
				response.addError(CLINICIAN_DOES_NOT_MATCH, inserts);
				return response;
			}
			if(clinician!=null){
				CollectionServiceResponse<gov.va.med.mhv.sm.wsclient.adminqueriessvc.User> clinicianResponse = getAdminService().findVistaClinician(clinician.getDuz(),clinician.getStationNo());
				try
				{
						if(LOG.isInfoEnabled()){
							LOG.info("Provider Login=>"+userName+" DUZ# "+duz+" & Facility# "+station);
						}
						List<gov.va.med.mhv.sm.wsclient.adminqueriessvc.User> providersList = (List)clinicianResponse.getCollection();
						if(providersList!=null && providersList.size()!=0){
							gov.va.med.mhv.sm.wsclient.adminqueriessvc.User vistAClinician = (gov.va.med.mhv.sm.wsclient.adminqueriessvc.User)providersList.get(0);
							if(LOG.isInfoEnabled()){
								LOG.info("Provider Login=>VistAClinician Found For =>"+userName+" DUZ# "+duz+" & Facility# "+station);
								LOG.info("Provider Login=>VistA FirstName=>"+vistAClinician.getFirstName()+" VistA LastName=>"+vistAClinician.getLastName());
								LOG.info("Provider Login=>SM FirstName=>"+clinician.getFirstName()+" SM LastName=>"+clinician.getLastName());
							}
							if(!(vistAClinician.getLastName().equalsIgnoreCase(clinician.getLastName())) || !(vistAClinician.getFirstName().equalsIgnoreCase(clinician.getFirstName()))){
								response.addError(PROVIDER_NAME_MIS_MATCH_ERROR);
								return response;
							}
						}else{
							if(LOG.isInfoEnabled()){
								LOG.info("VistAClinician Not Found or VistA Connection Error =>"+userName+" DUZ# "+duz+" & Facility# "+station);
							}
							response.addError(USER_LOOKUP_ERROR);
							return response;
						}
						updateLastLogin(clinician);
				}catch(Exception e1){
					if(LOG.isErrorEnabled()){
						LOG.info("Provider Login=>Error Occured=>"+userName+" DUZ# "+duz+" & Facility# "+station+" "+e1);
					}
					response.addError(USER_LOOKUP_ERROR);
					return response;
				}
			}
			response.setPayload(updateTriageAndDistributionGroups(clinician));
			return response;
		}

	private User updateLastLogin(User user){
		User updatedUser=null;
		try{
			if(user.getCurrentLogin()!=null){
				user.setLastLogin(user.getCurrentLogin());
			}
			user.setCurrentLogin(new Date());
			updatedUser =  userDao.save(user);
			if(LOG.isInfoEnabled()){
				LOG.error("AuthenticateServiceImpl Updated Last Login Successful for->"+user.getId());
			}
		}catch(Exception e){
			if(LOG.isErrorEnabled()){
				LOG.error("AuthenticateServiceImpl->An Error Occured while Update Last Login"+user.getId());
			}
		}
		return updatedUser;
	}

	public ServiceResponse<Clinician> authenticateClinician(String station, String duz)
		{
			ServiceResponse<Clinician> response = new ServiceResponse<Clinician>();

			Clinician clinician = userDao.findClinicianByStationAndDuz(station, duz, true);

			if (clinician == null) {
				if(LOG.isWarnEnabled()){
					LOG.warn("Authentication failed: Failed to find clinician with station='" + station + "' and DUZ='" + duz + "'");
				}
				Object[] inserts = { station, duz };
				response.addError(CLINICIAN_DOES_NOT_EXIST, inserts);
				return response;
			}

			response.setPayload(updateTriageAndDistributionGroups(clinician));
			return response;
		}

	public ServiceResponse<Administrator> authenticateAdministrator(
		MhvAuthenticationSubject subject)
	{
		MhvAuthenticationSubjectDecorator decorator =
			new MhvAuthenticationSubjectDecorator(subject, mhvIntegrationUtils);

		ServiceResponse<Administrator> response =
			new ServiceResponse<Administrator>();
		if (!decorator.isValidAdministrator()) {
			if(LOG.isErrorEnabled()){
				LOG.error("Invalid authentication subject found " +
						MhvAuthenticationSubjectDecorator.describe(decorator));
			}
			response.addError(AUTHENTICATION_FAILED);
			return response;
		}
		response.setPayload(createOrUpdateAdministratorFrom(decorator));
		createOrUpdateCredentials(response.getPayload(), subject);
		return response;
	}

	public ServiceResponse<Administrator> authenticateAdministrator(
		Credentials credentials)
	{
		ServiceResponse<Administrator> response =
			new ServiceResponse<Administrator>();
		ServiceResponse<User> serviceResponse = authenticate(credentials,
				UserTypeEnum.ADMINISTRATOR);
		if (hasError(serviceResponse)) {
			copyMessages(serviceResponse, response);
		} else {
			User user = serviceResponse.getPayload();
			if (user != null) {
				response.setPayload(userDao.findAdministratorById(user.
					getId()));
			}
		}
	    return response;
	}

	public ServiceResponse<Credentials> findCredentials(User user) {
		Precondition.assertNotNull("user", user);
		ServiceResponse<Credentials> response =
			new ServiceResponse<Credentials>();
		response.setPayload(credentialsDao.findForUser(user));
		return response;
	}

	public ServiceResponse<Patient> authenticatePatientById(Long id){
		ServiceResponse<Patient> response = new ServiceResponse<Patient>();
		if (isProductionMode()) {
			response.addError(AUTHENTICATION_FAILED);
			return response;
		}
		response.setPayload(updateRelationshipAndTriageGroups(userDao.
			findPatientById(id)));
		return response;
	}

	/**
	 * KJG - added this to support new API authentication.
	 *
	 * @param mhvCorrelationId
	 * @return
	 */
	public ServiceResponse<Patient> authenticatePatient(Long mhvCorrelationId)
		{
			ServiceResponse<Patient> response = new ServiceResponse<Patient>();

			MHVPatient mhvPatient = userDao.findMHVPatientByUserProfileId(mhvCorrelationId);
			if( mhvPatient == null ) {
				response.addError(AUTHENTICATION_FAILED);
				return response;
			}

			Patient patient = UserUtils.createOrUpdatePatientFrom(mhvPatient, userDao, facilityDao, patientFacilityDao);
			patient.setUserProfileId(mhvPatient.getUserProfileId());  //Pass along the userProfileId
			response.setPayload(updateRelationshipAndTriageGroups(patient));

			return response;
		}

	public ServiceResponse<Patient> authenticatePatientByUsername(
		String username)
	{
		ServiceResponse<Patient> response = new ServiceResponse<Patient>();

		if (isProductionMode()) {
			response.addError(AUTHENTICATION_FAILED);
			return response;
		}
		response.setPayload(updateRelationshipAndTriageGroups(userDao.
			findPatientByUsername(username)));
		return response;
	}

	public ServiceResponse<Clinician> authenticateClinicianById(Long id){
		ServiceResponse<Clinician> response = new ServiceResponse<Clinician>();
		if (isProductionMode()) {
			response.addError(AUTHENTICATION_FAILED);
			return response;
		}
		response.setPayload(updateTriageAndDistributionGroups(userDao.
			findClinicianById(id)));
		return response;
	}

	public ServiceResponse<Patient> fetchPatientById(Long id) {
		ServiceResponse<Patient> response = new ServiceResponse<Patient>();
		response.setPayload(userDao.findPatientById(id));
		return response;
	}
	public ServiceResponse<Clinician> fetchClinicianById(Long id) {
		ServiceResponse<Clinician> response = new ServiceResponse<Clinician>();
		response.setPayload(updateTriageAndDistributionGroups(userDao.
				findClinicianById(id)));
		return response;
	}

	public ServiceResponse<Clinician> fleshClinicianById(Long id) {
		ServiceResponse<Clinician> response = new ServiceResponse<Clinician>();
		response.setPayload(userDao.findClinicianById(id));
		return response;
	}

	public UserDao getUserDao() {
		return userDao;
	}
	public void setUserDao(UserDao userDao) {
		this.userDao = userDao;
	}
	public TriageGroupDao getTriageGroupDao() {
		return triageGroupDao;
	}
	public void setTriageGroupDao(TriageGroupDao triageGroupDao) {
		this.triageGroupDao = triageGroupDao;
	}
	public DistributionGroupDao getDistributionGroupDao() {
		return distributionGroupDao;
	}
	public void setDistributionGroupDao(DistributionGroupDao distributionGroupDao) {
		this.distributionGroupDao = distributionGroupDao;
	}
	public FacilityDao getFacilityDao() {
		return facilityDao;
	}
	public void setFacilityDao(FacilityDao facilityDao) {
		this.facilityDao = facilityDao;
	}
	public RelationshipManagementService getRelationshipManagementService() {
		return relationshipManagementService;
	}
	public void setRelationshipManagementService(
			RelationshipManagementService relationshipManagementService) {
		this.relationshipManagementService = relationshipManagementService;
	}
	public BaseEntityDaoHibernate<PatientFacility, Long> getPatientFacilityDao() {
		return patientFacilityDao;
	}
	public void setPatientFacilityDao(
			BaseEntityDaoHibernate<PatientFacility, Long> patientFacilityDao) {
		this.patientFacilityDao = patientFacilityDao;
	}

	public LoggingService getLoggingService() {
		return loggingService;
	}
	public void setLoggingService(LoggingService loggingService) {
		this.loggingService = loggingService;
	}



	private boolean isProductionMode() {
		MhvIntegrationSettings settings  = mhvIntegrationUtils.getSettings();
		return (settings == null) || settings.isProductionMode();
	}


	private ServiceResponse<User> authenticate(Credentials credentials,
		UserTypeEnum userType)
	{
		assert userType != null;
		ServiceResponse<User> response = new ServiceResponse<User>();
	    if (credentials == null) {
			response.addError(AUTHENTICATION_FAILED);
			return response;
	    }
	    UserCredentialsBuilder.Components components =  null;
	    try {
	    	components = credentialsBuilder.extract(credentials);
	    } catch (IllegalKeyException e) {
			response.addError(AUTHENTICATION_FAILED);
			return response;
	    }
	    if ((components == null) || !userType.equals(components.getUser().
	    		getUserType()))
	    {
			response.addError(AUTHENTICATION_FAILED);
			return response;
	    }
	    Credentials expectedCredentials = credentialsDao.findNonExpiredForUser(
	    	components.getUser());
	    if (expectedCredentials == null) {
			response.addError(AUTHENTICATION_FAILED);
			return response;
		}
	    // Credentials are to be used only once
	    response.setPayload(expectedCredentials.getUser());
	    credentialsDao.delete(expectedCredentials.getId());
	    return response;
	}







	private Patient updateRelationshipAndTriageGroups(Patient patient) {
		if (patient == null) {
			return null;
		}
		if (LOG.isDebugEnabled()) {
			LOG.debug("Update relationships for '" + patient.getUsername() +
				"'");
		}
		try {
			relationshipManagementService.syncPatient(patient);
		} catch(Exception e) {
			// TODO: Exceptions should probably not be captured
			// Error passing should occur through the response object
			// The response object should be inspected
			// However, keeping this for now
			if(LOG.isErrorEnabled()){
				LOG.error("Error syncing relationships for patient '" +
				patient.getUsername() + "'", e);
			}
		}
		// IMPORTANT: Get the triage groups AFTER the relationship update
		// otherwise the new relationships will not be mapped
		addTriageGroups(patient);
		return patient;
	}

	private void addTriageGroups(Patient patient) {
		if (patient == null) {
			return;
		}
		if (LOG.isDebugEnabled()) {
			LOG.debug("Set triage groups for '" + patient.getUsername() + "'");
		}
		patient.setGroups((List)triageGroupDao.getTriageGroupsForPatient(patient));
		Collection<TriageGroup> patientAssociatedGroups = triageGroupDao.getTriageGroupsForPatient(patient);
		if(LOG.isInfoEnabled()){
			LOG.info("<<<<<<<<<<<<<101-->Patient associated Groups for patient.."+patient.getId()+"^Name"+patient.getName()+"^"+patientAssociatedGroups);
		}
		CollectionServiceResponse<PatientBlockedTriageGroup> collectionResoponse =patientBlockedService.getPatientBlockedTriageGroupsByPatientId(patient.getId());
		Collection<PatientBlockedTriageGroup> patBlockedGroups = collectionResoponse.getCollection();
		for(PatientBlockedTriageGroup blockedTg:patBlockedGroups){
			if(LOG.isInfoEnabled()){
				LOG.info("<<<<<<<<<<<<<102-->Patient Blocked Triage Group..."+blockedTg.getTriageGroupId());
			}
			for(TriageGroup associatedTg:patientAssociatedGroups){
				if(blockedTg.getTriageGroupId().equals(associatedTg.getId())){
					if(LOG.isInfoEnabled()){
						LOG.info("<<<<<<<<<<<<<103-->Removing Patient Blocked Group from List..."+blockedTg.getTriageGroupId()+"^"+associatedTg.getName());
					}
					patientAssociatedGroups.remove(associatedTg);
					if(LOG.isInfoEnabled()){
						LOG.info("<<<<<<<<<<<<<104-AuthenticationServiceImpl->Removed...");
					}
					break;
				}
			}
	    }

		// Now remove the Facility Triage Groups from the list -
		CollectionServiceResponse<TriageGroup> collectionFacilityResoponse =patientBlockedService.getPatientBlockedAllFacilityTriageGroupsByPatientId(patient.getId());
		Collection<TriageGroup> patBlockedFacilityGroups = collectionFacilityResoponse.getCollection();
		for(TriageGroup blockedFacilityTg:patBlockedFacilityGroups){
			if(LOG.isInfoEnabled()){
				LOG.info("<<<<<<<<<<<<<102a-->Patient Blocked Facility Triage Group..."+blockedFacilityTg.getId());
			}
			for(TriageGroup associatedTg:patientAssociatedGroups){
				if(blockedFacilityTg.getId().equals(associatedTg.getId())){
					if(LOG.isInfoEnabled()){
						LOG.info("<<<<<<<<<<<<<103a-->Removing Patient Blocked Facility Triage Group from List..."+blockedFacilityTg.getId()+"^"+blockedFacilityTg.getName());
					}
					patientAssociatedGroups.remove(associatedTg);
					if(LOG.isInfoEnabled()){
						LOG.info("<<<<<<<<<<<<<104a-AuthenticationServiceImpl->Removed...");
					}
					break;
				}
			}
	    }

		patient.setUserAssociatedGroups(patientAssociatedGroups);
	}

	private Clinician updateTriageAndDistributionGroups(Clinician clinician) {
		Locale.setDefault(Locale.ENGLISH);
		if (clinician == null) {
			return clinician;
		}
		clinician.setGroups(triageGroupDao.getTriageGroupsForClinician(
			clinician));
		List<Object[]> distributionList = distributionGroupDao.getAllDistributionGroupsByUserOrShared(clinician);
		List<DistributionGroup> distList1 = new ArrayList<DistributionGroup>();
		List<DistributionGroup> distList2 = new ArrayList<DistributionGroup>();
		for(Object[] object : distributionList)
		{
			DistributionGroup distGroup = new DistributionGroup();
			distGroup.setId(new Long(object[0].toString()));
			distGroup.setName(object[1].toString());
			distGroup.setOwner(userDao.findClinicianById(new Long(object[2].toString())));

			if(object[3].toString().equals("1"))
				distGroup.setPublicGroup(true);
			else
				distGroup.setPublicGroup(false);

			if(object[4]!=null)
				distGroup.setVisnId(new Long(object[4].toString()));
			else
				distGroup.setVisnId(null);

			if(!distGroup.isPublicGroup()){
				distList1.add(distGroup);
			}else{
				distList2.add(distGroup);
			}
		}
		Collections.sort(distList1, DIST_GROUP_SORTER);
		Collections.sort(distList2, DIST_GROUP_SORTER);
		distList1.addAll(distList2);

		clinician.setDistGroups(distList1);
		distList2 = null;
		distributionList=null;

		return clinician;
	}


	private Administrator createOrUpdateAdministratorFrom(
		MhvAuthenticationSubjectDecorator subject)
	{
		assert subject != null;
		String username = StringUtils.trim(subject.getUserName());

		Administrator administrator = userDao.findAdministratorByUsername(
			username);
		if (administrator == null) {
			administrator = new Administrator();
			administrator.setLastName(subject.getLastName());
			administrator.setFirstName(subject.getFirstName());
			administrator.setUsername(username);
			administrator = (Administrator) userDao.save(administrator);
		}

		administrator.setLastName(subject.getLastName());
		administrator.setFirstName(subject.getFirstName());
		String emailAddress = getEmailAddress(username);
		if(emailAddress!=null){
			if(LOG.isInfoEnabled()){
				LOG.info("Email Address of user "+username+" from AD is"+emailAddress);
			}
			administrator.setEmail(emailAddress);
		}else{
			if(LOG.isInfoEnabled()){
				LOG.info("Email Address of user "+username+" from AD is null");
			}
		}
		if(administrator.getCurrentLogin()!=null){
			administrator.setLastLogin(administrator.getCurrentLogin());
		}
		administrator.setCurrentLogin(new Date());
		String userName = administrator.getUsername();
		administrator.setFacilities(mergeFacilities(administrator.
			getFacilities(), UserUtils.toSet(subject.getFacilities()),
			new FacilityIdentifier(), username));
		administrator.setVisns(mergeFacilities(administrator.getVisns(), UserUtils.toSet(
			subject.getVisns()), new VisnIdentifier(), userName));
		administrator.setNational(BooleanUtils.isTrue(subject.getNational()));
		administrator.setModifiedDate(new Date());
		administrator = (Administrator) userDao.save(administrator);
		return administrator;
	}

	@SuppressWarnings("unchecked")
	private String getEmailAddress(String userName) {
		String emailAddress=null;
		Properties configFile = ConfigurationManager.getConfiguration("/sm-active-directory.properties");
		String searchBase=null;
		Hashtable env=null;
		if(configFile!=null){
			if(LOG.isInfoEnabled()){
				LOG.info("<<<777>>>AD.principal....."+configFile.getProperty("ad.principal")+" AD.url..."+configFile.getProperty("ad.url")+" Ad.searchBase..."+configFile.getProperty("ad.searchBase"));
			}
			env = new Hashtable();
			env.put(Context.INITIAL_CONTEXT_FACTORY,"com.sun.jndi.ldap.LdapCtxFactory");
			env.put(Context.SECURITY_AUTHENTICATION, "simple");
			env.put(Context.SECURITY_PRINCIPAL,configFile.getProperty("ad.principal") );
			env.put(Context.SECURITY_CREDENTIALS, configFile.getProperty("ad.credentials"));
			env.put(Context.PROVIDER_URL, configFile.getProperty("ad.url"));
			searchBase = configFile.getProperty("ad.searchBase").toString();
			LdapContext ldapContext = null;
			SearchControls sc = new SearchControls();
			sc.setReturningAttributes(new String[] { "mail" }); // Specify
			sc.setSearchScope(SearchControls.SUBTREE_SCOPE);
			try
			{
					ldapContext = new InitialLdapContext(env, null);
					String searchFilter = "(&(objectClass=user)(sAMAccountName="+ ESAPI.encoder().encodeForLDAP(userName) + "))";
					NamingEnumeration searchResults = ldapContext.search(ESAPI.encoder().encodeForLDAP(searchBase), ESAPI.encoder().encodeForLDAP(searchFilter), sc);
					while (searchResults.hasMoreElements()) {
						SearchResult searchResult = (SearchResult) searchResults.next();
						Attributes attributes = searchResult.getAttributes();
						if (attributes != null) {
							if (attributes.get("mail") != null)
								emailAddress = (String)attributes.get("mail").get();
						}
					}

			}catch(IntrusionException exp1)
			 {
				if(LOG.isErrorEnabled()){
					LOG.error("AuthenticationServiceImpl->IntrusionException for user....."+userName+": "+exp1);
				}
		     }
 			catch (Exception exp2) 
 			{
					if(LOG.isErrorEnabled()){
						LOG.error("AuthenticationServiceImpl->Naming Exception Occured for userName:"+userName+": "+exp2);
					}
			}finally{
				try{
					if(ldapContext != null)
						ldapContext.close();
					}catch(Exception exp3){
						if(LOG.isErrorEnabled()){
							LOG.error("AuthenticationServiceImpl->Exception while closing ldapContext for user:"+userName+": "+exp3);
						}
					}
				}
		}
		return emailAddress;
	}

	private Collection<Facility> mergeFacilities(
		Collection<Facility> currentFacilities, Set<String> newFacilities,
		InstitutionIdentifier identifier, String userName)
	{
		if (currentFacilities == null) {
			currentFacilities = new HashSet<Facility>();
		}
		// Create a map of the facilities. Map each facility to false,
		// indicating it is (possibly) no longer current
		Map<String, Boolean> facilitiesMap = createFacilitiesMap(
			currentFacilities, identifier);

		if (newFacilities != null) {
			for(String facility : newFacilities) {
				String key = identifier.extract(facility);
				// This (vistA) facility is (still) current,
				// indicated by mapping the facility to true
				if (facilitiesMap.put(key, Boolean.TRUE) == null) {
					currentFacilities.add((identifier.getByStationNumber())
						? facilityDao.getFacilityByStationNumber(key)
						: facilityDao.getFacilityByName(key));
				}
			}
		}

		for (String key: facilitiesMap.keySet()){
			boolean patientNoLongerHasFacility = BooleanUtils.isFalse(
				facilitiesMap.get(key));
			if (patientNoLongerHasFacility) {
				// Remove the facilities that are mapped to false,
				// i.e. are no longer current
				Facility facilityToRemove = null;
				for(Facility facility: currentFacilities){
					if(identifier.getKey(facility).equals(key)) {
						facilityToRemove = facility;
						break;
					}
				}
				if(facilityToRemove != null){
					if (LOG.isDebugEnabled()) {
						LOG.debug("Remove facility '" + identifier.getKey(
							facilityToRemove) + "' from '" + userName + "'");
					}
					currentFacilities.remove(facilityToRemove);
				}
			}
		}
		return currentFacilities;
	}

	private Map<String,Boolean> createFacilitiesMap(
		Collection<Facility> facilities, InstitutionIdentifier identifier)
	{
		Map<String,Boolean> facilitiesMap = new Hashtable<String, Boolean>();
		if (facilities != null) {
			for(Facility facility: facilities) {
				facilitiesMap.put(identifier.getKey(facility), Boolean.FALSE);
			}
		}
		return facilitiesMap;
	}

	private Credentials createOrUpdateCredentials(User user,
		MhvAuthenticationSubject subject)
	{
		if (user == null) {
			return null;
		}
		if (!subject.getRequiresCredentials()) {
			return null;
		}
		Credentials credentials = credentialsBuilder.build(user, credentialsDao.
			findForUser(user));
		if (credentials == null) {
			throw new IllegalStateException(
				"Unable to create or update credentials of '" + user.
				getUsername() + "' for " + MhvAuthenticationSubject.describe(
				subject));
		}
		return credentialsDao.save(credentials);
	}



	private static interface InstitutionIdentifier {
		String getKey(Facility facility);
		String extract(String value);
		boolean getByStationNumber();
	};

	private static final class FacilityIdentifier
		implements InstitutionIdentifier
	{
		public String getKey(Facility facility) {
			return (facility != null) ? facility.getStationNumber() : null;
		}
		public String extract(String value) {
			Precondition.assertMinLength("value", value, 3);
			value = StringUtils.substring(value, 0, 3);
			try {
				Integer.parseInt(value);
			} catch (NumberFormatException e) {
				Precondition.fail("Station number not in correct format: " +
					value);
			}
			return value;
		}
		public boolean getByStationNumber() {
			return true;
		}
	}

	private static final class VisnIdentifier
		implements InstitutionIdentifier
	{
		public String getKey(Facility facility) {
			return (facility != null) ? facility.getName() : null;
		}
		public String extract(String value) {
			return value;
		}
		public boolean getByStationNumber() {
			return false;
		}
	}

	public static final Comparator<DistributionGroup> DIST_GROUP_SORTER = new Comparator<DistributionGroup>() {
		public int compare(DistributionGroup a, DistributionGroup b) {
			if(a == null || b == null) return 0;
			return a.getName().toUpperCase().compareTo(b.getName().toUpperCase());
		}
	};

	public PatientBlockedService getPatientBlockedService() {
		return patientBlockedService;
	}

	public void setPatientBlockedService(PatientBlockedService patientBlockedService) {
		this.patientBlockedService = patientBlockedService;
	}

	public AdminService getAdminService() {
		return adminService;
	}

	public void setAdminService(AdminService adminService) {
		this.adminService = adminService;
	}
	
}
