package gov.va.med.mhv.sm.service.impl;

import gov.va.med.mhv.foundation.service.response.CollectionServiceResponse;
import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.foundation.util.Precondition;
import gov.va.med.mhv.sm.ad.ADQuery;
import gov.va.med.mhv.sm.ad.ADQueryResult;
import gov.va.med.mhv.sm.dao.ClinicalUserTypeDao;
import gov.va.med.mhv.sm.dao.TriageGroupDao;
import gov.va.med.mhv.sm.dao.UserDao;
import gov.va.med.mhv.sm.dto.DiagnosisDTO;
import gov.va.med.mhv.sm.dto.LocationDssUnitsDTO;
import gov.va.med.mhv.sm.dto.PatientEligibilityClassifications;
import gov.va.med.mhv.sm.enumeration.UserStatusEnum;
import gov.va.med.mhv.sm.enumeration.EmailNotificationEnum;
import gov.va.med.mhv.sm.enumeration.WorkloadCaptureTypeEnum;
import gov.va.med.mhv.sm.model.Administrator;
import gov.va.med.mhv.sm.model.ClinicalUserType;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.TriageGroup;
import gov.va.med.mhv.sm.service.AdminService;
import gov.va.med.mhv.sm.service.response.messages.SmsServiceMessages;
import gov.va.med.mhv.sm.util.WorkloadUtil;
import gov.va.med.mhv.sm.wsclient.IntegrationServiceDelegate;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.Clinic;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.ClinicLookupResponse;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.ClinicPatientsResponse;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.PCMMProviderPatientsResponse;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.Provider;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.ProviderLookupResponse;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.SMClinic;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.SMClinicLookupResponse;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.SMNoteTitle;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.SMTIUTitleLookupResponse;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.Team;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.TeamLookupResponse;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.TeamPatientsResponse;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.User;
import gov.va.med.mhv.sm.wsclient.adminqueriessvc.UserLookupResponse;
import gov.va.med.mhv.sm.wsclient.tiusvc.DSSUnit;
import gov.va.med.mhv.sm.wsclient.tiusvc.DSSUnitQueryResponse;
import gov.va.med.mhv.sm.wsclient.tiusvc.Diagnosis;
import gov.va.med.mhv.sm.wsclient.tiusvc.DiagnosisQueryResponse;
import gov.va.med.mhv.sm.wsclient.tiusvc.ECSProcedure;
import gov.va.med.mhv.sm.wsclient.tiusvc.ECSProcedureQueryResponse;
import gov.va.med.mhv.sm.wsclient.tiusvc.PatientEClassQueryResponse;
import gov.va.med.mhv.sm.wsclient.tiusvc.PatientEligibility;
import gov.va.med.mhv.sm.wsclient.tiusvc.PatientProblemsQueryResponse;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.List;


import org.apache.commons.lang.time.DateUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.validator.EmailValidator;

public class AdminServiceImpl implements AdminService {

	@SuppressWarnings("unused")
	private static final Log log = LogFactory.getLog(AdminServiceImpl.class);

	private IntegrationServiceDelegate delegate;
	private ClinicalUserTypeDao clinicalUserTypeDao;
	private UserDao userDao;
	private TriageGroupDao triageGroupDao;
	private ADQuery adQuery;
	private final static String RESPONSE_OK="OK";
	private final static String RESPONSE_FAIL="FAIL";
	private final static String RESPONSE_ERROR="ERROR";
	private final static String RESPONSE_STATUS_MAXSTRING="MAXSTRING";
	private final static String RESPONSE_STATUS_NO_MATCH="No matches found.";


	public CollectionServiceResponse<User> findVistaClinician(String lastName, String firstName, String stationNo){

		CollectionServiceResponse<User> response = new CollectionServiceResponse<User>();

		Precondition.assertNotBlank("lastName", lastName);
		Precondition.assertNotBlank("stationNo", stationNo);

		Collection<User> users = new ArrayList<User>();

		try {

			UserLookupResponse userResponse = delegate.getUserByNameAndDUZ(lastName, firstName, null, stationNo);

			if(!userResponse.getStatus().equalsIgnoreCase("Ok")){
				if(log.isErrorEnabled()){
					log.error("Error searching for clinicians with user lookup ensemble web service: " +
						userResponse.getStatus() + "  " + userResponse.getReason());
				}
				response.addError(SmsServiceMessages.USER_LOOKUP_SERVICE_RETURNED_ERROR);
				response.setCollection(users);
				return response;
			}

			if (userResponse.getUsers() != null) {
				for(int i = 0; i < userResponse.getUsers().length; i++){
					users.add(userResponse.getUsers()[i]);
				}
			}
			response.setCollection(users);

		} catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("Error searching for clinicians with user lookup ensemble web service.", e);
			}
			response.addError(SmsServiceMessages.USER_LOOKUP_SERVICE_RETURNED_ERROR);
			response.setCollection(users);

		}

		return response;

	}

	public CollectionServiceResponse<User> findVistaClinician(String duz, String stationNo){

		CollectionServiceResponse<User> response = new CollectionServiceResponse<User>();

		Precondition.assertNotBlank("duz", duz);
		Precondition.assertNotBlank("stationNo", stationNo);

		Collection<User> users = new ArrayList<User>();

		try{
			UserLookupResponse userResponse = delegate.getUserByNameAndDUZ(null, null, duz, stationNo);
			if(!userResponse.getStatus().equalsIgnoreCase("Ok")){
				if(log.isErrorEnabled()){
					log.error("Error searching for clinicians with user lookup ensemble web service: " +
							userResponse.getStatus() + "  " + userResponse.getReason());
				}
				response.addError(SmsServiceMessages.USER_LOOKUP_SERVICE_RETURNED_ERROR);
				response.setCollection(users);
				return response;
			}

			if(userResponse.getUsers() != null){
				for(int i = 0; i < userResponse.getUsers().length; i++){
					users.add(userResponse.getUsers()[i]);
				}
			}
			response.setCollection(users);
		}catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("FindVistAClinician=>DUZ="+duz+"Station="+stationNo+" Remote Exception Occured",e);
			}
			response.addError(SmsServiceMessages.USER_LOOKUP_SERVICE_RETURNED_ERROR);
			response.setCollection(users);
		}
		catch(Exception e1){
			if(log.isErrorEnabled()){
				log.error("FindVistAClinician=>DUZ="+duz+"Station="+stationNo+" Exception Occured",e1);
			}
			response.addError(SmsServiceMessages.USER_LOOKUP_SERVICE_RETURNED_ERROR);
			response.setCollection(users);
		}
		return response;
	}



	public ServiceResponse<ADQueryResult> findADAccount(String username){

		ServiceResponse<ADQueryResult> response = new ServiceResponse<ADQueryResult>();

		Precondition.assertNotBlank("username", username);

		ADQuery adq = getAdQuery();
		List<ADQueryResult> adr = adq.queryUser(username);

		if(adr == null || adr.size() == 0){
			response.addError(SmsServiceMessages.ADQUERY_NO_RESULTS);
			return response;
		}

		if(adr.size() > 1){
			response.addError(SmsServiceMessages.ADQUERY_NON_UNIQUE_RESULTS);
			return response;
		}

		response.setPayload(adr.get(0));
		return response;
	}



	public ServiceResponse<Clinician> createClinicianAccount(String stationNo, User vistaUser, ADQueryResult adr,
			ClinicalUserType clinicalUserType, Administrator admin){

		Precondition.assertNotNull("admin", admin);
		Precondition.assertNotBlank("stationNo", stationNo);
		Precondition.assertNotNull("vistaUser", vistaUser);
		Precondition.assertNotNull("adquery", adr);
		Precondition.assertNotBlank("vistaUser.ien", vistaUser.getIEN());

		//TODO:  make sure we have a valid administrator

		ServiceResponse<Clinician> response;

		boolean provider = false;

		if(vistaUser.getProviderIndicator() != null && vistaUser.getProviderIndicator().equalsIgnoreCase("PROVIDER")){
			provider = true;
		}


		// Do *not* use the ad results for the first and last names.  We need
		// to use the names from vista so that the name match for TIU notes will succeed.
		response = createClinicianAccount(adr.getUserName(),
		 			 	stationNo, vistaUser.getFirstName(), vistaUser.getLastName(),
		 			 	adr.getEmail(), vistaUser.getSSN(), vistaUser.getIEN(),
		 			 	clinicalUserType, provider);



		return response;
	}

	  public ServiceResponse<Clinician> createClinicianAccount(String networkId, String stationNo, String firstName, String lastName,
						String email, String ssn, String duz, ClinicalUserType clinicalUserType, boolean isProvider){

		Precondition.assertNotBlank("stationNo", stationNo);
		Precondition.assertNotBlank("firstname", firstName);
		Precondition.assertNotBlank("lastname", lastName);
		Precondition.assertNotBlank("duz", duz);
		Precondition.assertNotBlank("networkId", networkId);
		//Precondition.assertNotBlank("ssn", ssn);

		ServiceResponse<Clinician> response = new ServiceResponse<Clinician>();
		EmailValidator emailV = EmailValidator.getInstance();

		Clinician c = null;

		// if clinician already exists return the old one
		// and put in an error message
		c = userDao.findAuthClinicianByStationAndDuz(stationNo, duz);
		if(c != null){

			if(c.isActive() == false){
				response.addError(SmsServiceMessages.CLINICIAN_ACCOUNT_INACTIVE);
				response.setPayload(null);
				return response;
			}

			response.addInfo(SmsServiceMessages.CLINICIAN_ACCOUNT_ALREADY_EXISTS);
			response.setPayload(c);
			return response;
		}

		c = new Clinician();

		c.setStationNo(stationNo);
		c.setFirstName(firstName);
		c.setLastName(lastName);
		c.setDuz(duz);
		c.setEmailNotification(EmailNotificationEnum.EACH_MESSAGE);
		c.setUsername(networkId);
		c.setSsn(ssn);
		c.setEmail(email);
		c.setStatus(UserStatusEnum.OPT_IN);
		c.setProvider(isProvider);
		c.setClinicalUserType(clinicalUserType);


		try {
			c = (Clinician) userDao.save(c);
			response.setPayload(c);

		} catch (Exception e) {
			if(log.isErrorEnabled()){
				log.error("Error saving new clinican: " + c.getUsername() + "^" + c.getStationNo(), e);
			}
			response.addError(SmsServiceMessages.CLINICIAN_ACCOUNT_CREATION_ERROR);
			response.setPayload(null);
		}

		return response;

	}

	public CollectionServiceResponse<ClinicalUserType> getClinicalUserTypes(){

		CollectionServiceResponse<ClinicalUserType> response = new CollectionServiceResponse<ClinicalUserType>();
		List<ClinicalUserType> l = clinicalUserTypeDao.findAll();
		response.setCollection(l);
		return response;

	}



	public CollectionServiceResponse<Provider> searchPcmmProviders(String stationNo, String lastName, String firstName){

		Precondition.assertNotBlank("stationNo", stationNo);
		Precondition.assertNotBlank("lastName", lastName);

		CollectionServiceResponse<Provider> response = new CollectionServiceResponse<Provider>();
		try {
			ProviderLookupResponse r = delegate.getProvidersByNameAndDUZ(lastName, firstName, null, stationNo);

			if (r == null || r.getProviders() == null || r.getProviders().length == 0) {
				return response;
			}

			List<Provider> list = new ArrayList<Provider>();
			for(int i = 0; i < r.getProviders().length; i++){
				list.add(r.getProviders()[i]);
			}
			response.setCollection(list);
		} catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("Error fetching providers by name: " + stationNo + "^" + firstName + " " + lastName,e);
			}
			response.addError(SmsServiceMessages.UNABLE_TO_FETCH_PROVIDERS);
		}
		return response;
	}


	public ServiceResponse<Provider> getPcmmProvider(String stationNo, String duz){

		Precondition.assertNotBlank("stationNo", stationNo);
		Precondition.assertNotBlank("duz", duz);

		ServiceResponse<Provider> response = new ServiceResponse<Provider>();
		try {
			ProviderLookupResponse r = delegate.getProvidersByNameAndDUZ(null, null, duz, stationNo);

			if(r == null || r.getProviders() == null){
				return response;
			}

			if(r.getProviders().length > 1){
				response.addError(SmsServiceMessages.DATA_INTEGRITY_NOT_UNIQUE_RESULT);
				return response;
			}

			response.setPayload(r.getProviders()[0]);

		} catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("Error fetching providers by duz: " + stationNo + "^" + duz,e);
			}
			response.addError(SmsServiceMessages.UNABLE_TO_FETCH_PROVIDERS);
		}
		return response;

	}

	public CollectionServiceResponse<Clinic> searchClinics(String stationNo, String clinicName){

		Precondition.assertNotBlank("stationNo", stationNo);
		Precondition.assertNotBlank("clinicName", clinicName);

		CollectionServiceResponse<Clinic> response = new CollectionServiceResponse<Clinic>();
		try {
			ClinicLookupResponse r = delegate.getClinicsByName(clinicName, stationNo);

			if (r == null || r.getClinics() == null || r.getClinics().length == 0) {
				return response;
			}

			List<Clinic> list = new ArrayList<Clinic>();

			for(int i = 0; i < r.getClinics().length; i++){
				list.add(r.getClinics()[i]);
			}
			response.setCollection(list);
		} catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("Error fetching clinics by name: " + stationNo + "^" + clinicName,e);
			}
			response.addError(SmsServiceMessages.UNABLE_TO_FETCH_CLINICS);
		}
		return response;
	}

	public CollectionServiceResponse<SMClinic> searchSMClinicsByStopCode(String stationNo, String clinicName,String stopCode){

		Precondition.assertNotBlank("stationNo", stationNo);
		Precondition.assertNotBlank("clinicName", clinicName);
		Precondition.assertNotBlank("stopCode", stopCode);

		CollectionServiceResponse<SMClinic> response = new CollectionServiceResponse<SMClinic>();
		try {
			SMClinicLookupResponse r = delegate.getClinicsByNameStopCode(clinicName, stationNo, stopCode);

			if (r == null || r.getClinics() == null || r.getClinics().length == 0) {
				return response;
			}

			List<SMClinic> list = new ArrayList<SMClinic>();

			for(int i = 0; i < r.getClinics().length; i++){
				list.add(r.getClinics()[i]);
			}
			response.setCollection(list);
		} catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("Error fetching SM clinics by name and stop code: " + stationNo + "^" + clinicName + "!!stopCode^"+stopCode,e);
			}
			response.addError(SmsServiceMessages.UNABLE_TO_FETCH_SMCLINICS);
		}
		return response;
	}

	public LocationDssUnitsDTO getDSSUnitsByClinicAndStation(String stationNo, String clinicIen,String duz,String workloadType){

		Precondition.assertNotBlank("stationNo", stationNo);
		Precondition.assertNotBlank("clinicName", clinicIen);
		Precondition.assertNotBlank("duz", duz);
		LocationDssUnitsDTO locationDssUnitsDTO =  new LocationDssUnitsDTO();

		try {
			DSSUnitQueryResponse dssResponse = delegate.getDSSUnits(stationNo, clinicIen, duz);
			if(log.isInfoEnabled()){
				log.info("101 getDSSUnits response for"+stationNo+"^"+clinicIen+"^"+duz+" ConversationId->"+dssResponse.getConversationID()+"^Status->"+dssResponse.getStatus());
			}
			if(dssResponse!=null && dssResponse.getConversationID()!=null){
				if(dssResponse.getStatus().equalsIgnoreCase("OK")){
					if(dssResponse.getDSSUnits()!=null && dssResponse.getDSSUnits().length!=0){
						List<DSSUnit> dssList = new ArrayList<DSSUnit>();
						for(int i = 0; i < dssResponse.getDSSUnits().length; i++){
							if(workloadType.equalsIgnoreCase(WorkloadCaptureTypeEnum.HISTORICAL_NOTE.getId().toString()))
							{
								if(dssResponse.getDSSUnits()[i].getPCEFlag().equalsIgnoreCase("N"))
								{
									dssList.add(dssResponse.getDSSUnits()[i]);
								}
							}else{
								if(dssResponse.getDSSUnits()[i].getPCEFlag()!=null && !dssResponse.getDSSUnits()[i].getPCEFlag().equalsIgnoreCase("N"))
								{
									dssList.add(dssResponse.getDSSUnits()[i]);
								}
							}
						}
						if(dssList.size()!=0){
							locationDssUnitsDTO.setDssUnits(dssList);
						}
						else{
							locationDssUnitsDTO.setError(SmsServiceMessages.DSS_UNITS_ERROR3);
						}
					}else{
						log.info("103 getDSSUnits response Location for"+stationNo+"^"+clinicIen+"^"+duz+" ConversationId->"+dssResponse.getConversationID()+"^Status->"+dssResponse.getStatus()+"^No Dss Units with response->");
						locationDssUnitsDTO.setError(SmsServiceMessages.WORKLOAD_ERROR);
					}

					if(dssResponse.getLocation()!=null){
						locationDssUnitsDTO.setLocationIen(dssResponse.getLocation().getLocIEN());
						locationDssUnitsDTO.setLocationName(dssResponse.getLocation().getLocName());
					}else{
						if(log.isInfoEnabled()){
							log.info("104 getDSSUnits response Location for"+stationNo+"^"+clinicIen+"^"+duz+" ConversationId->"+dssResponse.getConversationID()+"^Status->"+dssResponse.getStatus()+"^Location->"+dssResponse.getLocation());
						}
					}
				}
				if(dssResponse.getStatus().equalsIgnoreCase("ERROR")){
					locationDssUnitsDTO.setError(dssResponse.getReason());
				}
			}
		} catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("Error while connecting ensemble for getDSSUnits"+stationNo+"^"+clinicIen+"^"+duz);
			}
			locationDssUnitsDTO.setError(SmsServiceMessages.ERROR_OCCURED);
		}
		catch(Exception e){
			if(log.isErrorEnabled()){
				log.error("Error fetching DSS Units by Clinic and Station"+stationNo+"^"+clinicIen+"^"+duz);
			}
			locationDssUnitsDTO.setError(SmsServiceMessages.ERROR_OCCURED);
		}
		return locationDssUnitsDTO;
	}

	public CollectionServiceResponse<ECSProcedure> getECSProcedures(String stationNo, String location, String dssUnit){

		Precondition.assertNotBlank("stationNo", stationNo);
		Precondition.assertNotBlank("location", location);
		Precondition.assertNotBlank("dssUnit", dssUnit);

		CollectionServiceResponse<ECSProcedure> procedureResponse = new CollectionServiceResponse<ECSProcedure>();
		try
		{

			ECSProcedureQueryResponse response = delegate.getECSProcedures(stationNo, WorkloadUtil.getIen(location), WorkloadUtil.getIen(dssUnit));
			if(log.isInfoEnabled()){
				log.info("201 getECS Procedure response for"+stationNo+"^"+stationNo+"^Location"+location+" ConversationId->"+response.getConversationID()+"^Status->"+response.getStatus());
			}
			if(response!=null && response.getConversationID()!=null){
				if(response.getStatus().equalsIgnoreCase("OK")){
					if(response.getECSProcedures()!=null && response.getECSProcedures().length!=0){
						List<ECSProcedure> procedureList = new ArrayList<ECSProcedure>();
						for(int i = 0; i < response.getECSProcedures().length; i++){
							procedureList.add(response.getECSProcedures()[i]);
						}
						procedureResponse.setCollection(procedureList);
					}
				}else if(response.getStatus().equalsIgnoreCase("Error")){
					if(response.getReason()!=null && response.getReason().startsWith("No Procedures")){
						Object[] errorParams = { WorkloadUtil.getIen(dssUnit), WorkloadUtil.getIen(location), WorkloadUtil.getName(dssUnit), WorkloadUtil.getName(location) };
						procedureResponse.addError(SmsServiceMessages.PROCEDURE_NOT_FOUND, errorParams);
						return procedureResponse;
					}
					else{
						procedureResponse.addError(SmsServiceMessages.UNABLE_TO_FETCH_ECS_PROCEDURES);
					}
				}
			}
		}catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("Error occured while connecting to ensemble for getECSProcedures " + stationNo + "^" + location  + "DSS^"+dssUnit);
			}
			procedureResponse.addError(SmsServiceMessages.ERROR_OCCURED);
		}
		catch(Exception e1){
			if(log.isErrorEnabled()){
				log.error("Error fetching ECS Procedures by location and dss: " + stationNo + "^" + location + "DSS^"+dssUnit);
			}
			procedureResponse.addError(SmsServiceMessages.ERROR_OCCURED);
		}
		return procedureResponse;
	}

	public CollectionServiceResponse<Diagnosis> getPatientProblems(String stationNo, Long patientId){

		Precondition.assertNotBlank("stationNo", stationNo);
		Precondition.assertNotNull("patientId", patientId);
		
		if(log.isInfoEnabled()){
			log.info("PatientProblems->"+stationNo+"^patientId"+patientId);
		}
		
		CollectionServiceResponse<Diagnosis> response = new CollectionServiceResponse<Diagnosis>();
		try {
			Patient patient = userDao.findPatientById(patientId);
			PatientProblemsQueryResponse r = delegate.getPatientProblems(stationNo, patient.getIcn());

			if (r == null || r.getProblemList() == null || r.getProblemList().length == 0) {
				if(log.isInfoEnabled()){
					log.info("No Problem List Found for Patient "+patientId);
				}
				return response;
			}

			List<Diagnosis> list = new ArrayList<Diagnosis>();

			for(int i = 0; i < r.getProblemList().length; i++){
				list.add(r.getProblemList()[i]);
			}
			response.setCollection(list);
			patient=null;

		} catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("Error fetching Problem List by patient: " + patientId + "^" + patientId,e);
			}
			response.addError(SmsServiceMessages.UNABLE_TO_FETCH_PATIENT_PROBLEM_LIST);
		}
		return response;
	}

	public ServiceResponse<DiagnosisDTO> getDiagnosis(String stationNo, String diagnosisName){

		Precondition.assertNotBlank("stationNo", stationNo);
		Precondition.assertNotBlank("diagnosisName", diagnosisName);

		ServiceResponse<DiagnosisDTO> diagnosisResponse = new ServiceResponse<DiagnosisDTO>();
		DiagnosisDTO diagnosisDto = new DiagnosisDTO();
		try{
			DiagnosisQueryResponse response = delegate.getDiagnosis(stationNo, diagnosisName);
			if(response!=null && response.getStatus().equalsIgnoreCase(RESPONSE_OK)){
				List<Diagnosis> diagnosis = new ArrayList<Diagnosis>();
				for(int i = 0; i < response.getDiagnosisList().length; i++){
					diagnosis.add(response.getDiagnosisList()[i]);
				}
				//diagnosisResponse.setCollection(diagnosis);
				diagnosisDto.setDiagnosis(diagnosis);
			}else if(response!=null && response.getStatus().equalsIgnoreCase(RESPONSE_FAIL)){
					if(response.getReason().equalsIgnoreCase(RESPONSE_STATUS_MAXSTRING)){
						diagnosisDto.setDiagnosisError(SmsServiceMessages.DIAGNOSIS_ERR1);
					}else{
						diagnosisDto.setDiagnosisError(SmsServiceMessages.DIAGNOSIS_ERR4);
					}
			}else if(response!=null && response.getStatus().equalsIgnoreCase(RESPONSE_ERROR)){
				if(response.getReason().equalsIgnoreCase(RESPONSE_STATUS_NO_MATCH)){
					if(log.isInfoEnabled()){
						log.info("No Diagnosis Found for diagnosisName "+diagnosisName+" stationNo "+stationNo);
					}
					diagnosisDto.setDiagnosisError(SmsServiceMessages.DIAGNOSIS_ERR4);
				}
			}
		}catch(RemoteException e){
			if(log.isErrorEnabled()){
				log.error("Remote Exception->while search diagnosis for patient: " + stationNo + "^" + diagnosisName,e);
				diagnosisDto.setDiagnosisError(SmsServiceMessages.DIAGNOSIS_ERR1);
			}
		}
		catch(Exception e1){
			if(log.isErrorEnabled()){
				log.error("Exception Occured while search diagnosis for patient: " + stationNo + "^" + diagnosisName,e1);
				diagnosisDto.setDiagnosisError(SmsServiceMessages.ERROR_OCCURED);
			}
		}
		diagnosisResponse.setPayload(diagnosisDto);
		return diagnosisResponse;
	}

	public PatientEligibilityClassifications getPatientEligibilityAndClassification(String stationNo, String dssUnitIen, Long patientId){

		Precondition.assertNotBlank("stationNo", stationNo);
		Precondition.assertNotBlank("dssUnitIen", dssUnitIen);
		Precondition.assertNotNull("patientId", patientId);
		if(log.isInfoEnabled()){
			log.info("PatientEligibility->"+stationNo+"^DssUnit"+dssUnitIen+"^patientId"+patientId);
		}
		PatientEligibilityClassifications classification =  new PatientEligibilityClassifications();
		Patient patient = null;
		try{
			patient = userDao.findPatientById(patientId);
			if(patient!=null){
				PatientEClassQueryResponse patClassResponse = delegate.getPatientEClassifications(stationNo, dssUnitIen, patient.getIcn());
				if(patClassResponse!=null && patClassResponse.getStatus().equalsIgnoreCase(RESPONSE_OK)){
					List<PatientEligibility> patEligibilityList = new ArrayList<PatientEligibility>();
					for(int i = 0; i < patClassResponse.getPatientEligibilities().length; i++){
						if(patClassResponse.getPatientEligibilities()[i]!=null){
							patEligibilityList.add(patClassResponse.getPatientEligibilities()[i]);
							if(patClassResponse.getPatientEligibilities()[i].getPrimarySecondaryFlag().equals("1")){
								classification.setPrimaryEligibility(patClassResponse.getPatientEligibilities()[i].getEligCodeIEN());
							}
						}
					}
					classification.setPatientEligibilities(patEligibilityList);
					classification.setPatientClassification(patClassResponse.getPatientClassifications());				
				}else if(patClassResponse!=null && patClassResponse.getStatus().equalsIgnoreCase(RESPONSE_ERROR)){
					if(log.isErrorEnabled()){
						log.error("Eligibility Error=>100^" + stationNo + "^DSS Ien" + dssUnitIen + "!!patientIcn^"+patient.getIcn()+"^"+patClassResponse.getReason());
					}
				}
	
				if (patClassResponse == null || patClassResponse.getPatientEligibilities()== null || patClassResponse.getPatientEligibilities().length == 0) {
					classification.setError(SmsServiceMessages.UNABLE_TO_PATIENT_ELIGIBILITY_AND_CLASSIFICATION);
					return classification;
				}
		}else{
			if(log.isErrorEnabled()){
				log.error("Eligibility Error=>200^" + stationNo + "^DSS Ien" + dssUnitIen + patientId+"^PATIENT NOT FOUND^");
			}
	
		}
		}catch(RemoteException e){
			if(log.isErrorEnabled()){
				log.error("Eligibility Error=>300^" + stationNo + "^DSS Ien" + dssUnitIen + "!!patientIcn^"+patient.getIcn()+e);
				e.printStackTrace();
			}
			classification.setError(SmsServiceMessages.UNABLE_TO_PATIENT_ELIGIBILITY_AND_CLASSIFICATION);
		}
		catch(Exception e1){
			if(log.isErrorEnabled()){
				log.error("Eligibility Error=>400^" + stationNo + "^DSS Ien" + dssUnitIen + "!!patientIcn^"+patient.getIcn()+e1);
				e1.printStackTrace();
			}
		}
		
		patient=null;
		return classification;
	}

	/**
	 * Return a list of SM CPRS Titles for Station and documentClass .
	 *
	 * @param stationNo
	 * @param documentClass
	 * @return
	 */
	public CollectionServiceResponse<SMNoteTitle> getCPRSTIUTitles(java.lang.String station, java.lang.String documentClass){
		Precondition.assertNotBlank("stationNo", station);
		Precondition.assertNotBlank("documentClass", documentClass);

		CollectionServiceResponse<SMNoteTitle> response = new CollectionServiceResponse<SMNoteTitle>();
		try {
			SMTIUTitleLookupResponse r = delegate.getCPRSTIUTitles(station, documentClass);

			if (r == null || r.getSMTIUTitles() == null || r.getSMTIUTitles().length == 0) {
				return response;
			}

			List<SMNoteTitle> list = new ArrayList<SMNoteTitle>();

			for(int i = 0; i < r.getSMTIUTitles().length; i++){
				list.add(r.getSMTIUTitles()[i]);
			}
			response.setCollection(list);
		} catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("Error fetching SM CPRS TIU Titles by station and document Class: Station->" + station + "^" +"!!document Class^"+documentClass,e);
			}
			response.addError(SmsServiceMessages.UNABLE_TO_FETCH_SMCLINICS);
		}
		return response;
	}


	public CollectionServiceResponse<Team> searchTeams(String stationNo, String teamName){

		Precondition.assertNotBlank("stationNo", stationNo);
		Precondition.assertNotBlank("teamName", teamName);

		CollectionServiceResponse<Team> response = new CollectionServiceResponse<Team>();
		try {
			TeamLookupResponse r = delegate.getTeamsByName(teamName, stationNo);

			if (r == null || r.getTeams() == null || r.getTeams().length == 0) {
				return response;
			}

			List<Team> list = new ArrayList<Team>();

			for(int i = 0; i < r.getTeams().length; i++){
				list.add(r.getTeams()[i]);
			}
			response.setCollection(list);
		} catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("Error fetching teams by name: " + stationNo + "^" + teamName,e);
			}
			response.addError(SmsServiceMessages.UNABLE_TO_FETCH_TEAMS);
		}
		return response;
	}

	public CollectionServiceResponse<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient> getPatientsByOERRTeam(String teamIEN, String station){

		Precondition.assertNotBlank("IEN", teamIEN);
		Precondition.assertNotBlank("station", station);

		CollectionServiceResponse<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient> response = new CollectionServiceResponse<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient>();
		try {
			TeamPatientsResponse r = delegate.getPatientsByTeam(teamIEN, station);

			if (r == null || r.getPatients() == null || r.getPatients().length == 0) {
				return null;
			}

			for(int i = 0; i < r.getPatients().length; i++){
				gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient patient = r.getPatients()[i];
			}
		} catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("Error fetching Patients By OERR " + teamIEN + "^" + station,e);
			}
		}
		return response;
	}

	public CollectionServiceResponse<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient> getPatientsByProvider(String providerIEN, String station){

		Precondition.assertNotBlank("IEN", providerIEN);
		Precondition.assertNotBlank("station", station);

		CollectionServiceResponse<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient> response = new CollectionServiceResponse<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient>();
		try {

			PCMMProviderPatientsResponse r = delegate.getPatientsByProvider(providerIEN, station);
			if (r == null || r.getPatients() == null || r.getPatients().length == 0) {
				return null;
			}
			for(int i = 0; i < r.getPatients().length; i++){
				gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient patient = r.getPatients()[i];
				if(log.isInfoEnabled()){
					log.info("Patient ICN....."+patient.getICN());
					log.info("Patient DFN....."+patient.getDFN());
				}
			}
		} catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("Error fetching Patients By OERR " + providerIEN + "^" + station,e);
			}
		}
		return response;
	}

	public CollectionServiceResponse<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient> getPatientsByClinic(String clinicIEN, String station){

		Precondition.assertNotBlank("IEN", clinicIEN);
		Precondition.assertNotBlank("station", station);

		CollectionServiceResponse<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient> response = new CollectionServiceResponse<gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient>();
		try {
			Date beginDate = new Date();
			Date endDate = new Date();
			beginDate = DateUtils.addDays(beginDate, -365);
			endDate = DateUtils.addDays(endDate, 365);

			ClinicPatientsResponse clinicPatResponse = delegate.getPatientsByClinicAndDate(clinicIEN, beginDate, endDate, station);

			if (clinicPatResponse == null || clinicPatResponse.getPatients() == null || clinicPatResponse.getPatients().length == 0) {
				return null;
			}
			for(int i = 0; i < clinicPatResponse.getPatients().length; i++){
				gov.va.med.mhv.sm.wsclient.adminqueriessvc.Patient patient = clinicPatResponse.getPatients()[i];
			}
		} catch (RemoteException e) {
			if(log.isErrorEnabled()){
				log.error("Error fetching Patients By OERR " + clinicIEN + "^" + station,e);
			}
		}
		return response;
	}

	public ServiceResponse<TriageGroup> saveTriageGroup(TriageGroup tg){
		Precondition.assertNotNull("triageGroup", tg);
		Precondition.assertNotBlank("tg.name", tg.getName());
		Precondition.assertNotBlank("tg.vistaDiv", tg.getVistaDiv());
		ServiceResponse<TriageGroup> response = new ServiceResponse<TriageGroup>();
		tg = triageGroupDao.save(tg);
		response.setPayload(tg);
		return response;
	}

	public IntegrationServiceDelegate getDelegate() {
		return delegate;
	}
	public void setDelegate(IntegrationServiceDelegate delegate) {
		this.delegate = delegate;
	}
	public ClinicalUserTypeDao getClinicalUserTypeDao() {
		return clinicalUserTypeDao;
	}
	public void setClinicalUserTypeDao(ClinicalUserTypeDao clinicalUserTypeDao) {
		this.clinicalUserTypeDao = clinicalUserTypeDao;
	}
	public UserDao getUserDao() {
		return userDao;
	}
	public void setUserDao(UserDao userDao) {
		this.userDao = userDao;
	}
	public TriageGroupDao getTriageGroupDao() {
		return triageGroupDao;
	}
	public void setTriageGroupDao(TriageGroupDao triageGroupDao) {
		this.triageGroupDao = triageGroupDao;
	}


	public ADQuery getAdQuery() {
		return adQuery;
	}

	public void setAdQuery(ADQuery adQuery) {
		this.adQuery = adQuery;
	}



}
