package gov.va.med.mhv.sm.service;

import gov.va.med.mhv.foundation.service.Service;
import gov.va.med.mhv.foundation.service.response.CollectionServiceResponse;
import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.sm.dto.UserSurrogateDTO;
import gov.va.med.mhv.sm.enumeration.PerformerTypeEnum;
import gov.va.med.mhv.sm.model.Administrator;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.MHVPatient;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.Signature;
import gov.va.med.mhv.sm.model.Surrogate;
import gov.va.med.mhv.sm.model.User;
import gov.va.med.mhv.sm.model.UserProfileVW;
import gov.va.med.mhv.sm.wsclient.queriessvc.PatientDemographics;

import java.util.List;

public interface UserManagementService extends Service{

	/**
	 * OPT IN a user.  This user must be in the database.
	 * This cannot be a new user.
	 *
	 * @param u  User to update
	 * @return   true/false
	 */
	public ServiceResponse<Boolean> optInUser(User u);


	/**
	 * OPT IN a user.  This user must be in the database.
	 * This cannot be a new user.
	 *
	 * @param u  User to update, PerformerType to set
	 * @return   true/false
	 */

	public ServiceResponse<Boolean> optInUser(User u,PerformerTypeEnum performerType);


	/**
	 * OPT OUT a user.  This user must be in the database.
	 * This cannot be a new user.
	 *
	 * @param u  User to update
	 * @return   true/false
	 */
	public ServiceResponse<Boolean> optOutUser(User u);


	/**
	 * OPT OUT a user.  This user must be in the database.
	 * This cannot be a new user.
	 *
	 * @param u  User to update, PerformerTypeEnum to set
	 * @return   true/false
	 */
	public ServiceResponse<Boolean> optOutUser(User u,PerformerTypeEnum performerType);


	/*
	 * get patient by userId.
	 * @param userId
	 * @return Patient
	 */
	public ServiceResponse<Patient> getPatientById(Long userId);



	/**
	 * Return a list of patients matching the passed in criteria.
	 * The stationNo is required.
	 * The last name or the NSSN must be specified.  The first name
	 * is not required.
	 *
	 * @param fn    first name, full or partial
	 * @param ln    last name, full or partial
	 * @param nssn  nssn, first letter of last name and last 4 digits
	 *              of the social security number
	 * @param stationNumber  facility identifier
	 * @return      List<Patient>
	 */

	public CollectionServiceResponse<Patient> searchForPatients(String fn, String ln, String nssn, String stationNumber);

	/**
	 * Return a list of MHV patients matching the passed in criteria.
	 * The stationNo is required.
	 * The last name or the NSSN must be specified.  The first name
	 * is not required.
	 *
	 * @param fn    first name, full or partial
	 * @param ln    last name, full or partial
	 * @param nssn  nssn, first letter of last name and last 4 digits
	 *              of the social security number
	 * @param stationNumber  facility identifier
	 * @return      List<Patient>
	 */

	public CollectionServiceResponse<MHVPatient> searchForMHVPatients(String fn, String ln, String nssn, String stationNumber);

	public ServiceResponse<Patient> createOrUpdatePatientFrom(MHVPatient mhvPatient);


	/**
	 * Return a list of clinicians matching the passed in criteria.
	 * The last name must be specified.  The first name
	 * is not required.
	 *
	 * @param fn    first name, full or partial
	 * @param ln    last name, full or partial
	 * @param station a must
	 * @return      List<Clinician>
	 */

	public CollectionServiceResponse<Clinician> searchForClinicians(String fn, String ln, String station, boolean active);


	/**
	 * Return a list of clinicians includes active/inactive matching the passed in criteria.
	 * The last name must be specified.  The first name
	 * is not required.
	 *
	 * @param fn    first name, full or partial
	 * @param ln    last name, full or partial
	 * @param station a must
	 * @return      List<Clinician>
	 */

	public CollectionServiceResponse<Clinician> searchForClinicians(String fn, String ln, String station);


	/**
	 * Return a list of clinicians the current user is substituting for
	 * @param c
	 * @return
	 */
	public CollectionServiceResponse<Surrogate> getSurrogatesFor(Clinician c);

	/**
	 * A user can be a surrogate for multiple clinicians.  But if the user
	 * is a surrogate, he/she cannot set a surrogate.  We are only allowing
	 * one "generation" of surrogate.
	 *
	 * @param c         User being substituted
	 * @param surrogate User being the substitute
	 * @return
	 */
//	public ServiceResponse<Clinician> setUserSurrogate(Clinician c, MailParticipant surrogate);



	/**
	 * Fetch a clinician and set appropriate properties.  i.e.  Surrogate
	 * which are not fetched and populated by the DAO object
	 *
	 * @param userId
	 * @return
	 */
	public ServiceResponse<Clinician> fetchClinician(Long userId);


	/**
	 * Return a list of clinicians for the station
	 * with a clinician.status = UserStatusEnum.OPT_IN
	 *
	 * @param stationNo
	 * @return
	 */
	public CollectionServiceResponse<Clinician> getCliniciansForStation(String stationNo);


	/**
	 * Return a list of clinicians for the station
	 * (regardless of clinician.status)
	 *
	 * @param stationNo
	 * @return
	 */
	public CollectionServiceResponse<Clinician> getCliniciansForStationAll(String stationNo);


	/**
	 * Return a clinician for the given station and duz
	 * @param stationNo
	 * @param duz
	 * @return
	 */
	public ServiceResponse<Clinician> getClinicianForStationAndDuz(String stationNo, String duz, boolean active);


	/**
	 * Save changes to the user's preferences.
	 *
	 * Note that setting the surrogate must be done separately from this
	 * mehtod.
	 *
	 * @see setUserSurrogate(Clinician, MailParticipant)
	 *
	 * @param u  User saving preferences
	 * @return   the updated user object
	 */
	public ServiceResponse<User> savePreferences(User u);


	/**
	 * Block patient access
	 *
	 * Unblock by setting using optInUser(User) or optOutUser(User)
	 *
	 * @param p
	 * @return
	 */
	public ServiceResponse<Boolean> blockPatient(Patient p, Administrator admin, String reason);


	/**
	 * Unblock a patient's access.  Resets the status to OPT_OUT
	 * @param p
	 * @param admin
	 * @param reason
	 * @return
	 */
	public ServiceResponse<Boolean> unblockPatient(Patient p, Administrator admin, String reason);


	/**
	 * Fetch a patients demographics from Vista
	 *
	 * @param ssn
	 * @param stationNo
	 * @return
	 */
	public ServiceResponse<PatientDemographics> getPatientDemographics(String ssn, String stationNo);

	/**
	 * Populate the facilities in the patient object
	 */
	public ServiceResponse<Patient> fleshPatientFacilities(Patient p);

		/**
		 * Inactivate or Reactivate Clinician
		 *
		 * @param Clinician
		 * @param active
		 * @return
		 */
	public ServiceResponse<Clinician> inactivateReactivateClinician(Clinician c, boolean active);

	/**
	 * Add User Signature.
	 *
	 * @param Signature
	 * * @return
	 */
	public ServiceResponse<Boolean> addUserSignature(Signature s);

	/**
	 * Set Surrogate For Clinician
	 *
	 * @param clinician, List of Surrogates
	 * * @Boolean
	 */
	public ServiceResponse<Boolean> setSurrogatesFor(Clinician c, List<Surrogate> surrogateList);


	/**
	 * clearSurrogatesFor For Clinician
	 *
	 * @param clinician
	 * * @Boolean
	 */
	public ServiceResponse<Boolean> clearSurrogatesFor(Clinician c);

	/**
	 * Remove Surrogate
	 *
	 * @param surrogateId
	 * * @Boolean
	 */
	public ServiceResponse<Boolean> removeSurrogate(Long surrogateId);


	/**
	 * updateClinician method update the Clinician firtstName, lastName
	 *
	 * @param Clinician
	 * @return
	 */
	public ServiceResponse<Clinician> updateClinician(Clinician clinician);

	/**
	 * updateClinician method update the Clinician ClinicalUserType and networkId
	 * record which User made the change.
	 *
	 * @param User
	 * @param Clinician
	 * @return
	 */
	public ServiceResponse<Clinician> updateClinician(User u, Clinician clinician);

	public CollectionServiceResponse<UserSurrogateDTO>getTriageMembersWithCurrentSurrogate(Long triageGroupId);

	/**
	 * Get patient treatment facilities by userId
	 *
	 * @param username
	 * @return vistAName and stationNumber of the patient treatment facilities
	 * @return List<Object> return Object[0]=vistAName, Object[1]=stationNumber
	 */
	public List<Object[]> getPatientFacilitiesByUserId(Long userId);

	/* ADDING FOR API DAO CONVERSION TO SERVICE */

	public User findById(Long Id);
	public Patient findPatientById(Long Id);
	public Patient findApiPatientById(Long Id);
	public Surrogate getCurrentSurrogateFor(Clinician clinician);

	public ServiceResponse<gov.va.med.mhv.sm.wsclient.queriessvc.User> getClinicianDemographics(Clinician c);

	public String getActiveSurrogateByUser(Clinician provider);
	
	public ServiceResponse<List<Object[]>> getUserDetailsByIds(Long messageId, Long userId);

	
	public MHVPatient findMHVPatientByUserName(String userName);

	
	public int updateSMOptStatusByUserName(String userName, Boolean optStatus);
	
	public UserProfileVW getUserProfileByUserName(String userName);
	

}
