package gov.va.med.mhv.sm.service;

import gov.va.med.mhv.foundation.service.Service;
import gov.va.med.mhv.foundation.service.response.CollectionServiceResponse;
import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.sm.enumeration.SystemFolderEnum;
import gov.va.med.mhv.sm.model.Folder;
import gov.va.med.mhv.sm.model.Message;
import gov.va.med.mhv.sm.model.NewMessage;
import gov.va.med.mhv.sm.model.TriageGroup;
import gov.va.med.mhv.sm.model.User;

public interface SendMessageService extends Service {

	
	/**
	 * Send a <b>new</b> message.
	 * <p>
	 * A TriageGroup must be defined for any messaging involving
	 * a patient.  For patients initiating the message the recipient
	 * will be the TriageGroup itself.  For clinician->clinician
	 * messages a TriageGroup is not necessary.  
	 * <p>
	 * The TriageGroup may be <code>null</code> as long as a Patient
	 * is not involved.  The MessageTemplate may be <code>null</code>.
	 * All other fields of the NewMessage object must be non-null 
	 * and/or non-empty values.
	 * <p>
	 * If a patient is sending a message to a TriageGroup the TriageGroup
	 * defined as the recipient will be used. The TriageGroup member of the
	 * NewMessage will be ignored, and may be <code>null</code>.	 * 
	 * <p>
	 * @see send(NewMessage)
	 * @see reply(Message, NewMessage) 
	 * 
	 * @param nm the Message to send
	 * @return   <code>true</code> if the message was sent successfully
	 */
	public ServiceResponse<Message> send(NewMessage nm);
	
	
	/**
	 * Reply to an existing Message.
	 * <p>
	 * The recipient of the sent message will be derived from the Message being replied to.
	 * <ul>
	 *   <li>if Thread.TriageGroup != null and sender is Patient, recipient will be TriageGroup.</li>
	 *   <li>if Thread.TriageGroup != null and sender is Clinician, recipient will be Patient.</li>
	 *   <li>Otherwise Clinician-Clinician is assumed and the recipient will be the other Clinician</li>
	 * </ul>
	 * The NewMessage.to is ignored.
	 * <p>
	 * @see send(NewMessage)
	 * 
	 * @param orig  the Message being replied to.  the recipient of the message
	 *              will be derived from this. 
	 * @param nm    the User the message is from.
	 * @return      <code>true</code> if the message was sent successfully
	 */
	public ServiceResponse<Message> reply(Message orig, NewMessage nm);
	
	
	/**
	 * Save a message as a draft.  The subject, body, to pieces must be
	 * defined.  If the draft is to be saved again the messageId of the 
	 * original must be passed to prevent a duplicate from being created.  
	 * 
	 * @param nm        NewMessage object containing all pieces of the message
	 * @param messageId Id of draft message if being saved again
	 * @return          <code>true</code> if the message was saved successfully
	 */
	public ServiceResponse<Message> saveDraft (NewMessage nm, Long messageId);
	
	
	
	/**
     * Only address the message at this point since the triagegroup could have changed since the
     * message was created.
     * 
     * The message was already created prior to this but we need to make the Addressee's up to date.
     * 
     * 
     * @param orig
     * @param nm
     * @return
     */
    public ServiceResponse<Message> sendReplyDraft(Message orig, NewMessage nm) ;
    
    /*
     *  Use for Sending draft message saved previously
     */
    public ServiceResponse<Message> sendReplyDraftForAPI(Message orig, NewMessage nm) ;
    
    

	public ServiceResponse<Message> saveReplyDraftForWeb(Message orig, NewMessage nm,Folder folder);
	
	/**
    *
    *  Do the same as reply except push the message to the DRAFTS folder
    *
    *	Don't have to do all the addressing, etc. here because delay until the
    *  actual message is sent.  This prevents stale Addressee's.
    *
    * @param orig
    * @param nm
    * @return
    */
	public ServiceResponse<Message> saveReplyDraft(Message orig, NewMessage nm);
	
	
	
	
	/**
	 * This is the <b>only</b> way to send a 
	 * message that has been saved as a draft.
	 * <p>
	 * Note that this interface will not accept changes
	 * to the draft message.  In order to change the
	 * message use the {@link DraftService#save(Message)} method to save
	 * the message prior to sending the message.
	 * 
	 * @param nm        NewMessage that contains the contents of the message to send
	 * @param messageId Id of the existing draft message.  It will be deleted
     *                  if the message is successfully sent 
	 * @return          <code>true</code> if the message was sent successfully
	 */
	public ServiceResponse<Message> sendDraft(NewMessage nm, Long messageId);
	
	/**
	 * Reply to an existing Message.
	 * <p>
	 * The recipient of the sent message will be derived from the Message being replied to.
	 * <ul>
	 *   <li>if Thread.TriageGroup != null and sender is Patient, recipient will be TriageGroup.</li>
	 *   <li>if Thread.TriageGroup != null and sender is Clinician, recipient will be Patient.</li>
	 *   <li>Otherwise Clinician-Clinician is assumed and the recipient will be the other Clinician</li>
	 * </ul>
	 * The NewMessage.to is ignored.
	 * <p>
	 * @see send(NewMessage)
	 * 
	 * @param orig  the Message being replied to.  the recipient of the message
	 *              will be derived from this. 
	 * @param nm    the User the message is from.
	 * @return      <code>true</code> if the message was sent successfully
	 */
	public ServiceResponse<Message> replyFromSentFolder(Message orig, NewMessage nm);
	
	
	
	/**
	 * Convert a draft message into a NewMessage object for simple use
	 * within the web tier
	 * 
	 * @param u         User that owns the draft
	 * @param messageId Id of the draft message
	 * @return          NewMessage object
	 */
	public ServiceResponse<NewMessage>draftToNewMessage(User u, Long messageId);
	
	
	/**
	 * Return a list of all TriageGroups for a given station ordered by name
	 * 
	 * @param station
	 * @return
	 */
	public CollectionServiceResponse<TriageGroup> getTriageGroupsForStation(String station);
	
	
	/**
	 * Return a list of all TriageGroups that are active
	 * WARNING: this could be a large dataset
	 * @return
	 */
	public CollectionServiceResponse<TriageGroup> getTriageGroupsAllActive();
	
    
	/**
     * Send a message to a particular user.
     */	
	public void addressMessage(Message m, User u, SystemFolderEnum folder, boolean sendToSurrogate,boolean ccMessage);	
	
}
