package gov.va.med.mhv.sm.service;

import gov.va.med.mhv.foundation.service.Service;
import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.sm.enumeration.ClinicianStatusEnum;
import gov.va.med.mhv.sm.enumeration.ReAssignMessageActionEnum;
import gov.va.med.mhv.sm.model.Addressee;
import gov.va.med.mhv.sm.model.Annotation;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.DistGroupRecallMessage;
import gov.va.med.mhv.sm.model.Folder;
import gov.va.med.mhv.sm.model.Message;
import gov.va.med.mhv.sm.model.MessageAttachment;
import gov.va.med.mhv.sm.model.User;

import java.sql.Blob;
import java.util.Date;
import java.util.List;


public interface MessageService extends Service {

	/**
	 * Fetch a message.  Useful for fetching a new message
	 * or refreshing an existing message that might have
	 * been modified after service layer modification.
	 *
	 * @param id  Message id
	 * @return
	 */
	public ServiceResponse<Message> fetchMessage(Long id);


	/**
	 * Find a MessageBy Id.
	 * @param id  Message id
	 * @return
	 */
	public Message findMessageById(Long id);


	/**
	 * Read a message.  Fetches the message in question
	 * along with the entire thread and all annotations.
	 * Marks the message as having been read in the database
	 * if not already marked as such.
	 * <p>
	 * @see #readMessage(Long, Long)
	 *
	 *  @param m Message to be read
	 *  @param u User that owns the message
	 *  @return          a ServiceResponse&lt;Message&gt; with the message
	 *                   instance in the payload.
	 */
	public ServiceResponse<Message> readMessage(Message m, User u);


	/**
	 * Annotate a message thread.  Clinicians can annotate message threads so that
	 * private conversations can noted.  These annotations are not to be displayed
	 * in the patient's view, nor will they be added to the patient's record when
	 * the thread is saved as a progress note.
	 *
	 * @param a the Annotation that is to be saved.  The Annotation object
	 * 	        must have a.Thread, a.Annotation and the a.Author set.
	 * @return  <code>true</code> if the message was annotated successfully
	 */
	public ServiceResponse<Boolean> annotateThread(Annotation a);


	/**
	 * Assign the message to a Clinician.
	 * <p>
	 * If the assignTo parameter is <code>null</code> then any pending
	 * assignment is removed.
	 *
	 * If assignTo is not the same as assignedBy then the status will be
	 * set to IN_PROCESS.
	 *
	 * Note!! If the status is set to COMPLETED then the
	 * neither the status nor assignedTo fields may be changed.
	 *
	 * @param m           Message being assigned.
	 * @param assignTo    Clinician message being assigned to.
	 * @param assignedBy  Clinician assigning the message
	 * @return            <code>true</code> if the message was assigned successfully
	 */
	public ServiceResponse<Boolean> assignMessage(
			Message m, Clinician assignTo, Clinician assignedBy);

	/**
	 * Assign the message to a Clinician and Update the reassign History
	 * <p>
	 * @param m           Message being assigned.
	 * @param assignTo    Clinician message being assigned to.
	 * @param assignedBy  Clinician assigning the message
	 * @param assignedToWithSurrogate -> Name of the Surrogate acting for the assignedTo
	 * @return            <code>true</code> if the message was assigned successfully
	 */

	public ServiceResponse<Boolean> assignMessageAndUpdateHistory(Message message, Clinician assignedTo, Clinician assignedBy, String assignedToWithSurrogate);

	/**
	 * Set the Clinician's message status.
	 *
	 * The m.assignedTo clinician must be set to the same clinician as the
	 * setBy clinician otherwise the only acceptable status is IN_PROGRESS
	 *
	 * @param m      Message that is being modified.
	 * @param status new status
	 * @param setBy  the Clinician setting the status
	 * @return       <code>true</code> if the message status was changed successfully
	 */
	public ServiceResponse<Boolean> setMessageStatus(
			Message m, ClinicianStatusEnum status, Clinician setBy,Date reminderDate);


	/**
	 * Set the Clinician's All Previous Message (including the new reply message)  status.
	 *
	 * The m.assignedTo clinician must be set to the same clinician as the
	 * setBy clinician otherwise the only acceptable status is IN_PROGRESS
	 *
	 * @param m      Message that is being modified.
	 * @param status new status
	 * @param setBy  the Clinician setting the status
	 * @param newReplyMessageId  Clinicians reply messageId when they Reply and Complete.
	 * @return       <code>true</code> if the message status was changed successfully
	 */
	public ServiceResponse<Boolean> setAllPreviousMessageStatus(
			Message m, ClinicianStatusEnum status, Clinician setBy,Long newReplyMessageId,Date reminderDate);


	/**
	 * Move a message to a different folder.
	 * Use this method as a secondary option when the Addressee
	 * object is not directly available.
	 * <p>
	 * <b>NOTE:</b> Draft messages cannot be moved.  And
	 * messages cannot be moved into the Drafts folder.
	 * <p>
	 * @see deleteMessage(Message, User)
	 *
	 * @param m Message to be moved.
	 * @param u User that owns the message
	 * @param f Folder the message is to be moved to.
	 * @return  <code>true</code> if the message was moved successfully
	 */
	public ServiceResponse<Boolean> moveMessage(
			Message m, User u, Folder f);

	/**
	 * Move a message to a different folder.
	 * Use this method when the Addressee object is directly available
	 * <p>
	 * <b>NOTE:</b> Draft messages cannot be moved.  And
	 * messages cannot be moved into the Drafts folder.
	 * <p>
	 * @see deleteMessage(Message, User)
	 *
	 * @param a Addressee of the message to be moved.
	 * @return  <code>true</code> if the message was moved successfully
	 */
	public ServiceResponse<Boolean> moveMessage(
			Addressee a);

	/**
	 * Delete a message
	 * <p>
	 * This is the same as moving a message to the Deleted folder.
	 * This is just a convenience method.
	 * <p>
	 * @see moveMessage(Message, User, Folder)
	 *
	 * @param m  Message to be deleted
	 * @param u  User that owns the message
	 * @return
	 */
	public ServiceResponse<Boolean> deleteMessage(Message m, User u);


	/**
	 * delete a draft message
	 *
	 * User is the owner of the message, and hopefully the user
	 * actually logged in.
	 *
	 * @param u         the user logged in and owns the message
	 * @param messageId the Id of the message to delete
	 * @return          <code>true</code> if the message was deleted successfully
	 */
	public ServiceResponse<Boolean> deleteDraft(User u, Long messageId);

	/**
	 * retrieve the User that send the message
	 *
	 * @param Id     the user_id of the sender of the message
	 * @return          a ServiceResponse&lt;User&gt; with the user
	 *                   instance in the payload.
	 */
	public ServiceResponse<User> getSender(Long Id);


	/**
	 * completeIndividualMessage completes the particular message and move to completed folder
	 *
	 * @param Id    message - particular message needs to be completed
	 * @param user	user - assign the message to user
	*/
	public ServiceResponse<Boolean> completeIndividualMessage(Message message,Clinician user);

	/**
	 * completedMessageReassignment user can change the status of completed message to Reassigned.
	 *
	 * @param Id    message - particular message needs to be completed
	 * @param Clinician assigner
	 * @param Clinician assignee
	 *
	*/
	public ServiceResponse<Boolean> completedMessageReassignment(Message m,Clinician assignedBy, Long assignedTo,String annotation,String surrogateEnabled);



	/**
	 * updateReadReceipt set the readReceipt of the each message
	 *
	 * @param Id    messageId - messageId needs to be set to "READ" value.
	 *
	 */
	public ServiceResponse<Message> updateReadReceipt(Long messageId);

	/**
	 * updateAssignmentStatus assign the message to the user and update clinician_status
	 *
	 * @param Id    messageId - messageId, clinician
	 *
	 */
	public ServiceResponse<Message> updateAssignmentStatus(Long messageId, Clinician clinician);

	/**
	 * markReadAsUnread set the readReceipt of the each message
	 *
	 * @param Id    messageId - messageId needs to be set to "READ" value.
	 *
	 */

	/**
	 * UpdateAddresseeReAssignedFlag set the reassignedFlag to true of each addressee in the message.
	 *
	 * This is very important to view the message in Reassign folder.
	 */
	public void updateAddresseeReAssignedActive(Message message);
	
	public ServiceResponse<Message> updateMessageWithCPRSNotesStatus(Long messageId,String cprsNotesStatus,Date updatedDate);


	public void markReadAsUnread(Addressee ad);


	/**
	 * recall set the active value to false of the message.  Making it recalled.
	 *
	 * @param Id    messageId - messageId needs to be set to "inactive" value.
	 * @param clinician clinician to write an audit
	 *
	 */
	public ServiceResponse<Message> recall(Long messageId, Clinician clinician);



	
	/**
	 * reset the Reminders for the particular addressee record, so it will not show in the Reminders folder.
	 * @param Id    addresseeId - addresseeId needs to be set to null value.
	 *
	 */
	public void deleteAddresseeReminders(Long addresseeId);

	/**
	 * saveAttachment store the message attachment to table.
	 *
	 * @param attachment   blob
	 * @param fileName name of the File
	 * @param mimeType mimeType
	 *
	 */
	public ServiceResponse<MessageAttachment> saveAttachment(Blob attachment, String fileName, String mimeType, Long userId);

	/**
	 * saveAttachment - Method to support the SM API.
	 *
	 * @param fileBytes   byte array
	 * @param fileName name of the File
	 * @param mimeType mimeType
	 *
	 */
	public ServiceResponse<MessageAttachment> saveAttachment(byte[] fileBytes, String fileName, String mimeType);
	

	
	/**
	 * removeAttachment, remove Attachment delete the attchment from table.
	 *
	 * @param attachmentId   attachmentId
	 *
	 */
	public void removeAttachment(Long attachmentId);

	
	public ServiceResponse<Message> reAssignMessage(Long messageId,Long origTriageGroupId, Long reAssignedTriageGroupId, Long messageAssignedTo, Boolean manuallyAssociate, User user,ReAssignMessageActionEnum action);

	public ServiceResponse<MessageAttachment> fetchAttachment(Long attachmentId);
	
	public ServiceResponse<List<Object[]>> getAttachmentNamesByAttachmentIds(List attachmentIds) ;

	public ServiceResponse<Boolean> nofityFacilityAdminsAddPatientToFacility(Long messageId, Long station,User user);
	
	public ServiceResponse<Boolean> setReminderDate(Message message, Clinician user, Date reminderDate);
	
	public ServiceResponse<Date> fetchReminderDate(Message message, Clinician user);
	
	public ServiceResponse<List<Object[]>> getDistributionGroupMessagesByIds(Long senderId,Long recipientId,String checksum);
	
	/**
	 * recall set the active value to false of the distribution group messages.  Making it recalled.
	 *
	 * @param Id    - secureMessageId needs to be set to "inactive" value.
	 *
	 */
	public ServiceResponse<Boolean> recallDistributionGroupMessages(Long secureMessageId);
	
	public ServiceResponse<Boolean> updateDgRecallMessages(Long secureMessageId, Long userId,String recallStatus);
	
	public DistGroupRecallMessage  getDistGroupRecallMessageById(Long secureMessageId);
	
	public ServiceResponse<Boolean> resetCPRSNotesStatusByThread(Long threadId);
	
}
