package gov.va.med.mhv.sm.service;

import gov.va.med.mhv.foundation.service.Service;
import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.sm.model.Administrator;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.Credentials;
import gov.va.med.mhv.sm.model.MhvAuthenticationSubject;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.User;

public interface AuthenticationService extends Service {
	
	/**
	 * Message key indicating authentication failed.
	 */
	static final String AUTHENTICATION_FAILED = "authentication.failed";

	/**
	 * Message key indicating that no active clinician account could not 
	 * be found for a given station and DUZ.
	 * Message key shall be returned with inserts station, DUZ and userName.
	 * @see #authenticateClinician(String, String, String)
	 */
	static final String CLINICIAN_DOES_NOT_EXIST = "clinician.does.not.exist";

	/**
	 * Message key indicating that the clinician found for a given station 
	 * and DUZ does not match the given name.
	 * Message key shall be returned with inserts station, DUZ and userName.
	 * @see #authenticateClinician(String, String, String)
	 */
	static final String CLINICIAN_DOES_NOT_MATCH = "clinician.does.not.match";
	
	static final String USER_LOOKUP_ERROR = "user.lookup.error";

	static final String PROVIDER_NAME_MIS_MATCH_ERROR = "provider.name.does.not.match.error";
	
	/**
	 * Authenticate given subject as a patient.
	 * @param subject The subject being authenticated. Subject is expected to 
	 *    map to a patient. The subject's email can be null, but all other 
	 *    pieces are required for proper function.
	 *    The format of the subject's facilities expected is a list of 
	 *    strings that have the station number as the first 3 digits of the 
	 *    string. For instance all of the following values will be acceptable.
	 *  	605
	 *  	605AB
	 *  	605AB^Loma Linda
	 * @return A patient based upon a match of the subject's lastName, 
	 *    firstName, SSN and DOB.
	 *    If a match is not found a new Patient is created.
	 * 	  When an erro occured while authenticating the given subject,
	 *    the payload will be null and the response will contain an 
	 *    error message with key {@link #AUTHENTICATION_FAILED}  
	 */
	ServiceResponse<Patient> authenticatePatient(
		MhvAuthenticationSubject subject);

	ServiceResponse<Patient> authenticatePatient(Credentials credentials);
	
	/**
	 * Authenticate a clinician based on a match of the given user name 
	 * against clincian found for the given station number and duz.
	 * @param station The station number find the clinician for
	 * @param duz The DUZ find the clinician for
	 * @param userName The username to match against
	 * @return When a match is found, the payload of response will be the 
	 *     clinician with the given name and station number; 
	 *     otherwise the payload will be null and and the response will 
	 *     contain an error message with key {@link #AUTHENTICATION_FAILED}.
	 *     Additional it returns an error message key indicating the
	 *     particular problem ({@link #CLINICIAN_DOES_NOT_EXIST} or
	 *     {@link #CLINICIAN_DOES_NOT_MATCH})
	 */
	ServiceResponse<Clinician> authenticateClinician(String station, String duz, 
		String username);
	
	/**
	 * Authenticate a clinician based on a match of the given first name and last name
	 * against clinician found for the given station number and duz.
	 * @param station The station number find the clinician for
	 * @param duz The DUZ find the clinician for
	 * @return When a match is found, the payload of response will be the 
	 *     clinician with the given name and station number; 
	 *     otherwise the payload will be null and and the response will 
	 *     contain an error message with key {@link #AUTHENTICATION_FAILED}.
	 *     Additional it returns an error message key indicating the
	 *     particular problem ({@link #CLINICIAN_DOES_NOT_EXIST} or
	 *     {@link #CLINICIAN_DOES_NOT_MATCH})
	 */
	ServiceResponse<Clinician> authenticateClinician(String station, String duz);
	
	/**
	 * 
	 * @param networkId
	 * @param lastName
	 * @param firstName
	 * @param stations
	 * @param visns
	 * @param national
	 * @return
	 */
	ServiceResponse<Administrator> authenticateAdministrator(
		MhvAuthenticationSubject subject);

	ServiceResponse<Administrator> authenticateAdministrator(
		Credentials credentials);


	ServiceResponse<Credentials> findCredentials(User user);

	/**
	 * Authenticate a patient by the given id.
	 * @param id The id to find the patient for.
	 * @return A response with as payload the patient with the given id,
	 *      when the patient exists; null otherwise.
	 * 		When not in production mode the payload will be null and 
	 *      the response will contain an error message with key 
	 *      {@link #AUTHENTICATION_FAILED}  
	 */
	ServiceResponse<Patient> authenticatePatientById(Long id);
	
	/**
	 * Authenticate a patient by the given user name.
	 * @param username The user name to find the patient for.
	 * @return A response with as payload the patient with the given username,
	 *      when the patient exists; null otherwise.
	 * 		When not in production mode the payload will be null and 
	 *      the response will contain an error message with key 
	 *      {@link #AUTHENTICATION_FAILED}  
	 */
	ServiceResponse<Patient> authenticatePatientByUsername(String username);
	
	/**
	 * Authenticate a clinician by the given id.
	 * @param id The id to find the clinician for.
	 * @return A response with as payload the clinician with the given id,
	 *      when the clinician exists; null otherwise.
	 * 		When not in production mode the payload will be null and 
	 *      the response will contain an error message with key 
	 *      {@link #AUTHENTICATION_FAILED}  
	 */
	ServiceResponse<Clinician> authenticateClinicianById(Long id);
	
	
	/**
	 * KJG  for API support
	 */
	public ServiceResponse<Patient> authenticatePatient(Long mhvCorrelationId);
	public ServiceResponse<Patient> fetchPatientById(Long id);
	public ServiceResponse<Clinician> fetchClinicianById(Long id);
	
}
