/**
 * 
 */
package gov.va.med.mhv.sm.model.decorator;

import gov.va.med.mhv.foundation.crypto.MHVCipher;
import gov.va.med.mhv.foundation.util.Describeable;
import gov.va.med.mhv.foundation.util.DescriptionBuilder;
import gov.va.med.mhv.foundation.util.Precondition;
import gov.va.med.mhv.sm.model.MhvAuthenticationSubject;
import gov.va.med.mhv.sm.model.MhvIntegrationSettings;
import gov.va.med.mhv.sm.util.MhvIntegrationUtils;
import gov.va.med.mhv.sm.util.UserUtils;

import java.util.Date;

import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;

public final class MhvAuthenticationSubjectDecorator implements Describeable {
		
	public static String describe(MhvAuthenticationSubjectDecorator decorator) {
		return DescriptionBuilder.describe(decorator);
	}
	
	private final MhvAuthenticationSubject subject;
	private final MHVCipher cipher;
	private final MhvIntegrationUtils integrationUtils;
	
	public MhvAuthenticationSubjectDecorator(MhvAuthenticationSubject subject,
		MhvIntegrationUtils integrationUtils) 
	{
		Precondition.assertNotNull("subject", subject);
		Precondition.assertNotNull("integrationUtils", integrationUtils);
		this.subject = subject;
		this.integrationUtils = integrationUtils;
		MhvIntegrationSettings settings = integrationUtils.getSettings();
		Precondition.assertNotNull("integrationUtils.settings", settings);
		this.cipher = integrationUtils.createCipher();
	}
	
	public String createChecksum() {
		return integrationUtils.calculateChecksum(getTimestamp(), getLastName(), 
			getFacilities(), getVisns(), getNational());
	}
	
	public boolean isValidPatient() {
		return isValid() && isValidPatientSubject();
	}

	public boolean isValidAdministrator() {
		return isValid() && isValidAdministratorSubject();
	}
	    
	public String getUserName() {
		return subject.getUserName();
	}
	public String getFirstName() {
		return subject.getFirstName();
	}
	public String getLastName() {
		return subject.getLastName();
	}
	public String getEmail() {
		return subject.getEmail();
	}
	public String getIcn() {
		String icn = subject.getIcn(); 
		return (cipher != null) ? cipher.decrypt(icn) : icn;
	}
	public String getSsn() {
		String ssn = subject.getSsn(); 
		return (cipher != null) ? cipher.decrypt(ssn) : ssn;
	}
	public Date getDob() {
		return subject.getDob();
	}
	public String getSource() {
		return subject.getSource();
	}
	public Boolean getAuthenticated() {
		return subject.getAuthenticated();
	}
	public Boolean getNational() {
		return subject.getNational();
	}
	public String getChecksum() {
		return subject.getChecksum();
	}
	public Long getTimestamp() {
		return subject.getTimestamp();
	}
	public String[] getFacilities() {
		return subject.getFacilities();
	}
	public String[] getVisns() {
		return subject.getVisns();
	}
	
	//CR1160 Changes
	public boolean isTermsStatus() {
		return subject.isTermsStatus();
	}

	public Date getTermsStatusDateTime() {
		return subject.getTermsStatusDateTime();
	}
	
	public void describe(DescriptionBuilder builder) {
		if (builder == null) {
			return;
		}
		builder.header(this);
		// TODO - Refactor to newer version of DescriptionBuilder
		// using appendProperty etc. once newer version of foundation is used
		builder.append("[");
		builder.append("userName=").append(getUserName());
		builder.append("firstName=").append(getFirstName());
		builder.append("lastName=").append(getLastName());
		builder.append("email=").append(getEmail());
		builder.append("ssn=").append(UserUtils.maskSsn(getSsn()));
		builder.append("icn=").append(UserUtils.maskIcn(getIcn()));
		builder.append("dob=").append(getDob());
		builder.append("authenticated=").append(getAuthenticated());
		builder.append("national=").append(getNational());
		builder.append("timestamp=").append(getTimestamp());
		builder.append("checksum=").append(getChecksum());
		builder.append("source=").append(getSource());
		builder.append("stations=").append(getFacilities());
		builder.append("visns=").append(getVisns());
		builder.append("termsStatus").append(isTermsStatus());
		builder.append("termsStatusDateTime").append(getTermsStatusDateTime());
		builder.append("]");
	}
	

	private boolean isValid() {
		return !StringUtils.isBlank(subject.getUserName())
		    && !StringUtils.isBlank(subject.getLastName())
		    && (subject.getTimestamp() != null)
		    && !StringUtils.isBlank(subject.getSource())
		    && hasValidCheckSum();
	}

	private boolean hasValidCheckSum() {
		final String checksum = getChecksum();
		return !StringUtils.isBlank(checksum) && checksum.equals(
			createChecksum());
	}
	
	private boolean isValidAdministratorSubject() {
	    return (subject.getNational() != null);
	}

	private boolean isValidPatientSubject() {
		String ssn = getSsn();
	    return (subject.getDob() != null)
	        && !(StringUtils.isBlank(ssn) || (ssn.length() < 9))
			&& BooleanUtils.isTrue(subject.getAuthenticated());
	}

}