package gov.va.med.mhv.sm.dao.hibernate;

import gov.va.med.mhv.persistence.dao.hibernate.BaseEntityDaoHibernate;
import gov.va.med.mhv.sm.dao.UserDao;
import gov.va.med.mhv.sm.dao.exception.DataIntegrityException;
import gov.va.med.mhv.sm.enumeration.UserTypeEnum;
import gov.va.med.mhv.sm.model.Administrator;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.MHVPatient;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.User;
import gov.va.med.mhv.sm.model.UserProfileVW;

import java.util.Collection;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.LockMode;
import org.hibernate.Query;
import org.hibernate.criterion.Expression;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.springframework.dao.DataIntegrityViolationException;


public class UserDaoImpl extends BaseEntityDaoHibernate<User, Long> implements
		UserDao {

	@SuppressWarnings("unused")
	private static final Log log = LogFactory.getLog(UserDaoImpl.class);


	@SuppressWarnings("unchecked")
	public Clinician findAuthClinicianByStationAndDuz(String stationNo, String duz){

		String[] paramNames = new String[] {"stationNo","duz"};
		Object[] paramValues = new Object[] {stationNo, duz};

		List<Clinician> l =  getHibernateTemplate().findByNamedQueryAndNamedParam(
				UserDao.QRY_AUTH_CLINICIAN_BY_STATION_AND_DUZ, paramNames, paramValues );

		if (l == null || l.size() == 0) {
			/* no clinician matched */
			return null;
		}

		if(l.size() > 1){
			/* There is a real problem here.
			 * The database constraints should prevent this.
			 */
			throw new DataIntegrityViolationException("Duplicate clinicians found with given constraints.");
		}

		Clinician c = l.get(0);
		if(c.getUserType() != UserTypeEnum.CLINICIAN){
			/* The structure of the HQL appears to prevent
			 * this from ever happening but just in case...
			 */
			throw new DataIntegrityViolationException("User found is not a clinician.");
		}

		return c;

	}

	@SuppressWarnings("unchecked")
	public Clinician findClinicianByStationAndDuz(String stationNo, String duz, boolean active){

		String[] paramNames = new String[] {"stationNo","duz", "active"};
		Object[] paramValues = new Object[] {stationNo, duz, active};

		List<Clinician> l =  getHibernateTemplate().findByNamedQueryAndNamedParam(
				UserDao.QRY_CLINICIAN_BY_STATION_AND_DUZ, paramNames, paramValues );

		if (l == null || l.size() == 0) {
			/* no clinician matched */
			return null;
		}

		if(l.size() > 1){
			/* There is a real problem here.
			 * The database constraints should prevent this.
			 */
			throw new DataIntegrityViolationException("Duplicate clinicians found with given constraints.");
		}

		Clinician c = l.get(0);
		if(c.getUserType() != UserTypeEnum.CLINICIAN){
			/* The structure of the HQL appears to prevent
			 * this from ever happening but just in case...
			 */
			throw new DataIntegrityViolationException("User found is not a clinician.");
		}

		return c;

	}

	@SuppressWarnings("unchecked")
	public List<Clinician> findClinicianByStationAndDuz(String stationNo, String duz){

		String[] paramNames = new String[] {"stationNo","duz", };
		Object[] paramValues = new Object[] {stationNo, duz, };

		List<Clinician> l =  getHibernateTemplate().findByNamedQueryAndNamedParam(
				UserDao.QRY_CLINICIAN_FOR_STATION_AND_DUZ, paramNames, paramValues );

		if (l == null || l.size() == 0) {
			/* no clinician matched */
			return null;
		}

		return l;

	}

	@SuppressWarnings("unchecked")
	public Clinician findClinicianByStationAndUsername(String stationNo, String username){

		String[] paramNames = new String[] {"stationNo","username"};
		Object[] paramValues = new Object[] {stationNo, username.toUpperCase()};

		List<Clinician> l =  getHibernateTemplate().findByNamedQueryAndNamedParam(
				UserDao.QRY_AUTH_CLINICIAN_BY_STATION_AND_USERNAME, paramNames, paramValues );

		if (l == null || l.size() == 0) {
			/* no clinician matched */
			return null;
		}

		if(l.size() > 1){
			/* There is a real problem here.
			 * The database constraints should prevent this.
			 */
			throw new DataIntegrityViolationException("Duplicate clinicians found with given constraints.");
		}

		Clinician c = l.get(0);
		if(c.getUserType() != UserTypeEnum.CLINICIAN){
			/* The structure of the HQL appears to prevent
			 * this from ever happening but just in case...
			 */
			throw new DataIntegrityViolationException("User found is not a patient.");
		}

		return c;

	}


	/*
	 * (non-Javadoc)
	 * @see gov.va.med.mhv.sm.dao.UserDao#findPatient(java.lang.String, java.lang.String, java.lang.String, java.util.Date)
	 */
	@SuppressWarnings("unchecked")
	public Patient findPatient(String ssn, String lastName, String firstName, Date dob){

		String[] paramNames = new String[] {"ssn","lastName","firstName","dob"};
		Object[] paramValues = new Object[] {ssn,lastName.toUpperCase(),firstName.toUpperCase(),dob};

		List<Patient> l =  getHibernateTemplate().findByNamedQueryAndNamedParam(
				UserDao.QRY_AUTH_PATIENT, paramNames, paramValues );

		if(l == null || l.size() == 0){
			/* no patient matched */
			return null;
		}

		if(l.size() > 1){
			/* There is a real problem here.
			 * The database constraints should prevent this.
			 */
			throw new DataIntegrityViolationException("Duplicate patients found with given constraints.");
		}

		Patient p = l.get(0);
		if(p.getUserType() != UserTypeEnum.PATIENT){
			/* The structure of the HQL appears to prevent
			 * this from ever happening but just in case...
			 */
			throw new DataIntegrityViolationException("User found is not a patient.");
		}

		return p;
	}


	public void fleshFacilitiesExcludeLock(Patient p){
		getHibernateTemplate().initialize(p.getFacilities());
	}

	public void fleshFacilities(Patient p){
		getSession().lock(p,LockMode.NONE);
		getHibernateTemplate().initialize(p.getFacilities());
	}

	@SuppressWarnings("unchecked")
	public Patient findPatientById(Long id){

		String[] paramNames = new String[] { "id" };
		Object[] paramValues = new Object[] { id };

		List<Patient> o = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_PATIENT_BY_ID, paramNames, paramValues );

		if(o.size() == 0){
			return null;
		}

		if(o.size() > 1){
			throw new DataIntegrityException("Found users with duplicate ID");
		}
		return o.get(0);
	}

	@SuppressWarnings("unchecked")
	public MHVPatient findMHVPatientById(Long id,String stationNumber){

		String[] paramNames = new String[] { "id","stationNumber" };
		Object[] paramValues = new Object[] { id ,stationNumber};

		List<MHVPatient> o = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_MHV_PATIENT_BY_ID, paramNames, paramValues );

		if(o.size() == 0){
			return null;
		}

		if(o.size() > 1){
			throw new DataIntegrityException("Found users with duplicate ID");
		}
		return o.get(0);
	}

	@SuppressWarnings("unchecked")
	public MHVPatient findMHVPatientByUserName(String userName){

		String[] paramNames = new String[] { "userName" };
		Object[] paramValues = new Object[] { userName};

		List<MHVPatient> o = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_MHV_PATIENT_BY_USER_NAME, paramNames, paramValues );

		if(o.size() == 0){
			return null;
		}

		//if(o.size() > 1){
		//	throw new DataIntegrityException("Found users with duplicate ID");
		//}
		return o.get(0);
	}

	/**
	 * KJG ADDED FOR API CALL
	 */
	@SuppressWarnings("unchecked")
	public MHVPatient findMHVPatientByUserProfileId(Long userProfileId){

		String[] paramNames = new String[] { "userProfileId" };
		Object[] paramValues = new Object[] { userProfileId };

		List<MHVPatient> o = getHibernateTemplate().findByNamedQueryAndNamedParam(
				UserDao.QRY_MHV_PATIENT_BY_USER_PROFILE_ID, paramNames, paramValues );

		if(o.size() == 0){
			return null;
		}

		return o.get(0);
	}

	@SuppressWarnings("unchecked")
	public Patient findPatientByIcn(String icn){

		String[] paramNames = new String[] { "icn" };
		Object[] paramValues = new Object[] { icn };

		List<Patient> o = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_PATIENT_BY_ICN, paramNames, paramValues );

		if(o.size() == 0){
			return null;
		}

		if(o.size() > 1){
			throw new DataIntegrityException("Found patients with duplicate ICN");
		}
		return o.get(0);
	}


	@SuppressWarnings("unchecked")
	public Clinician findClinicianById(Long id){

		String[] paramNames = new String[] { "id" };
		Object[] paramValues = new Object[] { id };

		List<Clinician> o = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_CLINICIAN_BY_ID, paramNames, paramValues );

		if(o.size() == 0){
			return null;
		}

		if(o.size() > 1){
			throw new DataIntegrityException("Found users with duplicate ID");
		}
		return o.get(0);
	}

	@SuppressWarnings("unchecked")
	public Administrator findAdministratorById(Long id){

		String[] paramNames = new String[] { "id" };
		Object[] paramValues = new Object[] { id };

		List<Administrator> o = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_ADMINISTRATOR_BY_ID, paramNames, paramValues );

		if(o.size() == 0){
			return null;
		}

		if(o.size() > 1){
			throw new DataIntegrityException("Found users with duplicate ID");
		}
		return o.get(0);
	}


	/**
	 * Queries patient for send message addressee.
	 * @param fn - first name
	 * @param ln - last name
	 * @param nssn - nssn
	 * @param stationNumber - vista station number
	 */
	@SuppressWarnings("unchecked")
	public Collection<Patient> searchPatient(String fn, String ln, String nssn, String stationNumber)
	{
		Collection<Patient> result = null;
		Criteria  criteria = getSession().createCriteria(gov.va.med.mhv.sm.model.Patient.class);
		if (fn != null && fn.length()>0){
			criteria.add(Expression.ilike("firstName",fn,MatchMode.START));
		}
		if(ln != null && ln.length()>0){
			criteria.add(Expression.ilike("lastName",ln,MatchMode.START));
		}
		if(nssn != null && nssn.length()>0){
			criteria.add(Expression.ilike("nssn",nssn.toUpperCase(), MatchMode.START));
		}
		criteria.createAlias("facilities", "f");
		criteria.add(Expression.eq("f.stationNo", stationNumber));
		criteria.addOrder(Order.asc("lastName").ignoreCase());
		result = (Collection<Patient>)criteria.list();

		return result;
	}



	/**
	 * Queries MHV patient for send message addressee.
	 * @param fn - first name
	 * @param ln - last name
	 * @param nssn - nssn
	 * @param stationNumber - vista station number
	 */
	@SuppressWarnings("unchecked")
	public List<MHVPatient> searchMHVPatient(String fn, String ln, String nssn, String stationNumber)
	{
		List<MHVPatient> result = null;
		Criteria  criteria = getSession().createCriteria(gov.va.med.mhv.sm.model.MHVPatient.class);
		if (fn != null && fn.length()>0){
			criteria.add(Expression.ilike("firstName",fn,MatchMode.START));
		}
		if(ln != null && ln.length()>0){
			criteria.add(Expression.ilike("lastName",ln,MatchMode.START));
		}
		if(nssn != null && nssn.length()>0){
			criteria.add(Expression.ilike("nssn",nssn.toUpperCase(), MatchMode.START));
		}
		//criteria.createAlias("facilities", "f");
		criteria.add(Expression.eq("facility", stationNumber));
		result = (List<MHVPatient>)criteria.list();
		 System.out.println("out searchMHVPatient ... " );
		return result;
	}

	/**
	 * Queries clinician for send message addressee.
	 *
	 * @param fn - first name
	 * @param ln - last name
	 */
	@SuppressWarnings("unchecked")
	public Collection<Clinician> searchClinician(String fn, String ln, String station, boolean active){
		Collection<Clinician> result = null;
		Criteria  criteria = getSession().createCriteria(gov.va.med.mhv.sm.model.Clinician.class);
		if (fn != null && fn.length()>0){
			criteria.add(Expression.ilike("firstName",fn,MatchMode.START));
		}
		if(ln != null && ln.length()>0){
			criteria.add(Expression.ilike("lastName",ln,MatchMode.START));
		}
		criteria.add(Expression.ilike("stationNo",station,MatchMode.EXACT));
		criteria.add(Expression.eq("active", new Boolean(active)));
		criteria.addOrder(Order.asc("lastName").ignoreCase());
		result = (List<Clinician>)criteria.list();
		return result;
	}

	/**
	 * Queries All Clinician includes acvite/inactive
	 *
	 * @param fn - first name
	 * @param ln - last name
	 * @param station - station number
	 */
	@SuppressWarnings("unchecked")
	public List<Clinician> searchClinician(String fn, String ln, String station){
		List<Clinician> result = null;
		Criteria  criteria = getSession().createCriteria(gov.va.med.mhv.sm.model.Clinician.class);
		if (fn != null && fn.length()>0){
			criteria.add(Expression.ilike("firstName",fn,MatchMode.ANYWHERE));
		}
		if(ln != null && ln.length()>0){
			criteria.add(Expression.ilike("lastName",ln,MatchMode.ANYWHERE));
		}
		criteria.add(Expression.ilike("stationNo",station,MatchMode.EXACT));
		result = (List<Clinician>)criteria.list();
		return result;
	}

	public boolean isClinicianMemberOfTriage(Long userId, Long triageGroupId){
		String[] paramNames = new String[] { "userId","triageGroupId" };
		Object[] paramValues = new Object[] { userId,triageGroupId };

		List<Object> o = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_CLINICIAN_IS_MEMBER_OF_TRIAGE, paramNames, paramValues );

		if(o!=null){
			if(o.size()==1){
				return true;
			}
		}
		return false;
	}

	@SuppressWarnings("unchecked")
	public Patient findPatientByUsername(String username){
		String[] paramNames = new String[] { "username" };
		Object[] paramValues = new Object[] { username.toUpperCase() };

		List<Patient> l = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_PATIENT_BY_USERNAME, paramNames, paramValues );

		if(l.size() == 0){
			return null;
		}

		if(l.size() > 1){
			throw new DataIntegrityException("Found patients with duplicate username");
		}
		return l.get(0);

	}

	@SuppressWarnings("unchecked")
	public List<Patient> findAllPatients(){

		String[] paramNames = new String[] { };
		Object[] paramValues = new Object[] { };

		List<Patient> l = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_ALL_PATIENTS, paramNames, paramValues );

		return l;

	}

	@SuppressWarnings("unchecked")
	public Clinician findClinicianByUsername(String username){
		String[] paramNames = new String[] { "username" };
		Object[] paramValues = new Object[] { username.toUpperCase() };

		List<Clinician> l = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_CLINICIAN_BY_USERNAME, paramNames, paramValues );

		if(l.size() == 0){
			return null;
		}

		if(l.size() > 1){
			throw new DataIntegrityException("Found clinicians with duplicate username");
		}
		return l.get(0);

	}

	@SuppressWarnings("unchecked")
	public Administrator findAdministratorByUsername(String username){
		String[] paramNames = new String[] { "username" };
		Object[] paramValues = new Object[] { username.toUpperCase() };

		List<Administrator> l = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_ADMINISTRATOR_BY_USERNAME, paramNames, paramValues );

		if(l.size() == 0){
			return null;
		}

		if(l.size() > 1){
			throw new DataIntegrityException("Found administrators with duplicate username");
		}
		return l.get(0);

	}

	public User savePreferences(User u){

		User x = findById(u.getId());

		x.setEmail(u.getEmail());
		x.setEmailNotification(u.getEmailNotification());
		x.setStatus(u.getStatus());
		x.setMessageFilter(u.getMessageFilter());

		return save(x);


	}


	@SuppressWarnings("unchecked")
	public Patient findPatientByIen(String ien, String stationNo){

		String[] paramNames = new String[] { "ien", "stationNo" };
		Object[] paramValues = new Object[] { ien, stationNo };

		List<Patient> l = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_PATIENT_BY_IEN, paramNames, paramValues );

		if(l == null || l.size() == 0)
			return null;


		if(l.size() > 1){
			log.warn("more than one patient have same STATION^IEN: " + stationNo + "^" + ien);
		}

		return l.get(0);

	}



	@SuppressWarnings("unchecked")
	public List<Clinician> getCliniciansForStation(String stationNo){

		String[] paramNames = new String[] { "stationNo" };
		Object[] paramValues = new Object[] { stationNo };

		List<Clinician> l = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_CLINICIANS_FOR_STATION, paramNames, paramValues );

		return l;


	}


	@SuppressWarnings("unchecked")
	public List<Clinician> getCliniciansForStationAll(String stationNo){

		String[] paramNames = new String[] { "stationNo" };
		Object[] paramValues = new Object[] { stationNo };

		List<Clinician> l = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_CLINICIANS_FOR_STATION_ALL, paramNames, paramValues );

		return l;


	}


	public User activateUser(User u){

		User x = findById(u.getId());
		x.setActive(true);
		return save(x);
	}

	public User inactivateUser(User u){

		User x = findById(u.getId());
		x.setActive(false);
		return save(x);

	}

	public Patient updateRelationshipSyncDate(Patient p){
		Patient x = this.findPatientById(p.getId());
		x.setRelationshipUpdate(p.getRelationshipUpdate());
		return (Patient)this.save(x);
	}

	public List<Object[]> getTreatmentFacilitiesByPatient(Long userId){
		String[] paramNames = new String[] { "userId" };
		Object[] paramValues = new Object[] { userId };

		List<Object[]> l = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_GET_TREATMENT_FACILITIES_BY_PATIENT_ID, paramNames, paramValues );
		return l;
	}

	/**
	 * Get User's details by Ids
	 *
	 * @param messageId, userId
	 * @return List<Object[]>
	 */
	public List<Object[]> getUserDetailsByIds(Long messageId, Long userId){
		String[] paramNames = new String[] { "messageId","userId" };
		Object[] paramValues = new Object[] { messageId,userId };

		List<Object[]> l = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_GET_USER_DETAILS_BY_IDS, paramNames, paramValues );
		return l;
	}

	public int updateSMOptStatusByUserName(String userName,Boolean optStatus){
		Object[] paramValues = new Object[] {optStatus,userName};
		Query q = getSession().getNamedQuery("QryUpdateSMOptStatusByUserName");
		String sql = q.getQueryString();
		return getHibernateTemplate().bulkUpdate(sql,paramValues);
	}
	
	@SuppressWarnings("unchecked")
	public UserProfileVW getUserProfileByUserName(String userName){

		String[] paramNames = new String[] { "userName" };
		Object[] paramValues = new Object[] { userName };

		List<UserProfileVW> userProfileVwLst = getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_USER_PROFILE_BY_USERNAME, paramNames, paramValues );

		if(userProfileVwLst == null || userProfileVwLst.size() == 0)return null;
		return userProfileVwLst.get(0);
	}
}
