package gov.va.med.mhv.sm.dao.hibernate;

import gov.va.med.mhv.persistence.dao.hibernate.BaseEntityDaoHibernate;
import gov.va.med.mhv.sm.dao.AddresseeDao;
import gov.va.med.mhv.sm.dao.exception.DataIntegrityException;
import gov.va.med.mhv.sm.dto.SearchFormDTO;
import gov.va.med.mhv.sm.enumeration.AddresseeRoleEnum;
import gov.va.med.mhv.sm.enumeration.CPRSNotesStatusEnum;
import gov.va.med.mhv.sm.enumeration.ClinicianStatusEnum;
import gov.va.med.mhv.sm.enumeration.MessageCategoryTypeEnum;
import gov.va.med.mhv.sm.enumeration.MessageFilterEnum;
import gov.va.med.mhv.sm.enumeration.MessagesOrderByEnum;
import gov.va.med.mhv.sm.enumeration.SortOrderEnum;
import gov.va.med.mhv.sm.enumeration.SystemFolderEnum;
import gov.va.med.mhv.sm.model.Addressee;
import gov.va.med.mhv.sm.model.Folder;
import gov.va.med.mhv.sm.model.MessagesPage;
import gov.va.med.mhv.sm.model.priv.MessageCount;
import gov.va.med.mhv.sm.util.DateUtils;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.Criteria;
import org.hibernate.HibernateException;
import org.hibernate.ScrollMode;
import org.hibernate.ScrollableResults;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Expression;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;

public class AddresseeDaoImpl extends BaseEntityDaoHibernate<Addressee, Long> 
	implements AddresseeDao {

	@SuppressWarnings("unused")
	private static final Log log = LogFactory.getLog(AddresseeDaoImpl.class);
	private static final String EMPTY_STR = "";
	private static final String LIKE_STR = "%";

	@SuppressWarnings("unchecked")
	public List<MessageCount> getFolderCounts(Long ownerId){

		String[] paramNames = new String[] { "userId", "lastyear" }; 
		Object[] paramValues = new Object[] { ownerId, DateUtils.getLastYear()}; 
		
		List<Object[]> o = getHibernateTemplate().findByNamedQueryAndNamedParam(
				AddresseeDao.QRY_MESSAGE_COUNTS, paramNames, paramValues );
		
		List<MessageCount> l = new ArrayList<MessageCount>();
		for(Object[] c : o){
			MessageCount x = new MessageCount();
			x.setFolderId(Long.parseLong(c[0].toString()));
			x.setCount(Integer.parseInt(c[1].toString()));
			x.setUnreadCount(Integer.parseInt(c[1].toString()) - Integer.parseInt(c[2].toString()));
			l.add(x);
		}
		return l;
	}
	
	@SuppressWarnings("unchecked")
	public List<MessageCount> getCategoryCounts(Long ownerId){

		String[] paramNames = new String[] { "userId", "lastyear", "folderId" }; 
		Object[] paramValues = new Object[] { ownerId, DateUtils.getLastYear(),SystemFolderEnum.INBOX.getId()}; 
		
		List<Object[]> o = getHibernateTemplate().findByNamedQueryAndNamedParam(
				AddresseeDao.QRY_MESSAGE_CATEGORY_COUNTS, paramNames, paramValues );
		
		List<MessageCount> l = new ArrayList<MessageCount>();
		for(Object[] c : o){
			MessageCount x = new MessageCount();
			x.setMessageCategoryTypeId(Long.parseLong(c[0].toString()));
			x.setCount(Integer.parseInt(c[1].toString()));
			x.setUnreadCount(Integer.parseInt(c[1].toString()) - Integer.parseInt(c[2].toString()));
			l.add(x);
		}
		return l;
	}
	
	@SuppressWarnings("unchecked")
	public MessageCount getEscalatedCount(Long ownerId){
		
		String[] paramNames = new String[] { "userId", "lastyear" }; 
		Object[] paramValues = new Object[] { ownerId, DateUtils.getLastYear() }; 
		
		List<Object> l = getHibernateTemplate().findByNamedQueryAndNamedParam(
				AddresseeDao.QRY_ESCALATED_COUNT, paramNames, paramValues );
		
		MessageCount x = new MessageCount();
		x.setFolderId(SystemFolderEnum.ESCALATED.getId());
		x.setCount(Integer.parseInt(l.get(0).toString()));
		
		return x;
	}
	
	@SuppressWarnings("unchecked")
	public int getMessageReminderToday(Long userId){
		
		String[] paramNames = new String[] { "userId", }; 
		Object[] paramValues = new Object[] { userId }; 
		
		List<Object> l = getHibernateTemplate().findByNamedQueryAndNamedParam(
				AddresseeDao.QRY_COUNT_MESSAGE_REMINDERTODAY, paramNames, paramValues );
		
		return Integer.parseInt(l.get(0).toString());
	}

	public MessagesPage getMessages(Long userId, Long folderId, int pageNumber, int pageSize,
			MessagesOrderByEnum orderBy, MessageFilterEnum filter, 
			SortOrderEnum sortOrder, int messageCount, SearchFormDTO sf, Long subFolderId ){

		String o = null;
		Criteria  criteria = getSession().createCriteria(gov.va.med.mhv.sm.model.Addressee.class);
		criteria.createAlias("message", "m");
		criteria.createAlias("m.thread", "t");
		criteria.add(Restrictions.eq("m.active", new Boolean(true) ));

		switch(orderBy){
			
			case FROM:{
				o = "m.senderName";
				criteria.addOrder(sortOrder == SortOrderEnum.ASC ? Order.asc(o).ignoreCase() : Order.desc(o).ignoreCase());
				criteria.addOrder(Order.desc("m.sentDate"));
				break;
			}
			case TO:{
				o = "m.recipientName";
				criteria.addOrder(sortOrder == SortOrderEnum.ASC ? Order.asc(o).ignoreCase() : Order.desc(o).ignoreCase());
				criteria.addOrder(Order.desc("m.sentDate"));
				break;
			}
			case SUBJECT:{
				o = "t.subject";
				criteria.addOrder(sortOrder == SortOrderEnum.ASC ? Order.asc(o).ignoreCase()  : Order.desc(o).ignoreCase());
				criteria.addOrder(Order.desc("m.sentDate"));
				break;
			}
			case ASSIGNED:{
				criteria.createAlias("m.assignedTo", "c", Criteria.LEFT_JOIN);
				o = "c.lastName";
				criteria.addOrder(sortOrder == SortOrderEnum.ASC ? Order.asc(o).ignoreCase() : Order.desc(o).ignoreCase());
				o = "c.firstName";
				criteria.addOrder(sortOrder == SortOrderEnum.ASC ? Order.asc(o).ignoreCase() : Order.desc(o).ignoreCase());
				criteria.addOrder(Order.desc("m.sentDate"));
				break;
			}
			case STATUS:{
				o = "m.status";
				criteria.addOrder(sortOrder == SortOrderEnum.ASC ? Order.asc(o) : Order.desc(o));
				criteria.addOrder(Order.desc("m.sentDate"));
				break;
			}
			case CREATED_DATE:{
				o = "m.createdDate";
				criteria.addOrder(sortOrder == SortOrderEnum.ASC ? Order.asc(o) : Order.desc(o));
				criteria.addOrder(Order.desc("m.createdDate"));
				break;
			}
			
			case READ_RECEIPT:{
				o = "m.readReceipt";
				criteria.addOrder(sortOrder == SortOrderEnum.ASC ? Order.asc(o) : Order.desc(o));
				criteria.addOrder(Order.desc("m.readReceipt"));
				break;
			}
			
			case CPRSALERTS_DATE:{
				o = "m.cprsNotesFailedDate";
				criteria.addOrder(sortOrder == SortOrderEnum.ASC ? Order.asc(o) : Order.desc(o));
				criteria.addOrder(Order.desc("m.cprsNotesFailedDate"));
				break;
			}
			
			default:{
				// This should be done in the Action setting the default order.  But for now
				// we will set it up here.
				if(folderId == SystemFolderEnum.CPRSPROGRESSALERTS.getId()){
					o = "m.cprsNotesFailedDate";
					criteria.addOrder(sortOrder == SortOrderEnum.ASC ? Order.asc(o) : Order.desc(o));
					break;
				}
				if( folderId == SystemFolderEnum.DRAFTS.getId() ) {
					o = "m.createdDate";
				} else {
					o = "m.sentDate";
				}
				
				criteria.addOrder(sortOrder == SortOrderEnum.ASC ? Order.asc(o) : Order.desc(o));
				
			}
		}
		
		switch(filter){
			case ASSIGNED_TO_ME:{
				Criterion assignedToCriterion = Restrictions.eq("m.assignedTo.id", userId);
				Criterion ccAndSurrogate = Restrictions.or(Restrictions.eq("role", AddresseeRoleEnum.CC),Restrictions.eq("role", AddresseeRoleEnum.CC_SURROGATE));
				criteria.add(Restrictions.or(assignedToCriterion,ccAndSurrogate));
				break;
			}
			case UNASSIGNED:{
				Criterion unAssignedCriterion =Expression.isNull("m.assignedTo");
				Criterion ccAndSurrogate = Restrictions.or(Restrictions.eq("role", AddresseeRoleEnum.CC),Restrictions.eq("role", AddresseeRoleEnum.CC_SURROGATE));
				criteria.add(Restrictions.or(unAssignedCriterion, ccAndSurrogate));
				break;
			}
		}
		criteria.add(Expression.eq("owner.id", userId));
	
		if(folderId == SystemFolderEnum.ESCALATED.getId()){
			criteria.add(Expression.isNotNull("m.escalatedDate"));
			criteria.add(Expression.ne("m.status", ClinicianStatusEnum.COMPLETE));
		}
		if(folderId == SystemFolderEnum.REMINDER.getId()){
			criteria.add(Expression.isNotNull("reminderDate"));
		}
		else
		{	if(folderId != SystemFolderEnum.ESCALATED.getId() && folderId != SystemFolderEnum.REASSIGN.getId() && folderId != SystemFolderEnum.CPRSPROGRESSALERTS.getId()){
				criteria.add(Expression.eq("folderId",folderId));
			}
		}
		
		if(folderId.equals(SystemFolderEnum.INBOX.getId()) && subFolderId != null) {
			//6598
			criteria.add(Expression.eq("t.messageCategoryType", MessageCategoryTypeEnum.valueOf(subFolderId)));
		}
		
		if(folderId.equals(SystemFolderEnum.CPRSPROGRESSALERTS.getId())){
			criteria.add(Restrictions.eq("m.cprsNotesStatus", CPRSNotesStatusEnum.FAILED.getDescription()));
		}
		
		if(folderId.equals(SystemFolderEnum.REASSIGN.getId())){
			criteria.add(Restrictions.eq("reAssigned", new Boolean(true)));
			
		}else{
			// Fetch the active addressee for folders other than Reassign	
			criteria.add(Restrictions.eq("active", new Boolean(true)));		
		}
		
		// in the drafts folder messages have not been sent so only
		// add the criteria in all other folders
		if(!folderId.equals(SystemFolderEnum.DRAFTS.getId())){
			criteria.add(Expression.gt("m.sentDate", DateUtils.getLastYear()));
		}
		
		//Implement Search
		if( sf!=null ) {
			if(sf.getFromDate() != null && sf.getToDate() != null) {
				if( folderId == SystemFolderEnum.DRAFTS.getId() ) {
					o = "m.createdDate";
				} else {
					o = "m.sentDate";
				}
				criteria.add(Restrictions.between(o,sf.getFromDate(),sf.getToDate()));
			}
			else if(sf.getFromDate() != null && sf.getToDate() == null) {
				if( folderId == SystemFolderEnum.DRAFTS.getId() ) {
					o = "m.createdDate";
				} else {
					o = "m.sentDate";
				}
				criteria.add(Restrictions.ge(o,sf.getFromDate()));
			}
			else if(sf.getFromDate() == null && sf.getToDate() != null) {
				if( folderId == SystemFolderEnum.DRAFTS.getId() ) {
					o = "m.createdDate";
				} else {
					o = "m.sentDate";
				}
				criteria.add(Restrictions.le(o,sf.getToDate()));
			}
			
			if(sf.getFrom()!=null && sf.getFrom().length() > 0) {
				if(sf.isExactMatch()){
					criteria.add(Restrictions.eq("m.senderName", sf.getFrom()).ignoreCase());	
				}else{
					criteria.add(Restrictions.ilike("m.senderName", LIKE_STR+sf.getFrom()+LIKE_STR));
				}
			}
			if(sf.getTo()!=null && sf.getTo().length() > 0) {
				if(sf.isExactMatch()){
					criteria.add(Restrictions.eq("m.recipientName", sf.getTo()).ignoreCase());	
				}else{
					criteria.add(Restrictions.ilike("m.recipientName", LIKE_STR+sf.getTo()+LIKE_STR));
				}
			}
			if(sf.getSubject()!=null && sf.getSubject().length() > 0) {
				if(sf.isExactMatch()){
					criteria.add(Restrictions.eq("t.subject",sf.getSubject()).ignoreCase());
				}else{
					criteria.add(Restrictions.ilike("t.subject",LIKE_STR+sf.getSubject()+LIKE_STR));	
				}
			}
			
			if(sf.getMessageId()!=null && sf.getMessageId() > 0) {
				if(sf.isExactMatch()){
					criteria.add(Restrictions.eq("m.id", sf.getMessageId()));	
				}else{
					criteria.add(Restrictions.eq("m.id", sf.getMessageId()));
					//criteria.add(Restrictions.ilike("m.id", LIKE_STR+sf.getMessageId()+LIKE_STR));
				}
			}
		}
		if(log.isInfoEnabled()){
			log.info(criteria.toString());
		}
		
		return queryPage(criteria, pageNumber, pageSize, messageCount);
		
	}
	
	
	@SuppressWarnings("unchecked")
	private MessagesPage queryPage(Criteria q, int pageNumber, int pageSize, int messageCount){
		
		MessagesPage page = new MessagesPage();
		ScrollableResults sr=null;
		int elementCount;
		List<Addressee> elements;
		
		try {
			
		sr = q.scroll(ScrollMode.SCROLL_INSENSITIVE);
		sr.last();
		elementCount = sr.getRowNumber() + 1;
		
		sr.first();
		if (Integer.MAX_VALUE == pageNumber)
			pageNumber = calculateLastPageNumber(elementCount, pageSize);
		else if (pageNumber>(elementCount/pageSize))
			pageNumber = calculateLastPageNumber(elementCount, pageSize);
		
		q = q.setFirstResult(pageNumber * pageSize);
	    q = q.setMaxResults(pageSize);
	    elements = q.list();
		
		page.setPageNumber(pageNumber);
		page.setPageSize(pageSize);
		page.setElementCount(elementCount);
		page.setElements(elements);
		
		//initialize the underlying message and thread
		for(Addressee a : elements){
			getHibernateTemplate().initialize(a.getMessage());
			getHibernateTemplate().initialize(a.getMessage().getThread());
		}
		
		
		sr.close();
		} catch (HibernateException e) {
			log.error("Failed to create ScrollPage by getScrollPageInstanceWithTotalByScroll: " + e.getMessage());
			throw new RuntimeException(e);
		}
		
		finally{
			if(sr!=null){
				sr.close();
			}
		}
		return page;
		
	}
	
	/**
	 * This calculates the last page (zero based).
	 * 
	 * If it does NOT have any elements on the last page then we 
	 * need to calculate the last page - 1.
	 * 
	 * Make sure to return value is not less than '0'.
	 * 
	 * @param elementCount
	 * @param pageSize
	 * @return
	 */
	private int calculateLastPageNumber( int elementCount, int pageSize) {
		boolean elementsOnLastPage = elementCount%pageSize>0;
		int lastpage = ((elementCount/pageSize)-(elementsOnLastPage?0:1));
		return (lastpage>=0?lastpage:0);
	}
	
	@SuppressWarnings("unchecked")
	public Addressee getAddresseeForUser(Long userId, Long messageId){
		
		String[] paramNames = new String[] { "messageId", "userId" }; 
		Object[] paramValues = new Object[] { messageId, userId }; 
		
		List<Addressee> l =  getHibernateTemplate().findByNamedQueryAndNamedParam(
				QRY_ADDRESSEE_FOR_USER_MESSAGE, paramNames, paramValues );
		
		if(l.size() > 1){
			log.error("The user has multiple addressee records for the same message");
			log.error("User: " + userId + "  messageId: " + messageId);
			throw new DataIntegrityException("The user has multiple addressee records for the same message.");
		}

		if(l.size() == 0) return null;
		
		return l.get(0);
		
	}

	@SuppressWarnings("unchecked")
	public MessageCount getFolderCounts(Folder f) {
		
		String[] paramNames = new String[] { "folderId", "userId", "lastyear" }; 
		Object[] paramValues = new Object[] { f.getId(), f.getOwner().getId(), DateUtils.getLastYear()}; 
		
		List<Object[]> o = null;
		
		if( log.isDebugEnabled()) {
			log.debug("*****> getFolderCounts with " + f.getFilter() + " filter.");
		}
		
		switch (f.getFilter()) {
			case ASSIGNED_TO_ME: {
				o = getHibernateTemplate().findByNamedQueryAndNamedParam(
						AddresseeDao.QRY_MESSAGE_COUNT_ASSIGNED_TO, paramNames, paramValues );
				break;
			}
			case UNASSIGNED: {
				o = getHibernateTemplate().findByNamedQueryAndNamedParam(
						AddresseeDao.QRY_MESSAGE_COUNT_UNASSIGNED, paramNames, paramValues );
				break;
			}
			case ALL: {
				o = getHibernateTemplate().findByNamedQueryAndNamedParam(
						AddresseeDao.QRY_MESSAGE_COUNT_ALL, paramNames, paramValues );
			}
		}
		
		MessageCount x = new MessageCount();
		Object[] c = o.get(0);
		x.setCount(Integer.parseInt(c[0].toString()));
		x.setUnreadCount(Integer.parseInt(c[1].toString()));
		
		return x;
	}
	
	@SuppressWarnings("unchecked")
	public List<MessageCount> getCategoryCounts(Folder f){

		String[] paramNames = new String[] { "userId", "lastyear", "folderId" }; 
		Object[] paramValues = new Object[] { f.getOwner().getId(), DateUtils.getLastYear(),SystemFolderEnum.INBOX.getId()}; 
		
		List<Object[]> o = null;
		
		if( log.isDebugEnabled()) {
			log.debug("*****> getFolderCounts with " + f.getFilter() + " filter.");
		}
		
		switch (f.getFilter()) {
			case ASSIGNED_TO_ME: {
				o = getHibernateTemplate().findByNamedQueryAndNamedParam(
						AddresseeDao.QRY_MESSAGE_CATEGORY_COUNTS_ASSIGNED_TO, paramNames, paramValues );
				break;
			}
			case UNASSIGNED: {
				o = getHibernateTemplate().findByNamedQueryAndNamedParam(
						AddresseeDao.QRY_MESSAGE_CATEGORY_COUNTS_UNASSIGNED, paramNames, paramValues );
				break;
			}
			case ALL: {
				o = getHibernateTemplate().findByNamedQueryAndNamedParam(
						AddresseeDao.QRY_MESSAGE_CATEGORY_COUNTS, paramNames, paramValues );
			}
		}
		
		List<MessageCount> l = new ArrayList<MessageCount>();
		for(Object[] c : o){
			MessageCount x = new MessageCount();
			x.setMessageCategoryTypeId(Long.parseLong(c[0].toString()));
			x.setCount(Integer.parseInt(c[1].toString()));
			//x.setUnreadCount(Integer.parseInt(c[2].toString()));
			x.setUnreadCount(Integer.parseInt(c[1].toString()) - Integer.parseInt(c[2].toString()));
			l.add(x);
		}
		return l;
	}

	@SuppressWarnings("unchecked")
	public MessageCount getReminderCount(Long ownerId){
		
		String[] paramNames = new String[] { "userId", "lastyear" }; 
		Object[] paramValues = new Object[] { ownerId, DateUtils.getLastYear() }; 
		
		List<Object> l = getHibernateTemplate().findByNamedQueryAndNamedParam(
				AddresseeDao.QRY_REMINDER_COUNT, paramNames, paramValues );
		
		Object[] c = (Object[])l.get(0);
		MessageCount x = new MessageCount();
		x.setFolderId(SystemFolderEnum.REMINDER.getId());
		x.setCount(Integer.parseInt(c[0].toString()));
		x.setUnreadCount(Integer.parseInt(c[1].toString()));
		
		return x;
	}
	
	@SuppressWarnings("unchecked")
	public MessageCount getReassignedCount(Long ownerId){
		
		String[] paramNames = new String[] { "userId", "lastyear" }; 
		Object[] paramValues = new Object[] { ownerId, DateUtils.getLastYear() }; 
		
		List<Object> l = getHibernateTemplate().findByNamedQueryAndNamedParam(
				AddresseeDao.QRY_REASSIGNED_COUNT, paramNames, paramValues );
		
		Object[] c = (Object[])l.get(0);
		MessageCount x = new MessageCount();
		x.setFolderId(SystemFolderEnum.REASSIGN.getId());
		x.setCount(Integer.parseInt(c[0].toString()));
		x.setUnreadCount(Integer.parseInt(c[1].toString()));
		
		return x;
	}
	
	@SuppressWarnings("unchecked")
	public MessageCount getCPRSAlertsCount(Long userId){
		
		String[] paramNames = new String[] { "userId", "lastyear" }; 
		Object[] paramValues = new Object[] { userId, DateUtils.getLastYear() }; 
		
		List<Object> l = getHibernateTemplate().findByNamedQueryAndNamedParam(
				AddresseeDao.QRY_CPRSALERTS_COUNT, paramNames, paramValues );
		
		MessageCount x = new MessageCount();
		x.setFolderId(SystemFolderEnum.CPRSPROGRESSALERTS.getId());
		x.setCount(Integer.parseInt(l.get(0).toString()));
		
		return x;
	}
	
}

