package gov.va.med.mhv.sm.dao;

import gov.va.med.mhv.foundation.dao.BaseEntityDao;
import gov.va.med.mhv.sm.model.Administrator;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.MHVPatient;
import gov.va.med.mhv.sm.model.MailParticipant;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.Surrogate;
import gov.va.med.mhv.sm.model.User;
import gov.va.med.mhv.sm.model.UserProfileVW;

import java.util.Collection;
import java.util.Date;
import java.util.List;

public interface UserDao extends BaseEntityDao<User, Long> {
	
	public static final String QRY_AUTH_PATIENT = "QryAuthPatient";
	public static final String QRY_AUTH_CLINICIAN_BY_STATION_AND_DUZ = "QryAuthClinicianByStationAndDuz";
	public static final String QRY_CLINICIAN_BY_STATION_AND_DUZ = "QryClinicianByStationAndDuz";
	public static final String QRY_CLINICIAN_FOR_STATION_AND_DUZ = "QryClinicianForStationAndDuz";
	public static final String QRY_AUTH_CLINICIAN_BY_STATION_AND_USERNAME = "QryAuthClinicianByStationAndUsername";
	public static final String QRY_PATIENT_BY_ID = "QryPatientById";
	public static final String QRY_MHV_PATIENT_BY_ID = "QryMHVPatientById";
	public static final String QRY_MHV_PATIENT_BY_USER_NAME = "QryMHVPatientByUserName";
	public static final String QRY_CLINICIAN_BY_ID = "QryClinicianById";
	public static final String QRY_ADMINISTRATOR_BY_ID = "QryAdministratorById";
	//public static final String QRY_CLINICIAN_IS_SURROGATE_CLINICIAN = "QryClinicianIsSurrogateClinician";
	//public static final String QRY_CLINICIAN_IS_SURROGATE_TRIAGE = "QryClinicianIsSurrogateTriage";
	public static final String QRY_CLINICIAN_IS_MEMBER_OF_TRIAGE = "QryClinicianIsMemberOfTriage";
	public static final String QRY_PATIENT_BY_USERNAME = "QryPatientByUsername";
	public static final String QRY_CLINICIAN_BY_USERNAME = "QryClinicianByUsername";
	public static final String QRY_ADMINISTRATOR_BY_USERNAME = "QryAdministratorByUsername";
	public static final String QRY_CLINICIANS_FOR_STATION_ALL = "QryCliniciansForStationAll";
	public static final String QRY_CLINICIANS_FOR_STATION = "QryCliniciansForStation";
	public static final String QRY_PATIENT_BY_IEN = "QryPatientByIen";
	public static final String QRY_PATIENT_BY_ICN = "QryPatientByIcn";
	public static final String QRY_CLINICIANS_WITH_TG_SURROGATE ="QryClinicansWithTgSurrogate"; 
	public static final String QRY_ALL_PATIENTS = "QryAllPatients";
	public static final String QRY_MHV_PATIENT_BY_USER_PROFILE_ID = "QryMHVPatientByUserProfileId";
	public static final String QRY_GET_TREATMENT_FACILITIES_BY_PATIENT_ID = "QryGetTreatmentFacilitiesNameByPatient";
	public static final String QRY_GET_USER_DETAILS_BY_IDS = "QryGetUserDetailsByIds";
	public static final String QRY_UPDATE_SMOPT_STATUS_BY_USERNAME = "QryUpdateSMOptStatusByUserName";
	public static final String QRY_USER_PROFILE_BY_USERNAME = "QryUserProfileForUserName";
	
	
	
	/**
	 * Used to fetch a Clinician from the passed in values.
	 * 
	 * <b>Will return a clinician despite active status.</b>
	 * 
	 * @param stationNo
	 * @param duz
	 * @return
	 */
	public Clinician findAuthClinicianByStationAndDuz(String stationNo, String duz);
	
	/**
	 * Used to fetch a Clinician from the passed in values.
	 * May return null if the clinician not found.
	 * @param stationNo
	 * @param duz
	 * @return
	 */
	public Clinician findClinicianByStationAndDuz(String stationNo, String duz, boolean active);
	
	/**
	 * Used to fetch a Clinician from the passed in values.
	 * May return null if the clinician not found.
	 * @param stationNo
	 * @param duz
	 * @return
	 */
	public List<Clinician> findClinicianByStationAndDuz(String stationNo, String duz);
	
	
	/**
	 * Used to fetch a Clinician from the passed in values.
	 * May return null if the clinician not found.
	 * @param stationNo
	 * @param username
	 * @return
	 */
	
	public Clinician findClinicianByStationAndUsername(String stationNo, String username);
	
	/**
	 * Used to fetch a patient from the passed in values.
	 * May return null if patient not found.  Guaranteed
	 * to return a Patient.
	 * 
	 * @param lastName   Patients last name  
	 * @param firstName  Patients first name
	 * @param dob        Date of birth
	 * @return           Patient
	 */
	public Patient findPatient(String ssn, String lastName, String firstName, Date dob);
	
	
	public Patient findPatientById(Long id);
	public MHVPatient findMHVPatientById(Long id, String stationNumber);
	public Clinician findClinicianById(Long id);
	public Administrator findAdministratorById(Long id);
	
	/**
	 * Initialize the facilities for said Patient
	 * @param p
	 */
	public void fleshFacilities(Patient p);

	public void fleshFacilitiesExcludeLock(Patient p);

	
	/**
	 * Queries patient for send message addressee.
	 * @param fn             first name
	 * @param ln             last name
	 * @param nssn           nssn
	 * @param stationNumber  VistA facility
	 * return   List<Patient> matching criteria
	 */
	public Collection<Patient> searchPatient(String fn, String ln, String nssn, String stationNumber);
	
	/**
	 * Queries MHV patient for send message addressee.
	 * @param fn             first name
	 * @param ln             last name
	 * @param nssn           nssn
	 * @param stationNumber  VistA facility
	 * return   List<Patient> matching criteria
	 */
	public List<MHVPatient> searchMHVPatient(String fn, String ln, String nssn, String stationNumber);
	
	
	/**
	 * Queries clinician for send message addressee.
	 * @param fn             first name
	 * @param ln             last name
	 * @param station		 a mast	
	 * return   List<Clinician> matching criteria
	 */
	public Collection<Clinician> searchClinician(String fn, String ln, String station, boolean active);
	
	/**
	 * Queries All Clinician includes active/inactive
	 * @param fn             first name
	 * @param ln             last name
	 * @param station		 station number.	
	 * return   List<Clinician> matching criteria
	 */
	public List<Clinician> searchClinician(String fn, String ln, String station);
	
	/**
	 * Return a boolean. If the userId is memeber of the triage group, it will return true, else false.
	 * 
	 * @param userId,triageGroupId 
	 * @return boolean.
	 */
	
	public boolean isClinicianMemberOfTriage(Long userId, Long triageGroupId);
	
	/**
	 * return a patient based on the unique MHV username associated for each patient
	 * 
	 * @param username
	 * @return
	 */
	public Patient findPatientByUsername(String username);
	
	/**
	 * return a patient
	 * 
	 * @param ien
	 * @param stationNo
	 * @return
	 */
	public Patient findPatientByIen(String ien, String stationNo);
	
	/**
	 * return a patient
	 * 
	 * @param icn
	 * @return
	 */
	public Patient findPatientByIcn(String icn);
	
	/**
	 * return a clinician based on the unique network id associated for each clinician
	 * 
	 * @param username
	 * @return
	 */
	public User findClinicianByUsername(String username);
	
	
	/**
	 * return a administrator based on the unique network id associated for each administrator
	 * 
	 * @param username
	 * @return
	 */
	public Administrator findAdministratorByUsername(String username);
	
	
	/**
	 * return a list of clinicians for a station ordered by last,first name
	 * only clinicians with a clinician.status = UserStatusEnum.OPT_IN are returned
	 * 
	 * @see getCliniciansForStationAll(String)
	 * @param stationNo
	 * @return
	 */
	public List<Clinician> getCliniciansForStation(String stationNo);
	
	
	/**
	 * return a list of clinicians for a station ordered by last,first name
	 * all clinicians (regardless of clinician.status) are returned.  
	 * 
	 * @see getCliniciansForStation(String)
	 * @param stationNo
	 * @return
	 */
	public List<Clinician> getCliniciansForStationAll(String stationNo);
	
	
	
	
	/**
	 * Save a users preferences making sure that no
	 * changes are made to other parts of the user object
	 * 
	 * @param u  User to save
	 * 
	 * @return
	 */
	public User savePreferences(User u);
	
	
	
	/**
	 * Activate/Inactivate a user account
	 * 
	 * @param u
	 * @return
	 */
	public User activateUser(User u);
	public User inactivateUser(User u);

	/**
	 * Find clinicians that have the given triage group as a surrogate
	 */
	//public List<Clinician> findClinicansWithTriageGroupSurrogate(Long tgId);
	
	/**
	 * Fetch all patients from the database
	 * 
	 * @return
	 */
	public List<Patient> findAllPatients();
	
	
	
	/**
	 * Update the date/time that the last relationship sync was done.
	 *  
	 * @param p
	 * @return
	 */
	public Patient updateRelationshipSyncDate(Patient p);
	
	
	/**
	 * Find the corresponding patient view info from mhv for username.
	 * 
	 * @param username
	 * @return
	 */
	public MHVPatient findMHVPatientByUserProfileId(Long profileId);
	
	/**
	 * Get patient treatment facilities by userId
	 * 
	 * @param username
	 * @return vistAName and stationNumber of the patient treatment facilities
	 * @return List<Object> return Object[0]=vistAName, Object[1]=stationNumber
	 */
	public List<Object[]> getTreatmentFacilitiesByPatient(Long userId);

	/**
	 * Find the corresponding patient view info from mhv for username.
	 *
	 * @param username
	 * @return
	 */
	public MHVPatient findMHVPatientByUserName(String userName);
	
	/**
	 * Get User's details by Ids
	 *
	 * @param messageId, userId
	 * @return List<Object[]>
	 */
	public List<Object[]> getUserDetailsByIds(Long messageId, Long userId);
	
	public int updateSMOptStatusByUserName(String userName, Boolean optStatus);
	
	public UserProfileVW getUserProfileByUserName(String userName);
}
