package gov.va.med.mhv.sm.crypto;

import gov.va.med.mhv.foundation.crypto.MHVCipher;
import gov.va.med.mhv.foundation.util.CalendarExtUtils;
import gov.va.med.mhv.foundation.util.Precondition;
import gov.va.med.mhv.sm.enumeration.UserTypeEnum;
import gov.va.med.mhv.sm.model.Administrator;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.Credentials;
import gov.va.med.mhv.sm.model.MhvIntegrationSettings;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.User;
import gov.va.med.mhv.sm.util.MhvIntegrationUtils;

import java.util.Calendar;
import java.util.Date;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class UserCredentialsBuilder {
	
	private static final Log LOG = LogFactory.getLog(UserCredentialsBuilder.
		class);
	
	public static final class Components {
		private final User user;
		private final Date expirationDate;

		public Components(User user, Date expirationDate) {
			this.user = user;
			this.expirationDate = expirationDate;
		}
		
		public User getUser() {
			return user;
		}
		public Date getExpirationDate() {
			return expirationDate;
		}
	}

	private final MhvIntegrationUtils integrationUtils;
	
	public UserCredentialsBuilder(MhvIntegrationUtils integrationUtils) {
		Precondition.assertNotNull("integrationUtils", integrationUtils);
		this.integrationUtils = integrationUtils;
	}
	
	public Credentials build(User user, Credentials credentials) {
		Precondition.assertNotNull("user", user);
		Precondition.assertNotNull("user.userType", user.getUserType());
		MHVCipher cipher = integrationUtils.createCipher();
		if (cipher == null) {
			Precondition.fail("User Credentials cannot be be encrypted. " +
				" Specify an encryption password in the sm-mhv-integration " + 
				"properties.");
		}
		MhvIntegrationSettings settings = integrationUtils.getSettings();
		Precondition.assertNotNull("mhvIntegrationUtils.settings", settings);
		Date expirationDate = createExpirationDate(settings);
		String[] keyComponents = {
			Long.toString(user.getUserType().getId()),
			Long.toString(user.getId()), 
			user.getLastName(), 
			user.getFirstName(), 
			Long.toString(expirationDate.getTime())
		};
		
		if (credentials == null) {
			credentials = new Credentials();
		}
		credentials.setUser(user);
		credentials.setKey((new KeyGenerator(cipher, settings.getSeed())).
			build(keyComponents));
		credentials.setExpirationDate(expirationDate);
		return credentials;
	}

	public Components extract(Credentials credentials) {
		MhvIntegrationSettings settings = integrationUtils.getSettings();
		Precondition.assertNotNull("mhvIntegrationUtils.settings", settings);
		String[] keyComponents = (new KeyGenerator(getCipher(), settings.
			getSeed())).extract(credentials.getKey());
		assertComponents(keyComponents);
		Components components = new Components(getUser(keyComponents), 
			getExpirationDate(keyComponents));
		
		return components;
	}
	
	private Date createExpirationDate(MhvIntegrationSettings settings) {
		return CalendarExtUtils.add(Calendar.SECOND, settings.
			getCredentialsExpirationPeriod());
	}
	
	private User getUser(String[] keyComponents) {
		assertComponents(keyComponents);
		UserTypeEnum userType = null;
		try { 
			userType = UserTypeEnum.valueOf(Long.parseLong(keyComponents[0]));
		} catch (NumberFormatException e) {
			LOG.error("Unable to parse user type id from [" + keyComponents[0] + 
				"]", e);
			throw createIllegalKeyException(keyComponents, 
				"invalid user type key component");
		}
		User user = null;
		if (UserTypeEnum.ADMINISTRATOR.equals(userType)) {
			user = new Administrator();
		} else  if (UserTypeEnum.CLINICIAN.equals(userType)) {
			user = new Clinician();
		} else if (UserTypeEnum.PATIENT.equals(userType)) {
			user = new Patient();
		}
		if (user == null) {
			throw createIllegalKeyException(keyComponents, 
				"unknown user type key component [" + userType + "]");
		}
		Long id = null;
		try {
			id = Long.parseLong(keyComponents[1]);
		} catch (NumberFormatException e) {
			LOG.error("Unable to parse user id from [" + keyComponents[1] + "]", 
				e);
			throw createIllegalKeyException(keyComponents, 
				"invalid id key component");
		}
		user.setId(id);
		user.setLastName(keyComponents[2]);
		user.setFirstName(keyComponents[3]);
		return user;
	}

	private Date getExpirationDate(String[] keyComponents) {
		assertComponents(keyComponents);
		try {
			return new Date(Long.parseLong(keyComponents[4]));
		} catch (NumberFormatException e) {
			LOG.error("Unable to parse expiration date from key", e);
			throw createIllegalKeyException(keyComponents, 
				"invalid expiration date key");
		}
	}

	private void assertComponents(String[] keyComponents) {
		if ((keyComponents == null) || (keyComponents.length != 5)) {
			throw createIllegalKeyException(keyComponents, 
				"invalid number of key components");
		}
	}
	
	private IllegalKeyException createIllegalKeyException(String[] components, 
		String reason) 
	{
		LOG.error("Invalid key [" + components + "] found, because " + reason);
		return new IllegalKeyException();
	}
	
	private MHVCipher getCipher() {
		MHVCipher cipher = integrationUtils.createCipher();
		if (cipher == null) {
			Precondition.fail("User Credentials must be encrypted. Specify " + 
				"an encryption password in the sm-mhv-integration properties.");
		}
		return cipher;
	}

}
