package gov.va.med.mhv.sm.api.common.impl;

import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.foundation.service.response.messages.Message;
import gov.va.med.mhv.foundation.service.response.messages.Severity;
import gov.va.med.mhv.sm.api.cache.CacheHandler;
import gov.va.med.mhv.sm.api.common.SessionAPIService;
import gov.va.med.mhv.sm.api.enumeration.ErrorEnum;
import gov.va.med.mhv.sm.api.transfer.ClientApplication;
import gov.va.med.mhv.sm.api.transfer.Session;
import gov.va.med.mhv.sm.api.util.ClientApplicationHelper;
import gov.va.med.mhv.sm.api.util.SMApiUtility;
import gov.va.med.mhv.sm.enumeration.UserStatusEnum;
import gov.va.med.mhv.sm.enumeration.UserTypeEnum;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.service.AuthenticationService;
import gov.va.med.mhv.sm.thread.CacheSMDomainHandler;

import java.util.Collection;

import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.UriInfo;

import org.apache.axis.utils.StringUtils;
import org.apache.cxf.jaxrs.ext.MessageContext;

public class SessionAPIServiceImpl implements SessionAPIService {

	private static final String CLINICIAN_PATH = "/clinician/";
	private static final String PATIENT_PATH = "/patient/";
	private static final String EXPIRES_HEADER_KEY = "Expires";
	private static final String TOKEN_HEADER_KEY = "Token";
	private static final String MHV_CORRELATION_ID_HEADER_KEY = "mhvCorrelationId";
	private static final String DUZ_HEADER_KEY = "duz";
	private static final String STATION_NUMBER_HEADER_KEY = "stationNumber";
	
	private AuthenticationService authenticationService;
	
	private HttpHeaders httpHeaders;
	private UriInfo uriInfo;

	/**
	 * Error codes: 102,108,109,110,111,901,129
	 */
	@Override
	public Session checkSession(MessageContext mc) {
		httpHeaders = mc.getHttpHeaders();
		uriInfo = mc.getUriInfo();
		
		Session session = checkSession();
		
		//THIS IS SO WE CAN CAPTURE THE USER INFORMATION FOR LOGGING
		CacheHandler.getInstance().setSession(session);
		
		return session;
	}
	
	private Session checkSession() {
		Session session = null;

		String token = SMApiUtility.getValueOfHeaderElement(httpHeaders, TOKEN_HEADER_KEY);
		if (null != token) {
			try {
				session = SMApiUtility.decrypt(token);
			} catch (Exception e) {
				SMApiUtility.throwException(ErrorEnum.INVALID_SESSION_TOKEN_109);
			}
			
			//Validate that the session has data before proceeding
			if( session.getExpirationSeconds() < 0 || session.getTimestamp() == null ) {
				SMApiUtility.throwException(ErrorEnum.INVALID_SESSION_TOKEN_109);
			}
			
			//Check Application Permissions First
			if( !checkClientSourcePermissions(session.getClientApplication()) ) {
				SMApiUtility.throwException(ErrorEnum.APPLICATION_AUTHORIZATION_FAILED_102);
			}

			if( !checkClientUserPermissions(session.getUserTypeEnum())){
				SMApiUtility.throwException(ErrorEnum.INVALID_USER_PERMISSIONS_111);
			}
			
			if (session.isExpired()) {
				SMApiUtility.throwException(ErrorEnum.EXPIRED_SESSION_TOKEN_110);
			}

			if( session.getUserTypeEnum()==UserTypeEnum.PATIENT ) {
				//Load User object
				ServiceResponse<Patient> response = authenticationService.fetchPatientById(session.getUserId());
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.AUTHENTICATION_SERVICE_ERROR_901, response);
				Patient p = response.getPayload();
				session.setUser(p);
			} else if( session.getUserTypeEnum()==UserTypeEnum.CLINICIAN ) {
				//Load User object
				ServiceResponse<Clinician> response = authenticationService.fetchClinicianById(session.getUserId());
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.AUTHENTICATION_SERVICE_ERROR_901, response);
				Clinician c = response.getPayload();
				session.setUser(c);
			}
			
			//Make session available for any caller on this thread
			CacheHandler.getInstance().setSession(session);
			CacheSMDomainHandler.getInstance().setApplication(CacheHandler.getInstance().getSession().getClientApplication().getId());
			
		} else {
			SMApiUtility.throwException(ErrorEnum.MISSING_SESSION_TOKEN_108);
		}

		return session;
	}

	/**
	 * Session Error Codes Patient:  101,102,103,104,105,106,107,111,132
	 */
	@Override
	public Response createSession( HttpHeaders headers, UriInfo ui) {
		this.setHeaders(headers);
		this.setUi(ui);
		
		boolean isPatient = ui.getRequestUri().getPath().contains(PATIENT_PATH);
		boolean isClinician = ui.getRequestUri().getPath().contains(CLINICIAN_PATH);
		
		if( isPatient ) {
			return checkPatient();
		} else if( isClinician ) {
			return checkClinician();
		}
		
		return Response.serverError().build();
	}

	private boolean checkClientSourcePermissions( ClientApplication client) {
		String path = uriInfo.getRequestUri().getPath();
		for( String permission: client.getPermissions()) {
			if(path.contains(permission)) {
				return true;
			}
		}
		return false;
	}
	
	private boolean checkClientUserPermissions( UserTypeEnum userTypeEnum ) {
		String path = uriInfo.getRequestUri().getPath();
		if( userTypeEnum == null ) {
			return false;
		}
		
		String permission = ( UserTypeEnum.PATIENT.equals(userTypeEnum)?PATIENT_PATH:CLINICIAN_PATH);
		if(path.contains(permission)) {
			return true;
		}
		return false;
	}
	
	private ClientApplication getclientApplicationUsingHeaderValue() {
		String appToken = SMApiUtility.getValueOfHeaderElement(httpHeaders, "appToken");
		ClientApplication source = null;
		
		if (StringUtils.isEmpty(appToken)) {
			SMApiUtility.throwException(ErrorEnum.MISSING_APPLICATION_TOKEN_132);
		} else {
			try {
				source = ClientApplicationHelper.findClientApplication(appToken);
				if(null==source) {
					SMApiUtility.throwException(ErrorEnum.APPLICATION_AUTHENTICATION_FAILED_101);
				}
			} catch (Exception e) {
				SMApiUtility.throwException(ErrorEnum.APPLICATION_AUTHENTICATION_FAILED_101);
			}
		}
		
		return source;
	}

	private Response checkClinician() {
		Session session = null;

		//Check Application Authentication First
		ClientApplication clientApplication = getclientApplicationUsingHeaderValue();
		
		//Check Application Authorization Second
		if( !checkClientSourcePermissions( clientApplication ) ) {
			SMApiUtility.throwException(ErrorEnum.APPLICATION_AUTHORIZATION_FAILED_102);
		}
		
		// Get Header Attributes and find user checking the active, opted-in, etc.
		String duz = SMApiUtility.getValueOfHeaderElement(httpHeaders, DUZ_HEADER_KEY);
		String stationNumber = SMApiUtility.getValueOfHeaderElement(httpHeaders, STATION_NUMBER_HEADER_KEY);

		if (StringUtils.isEmpty(duz) || StringUtils.isEmpty(stationNumber)) {
			SMApiUtility.throwException(ErrorEnum.MISSING_USER_CREDENTIALS_104);
		} else {
			Long duzId = null;
			try {
				duzId = Long.getLong(duz);
			} catch(NumberFormatException e) {
				SMApiUtility.throwException(ErrorEnum.INVALID_USER_CREDENTIALS_103);
			}
			
			//NOT SURE ABOUT STATION NUMBER NUMERIC VALUE CONSTRAINT
			try {
				Long.valueOf(stationNumber);
			} catch(NumberFormatException e) {
				SMApiUtility.throwException(ErrorEnum.INVALID_USER_CREDENTIALS_103);
			}
			
			ServiceResponse<Clinician> response = authenticationService.authenticateClinician(stationNumber, duz);
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.AUTHENTICATION_SERVICE_ERROR_901, response);
			Clinician c = response.getPayload();
			
			if(null == c) {
				SMApiUtility.throwException(ErrorEnum.USER_NOT_FOUND_105);
			}
			
			//CHECK THESE BUSINESS RULES FOR CLINICIANS
			if (!c.isActive() || UserStatusEnum.BLOCKED.equals(c.getStatus()) ) {
				SMApiUtility.throwException(ErrorEnum.INELIGIBLE_USER_REASON_BLOCKED_106);
			}
			if( !UserStatusEnum.OPT_IN.equals(c.getStatus()) ) {
				SMApiUtility.throwException(ErrorEnum.INELIGIBLE_USER_REASON_NOT_OPTED_IN_OR_T_C_135);
			}
			if( !checkClientUserPermissions(c.getUserType())){
				SMApiUtility.throwException(ErrorEnum.INVALID_USER_PERMISSIONS_111);
			}
			
			//USER IS GOOD TO GO!
			session = new Session(c.getId(), clientApplication, c.getUserType());
			session.setUser(c);
		}
		
		//Make session available for any caller on this thread
		CacheHandler.getInstance().setSession(session);

		String token = null;
		try {
			token = SMApiUtility.encrypt(session);
		} catch (Exception e) {
			SMApiUtility.throwException(ErrorEnum.UNABLE_TO_CREATE_TOKEN_107);
		}

		return Response.ok().header(TOKEN_HEADER_KEY, token).header(EXPIRES_HEADER_KEY,session.getFormattedTimestamp()).build();
	}

	/**
	 * Checks the application token, path authorized for application.
	 * 
	 * Makes sure user is matching requested resource. (patient)
	 * 
	 * @return
	 */
	private Response checkPatient() {
		Session session = null;

		//Check Application Authentication First
		ClientApplication clientApplication = getclientApplicationUsingHeaderValue();
		
		//Check Application Authorization Second
		if( !checkClientSourcePermissions( clientApplication ) ) {
			SMApiUtility.throwException(ErrorEnum.APPLICATION_AUTHORIZATION_FAILED_102);
		}
		
		// Get Header Attributes and find user checking the active, opted-in, etc.
		String mhvCorrelationId = SMApiUtility.getValueOfHeaderElement(httpHeaders, MHV_CORRELATION_ID_HEADER_KEY);

		if (StringUtils.isEmpty(mhvCorrelationId)) {
			SMApiUtility.throwException(ErrorEnum.MISSING_USER_CREDENTIALS_104);
		} else {
			Long cId = null;
			try {
				cId = Long.valueOf(mhvCorrelationId);
			} catch(NumberFormatException e) {
				SMApiUtility.throwException(ErrorEnum.INVALID_USER_CREDENTIALS_103);
			}
			
			ServiceResponse<Patient> response = authenticationService.authenticatePatient(cId);
			if( response.getMessages().getErrorMessageCount() > 0 ) {
				Collection<Message> msgs = (Collection<Message>)response.getMessages().getErrorMessages();
				for(Message message: msgs) {
					if( message.getSeverity().getName().equals(Severity.ERROR) &&  message.getKey().equals("authentication.failed")) {
						SMApiUtility.throwException(ErrorEnum.USER_NOT_FOUND_105);
					}
				}
			}
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.AUTHENTICATION_SERVICE_ERROR_901, response);
			Patient p = response.getPayload();
			
			if (!p.isActive() || UserStatusEnum.BLOCKED.equals(p.getStatus()) ) {
				SMApiUtility.throwException(ErrorEnum.INELIGIBLE_USER_REASON_BLOCKED_106);
			}
			if( !UserStatusEnum.OPT_IN.equals(p.getStatus()) ) {
				SMApiUtility.throwException(ErrorEnum.INELIGIBLE_USER_REASON_NOT_OPTED_IN_OR_T_C_135);
			}
			if( !checkClientUserPermissions(p.getUserType())){
				SMApiUtility.throwException(ErrorEnum.INVALID_USER_PERMISSIONS_111);
			}
			
			//USER IS GOOD TO GO!
			session = new Session(p.getId(), clientApplication, p.getUserType());
			session.setUser(p);
		}
		
		//Make session available for any caller on this thread
		CacheHandler.getInstance().setSession(session);

		String token = null;
		try {
			token = SMApiUtility.encrypt(session);
		} catch (Exception e) {
			SMApiUtility.throwException(ErrorEnum.UNABLE_TO_CREATE_TOKEN_107);
		}

		return Response.ok().header(TOKEN_HEADER_KEY, token).header(EXPIRES_HEADER_KEY,session.getFormattedTimestamp()).build();
	}

	public HttpHeaders getHeaders() {
		return httpHeaders;
	}
	public void setHeaders(HttpHeaders headers) {
		this.httpHeaders = headers;
	}
	public UriInfo getUi() {
		return uriInfo;
	}
	public void setUi(UriInfo ui) {
		this.uriInfo = ui;
	}
	
	public AuthenticationService getAuthenticationService() {
		return authenticationService;
	}
	public void setAuthenticationService(AuthenticationService authenticationService) {
		this.authenticationService = authenticationService;
	}
	
}
