package gov.va.med.mhv.sm.api.common.impl;

import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.sm.api.common.FolderAPIService;
import gov.va.med.mhv.sm.api.common.SessionAPIService;
import gov.va.med.mhv.sm.api.enumeration.ErrorEnum;
import gov.va.med.mhv.sm.api.transfer.ClinicianMessageTO;
import gov.va.med.mhv.sm.api.transfer.ClinicianUserTO;
import gov.va.med.mhv.sm.api.transfer.FolderTO;
import gov.va.med.mhv.sm.api.transfer.FoldersTO;
import gov.va.med.mhv.sm.api.transfer.MessageTO;
import gov.va.med.mhv.sm.api.transfer.MessagesTO;
import gov.va.med.mhv.sm.api.transfer.PatientMessageTO;
import gov.va.med.mhv.sm.api.transfer.MessageSearchFormTO;
import gov.va.med.mhv.sm.api.transfer.Session;
import gov.va.med.mhv.sm.api.util.PropertiesHelper;
import gov.va.med.mhv.sm.api.util.SMApiUtility;
import gov.va.med.mhv.sm.dto.SearchFormDTO;
import gov.va.med.mhv.sm.enumeration.MessagesOrderByEnum;
import gov.va.med.mhv.sm.enumeration.MessagesPageEnum;
import gov.va.med.mhv.sm.enumeration.ParticipantTypeEnum;
import gov.va.med.mhv.sm.enumeration.SortOrderEnum;
import gov.va.med.mhv.sm.enumeration.SystemFolderEnum;
import gov.va.med.mhv.sm.model.Addressee;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.Folder;
import gov.va.med.mhv.sm.model.Message;
import gov.va.med.mhv.sm.model.MessagesPage;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.SystemFolder;
import gov.va.med.mhv.sm.model.User;
import gov.va.med.mhv.sm.service.MailboxService;
import gov.va.med.mhv.sm.service.MessageService;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.mail.Message.RecipientType;
import javax.ws.rs.core.Response;

import org.apache.commons.lang.StringUtils;
import org.apache.cxf.jaxrs.ext.MessageContext;

public class FolderAPIServiceImpl implements FolderAPIService {
	private SessionAPIService sessionApiService;
	private MailboxService mailboxService;
	private MessageService messageService;

	private static int MAX_PAGE_SIZE = 250;
	
	static {
		if( PropertiesHelper.getProperties() != null && PropertiesHelper.getProperties().containsKey("sm.api.maxpagesize")) {
			String expStr = PropertiesHelper.getProperties().getProperty("sm.api.maxpagesize");
			try {
				int expInt = Integer.parseInt(expStr);
				if( expInt > 0 ) {
					MAX_PAGE_SIZE = expInt;
				}
			} catch (NumberFormatException e) {
				//do nothing
			}
		}
	}
	
	@Resource
	MessageContext mc;

	/**
	 * Session Error Codes:  108, 109, 110, 111
	 * Error Codes:  112, 115, 900
	 */
	@Override
	public FolderTO getFolder( Long folderId ) {
		Session session = sessionApiService.checkSession(mc);
		User u = session.getUser();
		
		if( session.isPatient() ) {
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, mailboxService.getMailbox((Patient)u, false));
		} else if( session.isClinician() ) {
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, mailboxService.getMailbox((Clinician)u, false));	
		}
			
		Map<Long,Folder> folders = u.getMailbox().getFolders();

		Folder f = folders.get(folderId);
		if( null == f ) {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}
		else if( !f.getOwner().getId().equals(u.getId()) ) {
			SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
		}
		
		return new FolderTO(f.getId(), f.getName(), f.getCount(), f.getUnreadCount(), (f.isSystemFolder()?SystemFolderEnum.valueOf(f.getId()):null));
	}

	/**
	 * Session Error Codes:  108, 109, 110, 111
	 * Error Codes:  900
	 * 
	 * For Patient or Clinician
	 */
	@Override
	public FoldersTO getFolders() {
		Session session = sessionApiService.checkSession(mc);
		User u = session.getUser();
		
		if( session.isPatient() ) {
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, mailboxService.getMailbox((Patient)u, false));
		} else if( session.isClinician() ) {
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, mailboxService.getMailbox((Clinician)u, false, true));	
		}
		
		List<SystemFolder> systemFolders = u.getMailbox().getSystemFolders();
		List<Folder> userFolders = u.getMailbox().getUserFolders();
		
		List<FolderTO> folderList = new ArrayList<FolderTO>();
		for (Folder f : systemFolders) {
			folderList.add(new FolderTO(f.getId(), f.getName(), f.getCount(), f
					.getUnreadCount(), SystemFolderEnum.valueOf(f.getId()) ));
		}
		for (Folder f : userFolders) {
			folderList.add(new FolderTO(f.getId(), f.getName(), f.getCount(), f.getUnreadCount()));
		}

		return new FoldersTO(folderList);
	}
	
	/**
	 * Session Error Codes:  108, 109, 110, 111, 112, 115
	 * Error Codes:  900
	 * 
	 * For either Patient or Clinician
	 * 
	 * Special Requirement is to not populate the message body nor the attachment detail information.  
	 * That is received in the "read" action.
	 * 
	 */
	@Override
	public MessagesTO getMessagesForFolder(Long folderId, Long pageNumber, Long pageSize) {
		return getMessagesForFolderInt(folderId, pageNumber, pageSize, null, null, null);
	}
	
	/**
	 * Session Error Codes:  108, 109, 110, 111, 112, 115
	 * Error Codes:  900
	 * 
	 * For either Patient or Clinician
	 * 
	 * Special Requirement is to not populate the message body nor the attachment detail information.  
	 * That is received in the "read" action.
	 * 
	 */
	@Override
	public MessagesTO getMessagesForFolder(Long folderId, Long pageNumber, Long pageSize, MessagesOrderByEnum sortField, SortOrderEnum sortOrder) {
		return getMessagesForFolderInt(folderId, pageNumber, pageSize, null, sortField, sortOrder);
	}
	
	/**
	 * Session Error Codes:  108, 109, 110, 111, 112, 115
	 * Error Codes:  900
	 * 
	 * For either Patient or Clinician
	 * 
	 * Special Requirement is to not populate the message body nor the attachment detail information.  
	 * That is received in the "read" action.
	 * 
	 */
	@Override
	public MessagesTO searchForMessagesInFolder(Long folderId, Long pageNumber, Long pageSize, MessageSearchFormTO searchForm) {
		return getMessagesForFolderInt(folderId, pageNumber, pageSize, searchForm, null, null);
	}
	
	private MessagesTO getMessagesForFolderInt(Long folderId, Long pageNumber, Long pageSize, MessageSearchFormTO searchForm, MessagesOrderByEnum sortField, SortOrderEnum sortOrder ) {
		Session session = sessionApiService.checkSession(mc);
		User u = session.getUser();
		List<MessageTO>messages = new ArrayList<MessageTO>();
		
		//Offset the Page Number to 0 base.  (So -1)
		--pageNumber;
		
		if( pageSize > MAX_PAGE_SIZE ) {
			SMApiUtility.throwException(ErrorEnum.PAGE_SIZE_EXCEEDS_LIMIT_133);
		}
		
		if( session.isPatient() ) {
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, mailboxService.getMailbox((Patient)u, false));
		} else if( session.isClinician() ) {
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, mailboxService.getMailbox((Clinician)u, false, true));	
		}
		
		if( pageNumber < 0 ) {
			SMApiUtility.throwException(ErrorEnum.PAGE_NUMBER_MUST_BE_GREATER_THAN_ZERO_120);
		}
		if( pageSize <= 0 ) {
			SMApiUtility.throwException(ErrorEnum.PAGE_SIZE_MUST_BE_GREATER_THAN_ZERO_121);
		}
		
		Map<Long,Folder> map = u.getMailbox().getFolders();
		Folder f = map.get(folderId);
		if( null == f ) {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}
		else if( !f.getOwner().getId().equals(u.getId()) ) {
			SMApiUtility.throwException(ErrorEnum.NOT_OWNER_OF_ENTITY_112);
		}

		boolean elementsOnLastPage = f.getCount()%pageSize>0;
		long lastPage = ((f.getCount()/pageSize)-(elementsOnLastPage?0:1));
		//IF YOU TRY TO ACCESS PAGES BEYOND THE NUMBER OF ELEMENTS YOU WILL RECEIVE NO MESSAGES
		//HAVE TO CONFIRM THIS BEHAVIOR
		if( pageNumber <= lastPage ) {
			MessagesPage msgs = new MessagesPage();
			msgs.setPageNumber(pageNumber.intValue());
			msgs.setPageSize(pageSize.intValue());
			f.setMessages(msgs);
			
			if( null != sortField) {
				f.setOrderBy(sortField);
			}
			if( null != sortOrder) {
				f.setSortOrder(sortOrder);
			}	

			if( null == searchForm ) {
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900,mailboxService.getMessages(f, MessagesPageEnum.CURRENT));
			} else {
				/**
				 * PREPARE THE SEARCH FORM, and use it for the filter
				 */
				SearchFormDTO sf = new SearchFormDTO();
				sf.setExactMatch(searchForm.isExactMatch());
				sf.setFrom(searchForm.getSender());
				sf.setTo(searchForm.getRecipient());
				sf.setToDate(searchForm.getToDateAsDate());
				sf.setFromDate(searchForm.getFromDateAsDate());
				sf.setSubject(searchForm.getSubject());

				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900,mailboxService.getMessages(f, MessagesPageEnum.CURRENT, sf));
			}
			
			for( Addressee a : f.getMessages().getElements()) {
				Message m2 = a.getMessage();
				ServiceResponse<Message> response = messageService.fetchMessage(m2.getId());
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MESSAGE_SERVICE_ERROR_904, response);
				Message m = response.getPayload();
				
				if(session.isPatient()) {
					messages.add(new PatientMessageTO(m.getId(), m.getThread().getMessageCategoryType(), m.getThread().getSubject(), 
						null, m.isAttachment(), null, m.getRecipientId(), m.getRecipientName(), m.getSenderId(), m.getSenderName(),
						m.getSentDate(), m.getReadReceipt()));
				} else if(session.isClinician()) {
					messages.add(new ClinicianMessageTO(m.getId(), m.getThread().getMessageCategoryType(), m.getThread().getSubject(), 
							null, m.isAttachment(), null, m.getRecipientId(), m.getRecipientName(), m.getSenderId(), m.getSenderName(),
							m.getSentDate(), m.getReadReceipt(), m.getRecipientType(), m.getCcRecipientName(), m.getCcRecipientId(), 
							(m.getCcRecipientId()!=null?ParticipantTypeEnum.CLINICIAN:null), null,
							m.getEscalatedDate(), m.getCompletedDate(), (m.getAssignedTo()!=null? new ClinicianUserTO(m.getAssignedTo()):null), m.isEscalated(), m.getStatus(), a.getReminderDate(), null) );
				}
			}
		}
		
		return new MessagesTO(messages);
	}
	/**
	 * Session Error Codes:  108, 109, 110, 111
	 * Error Codes:  115, 116, 900
	 */
	@Override
	public Response deleteFolder(Long folderId) {
		//MAKE SURE IT IS A CUSTOM USER FOLDER
		Session session = sessionApiService.checkSession(mc);
		User u = session.getUser();
		
		if( session.isPatient() ) {
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, mailboxService.getMailbox((Patient)u, false));
		} else if( session.isClinician() ) {
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, mailboxService.getMailbox((Clinician)u, false));	
		}
		
		/**
		 * FIRST FIND THE FOLDER WE WANT TO DELETE
		 */
		List<Folder> userFolders = u.getMailbox().getUserFolders();
		Folder folderToDelete = null;
		for (Folder f : userFolders) {
			if( f.getId().equals(folderId) ) {
				folderToDelete = f;
				break;
			}
		}
		/**
		 * IF THE FOLDER IS FOUND AND THERE ARE NO MESSAGES IN THE FOLDER THEN DELETE IT
		 */
		if( null != folderToDelete ) {
			if (folderToDelete.getCount() < 1){
				ServiceResponse<Boolean> response = mailboxService.deleteUserFolder(folderToDelete.getId());
				SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, response);
			} else{
				SMApiUtility.throwException(ErrorEnum.FOLDER_MUST_BE_EMPTY_BEFORE_DELETE_116);
			}
		} else {
			SMApiUtility.throwException(ErrorEnum.NOT_FOUND_115);
		}
		
		return Response.ok().build();
	}
	
	/**
	 * Session Error Codes:  108, 109, 110, 111
	 * Error Codes:  900, 117, 125, 126, 127
	 */
	@Override
	public FolderTO createFolder(FolderTO folder) {
		//MAKE SURE IT IS A CUSTOM USER FOLDER
		Session session = sessionApiService.checkSession(mc);
		User u = session.getUser();

		if( session.isPatient() ) {
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, mailboxService.getMailbox((Patient)u, false));
		} else if( session.isClinician() ) {
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, mailboxService.getMailbox((Clinician)u, false));	
		}
		
		/**
		 * VALIDATE THE "NEW" FOLDER NAME
		 */
		if(StringUtils.isBlank(folder.getName())){
			SMApiUtility.throwException(ErrorEnum.FOLDER_REQUIRES_NAME_125);
		} else if(!StringUtils.isAlphanumericSpace(folder.getName())){
			SMApiUtility.throwException(ErrorEnum.FOLDER_REQUIRES_ALPHANUMERIC_NAME_127);
		}
		
		List<Folder> userFolders = u.getMailbox().getUserFolders();
		for (Folder f : userFolders) {
			if( f.getName().equalsIgnoreCase(folder.getName()) ) {
				SMApiUtility.throwException(ErrorEnum.FOLDER_ALREADY_EXISTS_126);
			}
		}
		
		/**
		 * CREATE THE NEW FOLDER IN THE DB WITH ONLY THE NAME FIELD FROM THE API REQUEST
		 */
		Folder newFolder = new Folder();
		newFolder.setName(folder.getName());
		newFolder.setOwner(u);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, mailboxService.addUserFolder(newFolder));

		/**
		 * RELOAD THE MAILBOX TO FETCH THE NEWLY CREATED FOLDER 
		 */
		if( session.isPatient() ) {
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, mailboxService.getMailbox((Patient)u, false));
		} else if( session.isClinician() ) {
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.MAILBOX_SERVICE_ERROR_900, mailboxService.getMailbox((Clinician)u, false));	
		}
		
		/**
		 * FIND THE FOLDER WE JUST CREATED
		 */
		userFolders = u.getMailbox().getUserFolders();
		Folder nf = null;
		for (Folder f : userFolders) {
			if( f.getName().equalsIgnoreCase(folder.getName()) ) {
				nf = f;
				break;
			}
		}
		/**
		 * IF WE CAN'T FIND THE FOLDER WE JUST CREATED THEN THERE IS A PROBLEM IN THE DB OR SOMEWHERE
		 */
		if( null == nf ) {
			SMApiUtility.throwException(ErrorEnum.DATA_INTEGRITY_ERROR_117);
		}
		
		return new FolderTO(nf.getId(), nf.getName(), nf.getCount(), nf.getUnreadCount());
	}
	
	public SessionAPIService getSessionApiService() {
		return sessionApiService;
	}
	public void setSessionApiService(SessionAPIService sessionApiService) {
		this.sessionApiService = sessionApiService;
	}
	public MailboxService getMailboxService() {
		return mailboxService;
	}
	public void setMailboxService(MailboxService mailboxService) {
		this.mailboxService = mailboxService;
	}
	public MessageService getMessageService() {
		return messageService;
	}
	public void setMessageService(MessageService messageService) {
		this.messageService = messageService;
	}

}
