package gov.va.med.mhv.sm.api.clinician.impl;

import gov.va.med.mhv.foundation.service.response.CollectionServiceResponse;
import gov.va.med.mhv.foundation.service.response.ServiceResponse;
import gov.va.med.mhv.sm.api.clinician.PatientAPIService;
import gov.va.med.mhv.sm.api.common.SessionAPIService;
import gov.va.med.mhv.sm.api.enumeration.ErrorEnum;
import gov.va.med.mhv.sm.api.transfer.FacilityTO;
import gov.va.med.mhv.sm.api.transfer.PatientSearchFormTO;
import gov.va.med.mhv.sm.api.transfer.PatientUserTO;
import gov.va.med.mhv.sm.api.transfer.Session;
import gov.va.med.mhv.sm.api.transfer.UserTO;
import gov.va.med.mhv.sm.api.transfer.UsersTO;
import gov.va.med.mhv.sm.api.util.PropertiesHelper;
import gov.va.med.mhv.sm.api.util.SMApiUtility;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.Facility;
import gov.va.med.mhv.sm.model.Patient;
import gov.va.med.mhv.sm.model.PatientBlockedFacility;
import gov.va.med.mhv.sm.model.PatientFacility;
import gov.va.med.mhv.sm.service.AuthenticationService;
import gov.va.med.mhv.sm.service.FacilityService;
import gov.va.med.mhv.sm.service.PatientBlockedService;
import gov.va.med.mhv.sm.service.UserManagementService;
import gov.va.med.mhv.sm.wsclient.queriessvc.PatientDemographics;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.annotation.Resource;

import org.apache.cxf.jaxrs.ext.MessageContext;

public class PatientAPIServiceImpl implements PatientAPIService {
	private SessionAPIService sessionApiService;
	private UserManagementService userManagementService;
	private AuthenticationService authenticationService;

	private FacilityService facilityService;
	private PatientBlockedService patientBlockedService;
	
	@Resource
	MessageContext mc;

	private static int MAX_PAGE_SIZE = 250;
	
	static {
		if( PropertiesHelper.getProperties() != null && PropertiesHelper.getProperties().containsKey("sm.api.maxpagesize")) {
			String expStr = PropertiesHelper.getProperties().getProperty("sm.api.maxpagesize");
			try {
				int expInt = Integer.parseInt(expStr);
				if( expInt > 0 ) {
					MAX_PAGE_SIZE = expInt;
				}
			} catch (NumberFormatException e) {
				//do nothing
			}
		}
	}
	
	@Override
	public UsersTO searchUsers(PatientSearchFormTO form, Long pageNumber, Long pageSize) {

		Session session = sessionApiService.checkSession(mc);
		Clinician c = (Clinician) session.getClinician();

		UsersTO usersTO = new UsersTO();

		//VALIDATION/MINIMUM FIELDS
		if( form.getFirstName().length() == 0 && form.getLastName().length() == 0 && form.getNssn().length() == 0 ) {
			SMApiUtility.throwException(ErrorEnum.PARAMETERS_CANNOT_BE_EMPTY_139);
		}
		
		//Offset the Page Number to 0 base.  (So -1)
		--pageNumber;
		
		if( pageSize > MAX_PAGE_SIZE ) {
			SMApiUtility.throwException(ErrorEnum.PAGE_SIZE_EXCEEDS_LIMIT_133);
		}
		
		CollectionServiceResponse<Patient> response = userManagementService.searchForPatients(form.getFirstName(), form.getLastName(), form.getNssn(), c.getStationNo());
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.USER_MANAGEMENT_SERVICE_ERROR_906, response);
		Collection<Patient> list = response.getCollection();

		//Page 1 with size 25 will result: start = 0; end = 25; or (0-24)
		long startElement = pageNumber.longValue()*pageSize.longValue();
		long endElement = (pageNumber.longValue()+1)*pageSize.longValue();
		long currElement = 0;
		
		List<UserTO> userArray = new ArrayList<UserTO>();
		for (Patient p : list) {
			if( currElement >= startElement && currElement < endElement) {
				userArray.add(new PatientUserTO(p.getId(),p.getFirstName(),p.getLastName(),p.getNssn(),p.getDob()));
			}
			currElement++;
		}
		usersTO.setUser(userArray);
		usersTO.setAvailable(new Long(list.size()));

		return usersTO;
	}

	@Override
	public PatientUserTO getPatient(Long userId) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = (Clinician) session.getClinician();

		ServiceResponse<Patient> response = authenticationService.fetchPatientById(userId);
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.AUTHENTICATION_SERVICE_ERROR_901, response);
		Patient p = response.getPayload();

		if(null==p) {
			SMApiUtility.throwException(ErrorEnum.USER_NOT_FOUND_105);
		}
		
		ServiceResponse<PatientDemographics> response2 = userManagementService.getPatientDemographics(p.getSsn(), c.getStationNo());
		SMApiUtility.throwExceptionOnErrors(ErrorEnum.USER_MANAGEMENT_SERVICE_ERROR_906, response2);
		PatientDemographics pd = response2.getPayload();
		//	MHVPatient mhvPatient = userManagementService.findMHVPatientByUserName(p.getUsername());
		return new PatientUserTO(p.getId(), p.getFirstName(), 
				p.getLastName(),  p.getNssn(), p.getDob(), 
				pd.getHomePhone(), pd.getHomeAddress().getAddress().getAddressLine1(),
				pd.getHomeAddress().getAddress().getAddressLine2(), 
				pd.getHomeAddress().getAddress().getCity(), 
				pd.getHomeAddress().getAddress().getState(), 
				pd.getHomeAddress().getAddress().getZipCode(),pd.getDFN().longValue());
	}

	@Override
	public List<FacilityTO> getPatientFacilityList(Long patientId) {
		Session session = sessionApiService.checkSession(mc);
		Clinician c = session.getClinician();
		Patient patient = userManagementService.findApiPatientById(patientId);
		if (patient == null) {
			SMApiUtility.throwException(ErrorEnum.USER_NOT_FOUND_105);
		}
		
		List<FacilityTO> facTOs = new ArrayList<FacilityTO>();
		
		for(PatientFacility patFacList : patient.getFacilities() ) {
			Long vistDiv = Long.valueOf(patFacList.getStationNo()).longValue();
			CollectionServiceResponse<PatientBlockedFacility> collectionResponse = patientBlockedService.getBlockedFacilityPatientsByPatientIdAndStation(patient.getId(), vistDiv);
			SMApiUtility.throwExceptionOnErrors(ErrorEnum.UNKNOWN_99, collectionResponse);
			Collection<PatientBlockedFacility> patBlockedFacilities = collectionResponse.getCollection();
			if(patBlockedFacilities == null || patBlockedFacilities.size() == 0){
				FacilityTO facTO = new FacilityTO();
				ServiceResponse<Facility> facResponse = facilityService.getFacilityByStationNumber(patFacList.getStationNo());
				Facility fac = facResponse.getPayload();
				facTO.setFacilityId(fac.getId());
				facTO.setFacilityName(fac.getName());
				facTOs.add(facTO);
			}

		}
		
		return facTOs;
	}
	
	public SessionAPIService getSessionApiService() {
		return sessionApiService;
	}
	
	public void setSessionApiService(SessionAPIService sessionService) {
		this.sessionApiService = sessionService;
	}

	public UserManagementService getUserManagementService() {
		return userManagementService;
	}

	public void setUserManagementService(UserManagementService userManagementService) {
		this.userManagementService = userManagementService;
	}

	public AuthenticationService getAuthenticationService() {
		return authenticationService;
	}

	public void setAuthenticationService(AuthenticationService authenticationService) {
		this.authenticationService = authenticationService;
	}

	public FacilityService getFacilityService() {
		return facilityService;
	}

	public void setFacilityService(FacilityService facilityService) {
		this.facilityService = facilityService;
	}

	public PatientBlockedService getPatientBlockedService() {
		return patientBlockedService;
	}

	public void setPatientBlockedService(PatientBlockedService patientBlockedService) {
		this.patientBlockedService = patientBlockedService;
	}
	

}
