package gov.va.med.mhv.sm.web.session;

import gov.va.med.mhv.foundation.util.Precondition;
import gov.va.med.mhv.sm.model.Clinician;
import gov.va.med.mhv.sm.model.Facility;
import gov.va.med.mhv.sm.model.MailParticipant;
import gov.va.med.mhv.sm.model.TriageGroup;
import gov.va.med.mhv.sm.web.util.ClinicianVO;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;

import org.apache.commons.collections.CollectionUtils;

/**
 * Represents items that are stored in the session for the manage 
 * user surrogates page flow. 
 * 
 * @author Rob Proper
 *
 */
public class ManageUserSurrogatesSession implements Serializable {

	private static final long serialVersionUID = -2821402322798673166L;

	// Set the following member variable to true, to allow the application
	// to automatically select a VISN and facility, if only one is available
	private static final boolean ALLOW_AUTO_SELECTION = true;

	/**
	 * Denotes the currently selected type of surrogate.
	 */
	public static enum SurrogateType {
		Clinicians, TriageGroups
	}

	/**
	 * Decorates the name of the clinician to include the clinical user type.
	 */
	public static class ClinicianDecorator {
		
		private final Clinician clinician;
		public ClinicianDecorator(Clinician clinician) {
			Precondition.assertNotNull("clinician", clinician);
			this.clinician = clinician;
		}
		public Long getId() {
			return clinician.getId();
		}
		public String getName() {
			return clinician.getName() + " (" + 
				((clinician.getClinicalUserType() != null) 
				 ? clinician.getClinicalUserType().getName() : "-") + ")"; 
		}
	}
	
	private Collection<Facility> visns = null;
	private Long selectedVisnId = null;
	
	private Collection<Facility> facilities = null;
	private Long selectedFacilityId = null;
	
	private Collection<Clinician> clinicians = null;
	private Long selectedClinicianId = null;
	
	private SurrogateType surrogateType = SurrogateType.Clinicians;
	
	private Collection<TriageGroup> triageGroups = null;

	private Boolean surrogateEnabled = null;

	private Long surrogateId = null;
	
	private String popWin="";
	
	public Collection<Facility> getVisns() {
		return visns;
	}
	public void setVisns(Collection<Facility> visns) {
		this.visns = visns;
	}

	public Long getSelectedVisnId() {
		return selectedVisnId;
	}
	public void setSelectedVisnId(Long selectedVisn) {
		this.selectedVisnId = selectedVisn;
	}
	public boolean getCanChangeVisn() {
		return !ALLOW_AUTO_SELECTION || (CollectionUtils.size(visns) > 1);
	}
	public Facility getSelectedVisn() {
		Long visnId = getSelectedVisnId();
		if ((visnId == null) || (visnId == 0L) || (visnId == -1L)) {
			return null;
		}
		if (getVisns() == null) { 
			return null;
		}
		for (Facility visn: getVisns()){
			if (visnId.equals(visn.getId())) {
				return visn;
			}
		}
		return null;
	}
	
	public Collection<Facility> getFacilities() {
		return facilities;
	}
	public void setFacilities(Collection<Facility> facilities) {
		this.facilities = facilities;
	}

	public Long getSelectedFacilityId() {
		return selectedFacilityId;
	}
	public void setSelectedFacilityId(Long selectedFacilityId) {
		this.selectedFacilityId = selectedFacilityId;
	}
	public boolean getCanChangeFacility() {
		return !ALLOW_AUTO_SELECTION || (CollectionUtils.size(facilities) > 1);
	}
	public Facility getSelectedFacility() {
		Long facilityId = getSelectedFacilityId();
		if ((facilityId == null) || (facilityId == 0L) 
			|| (facilityId == -1L)) 
		{
			return null;
		}
		if (getFacilities() == null) { 
			return null;
		}
		for (Facility facility: getFacilities()){
			if (facilityId.equals(facility.getId())) {
				return facility;
			}
		}
		return null;
	}
	public String getSelectedStationNumber() {
		Facility facility = getSelectedFacility(); 
		return (facility != null) ? facility.getStationNumber() : null;
	}
	

	public Collection<Clinician> getClinicians() {
		return clinicians;
	}
	public void setClinicians(Collection<Clinician> clinicians) {
		this.clinicians = clinicians;
	}
	public Collection<ClinicianDecorator> getClinicianDecorators() {
		Collection<ClinicianDecorator> decorators = null;
		if (clinicians != null) {
			decorators = new ArrayList<ClinicianDecorator>();
			for (Clinician clinician: clinicians) {
				decorators.add(new ClinicianDecorator(clinician));
			}
		}
		return decorators;
	}

	public Long getSelectedClinicianId() {
		return selectedClinicianId;
	}
	public void setSelectedClinicianId(Long selectedClinicianId) {
		this.selectedClinicianId = selectedClinicianId;
	}
	public Clinician getSelectedClinician() {
		Long id = getSelectedClinicianId();
		if ((id == null) || (id <= 0L)) {
			return null;
		}
		if (getClinicians() == null) { 
			return null;
		}
		for (Clinician clinician: getClinicians()){
			if (id.equals(clinician.getId())) {
				return clinician;
			}
		}
		return null;
	}


	/**
	 * Determines whether there are any clinicians. 
	 * @return True, if there is at least 1 clinician. 
	 */
	public boolean getHasClinicians() {
		return (getClinicians() != null) && !getClinicians().isEmpty();
	}
	
	/**
	 * Yields the list of candidate provider-type surrogates for 
	 * the selected clinician.
	 * @return The list of candidate provider-type surrogates; 
	 * null when no clinician has been selected (yet).
	 */
	public Collection<Clinician> getOtherClinicians() {	
		if (getClinicians() == null) { 
			return null;
		}
		Long id = getSelectedClinicianId();
		if (id == null) {
			return null;
		}
		Collection<Clinician> clinicians = new ArrayList<Clinician>();  
		
		// The assumption is made that the clinicians in the 
		// session data are ordered by last name, first name
		//Collection<Clinician> clinicians = 	new ArrayList<Clinician>();
		for (Clinician clinician: getClinicians()) {
			
			if (id.equals(clinician.getId())) {
				continue; 
				// The currently selected clinician cannot 
				// be a surrogate for herself
			}
			clinicians.add(clinician);	
		}
		return clinicians;
	}
	

	public Collection<TriageGroup> getTriageGroups() {
		return triageGroups;
	}
	public void setTriageGroups(Collection<TriageGroup> triageGroups) {
		this.triageGroups = triageGroups;
	}

	public SurrogateType getSurrogateType() {
		return surrogateType;
	}
	public void setSurrogateType(SurrogateType surrogateType) {
		
		this.surrogateType = (surrogateType != null) ? surrogateType 
			: SurrogateType.Clinicians;
	}
	public boolean getUseClinicianSurrogates() {
		return SurrogateType.Clinicians.equals(getSurrogateType());
	}

	public Long getSurrogateId() {
		return surrogateId;
	}
	public void setSurrogateId(Long surrogateId) {
		this.surrogateId = surrogateId;
	}

	public Boolean getSurrogateEnabled() {
		return surrogateEnabled;
	}
	public void setSurrogateEnabled(Boolean enableClinicianSurrogate) {
		this.surrogateEnabled = enableClinicianSurrogate;
	}

	/**
	 * Retrieves the currently selected clinician
	 * ({@link #getSelectedClinicianId()) from the list of clinicians
	 * ({@link #getClinicians()).
	 * @return The clinician, if found; null otherwise. 
	 */
	public MailParticipant getSelectedSurrogate() {
		if (surrogateId != null) {
			if (getUseClinicianSurrogates()) {
				assert (getClinicians() != null)
					: "No clinicians found when trying to find Clinician[id:" 
					  + surrogateId + "]";
				for (Clinician clinician: getClinicians()){
					if (surrogateId.equals(clinician.getId())) {
						return clinician;
					}
				}
			} else {
				assert (getTriageGroups() != null)
				: "No triage groups found when trying to find TriageGroup[id:" 
				  + surrogateId + "]";
				for (TriageGroup triageGroup: getTriageGroups()){
					if (surrogateId.equals(triageGroup.getId())) {
						return triageGroup;
					}
				}
			}
		} 
		return null;
	}
	
	/**
	 * Clears all properties of this ManageUserSurroagteSession.  
	 */
	public void clear() {
		visns = null;
		selectedVisnId = null;
		facilities = null;
		selectedFacilityId = null;
		clinicians = null;
		selectedClinicianId = null;
		triageGroups = null;
		surrogateId = null;
	}

	public String getPopWin() {
		return popWin;
	}
	public void setPopWin(String popWin) {
		this.popWin = popWin;
	}

}
