package gov.va.med.mhv.sm.web.reports;

import gov.va.med.mhv.sm.enumeration.ReportingFilterTypeEnum;
import gov.va.med.mhv.sm.model.report.ResultKey;
import gov.va.med.mhv.sm.model.report.SelectElement;
import gov.va.med.mhv.sm.service.ReportsService;
import gov.va.med.mhv.sm.util.DateUtils;
import gov.va.med.mhv.sm.util.PeriodsHelper;
import gov.va.med.mhv.sm.web.actions.BaseSMAdminAction;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts2.ServletActionContext;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

import com.opensymphony.xwork2.Preparable;

public class BaseReports extends BaseSMAdminAction implements Preparable {
	private static final long serialVersionUID = 1L;

	// Session paramater for holding result object
	public static final String REPORT_RESULTS = "Report Results";
	public static final String REPORT_RESULTS_DATA = "Report Results Data";
	
	// Common services
	private ReportsService reportsService;

	// Struts results
	protected static final String DISPLAY_RESULTS = "success";
	protected static final String CANCEL = "cancel";
	protected static final String INVALID_PARAMETERS = "invalidParameters";
	
	// Constants used for CSV formatting
	protected static final String NL = "\r\n";
	protected static final String SEPARATOR = ",";
	protected static final String QUOTE = "\"";
	protected static final String QSQ = QUOTE + SEPARATOR + QUOTE;

	@SuppressWarnings("unused")
	private static final Log log = LogFactory.getLog(BaseReports.class);
	
	// Every report has a start and end date, so these are stored in the base class
	protected Date startDate;
	protected Date endDate;
	private boolean datesRequired = true;
	
	//Could have multiple id's
	private List<ResultKey> subFilterIds = new ArrayList<ResultKey>();
	private ResultKey mainFilterId = null;
	private ReportingFilterTypeEnum filterType = null;
	
	private String createReport = null;
	private String printerFriendly = null;
	
	@Override
	public void prepare() throws Exception {
		super.prepare();
		WebApplicationContext ctx = WebApplicationContextUtils
			.getWebApplicationContext(ServletActionContext.getServletContext());
		setReportsService((ReportsService)ctx.getBean("reportsService"));
	}

	/**
	 * Does any calculations needed to display the initial input page
	 * @return
	 */
	public String preserveParameters() {
		reconstructParameters();
		return SUCCESS;
	}

	protected Date getDefaultStartDate() {
		Calendar c = Calendar.getInstance();
		c.add(Calendar.DAY_OF_MONTH, -6);
		return c.getTime();
	}

	protected Date getDefaultEndDate() {
		return new Date();
	}

	/**
	 * Returns the results formatted as comma separated values.
	 *
	 * This should really be abstract since the subclasses know the format
	 * of the results and are best suited to format the results as CSV.
	 * However, because this class is used in struts configuration - 
	 * struts instantiates this class - it cannot be abstract.
	 *
	 * @return The results formatted as CSV 
	 */
	public String getCsv() {
		return "";
	}
	
	/**
	 * Returns the default filename for the exported csv report.
	 *
	 * There's currently no requirement on the default filename for the
	 * report, but if such requirements get added, this can be overridden
	 * at the report level.
	 *
	 * @return Default filename for csv file.
	 */
	public String getCsvFilename() {
		String date = DateUtils.formatDate(new Date(),DateUtils.ENGLISH_DATE_DASH);
		return "SM_Report_"+date+".csv";
	}

	/**
	 * Runs the report and stores the result in the session.
	 *
	 * This method is the workhorse of the reports framework, although 
	 * much of the functionality can be overridden by the subclasses.
	 * It retrieves the report parameters from the session,
	 * validates the parameters, puts the report results into the session.
	 *
	 * @return Action result based on parameters
	 */
	public String runReport() {
		getReportParameters();
		
		if (readyToRunReport()) {
			if (!reportParametersValid()) {
				return INVALID_PARAMETERS;
			}
			BaseReports results = getReportResults();
			setSessionAttribute(REPORT_RESULTS, results);
			
			persistParameters();
			return DISPLAY_RESULTS;
		} else
			return CANCEL;
	}
	
	protected void reconstructParameters() {
		//Retrieves any stored options for another run
	}
	
	protected void persistParameters(){
		//Stores any stored options for another run
	}
	
	/**
	 * Retrieves the report parameters from the session.
	 *
	 * The only parameters that are in common with all reports is the start
	 * and end dates.  Subclasses should retrieve their own parameters in
	 * addition to calling this base method.
	 */
	protected void getReportParameters() {
	}
	
	/**
	 * Validates the report parameters.
	 *
	 * Subclasses should validate their own parameters in addition to
	 * calling this base method.  If any parameters are invalid, log the
	 * error and add it as an action error.
	 */
	protected boolean reportParametersValid() {
		boolean result = true;
		if ( startDate != null && endDate != null ) {
			// We have two dates, perform other tests
			if ( startDate.after(endDate)) {
				log.error("Start Date (" + startDate + ") must be after End Date (" + endDate + ")");
				addActionError("Start Date must not be after the End Date");
				result = false;
			}
		}
		return result;
	}
	
	/**
	 * Retrieves the start date from the session.
	 *
	 * @return the start date
	 */
	protected Date getStartDate() {
		return startDate;
	}

	/**
	 * Retrieves the end date from the session.
	 *
	 * @return the end date
	 */
	protected Date getEndDate() {
		return endDate;
	}

	/**
	 * Checks to see whether the button to create the report was
	 * clicked or the cancel button was clicked.
	 *
	 * @return true to create the report, false to cancel
	 */
	protected boolean readyToRunReport() {
		return createReport!=null;
	}

	/**
	 * This method generates the report results.
	 *
	 * This should really be abstract since the subclasses know the format
	 * of the results and are best suited to format the results as CSV.
	 * However, because this class is used in struts configuration - 
	 * struts instantiates this class - it cannot be abstract.
	 */	   
	protected BaseReports getReportResults() {
		// I wanted to make this abstract, but struts needs to instantiate
		// this class to use the selectReport method.
		return null;
	}
	
	public void setEndDate(Date endDate) {
		this.endDate = endDate;
	}

	public void setStartDate(Date startDate) {
		this.startDate = startDate;
	}
	
	public List<SelectElement> getYearOptions() {
		return PeriodsHelper.getListOfFiscalYears();
	}
	
	public List<SelectElement> getQuarterOptions() {
		return PeriodsHelper.getListOfQuarters();
	}
	
	public List<SelectElement> getMonthOptions() {
		return PeriodsHelper.getListOfMonths();
	}
	
	public List<SelectElement> getWeekOptions() {
		return PeriodsHelper.getListOfWeeks();
	}

	public boolean isDatesRequired() {
		return datesRequired;
	}

	public void setDatesRequired(boolean datesRequired) {
		this.datesRequired = datesRequired;
	}

	public ReportsService getReportsService() {
		return reportsService;
	}

	public void setReportsService(ReportsService reportsService) {
		this.reportsService = reportsService;
	}

	protected List<ResultKey> getFilterIds() {
		return subFilterIds;
	}

	protected void setFilterIds(List<ResultKey> filterIds) {
		this.subFilterIds = filterIds;
	}
	
	protected ResultKey getMainFilterId() {
		return mainFilterId;
	}

	protected void setMainFilterId(ResultKey mainFilterId) {
		this.mainFilterId = mainFilterId;
	}

	public void clearFilterIds() {
		this.subFilterIds.clear();
	}

	public ReportingFilterTypeEnum getFilterType() {
		return filterType;
	}

	public void setFilterType(ReportingFilterTypeEnum filterType) {
		this.filterType = filterType;
	}

	public String getCreateReport() {
		return createReport;
	}

	public void setCreateReport(String createReport) {
		this.createReport = createReport;
	}

	public String getPrinterFriendly() {
		return printerFriendly;
	}

	public void setPrinterFriendly(String printerFriendly) {
		this.printerFriendly = printerFriendly;
	}

	public String getSessionVariableName( Class clazz, String name ) {
		return clazz.getName()+"."+name;
	}
	
	public String resetParameters() {
		return SUCCESS;
	}
	
}

