package gov.va.fnod.view;

import gov.va.cem.common.modelutils.CaseAndSourceSystemMapper;
import gov.va.cem.common.modelutils.CaseOrigin;
import gov.va.fnod.business.CaseLinkValidator;
import gov.va.fnod.model.SampleCriteria;
import gov.va.fnod.model.SampleItem;
import gov.va.fnod.model.fnoddata.ActivityTypeCode;
import gov.va.fnod.service.SampleSession;
import gov.va.fnod.util.JSFMessageUtil;

import java.io.Serializable;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.ejb.EJB;
import javax.faces.bean.ManagedBean;
import javax.faces.bean.ManagedProperty;
import javax.faces.bean.RequestScoped;
import javax.faces.context.FacesContext;

@ManagedBean(name = "sample")
@RequestScoped
public class SampleBean implements Serializable {
	
	private static final long serialVersionUID = -635434547435748699L;
	private final static Logger logr = Logger.getLogger(SampleBean.class.getName());
	
	public enum SampleType {
		AWARD_AUDIT,
		PENDING_CASE,
		COMPLETED_FNOD_RECORD		
	}
	
	private static final String SAMPLE_TYPE_SESSION_KEY = "SAMPLE_TYPE_SESSION_KEY";
	public static final String NO_AVAILABLE_CASE = "NO_AVAILABLE_CASE";
	public static final String CANCELLED_SAMPLE = "CANCELLED_SAMPLE";
	public static final String NO_SAMPLE_AVAILABLE = "NO_SAMPLE_AVAILABLE";
	public static final String NEXT_AUDIT_CASE_LOADED = "NEXT_AUDIT_CASE_LOADED";
	public static final String NEXT_PENDING_CASE_LOADED = "NEXT_PENDING_CASE_LOADED";
	public static final String STAY_HERE = "STAY_HERE";
	public static final String SAMPLE_COMPLETE = "SAMPLE_COMPLETE";
	public static final String NEXT_COMPLETED_CASE_LOADED = "NEXT_COMPLETED_CASE_LOADED";
	public static final long ONE_YEAR = 365L * 24L * 60L * 60L * 1000L;
	
	@EJB
	private SampleSession sampleSession;
	
	@ManagedProperty(value="#{user}")
    private UserBean user;
	
	@ManagedProperty(value="#{caseBean}")
    private CaseBean caseBean;
	
	@ManagedProperty(value="#{lookup}")
    private LookupBean lookupBean;
	
	@ManagedProperty(value="#{sampleCriteria}")
    private SampleCriteriaUI sampleCriteriaBean;
	

	@ManagedProperty(value="#{caseLinkValidator}")
	CaseLinkValidator caseLinkValidator;
	
	
	public SampleBean() {
		super();
	}
	
	public UserBean getUser() {
		return user;
	}

	public void setUser(UserBean user) {
		this.user = user;
	}

	public CaseBean getCaseBean() {
		return caseBean;
	}

	public void setCaseBean(CaseBean caseBean) {
		this.caseBean = caseBean;
	}

	public LookupBean getLookupBean() {
		return lookupBean;
	}

	public void setLookupBean(LookupBean lookupBean) {
		this.lookupBean = lookupBean;
	}
	
	public String generateCaseSample() {
		String result;
		
		if (sampleCriteriaBean.getFromDate().compareTo(sampleCriteriaBean.getToDate()) > 0) {
			JSFMessageUtil.addError("DATE-01", "Invalid Date Range", "The 'To Date' must be greater than or equal to the 'From Date'.");
			result = null;
		}
		else {
			sampleSession.clearSample(user.getUsername());
			CaseOrigin caseOrigin = CaseAndSourceSystemMapper.convert(sampleCriteriaBean.getSourceCaseTypeMapId());
			
			// if zero (ALL), ignore the region selected and use the region associated with the chosen regional office
			Long regionId = sampleCriteriaBean.getRegionId();		
			Long regionalOfficeId = sampleCriteriaBean.getRegionalOfficeId();
			
			if (regionId == null || regionId == 0){
				regionId = null;
			}
			
			if (regionalOfficeId == null || regionalOfficeId == 0){
				regionalOfficeId = null;
			}
			
			SampleCriteria sampleCriteria = new SampleCriteria( 
					sampleCriteriaBean.getFromDate(),
					sampleCriteriaBean.getToDate(),
					caseOrigin.getSourceSystemId(),
					caseOrigin.getCaseTypeId(),
					regionId,
					regionalOfficeId,
					sampleCriteriaBean.getMaxRows() == null ? 0 : sampleCriteriaBean.getMaxRows(),
					sampleCriteriaBean.getSamplePercent() == null ? 0.0 : sampleCriteriaBean.getSamplePercent(),
					sampleCriteriaBean.getSelectedUserName() == null ? "%": sampleCriteriaBean.getSelectedUserName() );
			
			int sampleCount = sampleSession.generateCaseSample(user.getUsername(), sampleCriteria);
			
			if (sampleCount > 0) {
				result = nextLoadedPendingCase();
				setSampleType(SampleType.PENDING_CASE);
			}
			else {
				JSFMessageUtil.addWarning("REC-01", "No Results.", "No cases match the selected criteria.");
				result = NO_SAMPLE_AVAILABLE;
			}
		}
		return result;
	}
	
	public String generateFnodSample() {
		String result;

		if (sampleCriteriaBean.getFromDate().compareTo(sampleCriteriaBean.getToDate()) > 0) {
			JSFMessageUtil.addError("DATE-01", "Invalid Date Range", "The 'To Date' must be greater than or equal to the 'From Date'.");
			result = null;
		}
		else {
			sampleSession.clearSample(user.getUsername());
			CaseOrigin caseOrigin = CaseAndSourceSystemMapper.convert(sampleCriteriaBean.getSourceCaseTypeMapId());
			
			SampleCriteria sampleCriteria = new SampleCriteria(
					sampleCriteriaBean.getFromDate(),
					sampleCriteriaBean.getToDate(),
					caseOrigin.getSourceSystemId(),
					caseOrigin.getCaseTypeId(),
					sampleCriteriaBean.getRegionId(),
					sampleCriteriaBean.getRegionalOfficeId(),
					sampleCriteriaBean.getMaxRows() == null ? 0 : sampleCriteriaBean.getMaxRows(),
					sampleCriteriaBean.getSamplePercent() == null ? 0.0 : sampleCriteriaBean.getSamplePercent(),
					sampleCriteriaBean.getSelectedUserName() == null ? "%" : sampleCriteriaBean.getSelectedUserName() );
			
			int sampleCount = sampleSession.generateFnodSample(user.getUsername(), sampleCriteria);
			
			if (sampleCount > 0) {
				result = nextLoadedCompletedCase();
				setSampleType(SampleType.COMPLETED_FNOD_RECORD);
			}
			else {
				JSFMessageUtil.addWarning("REC-01", "No Results.", "No cases match the selected criteria.");
				result = NO_SAMPLE_AVAILABLE;
			}
		}
		
		return result;
	}
	
	
	public long getNext() {
		setSampleItem(sampleSession.getNext(user.getUsername(), getSampleItem()));		 
		return getSampleItem().getCaseId();
	}
	

	public long getNextAwardCase() {
		
		// If the user has skipped cases, this will bring them back to the top of the list
		SampleItem item = sampleSession.getNextAwardCase(user.getUsername(), getSampleItem());
		if ( item.getCaseId() == 0L && getSampleItem().getSampleId() > 0 ) {
			item = sampleSession.getNextAwardCase(user.getUsername(), new SampleItem());
		}

		// Record results
		setSampleItem(item);
		
		return getSampleItem().getCaseId();
	}
	
	public String nextLoadedPendingCase() {
		Long caseId = null;
		try {
			caseId = getNext();
			if (caseId == 0) {
				JSFMessageUtil.addInfo("CAS-00", "Sample Auditing Complete.", "No remaining cases in this sample.");
				return SAMPLE_COMPLETE;
			}
			else {
				caseBean.loadCaseLink(caseId);
				return NEXT_PENDING_CASE_LOADED;
			}
		}
		catch (Exception e){
			logr.log(Level.WARNING, "Unable to load next pending case.", e);
			JSFMessageUtil.addWarning("CAS-01", "No Results", "No remaining Pending Case to sample.");
			return NO_AVAILABLE_CASE;
		}
	}
	
	public String nextLoadedCompletedCase() {
		logr.log(Level.FINE, "Hit nextLoadedCompletedCase()");
		Long caseId = null;
		try {
			caseId = getNext();
			if (caseId == 0) {
				JSFMessageUtil.addWarning("CAS-00", "Sample Auditing Complete.", "All cases in this sample have been audited.");
				return SAMPLE_COMPLETE;
			}
			else {
				caseBean.loadCaseLink(caseId);
				return NEXT_COMPLETED_CASE_LOADED;
			}
		}
		catch (Exception e){
			logr.log(Level.WARNING, "Unable to load next completed case.", e);
			JSFMessageUtil.addWarning("CAS-02", "No Results", "No remaining Completed Case to sample.");
			return NO_AVAILABLE_CASE;
		}
	}
	
	public String nextLoadedAuditCase() {
		logr.log(Level.FINE, "Hit nextLoadedAuditCase()");
		try {
			Long caseId = getNextAwardCase();			
			caseBean.loadCaseLink(caseId);
			setSampleType(SampleType.AWARD_AUDIT);
			return NEXT_AUDIT_CASE_LOADED;
		}
		catch (Exception e){
			logr.log(Level.WARNING, "Unable to load next award audit case.", e);
			JSFMessageUtil.addWarning("CAS-01", "No Results", "No remaining Audit Case to sample.");
			return NO_AVAILABLE_CASE;
		}
	}
	
	/**
	 * All three audit types call this save method.  A session variable was saved by the setSampleType()
	 * method off of which we key when determining the pre-save activities (such as Activity capturing).
	 */
	public void save() {
		logr.log(Level.FINE, "Hit save()");
		try {
			
			switch (getSampleType()) {
				case AWARD_AUDIT:
					caseBean.setCurActivityTypeCode(ActivityTypeCode.AUDITED_AWARD);
					break;
				case PENDING_CASE:					
					caseBean.setCurActivityTypeCode(ActivityTypeCode.SAMPLED_PENDING);
					break;
				case COMPLETED_FNOD_RECORD:
					caseBean.setCurActivityTypeCode(ActivityTypeCode.AUDITED_FNOD);
					break;
				default :
					JSFMessageUtil.addError("ERR-05", 
							"No active sample!  Please re-initiate an audit workflow!", 
							"No session indicator was found to specify which type of audit was being performed.  Your session has timed out or this page was invoked out of order.");
					throw new RuntimeException("No active sample!  Please re-initiate an audit workflow!");
			}
						
			caseBean.saveCaseLink();	
		}
		catch (Exception e) {
			JSFMessageUtil.addError("ERR-02", "Unable to save FNOD Record!", e.getMessage());
			throw new RuntimeException(e);
		}
	}
	
	public String saveAndNextAwardAudit() {
		logr.log(Level.FINE, "Hit saveAndNextAwardAudit()");

		caseLinkValidator.setCreateMessages(true);
		boolean validated = caseLinkValidator.validateForComplete(caseBean.getCaseLink());
		if (validated) {
			save();
			return nextLoadedAuditCase();
		}
		else {
			JSFMessageUtil.addError("SAV-01", "Fnod record NOT saved!", "Please correct the errors and try again.");
			logr.log(Level.SEVERE, "Fnod record NOT saved!", "Please correct the errors and try again.");

			logr.log(Level.SEVERE, "faild in saveAndNextAwardAudit");
			logr.log(Level.SEVERE, "Error Occurred!", "Unable to save and load next award audit.");
			return STAY_HERE;

			//throw new Exception("Validation failed during Award Audit.  FNOD not saved.");
		}
			
		
		
	}
	
	public String skipAndNextAwardAudit() {
		caseBean.cancel();
		sampleSession.cancelAwardAudit(getSampleItem());
		return nextLoadedAuditCase();
	}
	
	public String saveAndNextPendingCase() {
		logr.log(Level.FINE, "Hit saveAndNextPendingCase()");
		try {
			save();
			return nextLoadedPendingCase();
		} catch (Exception e) {
			logr.log(Level.SEVERE, "failed in saveAndNextPendingCase", e);
			JSFMessageUtil.addError("SV-02", "Error Occurred!", "Unable to save and load next pending case.");
			return STAY_HERE;
		}		
	}
		
	public String saveAndNextCompletedCase() {
		logr.log(Level.FINE, "Hit saveAndNextCompletedCase()");
		try {
			caseLinkValidator.setCreateMessages(true);
			boolean validated = caseLinkValidator.validateForComplete(caseBean.getCaseLink());
			if (validated) {
				save();
			}
			else {
				JSFMessageUtil.addError("SAV-01", "Fnod record NOT saved!", "Please correct the errors and try again.");
				throw new Exception("Validation failed during Audit Completed.  FNOD not saved.");
			}						
			return nextLoadedCompletedCase();
		} catch (Exception e) {
			logr.log(Level.WARNING, "Failed to save and get next sample item.", e);
			JSFMessageUtil.addError("SV-02", "Error Occurred!", "Unable to save and load next completed case.");
			return STAY_HERE;
		}		
	}
	
	
	/**
	 * All three types of audits call this cancel method in order to escape from the audit loop.
	 * @return A string from which the application will use to navigate (such as to the main page.)
	 */
	public String cancel() {
		logr.log(Level.FINE, "Hit cancel()");
		try {
			caseBean.cancel(); 
			sampleSession.clearSample(user.getUsername());
		}
		catch (Exception e) {
			logr.log(Level.SEVERE, "Failed to cancel sample item.", e);
			JSFMessageUtil.addError("SV-01", "Error occurred while saving!", "Your changes have not been saved.");
			return STAY_HERE;
		}
		return CANCELLED_SAMPLE;
	}
	
	public String cancelAwardAudit() {
		logr.log(Level.FINE, "Hit cancelAwardAudit()");
		try {
			caseBean.cancel();
			sampleSession.cancelAwardAudit(getSampleItem());
		} catch (Exception e) {
			logr.log(Level.WARNING, "Failed to cancel award audit record", e);
			JSFMessageUtil.addError("SV-01", "Error occurred returning to record.", "Unable to return to the original record.");
			return STAY_HERE;
		}
		return CANCELLED_SAMPLE;
	}

	public SampleCriteriaUI getSampleCriteriaBean() {
		return sampleCriteriaBean;
	}

	public void setSampleCriteriaBean(SampleCriteriaUI sampleCriteriaBean) {
		this.sampleCriteriaBean = sampleCriteriaBean;
	}
	
	public SampleType getSampleType() {
		return (SampleType) FacesContext.getCurrentInstance().getExternalContext().getSessionMap().get(SAMPLE_TYPE_SESSION_KEY);
	}
	
	public void setSampleType(SampleType sampleType) {
		FacesContext.getCurrentInstance().getExternalContext().getSessionMap().put(SAMPLE_TYPE_SESSION_KEY, sampleType);
	}

	public String cancelPendingSample() {
		caseBean.cancel();
		return "samplePendingCasesCriteria";
	}
	
	public SampleItem getSampleItem() {
		return caseBean.getSampleItem();
	}
	
	public void setSampleItem(SampleItem sampleItem) {
		caseBean.setSampleItem(sampleItem);
	}

	public SampleSession getSampleSession() {
		return sampleSession;
	}

	public void setSampleSession(SampleSession sampleSession) {
		this.sampleSession = sampleSession;
	}

	public CaseLinkValidator getCaseLinkValidator() {
		return caseLinkValidator;
	}

	public void setCaseLinkValidator(CaseLinkValidator caseLinkValidator) {
		this.caseLinkValidator = caseLinkValidator;
	}
}