package gov.va.fnod.view;

import gov.va.cem.common.dateutils.DateHelper;
import gov.va.fnod.model.UserPrivilege;
import gov.va.fnod.model.fnoddata.Report;
import gov.va.fnod.security.exception.AccountLockedException;
import gov.va.fnod.service.ReportSession;
import gov.va.fnod.service.UserAuthorizationSession;
import gov.va.fnod.util.FNODConstants;
import gov.va.fnod.util.FileCacheBean;
import gov.va.fnod.util.JSFMessageUtil;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import javax.annotation.PostConstruct;
import javax.ejb.EJB;
import javax.faces.bean.ManagedBean;
import javax.faces.bean.ManagedProperty;
import javax.faces.bean.ViewScoped;
import javax.faces.context.FacesContext;
import javax.faces.model.SelectItem;

import org.apache.log4j.Logger;

@ManagedBean(name = "report")
@ViewScoped
public class ReportBean implements Serializable {

	private static final long serialVersionUID = -855567272552287003L;
	private final static Logger log = Logger.getLogger(ReportBean.class);

	@EJB
	private ReportSession reportSession;
	
	@EJB UserAuthorizationSession userAuthorization;
	
	@ManagedProperty(value="#{filecache}")
    private FileCacheBean filecache;
	
	@ManagedProperty(value="#{user}")
    private UserBean userBean;
	
	private String pathToReports; 
    private Date currentDate = new Date();	
    private Date minDate = DateHelper.get1900();	
    private Date maxDate = DateHelper.removeTime(DateHelper.addOneDay(currentDate));
	private Date date1 = currentDate;
	private Date date2 = null;							
	private Date twelveMonthMin = new Date();
	private Date twelveMonthMax = new Date();
	private String reportFilePath;
	private String csvFilePath;
	private boolean reportDisplayed = false;
	private boolean csvButtonDisplayed = false;
	private List<SelectItem> reportSelectList;
	private List<SelectItem> userSelectList;
	private Report selectedReport;
	private String selectedUser;
	
	public ReportBean() {
		super();
	}
	
	@PostConstruct
	public void init() {
		pathToReports = FacesContext.getCurrentInstance().getExternalContext().getInitParameter("report_file_path");
		
		reportSelectList = new ArrayList<SelectItem>();
		List<Report> reportList =  reportSession.getReportSelectList();

		// This is a Supervisor Privilege
		boolean hasRunReport = answerPrivQ(UserPrivilege.RUN_REPORT);
		
		// This is the general user privilege 
		boolean hasRunUserReport = answerPrivQ(UserPrivilege.RUN_USER_REPORT);
		
		for (Report report : reportList) {
			if (hasRunReport) {
				reportSelectList.add(new SelectItem(report));
			}
			else if(hasRunUserReport && report.getReportId().equals(FNODConstants.REPORT_ID_DAILY_USER_ACTIVITY)) {
				reportSelectList.add(new SelectItem(report));
			}
		}
		
		if (reportSelectList.size() > 0) {
			SelectItem selectItem = reportSelectList.get(0);
			selectedReport = (Report)selectItem.getValue();
		}
		
		/*
		 Build list of FNOD users, these consist of any users
		 that has processed FNOD records, or who can process FNOD
		 records.
		 Note: only supervisors can request information on other users.
		*/
		List<String> reportUsers;
		if ( hasRunReport ) {
			reportUsers = reportSession.getFnodRecordUsers();
			if ( ! reportUsers.contains(userBean.getUsername()) ) {
				reportUsers.add(userBean.getUsername());
			}
			Collections.sort(reportUsers);
			// Supervisors can also run all users
			reportUsers.add(0,"All");
		} else {
			reportUsers = new ArrayList<String>();
			reportUsers.add(userBean.getUsername());
		}
		
		/**
		 * Now build the select list
		 */
		userSelectList = new ArrayList<SelectItem>();
		for( int i=0; i< reportUsers.size(); i++) {
			if ( i==0 && reportUsers.get(i).equals("All")) {
				userSelectList.add(new SelectItem(null,reportUsers.get(i)));
			} else {
				userSelectList.add(new SelectItem(reportUsers.get(i)));
			}
		}
			
		selectedUser = userBean.getUsername();
		
	}
	
	public FileCacheBean getFilecache() {
		return filecache;
	}

	public void setFilecache(FileCacheBean filecache) {
		this.filecache = filecache;
	}

	public UserBean getUserBean() {
		return userBean;
	}

	public void setUserBean(UserBean userBean) {
		this.userBean = userBean;
	}

	public Report getReport() {
		return selectedReport;
	}

	public void setReport(Report report) {
		this.selectedReport = report;
		csvButtonDisplayed = false;
	}

	public Date getCurrentDate() {
		return currentDate;
	}

	public void setCurrentDate(Date currentDate) {
		this.currentDate = currentDate;
	}
	
	public Date getMinDate() {
		return minDate;
	}

	public void setMinDate(Date minDate) {
		this.minDate = minDate;
	}

	public Date getMaxDate() {
		return maxDate;
	}

	public void setMaxDate(Date maxDate) {
		this.maxDate = maxDate;
	}

	public Date getDate1() {
		return date1;
	}
	
	public void setDate1(Date date1) {
		this.date1 = date1;
		this.twelveMonthMax = DateHelper.getTwelveMonthMax(date1);
	}
	
	public Date getDate2() {
		return date2;
	}
	
	public void setDate2(Date date2) {
		this.date2 = date2;
	}

	public Date getTwelveMonthMin() {
		if (isEndDateDisplayed()) {
			return twelveMonthMin;
		}
		else {
			return minDate;
		}
	}

	public void setTwelveMonthMin(Date twelveMonthMin) {
		this.twelveMonthMin = twelveMonthMin;
	}

	public Date getTwelveMonthMax() {
		return twelveMonthMax;
	}

	public void setTwelveMonthMax(Date twelveMonthMax) {
		this.twelveMonthMax = twelveMonthMax;
	}

	public List<SelectItem> getReportSelectList() {
		return reportSelectList;
	}

	public void setReportSelectList(List<SelectItem> reportSelectList) {
		this.reportSelectList = reportSelectList;
	}

	public List<SelectItem> getUserSelectList() {
		return userSelectList;
	}

	public void setUserSelectList(List<SelectItem> userSelectList) {
		this.userSelectList = userSelectList;
	}

	public String getSelectedUser() {
		return selectedUser;
	}

	public void setSelectedUser(String selectedUser) {
		this.selectedUser = selectedUser;
	}

	public boolean isStartDateDisplayed() {
		try {
			switch (selectedReport.getParameterCode()) {
			case 1: case 2: case 3:
				return true;
			default:
				return false;
			}
		} catch (Exception e) {
			log.error(e.getMessage(), e.fillInStackTrace());
			return false;
		}
	}
	
	public boolean isEndDateDisplayed() {
		try {	
			switch (selectedReport.getParameterCode()) {
			case 2:
				return true;
			default:
				return false;
			}
		} catch (Exception e) {
			log.error(e.getMessage(), e.fillInStackTrace());
			return false;
		}
	}
	
	public String getReportFileName() {
		if (reportFilePath == null) {
			return "";
		}
		else {
			File reportFile = new File(reportFilePath);
			String reportFileName = "/showReport/" + reportFile.getName();
			return reportFileName;
		}
	}
	
	public String showReport() {
		try {
			csvButtonDisplayed = false;
			reportDisplayed = false;
			
			validateStartDate();
		    
			HashMap<String,Object> parameterValues = new HashMap<String, Object>();
			switch (selectedReport.getParameterCode()) {
			case 1:
				parameterValues.put("start_date", date1);
				break;
			case 2:
				validateEndDate();
				parameterValues.put("start_date", date1);
				parameterValues.put("end_date", date2);
				break;
			case 3:
				parameterValues.put("start_date", date1);
				if (selectedUser == null) {
					parameterValues.put("username", "%");
				}
				else {
					parameterValues.put("username", selectedUser);
				}
				break;
			}
			
			
			FileOutputStream reportOut = null;
			FileOutputStream csvOut = null;
			try {
				reportFilePath = filecache.createCacheFilePath(selectedReport.getReportIdRoot() + ".pdf");
				reportOut = new FileOutputStream(reportFilePath);
				
				try {
					csvFilePath = filecache.createCacheFilePath(selectedReport.getReportIdRoot() + ".csv");
					csvOut = new FileOutputStream(csvFilePath);
					
					reportSession.runReport(pathToReports, selectedReport.getReportId(), parameterValues, reportOut, csvOut);
					
				} finally {
					try {
						csvOut.close();
					} catch (IOException ex) {
						ex.printStackTrace();
						log.warn("unable to close csvOut",ex); 
					}
				}
			} finally {
				try {
					reportOut.close();
				} catch ( IOException ex) {
					ex.printStackTrace();
					log.warn("unable to close reportOut",ex);
				}
			}
			
			csvButtonDisplayed = true;
			reportDisplayed = true;
		} catch (Exception e) {
			e.printStackTrace();
			log.error(e.getMessage(), e.fillInStackTrace());
		}
		return null;
	}
	
	private void validateStartDate() {
		// detect empty start dates
		if (this.date1 == null) {
			JSFMessageUtil.addError("DATE-02", "Empty Start Date Encountered!", "Please provide a valid start date.");
			throw new RuntimeException("Empty Start Date Encountered!");
		}
		// detect start dates coming after end dates
		if (DateHelper.removeTime(this.date1).compareTo(new Date()) > 0) {
			JSFMessageUtil.addError("DATE-03", "Invalid Start Date!", "Start Date may not be greater than today.");
			throw new RuntimeException("Invalid Start Date!");
		}
	}

	private void validateEndDate() {
		// detect date ranges greater than 12 months inclusive
		if (DateHelper.getMonthsDifference(date1, date2) > 12) {
			JSFMessageUtil.addError("DATE-01", "Invalid Date Range!", "Date range cannot exceed 12 calendar months.");
			throw new RuntimeException("Invalid Date Range!");
		}
		// detect start dates greater than end dates
		if (DateHelper.removeTime(this.date1).compareTo(DateHelper.removeTime(date2)) > 0) {
			JSFMessageUtil.addError("DATE-04", "Invalid End Date!", "Start Date must be before or the same as the End Date.");
			throw new RuntimeException("Invalid End Date!");
		}
		// detect future end dates
		if (DateHelper.removeTime(new Date()).compareTo(DateHelper.removeTime(date2)) < 0) {
			JSFMessageUtil.addError("DATE-05", "Invalid End Date!", "End Date must not be a future date.");
			throw new RuntimeException("Invalid End Date!");
		}
		
	}

	
	public boolean isReportDisplayed() {
		return reportDisplayed;
	}
	
	public boolean isCsvButtonDisplayed() {
		return csvButtonDisplayed && answerPrivQ(UserPrivilege.DOWNLOAD_REPORT);
	}
	
	public boolean isUserSelectDisplayed() {
		return FNODConstants.REPORT_ID_DAILY_USER_ACTIVITY.equals(selectedReport.getReportId())  && answerPrivQ(UserPrivilege.RUN_REPORT);
	}
	
	public String getCsvFileName() {
		Date now = new Date();
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");
		String csvFileName = selectedReport.getReportId() + "_" + dateFormat.format(now) + ".csv";
		return csvFileName;
	}
	
	public void downloadCsv(FacesContext context, OutputStream out) throws IOException {
			BufferedReader reader = null;
			PrintWriter writer = null;
		try {
			reader = new BufferedReader(new FileReader(csvFilePath));
			writer = new PrintWriter(out);
			String line = reader.readLine();
			
			for (int i = 0; line != null; i++) {
				if (i == 0) {
					writer.print(line);
				}
				else {
					writer.print("\n" + line);
				}
				line = reader.readLine();
			}
		} catch (Exception e) {
			log.error(e.getMessage(), e.fillInStackTrace());
		} finally {
			try {
				reader.close();
				writer.close();
			} catch (Exception e) {
				log.error(e.getMessage(), e.fillInStackTrace());
			}
		}
    }
	
	private boolean answerPrivQ(UserPrivilege ... privileges) {
		try {
			return userAuthorization.isUserAuthorized(this.userBean
					.getUserContext(), userAuthorization
					.createSystemContext("ReportBeanPriv", privileges));
		} catch (AccountLockedException ex) {
			return false;
		}
	}
}