package gov.va.fnod.view;

/**
 * Backing Bean for handling Process Case
 */
import gov.va.fnod.model.FNODModelConstants;
import gov.va.fnod.model.SearchCriteria;
import gov.va.fnod.model.SourceCaseType;
import gov.va.fnod.model.fnoddata.Region;
import gov.va.fnod.model.fnoddata.RegionalOffice;
import gov.va.fnod.model.fnoddata.SourceCaseTypeMap;
import gov.va.fnod.util.FNODConstants;
import gov.va.fnod.util.JSFMessageUtil;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import javax.faces.bean.ManagedBean;
import javax.faces.bean.ManagedProperty;
import javax.faces.bean.ViewScoped;
import javax.faces.context.FacesContext;
import javax.faces.event.ValueChangeEvent;
import javax.faces.model.SelectItem;

import org.apache.log4j.Logger;

@ManagedBean(name = "processCase")
@ViewScoped
public class ProcessCaseBean implements Serializable {

	private static final long serialVersionUID = 2929135551872992187L;
	private final static Logger log = Logger.getLogger(ProcessCaseBean.class);

	public static final String RESULT_PROCESS_CASE_NO_RESULTS = "no_results";
	public static final String RESULT_PROCESS_CASE_RECORD_FOUND = "record_found";

	@ManagedProperty(value = "#{lookup}")
	private LookupBean lookupBean;

	@ManagedProperty(name = "userBean", value = "#{user}")
	private UserBean userBean;

	@ManagedProperty(name = "caseBean", value = "#{caseBean}")
	private CaseBean caseBean;

	private List<SelectItem> sourceCaseTypeMap;// list of Source Case Type

	private List<SelectItem> regions;// list of Regions
	private List<SelectItem> regionalOffices;// list of Regional Offices
	private SearchCriteria searchCriteria;
	private HashMap<String,String> regionalOfficeMap = null;
	
	private WizardSteps currentStep = WizardSteps.SelectCaseType;
	
	private enum WizardSteps {
		SelectCaseType,
		SelectDatesRegionalInfo
	}
	
	public ProcessCaseBean() {
		super();
		
	}
	
	private void populateRegionalOfficeMap(List<RegionalOffice> regionalOfficeList) { 
		
		
		if(regionalOfficeMap == null) { 
			regionalOfficeMap = new HashMap<String, String>();
			if(regionalOfficeList != null) { 
				for(RegionalOffice r : regionalOfficeList) { 
					regionalOfficeMap.put(r.getRegionalOfficeId() + "", r.getRegionalOfficeName());
				}
			}
			
		}
	}
	
	public String getSelectedCaseTypeDescription() {
		String result = null;
		if (searchCriteria != null && searchCriteria.getSourceCaseTypeMapId() != null && sourceCaseTypeMap != null) {
			for (SelectItem item : sourceCaseTypeMap) {
				if (searchCriteria.getSourceCaseTypeMapId().equals(item.getValue())) {
					result = item.getLabel();
					break;
				}
			}
		}
		return result;
	}
	
	/**
	 * @param lookupBean
	 *            the lookupBean to set
	 */
	public void setLookupBean(LookupBean lookupBean) {
		this.lookupBean = lookupBean;
	}

	public SearchCriteria getSearchCriteria() {
		if (searchCriteria == null) {
			searchCriteria = new SearchCriteria();
		}
		return searchCriteria;
	}

	public void setSearchCriteria(SearchCriteria searchCriteria) {
		this.searchCriteria = searchCriteria;
	}

	/**
	 * get the list of Source Case Type
	 * 
	 * @return the sourceCaseTypeMap
	 */
	public List<SelectItem> getSourceCaseTypeMap() {
		if (null == sourceCaseTypeMap) {
			List<SourceCaseTypeMap> sourceCaseTypes = lookupBean.getSourceCaseTypeMap();
			
			if (sourceCaseTypes != null) {
				sourceCaseTypeMap = new ArrayList<SelectItem>();
				for (SourceCaseTypeMap sourceCaseType : sourceCaseTypes) {
					sourceCaseTypeMap.add(new SelectItem(sourceCaseType.getSourceCaseTypeMapId(), sourceCaseType.getDescription()));								
				}
				// sets the 1st item as selected in the UI
				if (this.getSearchCriteria().getSourceCaseTypeMapId() == null) {
					this.getSearchCriteria().setSourceCaseTypeMapId(sourceCaseTypes.get(0).getSourceCaseTypeMapId());
				}
			}
		}
		return sourceCaseTypeMap;
	}

	/**
	 * get the list of Regions
	 * 
	 * @return the regions
	 */
	public List<SelectItem> getRegions() {
		if (null == regions) {
			List<Region> regionList = lookupBean.getRegions();
			if (regionList != null) {
				regions = new ArrayList<SelectItem>();
				regions.add(new SelectItem(0, FNODModelConstants.SELECT_ALL));
				for (Region reg : regionList) {
					regions.add(new SelectItem(reg.getRegionId(), reg
							.getDescription()));
				}
			}
		}
		return regions;
	}

	/**
	 * get the list of Regional Offices
	 * 
	 * @return the regionalOffices
	 */
	public List<SelectItem> getRegionalOffices() {
		if (null == regionalOffices) {
			List<RegionalOffice> regionalOfficeList = lookupBean.getRegionalOffices(0);
			if (regionalOfficeList != null) {
				populateRegionalOfficeMap(regionalOfficeList);
				regionalOffices = new ArrayList<SelectItem>();
				regionalOffices.add(new SelectItem(0,
						FNODModelConstants.SELECT_ALL));
				for (RegionalOffice regOffice : regionalOfficeList) {
					regionalOffices.add(new SelectItem(regOffice
							.getRegionalOfficeId(), regOffice
							.getRegionalOfficeName()));
				}
			}
		}
		return regionalOffices;
	}

	/**
	 * when 'Next' is clicked on the Source Case type selection screen
	 */
	public void next() {
		this.currentStep = WizardSteps.SelectDatesRegionalInfo;
	}

	/**
	 * valueChangeListener for Regions
	 * 
	 * @param e
	 */
	public void updateRegionalOffice(ValueChangeEvent e) {
		Long regionId = (Long) e.getNewValue();
		getSearchCriteria().setRegionId(regionId);
		regionalOffices = null;
		List<RegionalOffice> regionalOfficeList = lookupBean
				.getRegionalOffices(regionId);
		if (regionalOfficeList != null) {
			regionalOffices = new ArrayList<SelectItem>();
			regionalOffices
					.add(new SelectItem(0, FNODModelConstants.SELECT_ALL));
			for (RegionalOffice regOffice : regionalOfficeList) {
				regionalOffices.add(new SelectItem(regOffice
						.getRegionalOfficeId(), regOffice
						.getRegionalOfficeName()));
			}
		}
		FacesContext.getCurrentInstance().renderResponse();// repaint the screen
	}

	/**
	 * when 'Previous' is clicked on the Search criteria screen
	 */
	public void previousScreen() {
		this.currentStep = WizardSteps.SelectCaseType;
	}

	private String getROsSearched( Long id ) { 
		StringBuffer sb = new StringBuffer();
		sb.append( FNODConstants.WARN_NO_CASES_FOUND_4_RO );
		String office = regionalOfficeMap.get(id + "");
		if( office == null ) { office = "No Regional Office Selected";} 
		sb.append( office );
		return sb.toString();
	}
	
	/**
	 * When 'Search' is clicked on the Search criteria screen
	 */
	public String searchCase() {
		String result = null;

		try {
			// get the next caseLink
			// This creates a transaction and should return only the caseId that was assigned to the user.
			caseBean.searchNextCase(searchCriteria);
			
			if (caseBean.getCaseLink() == null) {
				result = "";//RESULT_PROCESS_CASE_NO_RESULTS;
				String msg = getROsSearched(searchCriteria.getRegionalOfficeId());
				JSFMessageUtil
						.addWarning("SRCH-01", "No Records Found!",
								"No search results were found for the given search criteria!\n" + 
								 msg );
			} else {			
				// New that we have a new case, open it for uses by the UI
				caseBean.setSearchCriteria(searchCriteria);
				this.searchCriteria = null;
				result = RESULT_PROCESS_CASE_RECORD_FOUND;
			}
		} catch (Exception e) {
			e.printStackTrace();
			log.error("Exception while trying to search in ProcessCase", e);
			JSFMessageUtil.addWarning("SRCH-02", "No Records Found!",
					"Problems may have been encountered while searching.");
		}

		return result;
	}

	/**
	 * Reset Search criteria
	 */
	/*
	 * private void resetSearchCriteria() { SearchCriteria sc =
	 * this.getSearchCriteria(); sc.setSourceCaseTypeMapId(null);
	 * sc.setDateRangeFrom(null); sc.setDateRangeTo(null); sc.setRegionId(null);
	 * sc.setRegionalOfficeId(null); regionalOffices = null; }
	 */

	/**
	 * @return the displaySearchCriteria
	 */
	public boolean isDisplaySearchCriteria() {
		return this.currentStep == WizardSteps.SelectDatesRegionalInfo;
	}

	/**
	 * @return the displayFlagAppSearchCriteria
	 */
	public boolean isDisplayFlagAppSearchCriteria() {
		return isDisplaySearchCriteria()
				&& getSearchCriteria().getSourceCaseTypeMapId() == SourceCaseType.FLAG_APPLICATION
						.getSourceCaseTypeId();
	}

	/**
	 * @param regionalOffices
	 *            the regionalOffices to set
	 */
	public void setRegionalOffices(List<SelectItem> regionalOffices) {
		this.regionalOffices = regionalOffices;
	}

	public UserBean getUserBean() {
		return userBean;
	}

	public void setUserBean(UserBean userBean) {
		this.userBean = userBean;
	}

	public LookupBean getLookupBean() {
		return lookupBean;
	}

	public CaseBean getCaseBean() {
		return caseBean;
	}

	public void setCaseBean(CaseBean caseBean) {
		this.caseBean = caseBean;
	}

	public HashMap<String, String> getRegionalOfficeMap() {
		return regionalOfficeMap;
	}

	public void setRegionalOfficeMap(HashMap<String, String> regionalOfficeMap) {
		this.regionalOfficeMap = regionalOfficeMap;
	}

}
