package gov.va.fnod.view;
import gov.va.fnod.security.exception.AuthenticationException;
import gov.va.fnod.security.exception.AuthorizationException;
import gov.va.fnod.security.exception.PasswordExpiredException;
import gov.va.fnod.security.exception.PswdValidationException;
import gov.va.fnod.service.UserAuthenticationSession;
import gov.va.fnod.util.JSFMessageUtil;
import gov.va.fnod.util.JsfMsgPageFlowHelper;

import java.text.SimpleDateFormat;
import java.util.Date;

import javax.ejb.EJB;
import javax.faces.bean.ManagedBean;
import javax.faces.bean.ManagedProperty;
import javax.faces.bean.RequestScoped;
import javax.faces.context.FacesContext;

import org.apache.log4j.Logger;

@ManagedBean(name="login")
@RequestScoped
public class LoginBean {

	private static final String LOGIN_SUCCESSFUL = "LOGIN_SUCCESSFUL";
	
	private final static Logger log = Logger.getLogger(LoginBean.class);
	
	private String username;
	private String password;
	private String newPassword;
	private String confirmNewPassword;

	@ManagedProperty(value = "#{user}")
	UserBean userBean;
	
	@ManagedProperty(value="#{jsfMsgPageFlowHelper}")
	private JsfMsgPageFlowHelper jsfMsgPageFlowHelper;
	
	@EJB
	private UserAuthenticationSession userAuthSession;
	
	public LoginBean() {
		super();
	}
	
	/**
	 * when the user clicks the 'Login' button on the login screen
	 * @return
	 */
	public String login() {
		
		String result = null;
		 
		userBean.setUserContext(null);
		
		try {
			if (username == null) {
				JSFMessageUtil.addError("USER-01", "Username may not be empty.", "Please enter a valid Username.");
			} 
			else if (password == null) {
				JSFMessageUtil.addError("USER-01", "Password may not be empty.", "Please enter a valid Password.");
			}
			else if (this.newPassword == null && this.confirmNewPassword == null) {
				userBean.setUserContext(userAuthSession.login(username, password));
				Date passwordExpiration = userAuthSession.getWarningDays(userBean.getUserContext());
				if ( passwordExpiration != null ) {
					String message = "Your password will expire on " + new SimpleDateFormat("E MMM d").format(passwordExpiration) + "."; 
					jsfMsgPageFlowHelper.addWarningMsg("EXP-01", "", message);
				}
				result = LOGIN_SUCCESSFUL;
        setLocale();
			}
			else if ( loginChangePassword() ) {
				result = LOGIN_SUCCESSFUL;					
			} 
		} catch (PasswordExpiredException e) {
			JSFMessageUtil.addError("LOGIN-01", "Login Failed!", "Your password has expired, please change your password now.");
			log.warn(e);
		}catch (AuthenticationException e) {
			JSFMessageUtil.addError("LOGIN-01", "Login Failed!", "Please check your credentials and try again.");
			log.warn(e);
		} catch (AuthorizationException e) {
			JSFMessageUtil.addError("LOGIN-02", "Login Failed!", e.getMessage());
			log.warn(e);
		} catch (PswdValidationException e) {
			JSFMessageUtil.addError("LOGIN-03", "Login Failed!", e.getMessage());
			log.warn(e);
		}		
		
		return result;
	}
  
  private void setLocale() {
     userBean.setUserLocale(FacesContext.getCurrentInstance().getExternalContext().getRequestLocale());
  }
	
	/**
	 * when login is clicked on the change password screen
	 * @throws PswdValidationException 
	 * @throws AuthorizationException 
	 * @throws AuthenticationException 
	 */
	public boolean loginChangePassword() throws AuthenticationException, AuthorizationException, PswdValidationException {
		
		boolean retval;
		
		if ( validatePasswords() ) { 
			userBean.setUserContext(userAuthSession.login(username, password, newPassword));		
			resetLoginVariables();
			retval = true;
		} else {
			retval = false;
		}
		
		return retval;
	}
	
	private boolean validatePasswords() {
		
		boolean result;
		
		if (this.newPassword == null || !this.newPassword.equals(confirmNewPassword)) {
			JSFMessageUtil.addError("PSWD-01", "Passwords do not match!", "Please enter the same valid password twice.");
			result = false;
		}
		else {
			result = true;
		}
		
		return result;
	}
	
	/**
	 * user clicks the 'cancel' on the Login screen
	 */
	public String cancelLogin() {
		resetLoginVariables();
		userBean.invalidateSession();		
		return "cancelLogin";
	}
	
	/**
	 * when Cancel button is clicked on the password change screen
	 */
	public void cancelPwdChange() {
		resetLoginVariables();
	}
	
	
	/**
	 * user clicks the 'Continue' link when the password change warning occurs
	 * 
	 * @return
	 */
	public String continueWithoutPwdChange() {
		return "main";
	}
	
	/**
	 * reset the username and password
	 */
	private void resetLoginVariables() {
		username = null;
		password = null;
		newPassword = null;		
		confirmNewPassword = null;
	}
	
	public UserBean getUserBean() {
		return userBean;
	}

	public void setUserBean(UserBean userBean) {
		this.userBean = userBean;
	}

	public String getUsername() {
		return username;
	}

	public void setUsername(String username) {
		this.username = username;
	}

	public String getPassword() {
		return password;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	public String getNewPassword() {
		return newPassword;
	}

	public void setNewPassword(String newPassword) {
		this.newPassword = newPassword;
	}

	public String getConfirmNewPassword() {
		return confirmNewPassword;
	}

	public void setConfirmNewPassword(String confirmNewPassword) {
		this.confirmNewPassword = confirmNewPassword;
	}

	public JsfMsgPageFlowHelper getJsfMsgPageFlowHelper() {
		return jsfMsgPageFlowHelper;
	}

	public void setJsfMsgPageFlowHelper(JsfMsgPageFlowHelper jsfPageFlowHelper) {
		this.jsfMsgPageFlowHelper = jsfPageFlowHelper;
	}

}
