package gov.va.fnod.view;

import gov.va.cem.docstorage.service.DocumentStorageSession;
import gov.va.fnod.business.CaseChangeProcessor;
import gov.va.fnod.model.SampleItem;
import gov.va.fnod.model.SearchCriteria;
import gov.va.fnod.model.fnoddata.ActivityTypeCode;
import gov.va.fnod.model.fnoddata.CaseAttachment;
import gov.va.fnod.model.fnoddata.CaseFlagApp;
import gov.va.fnod.model.fnoddata.CaseLink;
import gov.va.fnod.model.fnoddata.CaseTypeCode;
import gov.va.fnod.model.fnoddata.FnodRecord;
import gov.va.fnod.model.fnoddata.LockedStatus;
import gov.va.fnod.model.fnoddata.SourceSystemCode;
import gov.va.fnod.service.CaseSession;
import gov.va.fnod.service.FnodRecordSession;
import gov.va.fnod.service.LookupSession;
import gov.va.fnod.service.SourceCaseTypeMapSession;
import gov.va.fnod.util.FNODConstants;
import gov.va.fnod.util.FileCacheBean;
import gov.va.fnod.util.JSFMessageUtil;

import java.io.Serializable;

import javax.ejb.EJB;
import javax.faces.bean.ManagedBean;
import javax.faces.bean.ManagedProperty;
import javax.faces.bean.SessionScoped;

import org.apache.log4j.Logger;

@ManagedBean(name = "caseBean")
@SessionScoped
public class CaseBean implements Serializable, FnodReportDataSupplierInterface {

	private static final long serialVersionUID = 29832093740932L;
	private final static Logger log = Logger.getLogger(CaseBean.class);
	
	@EJB
	private CaseSession caseSession;
	
	@EJB
	private LookupSession lookupSession;
	
	@EJB
	private DocumentStorageSession documentStorageSession;
	
	@EJB
	private SourceCaseTypeMapSession sourceCaseTypeMapSession;
	
	@EJB
	private FnodRecordSession fnodRecordSession;
	
	
	
	@ManagedProperty(value="#{user}")
    private UserBean user;
	
	@ManagedProperty(value="#{filecache}")
    private FileCacheBean filecache;
	
	private LockedStatus originalLockedStatus = null;
	private FnodRecord originalFnodRecord = null;	
	private ActivityTypeCode curActivityTypeCode;

	private SearchCriteria searchCriteria;
	
	private String flagAppFileName = null;
	
	private AttachmentHelper attachmentHelper;
	
	private SampleItem sampleItem = new SampleItem();
	
	public CaseBean() {
		super();
	}

	/* (non-Javadoc)
	 * @see gov.va.fnod.view.FnodReportDataSupplierInterface#getLookupSession()
	 */
	@Override
	public LookupSession getLookupSession() {
		return lookupSession;
	}

	public void setLookupSession(LookupSession lookupSession) {
		this.lookupSession = lookupSession;
	}

	/* (non-Javadoc)
	 * @see gov.va.fnod.view.FnodReportDataSupplierInterface#getUser()
	 */
	@Override
	public UserBean getUser() {
		return user;
	}

	public void setUser(UserBean user) {
		this.user = user;
	}

	/* (non-Javadoc)
	 * @see gov.va.fnod.view.FnodReportDataSupplierInterface#getCaseId()
	 */
	@Override
	public Long getCaseId() {
		if ( getCaseLink() != null ) {
			return getCaseLink().getCaseId();	
		} else {
			return null;
		}		
	}
	
	//------------------------------------------------

	public CaseLink loadCaseLink(long caseId) {
		
		CaseLink caseLink = caseSession.openCaseLinkById(caseId);
		
		// Backup original state for auditing activities
		this.originalLockedStatus = caseLink.getLockedStatus();
		
		if ( caseLink.getFnodRecord() == null ) {
			FnodRecord fnodRecord = fnodRecordSession.getFnodRecordByCaseId(caseId);
			caseLink.setFnodRecord(fnodRecord);
			if(fnodRecord != null) { 
				this.originalFnodRecord = (FnodRecord) fnodRecord.clone();
			}
		} else {
			this.originalFnodRecord = (FnodRecord)caseLink.getFnodRecord().clone();
		}
		
		attachmentHelper = new AttachmentHelper();
		
		fetchFlagAppCaseDataDocument();
		
		return caseLink;
	}
	
	public FnodRecord getFnodRecord() {
		return getCaseLink().getFnodRecord();
	}

	//------------------------------------------------

	public void saveCaseLink() {
		CaseChangeProcessor changeProcessor = new CaseChangeProcessor(caseSession);
		
		boolean runAwardAudit = changeProcessor.processAudit(user.getUsername(),
				getCaseLink(), curActivityTypeCode, originalLockedStatus,
				originalFnodRecord, attachmentHelper);
		
		addAttachmentsToDocStorage();
		
		long caseId = getCaseId();
		caseSession.saveCaseLink(runAwardAudit, user.getUsername(), curActivityTypeCode);
		
		deleteAttachmentFromDocStorage();
		
		cleanFileCache();
		
		
		JSFMessageUtil.addInfo("SAV-01", "Success.", String.format("%s - %s Record %d Saved.", getCaseLink().getSourceSystem().getSourceSystemCd(), getCaseLink().getCaseType().getDescription(), caseId));
		// State has changed since values were committed to database
		curActivityTypeCode = null;
		originalFnodRecord = null;
		originalLockedStatus = null;
		attachmentHelper = null;
		caseSession.setCaseLinkNull();
	}

	/**
	 * Adds documents recorded in attachment helper to Document Storage.
	 */
	private void addAttachmentsToDocStorage() {

		for (AttachmentWrapper wrapper : attachmentHelper.getAdds()) {
			
			// Add the document to document storage, getting the id
			Long docStorageId = documentStorageSession.addDocument(
					FNODConstants.SYSTEM_NAME, 
							wrapper.getAttachment().getFileName(), 
							filecache.getFilePath(wrapper.getCacheId()));
			
			// Set the id on the CaseAttachment record to be saved
			wrapper.getAttachment().setDocumentStorageId(docStorageId);
		}

	}

	private void deleteAttachmentFromDocStorage() {
		
		for ( AttachmentWrapper wrapper: attachmentHelper.getDeletes() ) {
			
			// Remove document from document storage, if it was saved there previously
			Long documentId = wrapper.getAttachment().getDocumentStorageId(); 
			if (  documentId != null ) {
				documentStorageSession.removeDocument(FNODConstants.SYSTEM_NAME, documentId);
			}
					
		}		
	}
	
	private void cleanFileCache() {
	
		if ( attachmentHelper != null ) {
			for( AttachmentWrapper wrapper: attachmentHelper.getAttachments() ) {
				if ( wrapper.getCacheId() != null ) {
					filecache.removeFromCache(wrapper.getCacheId());
				}
			}
		 }
		
		// flagAppFileName holds the file cache id for a flag application.
		if ( flagAppFileName != null ) {
			filecache.removeFromCache(flagAppFileName);
			flagAppFileName = null;
		}
	}


	public void setCurActivityTypeCode(ActivityTypeCode activityTypeCode) {
		curActivityTypeCode = activityTypeCode;
	}
	
	//------------------------------------------------

	/* (non-Javadoc)
	 * @see gov.va.fnod.view.FnodReportDataSupplierInterface#isFlagApp()
	 */
	@Override
	public boolean isFlagApp() {
		return getCaseLink().getSourceSystem().getSourceSystemCd().equals(SourceSystemCode.FLAG_APPLICATION);
	}
	
	/* (non-Javadoc)
	 * @see gov.va.fnod.view.FnodReportDataSupplierInterface#isAmas()
	 */
	@Override
	public boolean isAmas() {
		return getCaseLink().getSourceSystem().getSourceSystemCd().equals(SourceSystemCode.AMAS);
	}
	
	/* (non-Javadoc)
	 * @see gov.va.fnod.view.FnodReportDataSupplierInterface#isBoss()
	 */
	@Override
	public boolean isBoss() {
		return getCaseLink().getSourceSystem().getSourceSystemCd().equals(SourceSystemCode.BOSS);
	}
	
	@Override
	public boolean isAnc() {
		return getCaseLink().getSourceSystem().getSourceSystemCd().equals(SourceSystemCode.ANC);
	}
	
	/* (non-Javadoc)
	 * @see gov.va.fnod.view.FnodReportDataSupplierInterface#isFnod()
	 */
	@Override
	public boolean isFnod() {
		return getCaseLink().getCaseType().getCaseTypeCd().equals(CaseTypeCode.FNOD);
	}
	
	/* (non-Javadoc)
	 * @see gov.va.fnod.view.FnodReportDataSupplierInterface#isFnodSpouse()
	 */
	@Override
	public boolean isFnodSpouse() {
		return getCaseLink().getCaseType().getCaseTypeCd().equals(CaseTypeCode.FNOD_SPOUSE);
	}
	
	//------------------------------------------------
	
	/* (non-Javadoc)
	 * @see gov.va.fnod.view.FnodReportDataSupplierInterface#isComplete()
	 */
	@Override
	public boolean isComplete() {
		return LockedStatus.COMPLETE.equals(getCaseLink().getLockedStatus());
	}
	
	/* (non-Javadoc)
	 * @see gov.va.fnod.view.FnodReportDataSupplierInterface#getCaseType()
	 */
	@Override
	public String getCaseType() {
		SourceSystemCode sourceSystemCode = getCaseLink().getSourceSystem().getSourceSystemCd();
		
		if (sourceSystemCode  == SourceSystemCode.FLAG_APPLICATION) {
			return "caseFlagApplication";
		}
		else if (sourceSystemCode == SourceSystemCode.INSURANCE_REPORT) {
			return "caseaInsurance";
		}
		else {
			return "caseData";
		}
	}
	
	public SearchCriteria getSearchCriteria() {
		if (searchCriteria == null) {
			searchCriteria = new SearchCriteria();
		}
		return searchCriteria;
	}

	public void setSearchCriteria(SearchCriteria searchCriteria) {
		this.searchCriteria = searchCriteria;
	}
	
	public boolean isFnodQualifiedForDeletion() {
		return this.isFlagApp() && !LockedStatus.COMPLETE.equals(getCaseLink().getLockedStatus());
	}

	
	public void saveAndReopen() {		
		Long caseId = getCaseLink().getCaseId();		
		saveCaseLink();				
		loadCaseLink(caseId);		
	}
	
	/* (non-Javadoc)
	 * @see gov.va.fnod.view.FnodReportDataSupplierInterface#getCaseLink()
	 */
	@Override
	public CaseLink getCaseLink() {
		return caseSession.getCaseLink();
	}

	public void searchNextCase(SearchCriteria searchCriteria) {
		
		originalFnodRecord = null;
		originalLockedStatus = null;
		curActivityTypeCode = null; 
		
		Long caseId = caseSession.searchNextCase(user.getUsername(), searchCriteria);
		if ( caseId != null ) {
			loadCaseLink(caseId);
		}
		else {
			JSFMessageUtil.addWarning("NOREC-01", "No records found", "No records were found for the given search criteria.");
		}
		
	}

	private void fetchFlagAppCaseDataDocument() {
		if (isFlagApp()) {
			CaseFlagApp caseFlagApp = caseSession.getCaseLink().getCaseFlagApp();
			Long docStorageId = caseFlagApp.getDocStorageId();
			if (docStorageId != null) {
				
				try {
					flagAppFileName = fetchFromDocumentStorage(docStorageId); 
				}
				catch (Exception e) {
					JSFMessageUtil.addError("FILE-01", "Could not Create File!", "Error downloading file to hold the flag app case data image from document storage.");
					log.error("Error downloading file for flag app case data image from doc storage.", e);
				}		
				
			}
		}
	}

	public void setFilecache(FileCacheBean filecache) {
		this.filecache = filecache;
	}

	/* (non-Javadoc)
	 * @see gov.va.fnod.view.FnodReportDataSupplierInterface#getFlagAppFileName()
	 */
	@Override
	public String getFlagAppFileName() {
		return flagAppFileName;
	}

	public void setFlagAppFileName(String flagAppFileName) {
		this.flagAppFileName = flagAppFileName;
	}
	
	public void cancel() {
		caseSession.cancel();
		
		// clear state
		originalLockedStatus = null;
		originalFnodRecord = null;	
		curActivityTypeCode = null;
		
		//flagAppFileName is cleared in cleanFileCache;		
		cleanFileCache();
		
		attachmentHelper = null;
	}
	
	public void addAttachment(CaseAttachment attachment, String fileCacheId) {
		getCaseLink().addCaseAttachment(attachment);
		attachmentHelper.addAttachment(attachment, fileCacheId);
	}
	
	public void deleteAttachment(CaseAttachment attachment) {
		getCaseLink().getCaseAttachments().remove(attachment);
		attachmentHelper.deleteAttachment(attachment);
	}
	
	/* (non-Javadoc)
	 * @see gov.va.fnod.view.FnodReportDataSupplierInterface#fetchAttachmentDocument(gov.va.fnod.model.fnoddata.CaseAttachment)
	 */
	@Override
	public String fetchAttachmentDocument(CaseAttachment attachment) {
		String cacheId;
		cacheId = attachmentHelper.getCacheId(attachment);
		if ( cacheId == null ) {
			cacheId = fetchFromDocumentStorage(attachment.getDocumentStorageId());
			attachmentHelper.fetchedAttachment(attachment,cacheId);
		}
		return cacheId;	 
	}
	
	/* (non-Javadoc)
	 * @see gov.va.fnod.view.FnodReportDataSupplierInterface#fetchFromDocumentStorage(java.lang.Long)
	 */
	@Override
	public String fetchFromDocumentStorage(Long docStorageId) {
		
		String cacheId = null;
		
		if (docStorageId != null) {			
			cacheId = filecache.createCacheFilename(".pdf");
			String filecacheFilePath = filecache.getFilePath(cacheId);
	
			documentStorageSession.getDocument(FNODConstants.SYSTEM_NAME, docStorageId, filecacheFilePath);
			
		}
		
		return cacheId;
	}

	public FileCacheBean getFilecache() {
		return filecache;
	}

	public AttachmentHelper getAttachmentHelper() {
		return attachmentHelper;
	}

	public SourceCaseTypeMapSession getSourceCaseTypeMapSession() {
		return sourceCaseTypeMapSession;
	}

	public SampleItem getSampleItem() {
		return sampleItem;
	}

	public void setSampleItem(SampleItem sampleItem) {
		this.sampleItem = sampleItem;
	}

	@Override
	public boolean isEligibilityVerifiedDisabled() {
		return isAmas() || isBoss() || isAnc();
	}
	
}