package gov.va.fnod.view;



import gov.va.fnod.model.fnoddata.CaseAttachment;



import java.io.Serializable;

import java.util.ArrayList;

import java.util.List;

import java.util.Iterator;



/*

 * Delete -

 * 		Delete item already saved to document storage

 * 			documentStorageId will be present, cacheId optional

 * 		Delete item just added (not saved to document storage)

 * 			documentStorageId will be null

 * 			cacheId must be present

 */

public class AttachmentHelper implements Serializable {

	

	private static final long serialVersionUID = -6554837674661989580L;

	

	private List<AttachmentWrapper> attachments;

	private List<AttachmentWrapper> adds;

	private List<AttachmentWrapper> deletes;



	public AttachmentHelper() {

		attachments = new ArrayList<AttachmentWrapper>();

		adds = new ArrayList<AttachmentWrapper>();

		deletes = new ArrayList<AttachmentWrapper>();

	}

	

	

	/**

	 * Attachments being added are new and will not have a documentStorageId; reference to 

	 * the document file will be thru the cacheId only.  An exception will be thrown if the

	 * attachment is added more than once or has a document storage id or the cacheId is null.

	 * 

	 * @param attachment

	 * @param cacheId

	 * @throws IllegalArgumentException

	 */

	public void addAttachment( CaseAttachment attachment, String cacheId ) {

		AttachmentWrapper wrapper = locateAttachment(attachment);

		

		if ( attachment == null ) {

			throw new IllegalArgumentException("Attachment cannot be null.");

		}

		

		if ( attachment.getDocumentStorageId() != null ) {

			throw new IllegalArgumentException("Document Storage Id must be null.");

		}

		

		if ( cacheId == null || cacheId.isEmpty() ) {

			throw new IllegalArgumentException("cacheId is a required value.");

		}

		

		if ( wrapper != null ) {

			throw new IllegalArgumentException("Attachment already listed");

		}

		

		// Record action

		wrapper = new AttachmentWrapper(attachment,AttachmentAction.ADD, cacheId);

		

		attachments.add(wrapper);

		adds.add(wrapper);

	}

	

	/**

	 * There are two use cases for deleting an attached document, the first is deleting a document that

	 * is already in document storage and the second is deleting a document that was just added, but not 

	 * yet persisted.

	 * 

	 * @param attachment

	 * @param cacheId

	 */

	public void deleteAttachment( CaseAttachment attachment ) {

		

		if ( attachment == null ) {

			throw new IllegalArgumentException("Attachment cannot be null.");

		}

		

		// See if attachment was recently added

		AttachmentWrapper wrapper = locateAttachment(attachment);

		

		if ( wrapper == null ) {

			wrapper = new AttachmentWrapper(attachment, AttachmentAction.DELETE);

		} else {

			// attachment was located so if it exists as an add, 

			// move it to a delete so we can remove it from cache			

			if ( AttachmentAction.ADD.equals(wrapper.getAction()) ) {

				adds.remove(wrapper);						

			}

			// We don't track FETCH as a separate action category, so if its found, 

			// it need to be updated to DELETE and moved to the delete queue.

			wrapper.setAction(AttachmentAction.DELETE);

			deletes.add(wrapper);

		}

	}

	

	public String getCacheId(CaseAttachment attachment) {

		if ( attachment == null ) {

			throw new IllegalArgumentException("attachment is required");

		}

		

		AttachmentWrapper wrapper = locateAttachment(attachment);

		if ( wrapper == null ) {

			return null;			

		} else {

			return wrapper.getCacheId();

		}		

	}

	

	public void fetchedAttachment( CaseAttachment attachment, String cacheId ) {

		

		if ( attachment == null ) {

			throw new IllegalArgumentException("attachment is required");

		}

		

		if ( cacheId == null || cacheId.isEmpty() ) {

			throw new IllegalArgumentException("cacheId is required");

		}

		

		AttachmentWrapper wrapper;

		wrapper = locateAttachment(attachment);

		if ( wrapper == null ) {

			attachments.add(new AttachmentWrapper(attachment, AttachmentAction.FETCH, cacheId));

		}

		

	}

	

	private AttachmentWrapper locateAttachment(CaseAttachment attachment) {

		

		AttachmentWrapper retval = null;

		

		if ( attachment == null ) {

			throw new IllegalArgumentException("attachment cannot be null");

		}

		

		// Provide an object for comparison.  

		// The ADD action is arbitrary and not part of the comparison.

		AttachmentWrapper searchFor = new AttachmentWrapper(attachment,AttachmentAction.ADD);

		

		// Exit after finding first instance.

		for(Iterator<AttachmentWrapper> iter = attachments.iterator(); retval==null && iter.hasNext();) {

			AttachmentWrapper cur = iter.next();

			if ( cur.equals(searchFor) ) {

				retval = cur;

			}

		}

				

		return retval;

	}

	

	public List<AttachmentWrapper> getAdds() {

		return adds;

	}

	

	public List<AttachmentWrapper> getDeletes() {

		return deletes;

	}

	

	public List<AttachmentWrapper> getAttachments() {

		return attachments;

	}

	

	public void clear() {

		attachments.clear();

		adds.clear();

		deletes.clear();

	}

}

