package gov.va.fnod.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.UUID;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.faces.bean.ManagedBean;
import javax.faces.bean.SessionScoped;
import javax.faces.context.FacesContext;

import org.apache.log4j.Logger;

/**
 * Add/remove file to the cache directory
 */
@ManagedBean(name = "filecache")
@SessionScoped
public class FileCacheBean implements Serializable {

	private static final long serialVersionUID = -14221369845712L;
    private static Logger log = Logger.getLogger(FileCacheBean.class);
	
	private final static String CACHE = "cache";
    
    // Should always be created - don't need to test for null
    private List<String> trackingList = new ArrayList<String>(); //tracking list	
    private String cachePath;

    public FileCacheBean() {
        super();
    }

    /**
     * Initialize instance by getting directory from web.xml
     */
    @PostConstruct
    public void init() {
    	File dir = new File(FacesContext.getCurrentInstance().getExternalContext().getInitParameter("application_file_path")); 
    	System.out.println("++++++++++++++++++++ CACHEDIR ==> " + dir.toString());
    	setCachePath(new File(dir, CACHE).getAbsolutePath());
    }

    /**
     * Simple method to set cachePath; this is expected to be used in unit testing only, the @PostConstruct method with
     * initialize the class during runtime.
     * @param cachePath - path to directory to store cached files.
     */
    public void setCachePath(String cachePath) {
        this.cachePath = cachePath;
    }

    /**
     * Returns the tracking list.
     * <p />
     * This routing is not expected to be called often.
     */
    public List<String> getTrackingList() {
        return trackingList;
    }

    /**
     * Creates a unique file name in the cache directory and returns the file name.
     *
     * @param fileExt
     * @return
     */
    public String createCacheFilename(String fileExt) {

        log.debug("In createCacheFilename()");

        if (fileExt == null || fileExt.isEmpty()) {
            throw new IllegalArgumentException("fileExt is required");
        }

        //if fileExt starts with '.', remove '.' from fileExt
        if (fileExt.startsWith(".")) {
            fileExt = fileExt.substring(1);
        }

        String fileName;
        fileName = UUID.randomUUID().toString() + "." + fileExt;

        // TODO Timing concern - name registered before file is copied into cache
        trackingList.add(fileName);
        log.debug("tracking fileName ["+fileName+"]");

        return fileName;
    }

    /**
     * Computes the fully qualified file path given the name of a file
     * that is in the cache.
     * @param fileName string containing file name returned from cache.
     */
    public String getFilePath(String fileName) {

        if (fileName == null || fileName.isEmpty()) {
            throw new IllegalArgumentException("fileName is required");
        }

        return new File(new File(cachePath), fileName).getAbsolutePath();
    }

    /**
     * Add file to the cache directory and tracking list
     * @param stream
     * @param fileExt
     * @return fileName (guid.fileExt)
     */
    public String addToCache(InputStream input, String fileExt) {

        log.debug("In addToCache(): ["+fileExt+"]");

        if (input == null) {
            throw new IllegalArgumentException("input is required");
        }

        String fileName = createCacheFilename(fileExt);        
        try {
        	FileOutputStream output = new FileOutputStream(getFilePath(fileName));
        	try {
                byte[] buf = new byte[4096];
                int bytesRead;
                while ((bytesRead = input.read(buf)) > 0) {
                    output.write(buf, 0, bytesRead);
                }
            } catch (IOException ex) {
                throw new RuntimeException("Failed to add file to cache.");
            } finally {
                try {
                    output.close();
                } catch (IOException ex) {
                    throw new RuntimeException("Failed to close file.");
                }
            }
        } catch (FileNotFoundException ex) {
            throw new RuntimeException("Unable to create output file.");
        }
		return fileName;
    }

    /**
     * Get a stream to read a file from cache; the stream is opened and the requester is responsible for
     * closing the stream properly.
     * @param fileName
     * @return InputStream for reading cached item.
     */
    public InputStream getInputStream(String fileName) {

        log.debug("In getInputStream()");
        InputStream input = null;
        try {
        	input = new FileInputStream(getFilePath(fileName));
        } catch ( FileNotFoundException ex ) {
            throw new RuntimeException("File not found in cache.");
        }
        return input;
    }
    
    /**
     * Get a stream to write a file to cache; the stream is opened and the requester is responsible for
     * closing the stream properly.
     * @param fileName
     * @return OutputStream for writing a cached item.
     */
    public FileOutputStream getFileOutputStream(String fileExt) {
    	FileOutputStream output = null;

        log.debug("In getOutputStream()");
        try {
        	String fileName = createCacheFilename(fileExt);
        	output = new FileOutputStream(getFilePath(fileName));
        } catch ( Exception ex ) {
            throw new RuntimeException(ex);
        }
        
        return output;
    }
    
    public String createCacheFilePath(String fileExt) {
    	String filePath = null;
        try {
        	String fileName = createCacheFilename(fileExt);
        	filePath = getFilePath(fileName);
        } catch ( Exception ex ) {
            throw new RuntimeException(ex);
        }
        
        return filePath;
    }
    
    /**
     * Remove file from the tracking list and disk
     * @param String - fileName
     */
    public void removeFromCache(String fileName) {
        log.debug("In removeFromCache()");

        // Make sure file is in the tracking list
        if (trackingList.contains(fileName)) {
            // remove file from disk
            File file = new File(getFilePath(fileName));
            if (file.exists() && file.isFile()) {
                // TODO - may want to implement retries
                if (file.delete()) {
                    // then remove file from tracking list
                    trackingList.remove(fileName);
                }
            }
        }

        log.debug("trackingList: " + trackingList);
    }

    

    /**
     * cleanup code - This code is being called while the session is being torn down.  There is no
     * guarantee that the JSF framework will still be up.
     */
    @PreDestroy
    public void flushCache() {        
        File cacheDir = new File(cachePath);

        if (!cacheDir.isDirectory()) {  
        	System.out.println("cacheDir ==> " + cacheDir.getPath());
            throw new IllegalArgumentException("cacheDir is not a directory");
        }
        
        // Attempt to delete session files from cache.
        if(trackingList != null) {
	        Iterator<String> pathIter = trackingList.iterator();
	        while( pathIter.hasNext() ) {
	            File file = new File(cacheDir,pathIter.next());
	            if ( file.isFile() ) {
	                // going to enforce the issue
	                if ( file.delete() ) {
	                    pathIter.remove();
	                }
	            }
	        }        
	        if ( ! trackingList.isEmpty() ) {
	            System.out.println("not all files removed from cache");
	        }
        }
        
        trackingList = null;        
    }    
}