package gov.va.fnod.business;

import static gov.va.fnod.business.FnodValidationMessages.BA_BEN_CAN;
import static gov.va.fnod.business.FnodValidationMessages.BA_BU;
import static gov.va.fnod.business.FnodValidationMessages.BA_DODP;
import static gov.va.fnod.business.FnodValidationMessages.BA_FNOD;
import static gov.va.fnod.business.FnodValidationMessages.BU_BEN_CAN;
import static gov.va.fnod.business.FnodValidationMessages.DODP_BC;
import static gov.va.fnod.business.FnodValidationMessages.FNOD_DODP;
import static gov.va.fnod.business.FnodValidationMessages.SPOUSE_IRIS_BC;
import static gov.va.fnod.business.FnodValidationMessages.VDOD_REQ_BA;
import static gov.va.fnod.business.FnodValidationMessages.VDOD_REQ_BC;
import static gov.va.fnod.business.FnodValidationMessages.VDOD_REQ_BU;
import static gov.va.fnod.business.FnodValidationMessages.VDOD_REQ_DOD_PRESENT;
import static gov.va.fnod.business.FnodValidationMessages.VDOD_REQ_FNOD;
import gov.va.fnod.model.fnoddata.CaseLink;
import gov.va.fnod.model.fnoddata.CaseTypeCode;
import gov.va.fnod.model.fnoddata.FnodRecord;
import gov.va.fnod.model.fnoddata.Person;
import gov.va.fnod.model.fnoddata.SourceSystemCode;
import gov.va.fnod.util.FNODConstants;
import gov.va.fnod.util.JSFMessageUtil;
import gov.va.fnod.view.UserBean;

import java.io.Serializable;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import javax.faces.bean.ManagedBean;
import javax.faces.bean.ManagedProperty;
import javax.faces.bean.SessionScoped;

@ManagedBean(name = "caseLinkValidator")
@SessionScoped
public class CaseLinkValidator implements Serializable{

	/**
	 * 
	 */
	private static final long serialVersionUID = -5167832335599082285L;

	@ManagedProperty(value="#{user}")
	private UserBean user;

	private List<String> errorsLst = new ArrayList<String>();
	boolean createMessages = true;

	public CaseLinkValidator() {  }

	/**
	 * Use this constructor if you wish to control whether messages get added to the JSF response as part of the
	 * validation life cycle.
	 *
	 * @param createMessages
	 */
	public CaseLinkValidator(boolean createMessages) {
		this.createMessages = createMessages;
	}

	public final void setUser(UserBean value) {
		this.user = value;
	}

	public boolean validateForPark(CaseLink caseLink) {
		boolean result = false;
		if (isFlagIssuedValid(caseLink)
				&& isPMCOnlyOnFlagApp(caseLink)) {
			result = true;
		}
		return result;
	}

	/**
	 * Validates a Case/FNOD record against a number of business rules before permitting it to be marked
	 * complete.
	 *
	 * @param caseLink
	 * @return
	 */
	public boolean validateForComplete(CaseLink caseLink) {
		boolean result = false;

		// NOTE: Using non-short circuit and (& verses &&) to ensure all messages are displayed
		if (isDistinguishingIDSatisfied(caseLink)
				& isNMISuppliedAndValid(caseLink)
				& isTooOldToCreateSuppliedAndValid(caseLink)
				& isNameSatisfied(caseLink)
				& isPMCOnlyOnFlagApp(caseLink)
				& isMontlyAmountCompatibleWithBenefitCancelled(caseLink)
				& isFlagIssuedValid(caseLink)
				& isVeteranDoBDoDValid(caseLink)
				& isIrisValid(caseLink)
				& isNisValid(caseLink)
				& isSensitiveValid(caseLink)
				& isFolderOnlyBenefitValid(caseLink)
				& isReqiredVeteranDODSatisfied(caseLink)
				& isBirlsAddValid(caseLink)
				& isBirlsUpdateValid(caseLink)
				& isFnodValid(caseLink)
				& isDODAlreadyPresentValid(caseLink)
				& isCaseTypeSpouse(caseLink)) {

			if (isNormalUserConstraintsMet(caseLink)) {
				if(user != null && user.isSupervisor()) {
					result = isSupervisorConstraintsMet(caseLink);
				} else {
					// SPPT_CodeCR2453 1.5 & 1.6
					if(isNmiChecked(caseLink) || (isSensitiveChecked(caseLink) && isSensitiveValid(caseLink) )) {
						JSFMessageUtil.addError("FLD-06",
								"NMI or Sensitive Checked: ",
								"User must have supervisory privileges in order to proceed through completion. The case must be Parked for later update or Supervisor completion of the FNOD record.");
						result = false;
					} else {
						result = true;
					}
				}
				if (result && caseLink.getCaseType().getCaseTypeCd().equals(CaseTypeCode.FLAG_APPLICATION)) {
					result = isUserFlagAppConstraintsMet(caseLink);
				}
			}
		}
		return result;
	}

	private boolean isSupervisorConstraintsMet(final CaseLink caseLink) {
		boolean retval = true;
		// SPPT_CodeCR2453 1.1
		FnodRecord fnod = caseLink.getFnodRecord();
		if (user.isSupervisor()) {
			retval = (isNormalUserConstraintsMet(caseLink) || charToBoolean(fnod.getSensitive()));
			if(!retval) {
				JSFMessageUtil.addError("ISD-01",
						"Insufficient Data",
						"Must have additional data as a supervisor in order to proceed.");
			}
		}
		return retval;
	}

	private boolean isNormalUserConstraintsMet(final CaseLink caseLink) {
		// SPPT_CodeCR2453 1.2 & 1.3
		FnodRecord fnod = caseLink.getFnodRecord();
		boolean retval = false;
		if(FNODConstants.Y.equalsIgnoreCase(fnod.getNmi())) { 
			retval = true;
		}else { 
			retval = (charToBoolean(fnod.getBenefitCancelled())
					| charToBoolean(fnod.getFnod())
					| charToBoolean(fnod.getBirlsAdd())
					| charToBoolean(fnod.getBirlsUpdate())
					| charToBoolean(fnod.getDodAlreadyPresent())
					| charToBoolean(fnod.getIris())
					| charToBoolean(fnod.getNis()));
			if(!retval) {
				JSFMessageUtil.addError("ISD-02",
						"Insufficient Data",
						"Must have additional data as an auditor in order to proceed.\n" + 
								"Please select one of the following: \n" + 
								" * Benefit Cancelled\n" + 
								" * FNOD\n" + 
								" * Birls Add\n" + 
								" * Birls Update\n" + 
								" * DOD Already Present\n" + 
								" * IRIS\n" +
								" * NMI\n" +
								" * Sensitive\n" +
						" * NIS\n");
			}
		}
		return retval;
	}

	private boolean isUserFlagAppConstraintsMet(final CaseLink caseLink) {
		// SPPT_CodeCR2453 1.4
		FnodRecord fnod = caseLink.getFnodRecord();
		boolean retval = (charToBoolean(fnod.getFlagIssued())
				| charToBoolean(fnod.getPmc())
				| charToBoolean(fnod.getTooOldToCreate())
				| charToBoolean(fnod.getFolderOnlyBenefit()));
		return retval;
	}

	public void setFnodFieldsForComplete(CaseLink caseLink) {
		setNMIFieldsForComplete(caseLink);
		setTooOldFieldsForComplete(caseLink);
		setIrisForComplete(caseLink);
		setNisForComplete(caseLink);
		setSensitiveForComplete(caseLink);
		setFolderOnlyBenefitForComplete(caseLink);

	}

	/**
	 * This method ensure that if both a Date of Birth and Date of Death are present, that the date of birth
	 * proceeds the date of death.
	 *
	 * @param caseLink
	 * @return
	 */
	private boolean isVeteranDoBDoDValid(CaseLink caseLink) {

		boolean isSatisfied = true;

		Person veteran = caseLink.getFnodRecord().getVeteranPerson();

		if (veteran != null && veteran.getBirthDt() != null && veteran.getDeathDt() != null) {
			if (veteran.getBirthDt().after(veteran.getDeathDt()) || veteran.getBirthDt().equals(veteran.getDeathDt()) ) {
				isSatisfied = false;

				JSFMessageUtil.addError("VDATES-01",
						"Date Range",
						"Veteran date of birth must be before Veteran date of death.");
			}
		}
		return isSatisfied;
	}

	/**
	 * Checks if Veteran DOD is required and if so, if it has been provided.
	 *
	 * @param caseLink
	 * @return
	 */
	private boolean isReqiredVeteranDODSatisfied(CaseLink caseLink) {
		boolean result = true;
		FnodRecord fnodRec = caseLink.getFnodRecord();
		if (fnodRec != null) {
			Person vet = caseLink.getFnodRecord().getVeteranPerson();
			if (isEmpty(vet) || isEmpty(vet.getDeathDt())) {

				result = isSelectValidationError(fnodRec.getBirlsAdd(), VDOD_REQ_BA)
						& isSelectValidationError(fnodRec.getBirlsUpdate(), VDOD_REQ_BU)
						& isSelectValidationError(fnodRec.getBenefitCancelled(), VDOD_REQ_BC)
						& isSelectValidationError(fnodRec.getFnod(), VDOD_REQ_FNOD)
						& isSelectValidationError(fnodRec.getDodAlreadyPresent(), VDOD_REQ_DOD_PRESENT);
			}
		}
		return result;
	}

	private boolean isSelectValidationError(String selectValue, FnodValidationMessages msg) {
		boolean result = true;
		if (FNODConstants.Y.equals(selectValue)) {
			JSFMessageUtil.addError(msg.name(), msg.getSummaryMsg(), msg.getDetailMsg());
			result = false;
		}
		return result;
	}

	private boolean isNameSatisfied(CaseLink caseLink) {
		boolean isSatisfied = false;
		Person per = caseLink.getFnodRecord().getVeteranPerson();
		if (!(isEmpty(per) || isEmpty(per.getFirstName()) || isEmpty(per.getLastName()))) {
			isSatisfied = true;
		} else {
			if (!isNmiChecked(caseLink)) {
				if (createMessages) {
					JSFMessageUtil.addError("FLD-06",
							"Name Required!",
							"The veteran's name must be supplied.  First and Last name are required.  Middle name is optional.");
				}
			} else {
				isSatisfied = true;
			}
		}
		return isSatisfied;
	}

	private boolean isNmiChecked(CaseLink caseLink) {
		boolean retval = false;
		if (caseLink.getFnodRecord() != null) {
			retval = FNODConstants.Y.equals(caseLink.getFnodRecord().getNmi());
		}
		return retval;
	}

	private boolean isTooOldToCreateChecked(CaseLink caseLink) {
		boolean retval = false;
		if (caseLink.getFnodRecord() != null) {
			retval = FNODConstants.Y.equals(caseLink.getFnodRecord().getTooOldToCreate());
		}
		return retval;
	}

	/**
	 * This should return true when one of the following id's are provided: SSN, Veteran Claim Number or
	 * Military Service Number or if NMI has been checked (Y value).
	 *
	 * @param caseLink
	 * @return
	 */
	public boolean isDistinguishingIDSatisfied(CaseLink caseLink) {
		boolean result = false;

		if (isSSNSupplied(caseLink) || isVCNSupplied(caseLink) || isMSNSupplied(caseLink) || isNmiChecked(caseLink)) {
			result = true;
		} else {
			if (createMessages) {
				JSFMessageUtil.addError("FLD-06",
						"Required: Any Distinguishing ID!",
						"A distinguishing identifier must be provided: Social Security Number, Military Service Number or Veteran Claim Number.");
			}
		}
		return result;
	}

	private boolean isBirlsAddValid(CaseLink caseLink) {
		boolean result = true;
		FnodRecord fnodRec = caseLink.getFnodRecord();

		if (FNODConstants.Y.equals(fnodRec.getBirlsAdd())) {
			if (FNODConstants.Y.equals(fnodRec.getBirlsUpdate())) {
				JSFMessageUtil.addError(BA_BU.name(), BA_BU.getSummaryMsg(), BA_BU.getDetailMsg());
				result = false;
			}
			if (FNODConstants.Y.equals(fnodRec.getBenefitCancelled())) {
				JSFMessageUtil.addError(BA_BEN_CAN.name(), BA_BEN_CAN.getSummaryMsg(), BA_BEN_CAN.getDetailMsg());
				result = false;
			}
			if (FNODConstants.Y.equals(fnodRec.getDodAlreadyPresent())) {
				JSFMessageUtil.addError(BA_DODP.name(), BA_DODP.getSummaryMsg(), BA_DODP.getDetailMsg());
				result = false;
			}
			
			if (FNODConstants.N.equals(fnodRec.getFnod())) {
				JSFMessageUtil.addError(BA_FNOD.name(), BA_FNOD.getSummaryMsg(), BA_FNOD.getDetailMsg());
				result = false;
			}
		}
		return result;
	}

	private boolean isBirlsUpdateValid(CaseLink caseLink) {
		boolean result = true;
		FnodRecord fnodRec = caseLink.getFnodRecord();

		if (FNODConstants.Y.equals(fnodRec.getBirlsUpdate())) {
			if (FNODConstants.Y.equals(fnodRec.getBenefitCancelled())) {
				JSFMessageUtil.addError(BU_BEN_CAN.name(), BU_BEN_CAN.getSummaryMsg(), BU_BEN_CAN.getDetailMsg());
				result = false;
			}
		}
		return result;
	}

	private boolean isFnodValid(CaseLink caseLink) {
		boolean result = true;
		FnodRecord fnodRec = caseLink.getFnodRecord();

		if (FNODConstants.Y.equalsIgnoreCase(fnodRec.getFnod())) {
			if (FNODConstants.Y.equalsIgnoreCase(fnodRec.getDodAlreadyPresent())) {
				JSFMessageUtil.addError(FNOD_DODP.name(), FNOD_DODP.getSummaryMsg(), FNOD_DODP.getDetailMsg());
				result = false;
			}
		}
		return result;
	}

	private boolean isDODAlreadyPresentValid(CaseLink caseLink) {
		boolean result = true;
		FnodRecord fnodRec = caseLink.getFnodRecord();

		if (FNODConstants.Y.equals(fnodRec.getDodAlreadyPresent())) {
			if (FNODConstants.Y.equals(fnodRec.getBenefitCancelled())) {
				JSFMessageUtil.addError(DODP_BC.name(), DODP_BC.getSummaryMsg(), DODP_BC.getDetailMsg());
				result = false;
			}
		}
		return result;
	}

	private boolean isCaseTypeSpouse(CaseLink caseLink) {
		boolean result = true;
		FnodRecord fnodRec = caseLink.getFnodRecord();
		if (caseLink.getCaseType().getCaseTypeCd().equals(CaseTypeCode.FNOD_SPOUSE)) {
			if (FNODConstants.Y.equals(fnodRec.getIris())) {
				if (FNODConstants.N.equals(fnodRec.getBenefitCancelled())) {
					JSFMessageUtil.addError(SPOUSE_IRIS_BC.name(), SPOUSE_IRIS_BC.getSummaryMsg(), SPOUSE_IRIS_BC.getDetailMsg());
					result = false;
				}
			}
		}
		return result;
	}

	private boolean isNMISuppliedAndValid(CaseLink caseLink) {
		boolean result = true;
		FnodRecord fnod = caseLink.getFnodRecord();

		if (isNmiChecked(caseLink)) {

			if (FNODConstants.Y.equals(fnod.getDodAlreadyPresent())) {
				JSFMessageUtil.addError("NMI-01", "NMI invalid!", "If NMI is chosen, 'DOD Already Present' may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getFlagIssued())) {
				JSFMessageUtil.addError("NMI-02", "NMI invalid!", "If NMI is chosen, 'Flag Issued' may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getFnod())) {
				JSFMessageUtil.addError("NMI-03", "NMI invalid!", "If NMI is chosen, 'FNOD' may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getBenefitCancelled())) {
				JSFMessageUtil.addError("NMI-04", "NMI invalid!", "If NMI is chosen, 'Benefit Cancelled' may not be chosen.");
				result = false;
			}

			if (fnod.getMonthlyAmount() != null) {
				JSFMessageUtil.addError("NMI-05", "NMI invalid!", "If NMI is chosen, 'Monthly Amount' may not be supplied.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getBirlsAdd())) {
				JSFMessageUtil.addError("NMI-06", "NMI invalid!", "If NMI is chosen, 'BIRLS Add' may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getBirlsUpdate())) {
				JSFMessageUtil.addError("NMI-07", "NMI invalid!", "If NMI is chosen, 'BIRLS Update' may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getPmc())) {
				JSFMessageUtil.addError("NMI-08", "NMI invalid!", "If NMI is chosen, 'PMC' may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getTooOldToCreate())) {
				JSFMessageUtil.addError("NMI-09", "NMI invalid!", "If NMI is chosen, 'Too Old to Create' may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getNis())) {
				JSFMessageUtil.addError("NMI-10", "NMI invalid!", "If NMI is chosen, 'NIS' may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getSensitive())) {
				JSFMessageUtil.addError("NMI-11", "NMI invalid!", "If NMI is chosen, 'Sensitive' may not be chosen.");
				result = false;
			}
		}
		return result;
	}

	private void setNMIFieldsForComplete(CaseLink caseLink) {
		FnodRecord fnod = caseLink.getFnodRecord();
		if (FNODConstants.Y.equalsIgnoreCase(fnod.getNmi())) {
			fnod.setDodAlreadyPresent(null);
			fnod.setFlagIssued(null);
			fnod.setFnod(null);
			fnod.setBenefitCancelled(null);
			fnod.setMonthlyAmount(null);
			fnod.setBirlsAdd(null);
			fnod.setBirlsUpdate(null);
			fnod.setPmc(null);
			fnod.setTooOldToCreate(null);
		}
	}

	private boolean isTooOldToCreateSuppliedAndValid(CaseLink caseLink) {
		boolean result = true;
		FnodRecord fnod = caseLink.getFnodRecord();

		if (isTooOldToCreateChecked(caseLink)) {

			if (FNODConstants.Y.equals(fnod.getDodAlreadyPresent())) {
				JSFMessageUtil.addError("TOC-01", "'Too Old to Create' invalid!", "If 'Too Old to Create' is chosen, 'DOD Already Present' may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getFlagIssued())) {
				JSFMessageUtil.addError("TOC-02", "'Too Old to Create' invalid!", "If 'Too Old to Create' is chosen, 'Flag Issued' may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getFnod())) {
				JSFMessageUtil.addError("TOC-03", "'Too Old to Create' invalid!", "If 'Too Old to Create' is chosen, 'FNOD' may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getBenefitCancelled())) {
				JSFMessageUtil.addError("TOC-04", "'Too Old to Create' invalid!", "If 'Too Old to Create' is chosen, 'Benefit Cancelled' may not be chosen.");
				result = false;
			}

			if (fnod.getMonthlyAmount() != null) {
				JSFMessageUtil.addError("TOC-05", "'Too Old to Create' invalid!", "If 'Too Old to Create' is chosen, 'Monthly Amount' may not be supplied.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getBirlsAdd())) {
				JSFMessageUtil.addError("TOC-06", "'Too Old to Create' invalid!", "If 'Too Old to Create' is chosen, 'BIRLS Add' may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getBirlsUpdate())) {
				JSFMessageUtil.addError("TOC-07", "'Too Old to Create' invalid!", "If 'Too Old to Create' is chosen, 'BIRLS Update' may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getPmc())) {
				JSFMessageUtil.addError("TOC-08", "'Too Old to Create' invalid!", "If 'Too Old to Create' is chosen, 'PMC' may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getNmi())) {
				JSFMessageUtil.addError("TOC-09", "'Too Old to Create' invalid!", "If 'Too Old to Create' is chosen, NMI may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getIris())) {
				JSFMessageUtil.addError("TOC-10", "'Too Old to Create' invalid!", "If 'Too Old to Create' is chosen, IRIS may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getFolderOnlyBenefit())) {
				JSFMessageUtil.addError("TOC-11", "'Too Old to Create' invalid!", "If 'Too Old to Create' is chosen, Folder Only Benefit may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getSensitive())) {
				JSFMessageUtil.addError("TOC-12", "'Too Old to Create' invalid!", "If 'Too Old to Create' is chosen, Sensitive may not be chosen.");
				result = false;
			}
		}
		return result;
	}

	private void setTooOldFieldsForComplete(CaseLink caseLink) {
		FnodRecord fnod = caseLink.getFnodRecord();
		if (FNODConstants.Y.equalsIgnoreCase(fnod.getTooOldToCreate())) {
			fnod.setDodAlreadyPresent(null);
			fnod.setFlagIssued(null);
			fnod.setFnod(null);
			fnod.setBenefitCancelled(null);
			fnod.setMonthlyAmount(null);
			fnod.setBirlsAdd(null);
			fnod.setBirlsUpdate(null);
			fnod.setPmc(null);
			fnod.setNmi(null);
			fnod.setIris(null);
		}
	}

	private boolean isIrisValid(CaseLink caseLink) {
		FnodRecord fnod = caseLink.getFnodRecord();
		boolean result = true;
		if (FNODConstants.Y.equals(fnod.getIris())) {
			if (FNODConstants.Y.equals(fnod.getBirlsAdd())) {
				JSFMessageUtil.addError("IRIS-01", "'IRIS' invalid!", "If 'IRIS' is chosen, Birls Add may not be chosen.");
				result = false;
			}
		}
		return result;
	}

	private void setIrisForComplete(CaseLink caseLink) {
		FnodRecord fnod = caseLink.getFnodRecord();
		if (FNODConstants.Y.equalsIgnoreCase(fnod.getIris())) {
			fnod.setBirlsAdd(null);
			fnod.setTooOldToCreate(null);
		}
	}

	private boolean isNisValid(CaseLink caseLink) {
		FnodRecord fnod = caseLink.getFnodRecord();
		boolean result = true;
		if (FNODConstants.Y.equals(fnod.getNis())) {
			if (FNODConstants.Y.equals(fnod.getDodAlreadyPresent())) {
				JSFMessageUtil.addError("NIS-01", "'NIS' invalid!", "If 'NIS' is chosen, DOD Already Present may not be chosen.");
				result = false;
			}
			if (FNODConstants.Y.equals(fnod.getBenefitCancelled())) {
				JSFMessageUtil.addError("NIS-02", "'NIS' invalid!", "If 'NIS' is chosen, Benefit Cancelled may not be chosen.");
				result = false;
			}
			if (fnod.getMonthlyAmount() != null) {
				JSFMessageUtil.addError("NIS-03", "NIS invalid!", "If NIS is chosen, 'Monthly Amount' may not be supplied.");
				result = false;
			}
			if (FNODConstants.Y.equals(fnod.getBirlsUpdate())) {
				JSFMessageUtil.addError("NIS-04", "'NIS' invalid!", "If 'NIS' is chosen, Birls Update may not be chosen.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getSensitive())) {
				JSFMessageUtil.addError("NIS-05", "'NIS' invalid!", "If 'NIS' is chosen, Sensitive may not be chosen.");
				result = false;
			}
		}

		return result;
	}

	private void setNisForComplete(CaseLink caseLink) {
		FnodRecord fnod = caseLink.getFnodRecord();
		if (FNODConstants.Y.equals(fnod.getNis())) {
			fnod.setDodAlreadyPresent(null);
			fnod.setBenefitCancelled(null);
			fnod.setMonthlyAmount(null);
			fnod.setBirlsUpdate(null);
		}
	}

	private boolean isSensitiveValid(CaseLink caseLink) {
		FnodRecord fnod = caseLink.getFnodRecord();
		boolean result = true;
		if (FNODConstants.Y.equals(fnod.getSensitive())) {
			if (FNODConstants.Y.equals(fnod.getBirlsAdd())) {
				JSFMessageUtil.addError("SEN-01", "'Sensitive' invalid!", "If 'Sensitive' is chosen, Birls Add may not be chosen.");
				result = false;
			}
			//  Sensitive and NMI are mutually exclusive, rule enforced by NMI
		}

		if (FNODConstants.Y.equals(fnod.getSensitive()) && !user.isSupervisor()) {
			JSFMessageUtil.addError("SEN-02", "'Sensitive' invalid!", "Only supervisors may complete or modify if 'Sensitive' is chosen");
			result = false;
		}

		return result;
	}

	private boolean isSensitiveChecked(CaseLink caseLink) {
		boolean retval = false;
		if (caseLink.getFnodRecord() != null) {
			retval = FNODConstants.Y.equals(caseLink.getFnodRecord().getSensitive());
		}
		return retval;
	}

	private void setSensitiveForComplete(CaseLink caseLink) {
		FnodRecord fnod = caseLink.getFnodRecord();
		if (FNODConstants.Y.equals(fnod.getSensitive())) {
			fnod.setBirlsAdd(null);
			fnod.setNmi(null);
		}
	}

	private boolean isFolderOnlyBenefitValid(CaseLink caseLink) {
		FnodRecord fnod = caseLink.getFnodRecord();
		boolean result = true;
		if (FNODConstants.Y.equals(fnod.getFolderOnlyBenefit())) {

			// Folder Only Benefit and Too Old to Create mutually exclusive, rule enforced by Too Old to Create

			if (FNODConstants.Y.equals(fnod.getBenefitCancelled())) {
				JSFMessageUtil.addError("FOB-01", "'Folder Only Benefit' invalid!", "If 'Folder Only Benefit' is chosen, Benefit Cancelled may not be chosen.");
				result = false;
			}

			if (fnod.getMonthlyAmount() != null) {
				JSFMessageUtil.addError("FOB-02", "'Folder Only Benefit' invalid!", "If 'Folder Only Benefit' is chosen, Monthly Amount may not be supplied.");
				result = false;
			}

			if (FNODConstants.Y.equals(fnod.getBirlsAdd())) {
				JSFMessageUtil.addError("FOB-03", "'Folder Only Benefit' invalid!", "If 'Folder Only Benefit' is chosen, Birls Add may not be chosen.");
				result = false;
			}
		}
		return result;
	}

	private void setFolderOnlyBenefitForComplete(CaseLink caseLink) {
		FnodRecord fnod = caseLink.getFnodRecord();
		if (FNODConstants.Y.equals(fnod.getFolderOnlyBenefit())) {
			fnod.setTooOldToCreate(null);
			fnod.setBenefitCancelled(null);
			fnod.setMonthlyAmount(null);
			fnod.setBirlsAdd(null);
		}
	}

	private boolean isSSNSupplied(CaseLink caseLink) {
		boolean isSuppliedAndValid = false;
		Person per = caseLink.getFnodRecord().getVeteranPerson();
		boolean ssnProvided = !isEmpty(per) && !isEmpty(per.getSocialSecurityNumber());
		if (ssnProvided) {
			String ssn = per.getSocialSecurityNumber();
			if (ssn.matches("(\\d{3}-\\d{2}-\\d{4})|(\\d{9})")) {
				isSuppliedAndValid = true;
			} else {
				if (createMessages) {
					JSFMessageUtil.addError("FLD-11",
							"Required: SSN!",
							"SSN must be entered as ###-##-#### or 9 digits");
				}
			}
		}
		return isSuppliedAndValid;
	}

	private boolean isVCNSupplied(CaseLink caseLink) {
		boolean isSupplied = false;
		isSupplied = !isEmpty(caseLink.getFnodRecord().getVeteranClaimNum());
		return isSupplied;
	}

	private boolean isMSNSupplied(CaseLink caseLink) {
		boolean isSupplied = false;
		isSupplied = !isEmpty(caseLink.getFnodRecord().getMilitaryServiceNum());
		return isSupplied;
	}

	private boolean isPMCOnlyOnFlagApp(CaseLink caseLink) {
		boolean isValid = true;
		// check if PMC is selected
		if (charToBoolean(caseLink.getFnodRecord().getPmc())) {
			// check if Case is Flag Application
			if (SourceSystemCode.FLAG_APPLICATION.equals(caseLink.getSourceSystem().getSourceSystemCd())) {
				isValid = true;
			} else {
				isValid = false;
				if (createMessages) {
					JSFMessageUtil.addError("FLD-07",
							"PMC Flag App Only!",
							"PMC may only be selected when Source System is Flag Application.");
				}
			}
		}
		return isValid;
	}

	private boolean isMontlyAmountCompatibleWithBenefitCancelled(CaseLink caseLink) {
		boolean result = true;

		BigDecimal monthlyAmount = caseLink.getFnodRecord().getMonthlyAmount();
		if (monthlyAmount == null) {
			monthlyAmount = new BigDecimal(0.0);
		}

		if (monthlyAmount.doubleValue() > 0.0D && !charToBoolean(caseLink.getFnodRecord().getBenefitCancelled())) {
			result = false;
			if (createMessages) {
				JSFMessageUtil.addError(
						"FLD-08",
						"Mismatch between Benefit Cancelled and Monthly Amount!",
						"If Monthly Amount is > 0, then Benefit Cancelled must be selected.");
			}
		}
		if (charToBoolean(caseLink.getFnodRecord().getBenefitCancelled()) && monthlyAmount.doubleValue() <= 0.0D) {
			result = false;
			if (createMessages) {
				JSFMessageUtil.addError(
						"FLD-010",
						"Invalid Monthly Amount!",
						"If Benefit Cancelled is selected, Monthly Amount must be greater than 0.00.");
			}
		}
		return result;
	}

	private boolean isEmpty(Object o) {
		boolean result = false;
		if (o == null) {
			result = true;
		} else if (o instanceof String) {
			if (o.toString().trim().length() == 0) {
				result = false;
			}
		}
		return result;
	}

	public boolean isFlagIssuedValid(CaseLink caseLink) {
		boolean result = true;
		if (!isFlagApp(caseLink) && FNODConstants.Y.equals(caseLink.getFnodRecord().getFlagIssued())) {
			result = false;
		}
		return result;
	}

	public boolean isFlagApp(CaseLink caseLink) {
		return caseLink.getSourceSystem().getSourceSystemCd().equals(SourceSystemCode.FLAG_APPLICATION);
	}

	public boolean isEligibilityVerificationValid(CaseLink caseLink) {
		boolean result = true;
		if (!(SourceSystemCode.FLAG_APPLICATION.equals(caseLink.getSourceSystem().getSourceSystemCd().toString())
				|| SourceSystemCode.INSURANCE_REPORT.equals(caseLink.getSourceSystem().getSourceSystemCd().toString()))) {
			if (FNODConstants.Y.equals(caseLink.getFnodRecord().getEligibilityVerification())) {
				result = false;
				if (createMessages) {
					JSFMessageUtil.addError("FLD-09",
							"Eligibiliy Verification Flag App Only!",
							"The Eligibiliy Verification of can only be modified for Flag Applicatoins and Insurance Reports.");
				}
			}
		}
		return result;
	}

	/**
	 * Converts the database Y or N to true or false for validation purposes.
	 *
	 * @param s
	 * @return
	 */
	private boolean charToBoolean(String s) {
		boolean result;
		if (s == null || FNODConstants.N.equals(s)) {
			result = false;
		} else if (FNODConstants.Y.equalsIgnoreCase(s)) {
			result = true;
		} else {
			throw new IllegalArgumentException("Expected: null, 'Y' or 'N'. Found: " + s);
		}
		return result;
	}

	public boolean isCreateMessages() {
		return createMessages;
	}

	public void setCreateMessages(boolean createMessages) {
		this.createMessages = createMessages;
	}
}
