package gov.va.fnod.business;

import gov.va.fnod.model.fnoddata.AuditActivity;
import gov.va.fnod.model.fnoddata.AuditChange;
import gov.va.fnod.model.fnoddata.CaseAttachment;
import gov.va.fnod.model.fnoddata.CaseLink;
import gov.va.fnod.model.fnoddata.CaseType;
import gov.va.fnod.model.fnoddata.CaseTypeCode;
import gov.va.fnod.model.fnoddata.FnodRecord;
import gov.va.fnod.model.fnoddata.Person;
import gov.va.fnod.model.fnoddata.Region;
import gov.va.fnod.model.fnoddata.RegionalOffice;
import gov.va.fnod.model.fnoddata.SourceCaseTypeMap;
import gov.va.fnod.model.fnoddata.SourceSystem;
import gov.va.fnod.model.fnoddata.SourceSystemCode;
import gov.va.fnod.util.JSFMessageUtil;
import gov.va.fnod.view.FnodReportDataSupplierInterface;
import gov.va.fnod.view.converter.RegionConverter;
import gov.va.fnod.view.converter.RegionalOfficeConverter;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import org.apache.log4j.Logger;

import com.lowagie.text.Chunk;
import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.Element;
import com.lowagie.text.Font;
import com.lowagie.text.PageSize;
import com.lowagie.text.Paragraph;
import com.lowagie.text.Phrase;
import com.lowagie.text.Rectangle;
import com.lowagie.text.pdf.PdfPCell;
import com.lowagie.text.pdf.PdfPTable;
import com.lowagie.text.pdf.PdfReader;
import com.lowagie.text.pdf.PdfSmartCopy;
import com.lowagie.text.pdf.PdfWriter;
import com.lowagie.text.pdf.draw.LineSeparator;

public class CaseLinkPrimaryPdf {
	
	private Font fontHeader = new Font(Font.HELVETICA, 14, Font.BOLD);
	private Font font = new Font(Font.HELVETICA, 10);
	private Font fontBold = new Font(Font.HELVETICA, 10, Font.BOLD);
	private Font fontFixed = new Font(Font.COURIER, 10);
	private Font fontFixedBold = new Font(Font.COURIER, 10, Font.BOLD);
	private List<String> pdfFilePathsToAppend = new ArrayList<String>();
	
	private final static Logger log = Logger.getLogger(CaseLinkPrimaryPdf.class);
	
	FnodReportDataSupplierInterface caseBean = null;
	
	public CaseLinkPrimaryPdf(FnodReportDataSupplierInterface caseBean) {
		super();
		this.caseBean = caseBean;
	}
	
	enum FontType {
		Normal,
		Fixed
	}
	
	public byte[] createPrimaryPdf() {
		byte[] results;
		
		CaseLink caseLink = caseBean.getCaseLink();
	
		FnodRecord fnodRecord = caseLink.getFnodRecord();
		Person fnodVeteran = fnodRecord.getVeteranPerson();
		
		Document document = new Document(PageSize.LETTER, 36, 36, 36, 36);
		ByteArrayOutputStream os = new ByteArrayOutputStream();
		try {
			PdfWriter.getInstance(document, os);
			
			document.open();
			
			Paragraph headerPar = new Paragraph(new Phrase("FNOD RECORD", fontHeader));
			headerPar.setAlignment(Element.ALIGN_CENTER);
			document.add(headerPar);
			document.add(Chunk.NEWLINE);
			
			//===============================================
			// 1st Section
			//===============================================
			PdfPTable firstTable = table(3);
			
			// Locked Date
			String formattedFnodDt = "";
			Timestamp fnodCreatedDt = fnodRecord.getFnodCreatedDt();
			if (fnodCreatedDt != null) {
				formattedFnodDt = new SimpleDateFormat("MM/dd/yyyy kk:mm").format(fnodCreatedDt);
			}
			
			String lockedValue = caseLink.getLockedStatus() == null ? "" : caseLink.getLockedStatus().toString();
			
			String formattedCaseCreatedDt = "";
			Timestamp caseCreatedDt = fnodRecord.getCaseLink().getLoadDate();
			if (caseCreatedDt != null) {
				formattedCaseCreatedDt = new SimpleDateFormat("MM/dd/yyyy kk:mm").format(caseCreatedDt);
			}
			
			String formattedCaseLockedDt = "";
			Timestamp caseLockedDt = fnodRecord.getCaseLockedDt();
			if (caseLockedDt != null) {
				formattedCaseLockedDt = new SimpleDateFormat("MM/dd/yyyy kk:mm").format(caseLockedDt);
			}
			
			// row 1
			firstTable.addCell(cell("User Id: ", fnodRecord.getUsername()));
			firstTable.addCell(cell("Case: ", String.valueOf(caseLink.getCaseId())));
			firstTable.addCell(cell("FNOD Created Date: ", formattedFnodDt));
			
			
			// row 2
			firstTable.addCell(new BlankCell());
			firstTable.addCell(cell("Locked: ", lockedValue));
			firstTable.addCell(cell("Case Locked Date: ", formattedCaseLockedDt));
			
			// row 3
			firstTable.addCell(new BlankCell());
			firstTable.addCell(new BlankCell());
			firstTable.addCell(cell("Case Created Date: ", formattedCaseCreatedDt));
			
			document.add(firstTable);
			
			//===============================================
			// 2nd Section
			//===============================================
			
			PdfPTable secondTable = table(2);
			
			// Region
			RegionalOffice regionalOffice = fnodRecord.getRegionalOffice();
			String regOffDesc = "";
			if (regionalOffice != null) {
				Region region = regionalOffice.getRegion();
				if (region != null) {
					regOffDesc = region.getDescription();
				}
			}
			
			// Regional Office
			String regionalOfficeName = "";
			if (regionalOffice != null) {
				regionalOfficeName = regionalOffice.getRegionalOfficeName() == null ? "" : regionalOffice.getRegionalOfficeName();
			}
			
			// Regional Office Number
			String regionalOfficeNum = "";
			if (regionalOffice != null) {
				regionalOfficeNum = regionalOffice.getRegionalOfficeNum() == null ? "" : regionalOffice.getRegionalOfficeNum().toString();
			}
			
			
			CaseType caseType = caseLink.getCaseType();
			SourceSystem sourceSystem = caseLink.getSourceSystem();
			String sourceCaseTypeMapDesc = caseBean.getSourceCaseTypeMapSession().getSourceCaseTypeMap(sourceSystem, caseType).getDescription();
			secondTable.addCell(cell("Source: ", sourceCaseTypeMapDesc));
			secondTable.addCell(new BlankCell());
			
			if (isFlagApp(caseLink)) {
				secondTable.addCell(cell("Region: ", regOffDesc));
				secondTable.addCell(cell("RO #: ", regionalOfficeNum + " " + regionalOfficeName));
			}
			else {
				secondTable.addCell(new BlankCell());
				secondTable.addCell(new BlankCell());
			}
			
			document.add(secondTable);
			
				
			//===============================================
			// 3rd Section
			//===============================================
			PdfPTable thirdTable = table(4);
			
			// Name
			String firstName = fnodVeteran.getFirstName() == null ? "" : fnodVeteran.getFirstName();
			String middleName = fnodVeteran.getMiddleName() == null ? "" : fnodVeteran.getMiddleName();
			String lastName = fnodVeteran.getLastName() == null ? "" : fnodVeteran.getLastName();
			String suffixName = fnodVeteran.getSuffixName() == null ? "" : fnodVeteran.getSuffixName();
			
			
			// Veteran Name
			thirdTable.addCell(cell(label("Veteran Name"), 4));
			thirdTable.addCell(cell("First: ", firstName));
			thirdTable.addCell(cell("Middle: ", middleName));
			thirdTable.addCell(cell("Last: ", lastName));
			thirdTable.addCell(cell("Suffix: ", suffixName));
			
			document.add(thirdTable);
			
			//===============================================
			// 4th Section
			//===============================================
			PdfPTable fourthTable = table(4);
			
			// Date of Birth
			Date birthDt = fnodVeteran.getBirthDt();
			String formattedBirthDt = "";
			if (birthDt != null) {
				try {
					formattedBirthDt = birthDt == null ? "" : new SimpleDateFormat("MM/dd/yyyy").format(birthDt);			
				} catch (Exception e) {
					log.error("Unexpected birth date format: " + birthDt, e);
				}
			}
			
			// Date of Death
			String formattedDeathDt = "";
			Date deathDt = fnodVeteran.getDeathDt();
			if (deathDt != null) {
				try {
					formattedDeathDt = deathDt == null ? "" : new SimpleDateFormat("MM/dd/yyyy").format(deathDt);
				} catch (Exception e) {
					log.error("Unexpected death date format: " + deathDt, e);
				}
			}
			
			// Need More Information (NMI)
			String nmi = fnodRecord.getNmi() == null ? "" : fnodRecord.getNmi();
			
			// Too Old to Create
			String tooOldToCreate = fnodRecord.getTooOldToCreate() == null ? "" : fnodRecord.getTooOldToCreate();
			
			fourthTable.addCell(cell("DOB: ", formattedBirthDt));
			fourthTable.addCell(cell("DOD: ", formattedDeathDt));
			fourthTable.addCell(cell("NMI: ", nmi));
			fourthTable.addCell(cell("Too old to create? ", tooOldToCreate));
			
			document.add(fourthTable);
			
			//===============================================
			// 5th Section
			//===============================================
			PdfPTable fifthTable = table(3);
			
			// SSN
			String ssn = fnodVeteran.getSocialSecurityNumber() == null ? "" : fnodVeteran.getSocialSecurityNumber();
			if (ssn.trim().length() == 9) {
				ssn = new StringBuilder(ssn).insert(5, '-').insert(3, '-').toString();
			}
			
			// SV # (Military Service Number) 9 or 10 length
			String militaryServiceNum = fnodRecord.getMilitaryServiceNum();
			
			// Claim Number
			String veteranClaimNumber = fnodRecord.getVeteranClaimNum();
			
			fifthTable.addCell(cell("SSN: ", ssn));
			fifthTable.addCell(cell("SV #: ", militaryServiceNum));
			fifthTable.addCell(cell("Claim #: ", veteranClaimNumber));
			
			document.add(fifthTable);
			 
			//===============================================
			// 6th Section
			//===============================================
			
			// Eligibility Verification (Y/N)
			String eligibilityVerification = fnodRecord.getEligibilityVerification() == null ? "" : fnodRecord.getEligibilityVerification();

			//DoD Already Present (Y/N)
			String dodAlreadyPresent = fnodRecord.getDodAlreadyPresent() == null ? "" : fnodRecord.getDodAlreadyPresent();
			
			// Flag Issued (Y/N)
			String flagIssued = fnodRecord.getFlagIssued() == null ? "" : fnodRecord.getFlagIssued();
			
			document.add(paragraph("Eligibility Verification: ", eligibilityVerification));
			document.add(paragraph("DOD already present: ", dodAlreadyPresent));
			document.add(paragraph("Flag Issued: ", flagIssued));
			document.add(paragraph("FNOD: ", fnodRecord.getFnod()));
			
			//===============================================
			// 7th Section
			//===============================================
			PdfPTable sixthTable = table(2);
			
			// Benefit Cancelled (Y/N)
			String benefitCancelled = fnodRecord.getBenefitCancelled() == null ? "" : fnodRecord.getBenefitCancelled();
			
			// Monthly Amount
			String monthlyAmount = fnodRecord.getMonthlyAmount() == null ? "" : fnodRecord.getMonthlyAmount().toString();
			
			sixthTable.addCell(cell("Benefit Cancelled: ", benefitCancelled));
			sixthTable.addCell(cell("Monthly Amount: ", monthlyAmount));

			document.add(sixthTable);
			
			//===============================================
			// 8th Section
			//===============================================
			
			// BIRLS Added (Y/N)
			// IRIS (Y/N)
			
			//Paragraph birlsAddPar = new Paragraph();
			//birlsAddPar.add(new Phrase("BIRLS Add: ", fontBold));
			//birlsAddPar.add(new Phrase(fnodRecord.getBirlsAdd(), font));
			PdfPTable seventhTable = table(2);
			
			String birlsAdd = fnodRecord.getBirlsAdd() == null ? "" : fnodRecord.getBirlsAdd();
			String iris = fnodRecord.getIris() == null ? "" : fnodRecord.getIris();
			String birlsUpdate = fnodRecord.getBirlsUpdate() == null ? "" : fnodRecord.getBirlsUpdate();
			String nis = fnodRecord.getNis() == null ? "" : fnodRecord.getNis();
			String pmc = fnodRecord.getPmc() == null ? "" : fnodRecord.getPmc();
			String sensitive = fnodRecord.getSensitive() == null ? "" : fnodRecord.getSensitive();
			String folderOnlyBenefit = fnodRecord.getFolderOnlyBenefit() == null ? "" : fnodRecord.getFolderOnlyBenefit();
			//Row 1
			seventhTable.addCell(cell("BIRLS Add: ", birlsAdd));
			seventhTable.addCell(cell("IRIS: ", iris));
			
			//Row 2
			seventhTable.addCell(cell("BIRLS Update: ", birlsUpdate));
			seventhTable.addCell(cell("NIS: ", nis));
			
			//Row 3
			seventhTable.addCell(cell("PMC: ", pmc));
			seventhTable.addCell(cell("Sensitive: ", sensitive));
			
			//Row 4
			seventhTable.addCell(cell(" ", " "));
			seventhTable.addCell(cell("Folder Only Benefit: ", folderOnlyBenefit));
			
			
			
			document.add(seventhTable);
			
			//===============================================
			// 9th Section
			//===============================================
			// BIRLS Update (Y/N)
			// NIS (Y/N)
			// BIRLS Update (Y/N)
			//document.add(paragraph("BIRLS Update: ", fnodRecord.getBirlsUpdate()));
			
			
			
			// PMC (Y/N)
			//document.add(paragraph("PMC: ", fnodRecord.getPmc()));

			//document.add(Chunk.NEWLINE);
			
			// Comments (Free-form Text Field)
			document.add(paragraph("Comments: ", fnodRecord.getComments(), 20));
			
			addAuditActivityToDocument(document, caseLink);
			
			addCaseData(document, caseLink);
			
			addAttachmentsToDocument(document, caseLink);
			document.close();
			
			// Save the header to disk for final assembly
			String mainDoc = caseBean.getFilecache().addToCache(new ByteArrayInputStream(os.toByteArray()), ".pdf");
			pdfFilePathsToAppend.add(0,mainDoc);
			results = assembleFinalDocument();
			caseBean.getFilecache().removeFromCache(mainDoc);
			
		} catch (DocumentException e) {
			e.printStackTrace();
			log.error(e.getMessage(), e.fillInStackTrace());
			throw new RuntimeException("A serious failure occured in creating the FNOD Record PDF.");
		}
		return results;
	}

	class BlankCell extends PdfPCell {
		BlankCell() {
			super(new Phrase(" "));
			setBorder(Rectangle.NO_BORDER);
			setPadding(0);
		}
		
		BlankCell(int colspan) {
			super(new Phrase(" "));
			setBorder(Rectangle.NO_BORDER);
			setPadding(0);
			setColspan(colspan);
		}
	}

	
	
	
	private void addAuditActivityToDocument(Document document, CaseLink caseLink) throws DocumentException {
		if (caseBean.getUser().isAuditor() && caseLink.getAuditActivity() != null && caseLink.getAuditActivity().size() > 0) {
			PdfPTable auditTable = table(3);
			
			// Audit Activity Header
			auditTable.addCell(cell(label("Audit Activity"), 3));
			document.add(line());
			
			for (AuditActivity auditActivity : caseLink.getAuditActivity()) {
				String activityDate = new SimpleDateFormat("MM/dd/yyyy hh:mm:ss").format(auditActivity.getActivityDt()); 
				
				// line #1
				auditTable.addCell(cell("Username: ", auditActivity.getUsername()));
				auditTable.addCell(cell("Date : ", activityDate));
				auditTable.addCell(cell("Description: ", auditActivity.getActivityDescription()));
				
				// audit field changes
				for (AuditChange change : auditActivity.getAuditChanges()) {
					auditTable.addCell(cell("   Field: ", change.getFieldName()));
					auditTable.addCell(cell("Original Value: ", change.getOrigValue()));
					auditTable.addCell(cell("New Value: ", change.getNewValue()));
				}
			}
			
			document.add(auditTable);
		}
	}
	
	private void addCaseData(Document document, CaseLink caseLink) throws DocumentException {
		document.add(line());
		
		PdfPTable caseDataTitleTable = table(1);
		
		// source system code
		//SourceCaseTypeMapConverter srcCaseTypConv = new SourceCaseTypeMapConverter();
		SourceCaseTypeMap sourceCaseTypeMap = caseBean.getSourceCaseTypeMapSession().getSourceCaseTypeMap(caseLink.getSourceSystem(), caseLink.getCaseType());
		//String sourceCaseType = srcCaseTypConv.getAsString(null, null, sourceCaseTypeMap);
		caseDataTitleTable.addCell(cell(label(sourceCaseTypeMap.getDescription()), 6));
		document.add(caseDataTitleTable);
		
		
		if ( caseLink.getCaseData() != null ) {
			PdfPTable caseDataTable = table(100);
			addCaseDataNameInfo(caseDataTable, caseLink);
			addCaseDataServiceInfo(caseDataTable, caseLink);
			addVeterancyInfoAmas(caseDataTable, caseLink);
			addVeterancyInfoBoss(caseDataTable, caseLink);
			addDecedentInfoBoss(caseDataTable, caseLink);
			addHomeOfRecordInfoBoss(caseDataTable, caseLink);
			addSpouseInfoFnod(caseDataTable, caseLink);
			addSpouseDecedentInfoFnod(caseDataTable, caseLink);
			document.add(caseDataTable);
		} else if ( caseLink.getCaseFlagApp() != null ) {
			addFlagAppCaseDataInfo(document, caseLink);	
		} else if ( caseLink.getCaseInsurance() != null ) {		
			addInsuranceInfo(document, caseLink);
		}
		
	}
	
	/**
	 * This method loops through all the AttachmentWrappers and adds their full file path, via quePdfFileToAppend(filePath)
	 *  to the list of PDF files to be appended to the primary PDF document.  Appends are all performed at once via the 
	 * addQueuedFilesToEndOfDocument() method.
	 * 
	 * @param document
	 * @param caseLink
	 * @throws DocumentException
	 */
	private void addAttachmentsToDocument(Document document, CaseLink caseLink) throws DocumentException {
		if (caseLink.getCaseAttachments() != null && caseLink.getCaseAttachments().size() > 0) {
			
			// For each attachment
			for(CaseAttachment attachment : caseBean.getCaseLink().getCaseAttachments()) {
				quePdfFileToAppend(caseBean.fetchAttachmentDocument(attachment));
			}
	
		}
	}
	
	
	private void addCaseDataNameInfo(PdfPTable table, CaseLink caseLink) {
		// remember, there are 6 columns, not 4, so fill right with blanks
		// row #1
		table.addCell(cell(label("LAST NAME", FontType.Fixed), 16));
		table.addCell(cell(label("FIRST NAME", FontType.Fixed), 16));
		table.addCell(cell(label("MIDDLE NAME", FontType.Fixed), 16));
		table.addCell(cell(label("SUFFIX", FontType.Fixed), 16));
		table.addCell(new BlankCell(36));
		
		// row #2
		table.addCell(cell(field(caseLink.getCaseData().getVeteranPerson().getLastName(), FontType.Fixed), 16));
		table.addCell(cell(field(caseLink.getCaseData().getVeteranPerson().getFirstName(), FontType.Fixed), 16));
		table.addCell(cell(field(caseLink.getCaseData().getVeteranPerson().getMiddleName(), FontType.Fixed), 16));
		table.addCell(cell(field(caseLink.getCaseData().getVeteranPerson().getSuffixName(), FontType.Fixed), 16));
		table.addCell(new BlankCell(36));
	}

	private void addCaseDataServiceInfo(PdfPTable table, CaseLink caseLink) {
		table.addCell(cell(label("SSN", FontType.Fixed), 12));
		table.addCell(cell(label("SVC NBR", FontType.Fixed), 12));
		table.addCell(cell(label("DOB", FontType.Fixed), 12));
		table.addCell(cell(label("DOD", FontType.Fixed), 12));
		table.addCell(cell(label("BOS", FontType.Fixed), 12));
		table.addCell(cell(label("RANK", FontType.Fixed), 11));
		table.addCell(new BlankCell(29));
		
		SimpleDateFormat format = new SimpleDateFormat("M/d/yyyy");
		table.addCell(cell(field(caseLink.getCaseData().getVeteranPerson().getSocialSecurityNumber(), FontType.Fixed), 12));
		table.addCell(cell(field(caseLink.getCaseData().getVeteran().getMilitaryServiceNum(), FontType.Fixed), 12));
		table.addCell(cell(field(format.format(caseLink.getCaseData().getVeteranPerson().getBirthDt()), FontType.Fixed), 12));
		table.addCell(cell(field(format.format(caseLink.getCaseData().getVeteranPerson().getDeathDt()), FontType.Fixed), 12));
		table.addCell(cell(field(caseLink.getCaseData().getVeteran().getBranchOfServiceCd(), FontType.Fixed), 12));
		table.addCell(cell(field(caseLink.getCaseData().getVeteran().getRankCd(), FontType.Fixed), 29));
		table.addCell(new BlankCell(31));
	}
	
	private void addVeterancyInfoAmas(PdfPTable table, CaseLink caseLink) {
		if (isAmas(caseLink)) {
			table.addCell(cell(label("EOD", FontType.Fixed), 12));
			table.addCell(cell(label("RAD", FontType.Fixed), 12));
			table.addCell(cell(label("WAR PERIOD", FontType.Fixed), 18));
			table.addCell(cell(label("VETERAN ID", FontType.Fixed), 15));
			table.addCell(cell(label("USER ID", FontType.Fixed), 15));
			table.addCell(new BlankCell(28));
			
			SimpleDateFormat format = new SimpleDateFormat("M/d/yyyy");
			table.addCell(cell(field(format.format(caseLink.getCaseData().getVeteran().getEnteredOnActiveDutyDt()), FontType.Fixed), 12));
			table.addCell(cell(field(format.format(caseLink.getCaseData().getVeteran().getReleasedFromActiveDutyDt()), FontType.Fixed), 12));
			table.addCell(cell(field(caseLink.getCaseData().getVeteran().getWarPeriodCd(), FontType.Fixed), 18));
			table.addCell(cell(field(caseLink.getCaseData().getVeteran().getVeteranId(), FontType.Fixed), 15));
			table.addCell(cell(field(caseLink.getCaseData().getSourceUserId(), FontType.Fixed), 15));
			table.addCell(new BlankCell(28));
		}
	}

	private void addVeterancyInfoBoss(PdfPTable table, CaseLink caseLink) {
		if (isBoss(caseLink) || isAnc(caseLink) ) {
			table.addCell(cell(label("EOD", FontType.Fixed), 12));
			table.addCell(cell(label("RAD", FontType.Fixed), 12));
			table.addCell(cell(label("WAR PERIOD", FontType.Fixed), 18));
			table.addCell(cell(label("CLAIM NBR", FontType.Fixed), 15));
			table.addCell(cell(label("ALIAS NAME", FontType.Fixed), 30));
			table.addCell(new BlankCell(13));
			
			SimpleDateFormat format = new SimpleDateFormat("M/d/yyyy");
			table.addCell(cell(field(format.format(caseLink.getCaseData().getVeteran().getEnteredOnActiveDutyDt()), FontType.Fixed), 12));
			table.addCell(cell(field(format.format(caseLink.getCaseData().getVeteran().getReleasedFromActiveDutyDt()), FontType.Fixed), 12));
			table.addCell(cell(field(caseLink.getCaseData().getVeteran().getWarPeriodCd(), FontType.Fixed), 18));
			table.addCell(cell(field(caseLink.getCaseData().getVeteran().getClaimNum(), FontType.Fixed), 15));
			table.addCell(cell(field(caseLink.getCaseData().getVeteran().getAliasLastName(), FontType.Fixed), 30));
			table.addCell(new BlankCell(13));
		}
	}

	private void addDecedentInfoBoss(PdfPTable table, CaseLink caseLink) {
		if (isBoss(caseLink) || isAnc(caseLink) ) {
			table.addCell(cell(label("DECEDENT ID", FontType.Fixed), 15));
			table.addCell(cell(label("DOI", FontType.Fixed), 12));
			table.addCell(cell(label("CEMETERY NAME", FontType.Fixed), 42));
			table.addCell(cell(label("CEM TYPE", FontType.Fixed), 10));
			table.addCell(cell(label("CEM NBR", FontType.Fixed), 21));
			
			SimpleDateFormat format = new SimpleDateFormat("M/d/yyyy");
			table.addCell(cell(field(caseLink.getCaseData().getVeteranPerson().getDecedentId(), FontType.Fixed), 15));
			table.addCell(cell(field(format.format(caseLink.getCaseData().getVeteranPerson().getRecordOfIntermentDt()), FontType.Fixed), 12));
			table.addCell(cell(field(caseLink.getCaseData().getCemeteryName(), FontType.Fixed), 42));
			table.addCell(cell(field(caseLink.getCaseData().getCemeteryType(), FontType.Fixed), 10));
			table.addCell(cell(field(caseLink.getCaseData().getCemeteryNum(), FontType.Fixed), 21));
		}
	}
	
	private void addHomeOfRecordInfoBoss(PdfPTable table, CaseLink caseLink) {
		if (isBoss(caseLink) || isAnc(caseLink) ) {
			table.addCell(cell(label("CITY HOME OF RCD", FontType.Fixed), 30));
			table.addCell(cell(label("STATE HOME OF RCD", FontType.Fixed), 60));
			table.addCell(new BlankCell(10));

			table.addCell(cell(field(caseLink.getCaseData().getVeteranHomeOfRecordCity(), FontType.Fixed), 30));
			table.addCell(cell(field(caseLink.getCaseData().getVeteranHomeOfRecordState(), FontType.Fixed), 60));
			table.addCell(new BlankCell(10));
		}
	}
	
	private void addSpouseInfoFnod(PdfPTable table, CaseLink caseLink) {
		if (isFnodSpouse(caseLink)) {
			table.addCell(cell(label("SPOUSE LAST NAME", FontType.Fixed), 20));
			table.addCell(cell(label("FIRST NAME", FontType.Fixed), 20));
			table.addCell(cell(label("MIDDLE NAME", FontType.Fixed), 20));
			table.addCell(cell(label("DECEDENT ID", FontType.Fixed), 40));
			
			table.addCell(cell(field(caseLink.getCaseData().getSpousePerson().getLastName(), FontType.Fixed), 20));
			table.addCell(cell(field(caseLink.getCaseData().getSpousePerson().getFirstName(), FontType.Fixed), 20));
			table.addCell(cell(field(caseLink.getCaseData().getSpousePerson().getMiddleName(), FontType.Fixed), 20));
			table.addCell(cell(field(caseLink.getCaseData().getSpousePerson().getDecedentId(), FontType.Fixed), 40));
		}
	}
	
	private void addSpouseDecedentInfoFnod(PdfPTable table, CaseLink caseLink) {
		if (isFnodSpouse(caseLink)) {
			table.addCell(cell(label("DOB", FontType.Fixed), 12));
			table.addCell(cell(label("DOD", FontType.Fixed), 12));
			table.addCell(cell(label("DOI", FontType.Fixed), 76));
	
			
			SimpleDateFormat format = new SimpleDateFormat("M/d/yyyy");
			Date birthDt = caseLink.getCaseData().getSpousePerson().getBirthDt();
			table.addCell(cell(field(birthDt == null ? "" : format.format(birthDt), FontType.Fixed), 12));
			Date deathDt = caseLink.getCaseData().getSpousePerson().getDeathDt();
			table.addCell(cell(field(deathDt == null ? "" : format.format(deathDt), FontType.Fixed), 12));
			Date recordOfIntermentDt = caseLink.getCaseData().getSpousePerson().getRecordOfIntermentDt();
			table.addCell(cell(field(recordOfIntermentDt == null ? "" : format.format(recordOfIntermentDt), FontType.Fixed), 76));
		}
	}
	
	
	private void addInsuranceInfo(Document document, CaseLink caseLink) throws DocumentException {
		if (isCaseInsurance(caseLink)) {
			PdfPTable table = table(100);
			
			table.addCell(cell(label("Vet Name:", FontType.Fixed), 11));
			table.addCell(cell(field(caseLink.getCaseInsurance().getVeteranName(), FontType.Fixed), 89));
			 
			table.addCell(cell(label("Name and Address Line 1:", FontType.Fixed), 30));
			table.addCell(cell(field(caseLink.getCaseInsurance().getNameAddressLine1(), FontType.Fixed), 70));
	
			table.addCell(cell(label("Name and Address Line 2:", FontType.Fixed), 30));
			table.addCell(cell(field(caseLink.getCaseInsurance().getNameAddressLine2(), FontType.Fixed), 70));
	
			table.addCell(cell(label("Name and Address Line 3:", FontType.Fixed), 30));
			table.addCell(cell(field(caseLink.getCaseInsurance().getNameAddressLine3(), FontType.Fixed), 70));
	
			table.addCell(cell(label("Name and Address Line 4:", FontType.Fixed), 30));
			table.addCell(cell(field(caseLink.getCaseInsurance().getNameAddressLine4(), FontType.Fixed), 70));
	
			table.addCell(cell(label("Name and Address Line 5:", FontType.Fixed), 30));
			table.addCell(cell(field(caseLink.getCaseInsurance().getNameAddressLine5(), FontType.Fixed), 70));
	
			table.addCell(cell(label("City and State:", FontType.Fixed), 20));
			table.addCell(cell(field(caseLink.getCaseInsurance().getCityAndState(), FontType.Fixed), 80));
			
			table.addCell(cell(label("Zip or Foreign Code:", FontType.Fixed), 25));
			table.addCell(cell(field(caseLink.getCaseInsurance().getZipOrForeignCode(), FontType.Fixed), 75));
			
			table.addCell(cell(label("Date of Birth:", FontType.Fixed), 18));
			table.addCell(cell(field(caseLink.getCaseInsurance().getVeteranDateOfBirth(), FontType.Fixed), 82));
			
			table.addCell(cell(label("Date of Death:", FontType.Fixed), 18));
			table.addCell(cell(field(caseLink.getCaseInsurance().getVeteranDateOfDeath(), FontType.Fixed), 82));
			
			table.addCell(cell(label("Insurance File Number:", FontType.Fixed), 25));
			table.addCell(cell(field(caseLink.getCaseInsurance().getInsuranceFileNumber(), FontType.Fixed), 75));
			
			table.addCell(cell(label("SSN:", FontType.Fixed), 6));
			table.addCell(cell(field(caseLink.getCaseInsurance().getVeteranSocialSecurityNumber(), FontType.Fixed), 94));
			
			table.addCell(cell(label("Eight Digit Claim Number:", FontType.Fixed), 30));
			table.addCell(cell(field(caseLink.getCaseInsurance().getVeteranClaimNumber(), FontType.Fixed), 70));
			
			table.addCell(cell(label("Service Number:", FontType.Fixed), 18));
			table.addCell(cell(field(caseLink.getCaseInsurance().getVeteranMilitaryServiceNumber(), FontType.Fixed), 82));
			
			table.addCell(cell(label("Spouse Name:", FontType.Fixed), 15));
			table.addCell(cell(field(caseLink.getCaseInsurance().getSpouseName(), FontType.Fixed), 85));
			
			table.addCell(cell(label("Spouse Address Line 1:", FontType.Fixed), 27));
			table.addCell(cell(field(caseLink.getCaseInsurance().getSpouseAddressLine1(), FontType.Fixed), 73));
			
			table.addCell(cell(label("Spouse Address Line 2:", FontType.Fixed), 27));
			table.addCell(cell(field(caseLink.getCaseInsurance().getSpouseAddressLine2(), FontType.Fixed), 73));
			
			table.addCell(cell(label("Spouse Address Line 3:", FontType.Fixed), 27));
			table.addCell(cell(field(caseLink.getCaseInsurance().getSpouseAddressLine3(), FontType.Fixed), 73));
			
			table.addCell(cell(label("Spouse City and State:", FontType.Fixed), 27));
			table.addCell(cell(field(caseLink.getCaseInsurance().getSpouseCityAndState(), FontType.Fixed), 73));
			
			table.addCell(cell(label("Spouse Zip or Foreign Code:", FontType.Fixed), 32));
			table.addCell(cell(field(caseLink.getCaseInsurance().getSpouseZip(), FontType.Fixed), 68));
			
			document.add(table);
		}
	}
	
	private void addFlagAppCaseDataInfo(Document document, CaseLink caseLink) throws DocumentException {
		if (isFlagApp(caseLink)) {
			PdfPTable table = table(100);
			
			long regionId = caseLink.getCaseFlagApp().getRegionalId();
			long regionalOfficeId = caseLink.getCaseFlagApp().getRegionalOfficeId();
			
			String regionalOfficeName = new RegionConverter().getAsString(null, null, regionId);
			String roDescription = new RegionalOfficeConverter().getAsString(null, null, regionalOfficeId);
			
			table.addCell(cell("Region:", regionalOfficeName, 20, FontType.Fixed));
			
			table.addCell(cell("RO#:", roDescription, 20, FontType.Fixed));
			
			SimpleDateFormat format = new SimpleDateFormat("M/d/yyyy");
			table.addCell(cell("Entry Date:", format.format(caseBean.getCaseLink().getCaseFlagApp().getEntryDt()), 60, FontType.Fixed));
			
			document.add(table);
			
			// queue the FlagApp PDF to append later.
			quePdfFileToAppend(caseBean.getFlagAppFileName());
		}
	}
	
	
	/**
	 * This method merely accepts file paths to be added later to the end of the generated PDF document.
	 * The actual concatenation process is done in the assembleFinalDocument method.
	 * 
	 * @param filePath
	 */
	private void quePdfFileToAppend(String filePath) {
		this.pdfFilePathsToAppend.add(filePath);
	}
	
	private byte[] assembleFinalDocument() {
		byte[] result = null;
		
		if (pdfFilePathsToAppend != null && ! pdfFilePathsToAppend.isEmpty()) {
			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			String filePath = null;
			try {			
				Document document = new Document();		
				PdfSmartCopy copy = new PdfSmartCopy(document, baos);
				document.open();
				
				for (String filename : pdfFilePathsToAppend) {
					try {
						filePath = caseBean.getFilecache().getFilePath(filename);
						log.info("Appending attachment PDF to end of in-memory document: " + filePath);
						PdfReader reader = new PdfReader(filePath);
						
						int pages = reader.getNumberOfPages();
						for (int page = 0; page < pages;) {
							copy.addPage(copy.getImportedPage(reader, ++page));
						}
						
						copy.freeReader(reader);
					}
					catch (RuntimeException e) {
						// try to insulate one attachment failure from others.
						log.error("Unable to load file for attachment to PDF: " + filePath);
						e.printStackTrace();
					}					
				}
				document.close();
				
				result = baos.toByteArray();
				
			} catch (DocumentException e) {
				e.printStackTrace();
				log.error("Unable to add queued file to end of document.", e);
				JSFMessageUtil.addError("PDF-01", "PDF Error.", "Error attaching document to FNOD Report.");
			} catch (IOException e) {
				log.error("Unable to add queued file to end of document.", e);
				e.printStackTrace();
				JSFMessageUtil.addError("PDF-01", "PDF Error.", "Error attaching document to FNOD Report.");
			}			
		}
		else {
			result = null;
		}
		return result;
	}
	

	private Phrase label(String caption) {
		return label(caption, FontType.Normal);
	}
	
	private Phrase label(String caption, FontType fontType) {
		return new Phrase(caption, fontType == FontType.Normal ? fontBold : fontFixedBold);
	}
	
	private Phrase field(String value) {
		return field(value, FontType.Normal);
	}
	
	private Phrase field(String value, FontType fontType) {
		return new Phrase(value, fontType == FontType.Normal ? font : fontFixed);
	}
	
	private Phrase group(Phrase label, Phrase field) {
		Phrase p = new Phrase();
		p.add(label);
		p.add(field);
		return p;
	}
	
	private PdfPCell cell(String label, String value) {
		return cell(label, value, FontType.Normal);
	}
	
	private PdfPCell cell(String label, String value, FontType fontType) {
		return cell(label, value, 1, fontType);
	}
	
	private PdfPCell cell(String label, String value, int colspan, FontType fontType) {
		PdfPCell cell = new PdfPCell();
		cell.setBorder(Rectangle.NO_BORDER);
		cell.setPadding(0);
		cell.addElement(group(label(label, fontType), field(value, fontType)));
		return cell;
	}
	
	private PdfPCell cell(Phrase phrase, int colspan) {
		PdfPCell cell = new PdfPCell();
		cell.addElement(phrase);
		cell.setColspan(colspan);
		cell.setBorder(Rectangle.NO_BORDER);
		cell.setPadding(0);
		return cell;
	}
	
	private PdfPTable table(int columns) {
		PdfPTable table = new PdfPTable(columns);
		table.setKeepTogether(true);
		table.setWidthPercentage(100);
		table.setSpacingAfter(20);
		return table;
	}
	
	private Paragraph paragraph(String label, String value) {
		Paragraph par = new Paragraph();
		par.add(label(label));
		par.add(field(value));
		return par;
	}
	
	private Paragraph paragraph(String label, String value, int spacingAfter) {
		Paragraph par = paragraph(label, value);
		par.setSpacingAfter(spacingAfter);
		return par;
	}
	
	private Chunk line() {
		return new Chunk(new LineSeparator());
	}
	
	
	
	public boolean isFlagApp(CaseLink caseLink) {
		return caseLink.getSourceSystem().getSourceSystemCd().equals(SourceSystemCode.FLAG_APPLICATION);
	}
	
	public boolean isCaseInsurance(CaseLink caseLink) {
		return caseLink.getCaseInsurance() != null;
	}
	
	public boolean isAmas(CaseLink caseLink) {
		return caseLink.getSourceSystem().getSourceSystemCd().equals(SourceSystemCode.AMAS);
	}
	
	public boolean isBoss(CaseLink caseLink) {
		return caseLink.getSourceSystem().getSourceSystemCd().equals(SourceSystemCode.BOSS);
	}
	
	public boolean isAnc(CaseLink caseLink ) {
		return caseLink.getSourceSystem().getSourceSystemCd().equals(SourceSystemCode.ANC);
	}
	
	public boolean isFnod(CaseLink caseLink) {
		return caseLink.getCaseType().getCaseTypeCd().equals(CaseTypeCode.FNOD);
	}
	
	public boolean isFnodSpouse(CaseLink caseLink) {
		return caseLink.getCaseType().getCaseTypeCd().equals(CaseTypeCode.FNOD_SPOUSE);
	}
}
