package gov.va.fnod.business;



import gov.va.fnod.model.fnoddata.ActivityTypeCode;

import gov.va.fnod.model.fnoddata.AuditActivity;

import gov.va.fnod.model.fnoddata.AuditChange;

import gov.va.fnod.model.fnoddata.CaseLink;

import gov.va.fnod.model.fnoddata.FnodRecord;

import gov.va.fnod.model.fnoddata.LockedStatus;

import gov.va.fnod.model.fnoddata.Person;

import gov.va.fnod.service.CaseSession;

import gov.va.fnod.view.AttachmentHelper;

import gov.va.fnod.view.AttachmentWrapper;



import java.math.BigDecimal;

import java.sql.Timestamp;

import java.text.SimpleDateFormat;

import java.util.ArrayList;

import java.util.Date;

import java.util.List;



/**

 * @author Jim Lindelof

 */

public class CaseChangeProcessor {

	

	private CaseSession caseSession;

	

	public CaseChangeProcessor(CaseSession caseSession) {		

		this.caseSession = caseSession;

	}

		

	public boolean processAudit(String username, CaseLink caseLink,

			ActivityTypeCode activityType, LockedStatus origLockedStatus,

			FnodRecord origFnodRecord, AttachmentHelper attachmentHelper) {

		

		// Setting default, is an award audit required

		boolean awardAuditRequired = determineAwardAuditRequired(caseLink, origLockedStatus, origFnodRecord);

		

		if ( activityType == null && LockedStatus.COMPLETE.equals(origLockedStatus) ) {

			activityType = ActivityTypeCode.CORRECTION;

		}

		

		if ( activityType != null ) {

			

			List<AuditChange> fieldChanges;

			fieldChanges = checkFieldChanges(origFnodRecord,caseLink.getFnodRecord(), attachmentHelper);		



			if ( activityType.isForceActivityIfNoChange() || !fieldChanges.isEmpty() ) {

				AuditActivity audit = new AuditActivity();

				audit.setUsername(username);

				audit.setActivityDt(new Timestamp(System.currentTimeMillis()));			

				audit.setActivityType( caseSession.getActivityType(activityType) );

				audit.setAuditChanges(fieldChanges);

				caseLink.addAuditActivity(audit);

			}



			if ( ActivityTypeCode.AUDITED_AWARD.equals(activityType)) {

				awardAuditRequired = false;

			}

		

		}

		

		return awardAuditRequired;



	}

	

	private boolean determineAwardAuditRequired(CaseLink caseLink, LockedStatus origLockedStatus, FnodRecord origFnodRecord) {

		boolean retval = false;

		

		if ( !(origFnodRecord == null || caseLink.getFnodRecord() == null) ) {

			// If my current lock status is complete

			if ( LockedStatus.COMPLETE.equals(caseLink.getLockedStatus()) ) {			

				// So we changed from something (PENDING) to complete and recorded a monthly amount > 0

				if ( !LockedStatus.COMPLETE.equals(origLockedStatus) && hasMonthlyAmount(caseLink.getFnodRecord()) ) {

					retval = true;

				} else if ( LockedStatus.COMPLETE.equals(origLockedStatus) ) {

					retval = amountChanged(origFnodRecord.getMonthlyAmount(), caseLink.getFnodRecord().getMonthlyAmount());

				}

			}

		}

		

		return retval;		

	}

	

	private boolean hasMonthlyAmount(FnodRecord fnodRecord) {

		boolean retval = false;

		

		if ( fnodRecord != null ) {

			if ( fnodRecord.getMonthlyAmount() != null ) {

				// if there is more than a penny, then there is money

				if ( fnodRecord.getMonthlyAmount().doubleValue() > 0.005 ) {

					retval = true;

				}

			}

		}

		

		return retval;

	}

	

	private List<AuditChange> checkFieldChanges(FnodRecord orig, FnodRecord cur, AttachmentHelper attachmentHelper) {

				

		List<AuditChange> fieldChanges = new ArrayList<AuditChange>();

		

		// No comparisons to be done

		if ( !(orig == null || cur == null) ) {

			

			addChange("USER", orig.getUsername(), cur.getUsername(), fieldChanges);

			

			// Otherwise check all FNOD fields

			addChange("BENEFIT_CANCELLED", orig.getBenefitCancelled(), cur.getBenefitCancelled(), fieldChanges);

			addChange("BIRLS_ADD", orig.getBirlsAdd(), cur.getBirlsAdd(), fieldChanges);

			addChange("BIRLS_UPDATE", orig.getBirlsUpdate(), cur.getBirlsUpdate(), fieldChanges);

			addChange("COMMENTS", orig.getComments(), cur.getComments(), fieldChanges);

			addChange("DOD_ALREADY_PRESENT", orig.getDodAlreadyPresent(), cur.getDodAlreadyPresent(), fieldChanges);		

			addChange("ELIGIBILITY_VERIFICATION", orig.getEligibilityVerification(), cur.getEligibilityVerification(), fieldChanges);

			addChange("FLAG_ISSUED", orig.getFlagIssued(), cur.getFlagIssued(), fieldChanges);		

			addChange("FNOD", orig.getFnod(), cur.getFnod(), fieldChanges);



			addChange("MONTHLY_AMOUNT", orig.getMonthlyAmount(), cur.getMonthlyAmount(), fieldChanges);



			addChange("NMI", orig.getNmi(), cur.getNmi(), fieldChanges);

			addChange("PMC", orig.getPmc(), cur.getPmc(), fieldChanges);

			

			addChange("IRIS", orig.getIris(), cur.getIris(), fieldChanges);

			addChange("NIS", orig.getNis(), cur.getNis(), fieldChanges);			

			addChange("SENSITIVE", orig.getSensitive(), cur.getSensitive(), fieldChanges);

			addChange("FOLDER_ONLY", orig.getFolderOnlyBenefit(), cur.getFolderOnlyBenefit(), fieldChanges);			



			addChange("TOO_OLD_TO_CREATE", orig.getTooOldToCreate(), cur.getTooOldToCreate(), fieldChanges);

			addChange("USERNAME", orig.getUsername(), cur.getUsername(), fieldChanges);



			addChange("REGIONAL_OFFICE", orig.getRegionalOffice(), cur.getRegionalOffice(), fieldChanges);		

			

			addChange("VETERAN_CLAIM_NUM", orig.getVeteranClaimNum(), cur.getVeteranClaimNum(), fieldChanges);

			addChange("MILITARY_SERVICE_NUM", orig.getMilitaryServiceNum(), cur.getMilitaryServiceNum(), fieldChanges);

			

			if ( orig.getVeteranPerson() != null && cur.getVeteranPerson() != null ) {

				Person origPerson = orig.getVeteranPerson();

				Person curPerson = cur.getVeteranPerson();

				addChange("VETERAN_FIRST_NAME", origPerson.getFirstName(), curPerson.getFirstName(), fieldChanges);

				addChange("VETERAN_MIDDLE_NAME", origPerson.getMiddleName(), curPerson.getMiddleName(), fieldChanges);

				addChange("VETERAN_LAST_NAME", origPerson.getLastName(), curPerson.getLastName(), fieldChanges);

				addChange("VETERAN_SUFFIX_NAME", origPerson.getSuffixName(), curPerson.getSuffixName(), fieldChanges);			

				addChange("VETERAN_SOCIAL_SECURITY_NUMBER", origPerson.getSocialSecurityNumber(), curPerson.getSocialSecurityNumber(), fieldChanges);

				addChange("VETERAN_BIRTH_DT", origPerson.getBirthDt(), curPerson.getBirthDt(), fieldChanges);

				addChange("VETERAN_DEATH_DT", origPerson.getDeathDt(), curPerson.getDeathDt(), fieldChanges);

				addChange("VETERAN_DECEDENT_ID", origPerson.getDecedentId(), curPerson.getDecedentId(), fieldChanges);

				addChange("VETERAN_RECORD_OF_INTERMENT_DT", origPerson.getRecordOfIntermentDt(), curPerson.getRecordOfIntermentDt(), fieldChanges);

			}



			if (attachmentHelper != null) {

				// create an audit entry for each added attachment

				for (AttachmentWrapper added : attachmentHelper.getAdds()) {

					addChange("ATTACHMENT", "ADDED", added.getAttachment().getFileName(),  fieldChanges);

				}				

				

				// create an audit entry for each removed attachment

				for (AttachmentWrapper deleted : attachmentHelper.getDeletes()) {

					if (deleted.getAttachment().getDocumentStorageId() != null) {

						addChange("ATTACHMENT", "REMOVED", deleted.getAttachment().getFileName(),  fieldChanges);

					}

				}

			}

		}

		

		return fieldChanges;

		

	}

	

	private void addChange(String fieldName, String origValue, String curValue, List<AuditChange> changes) {

		if ( (origValue == null && curValue != null) || 

			 (origValue != null && ! origValue.equals(curValue)) ) {

			AuditChange change = new AuditChange();

			change.setFieldName(fieldName);

			change.setOrigValue(origValue);

			change.setNewValue(curValue);

			changes.add(change);

		}

	}

	

	private <T> void addChange(String fieldName, T origValue, T curValue, List<AuditChange> changes) {

		if ( (origValue == null && curValue != null) || 

			 (origValue != null && ! origValue.equals(curValue)) ) {

			AuditChange change = new AuditChange();

			change.setFieldName( fieldName );

			change.setOrigValue( origValue!=null ? origValue.toString(): null );

			change.setNewValue( curValue!=null ? curValue.toString(): null );

			changes.add(change);

		}

	}



	private void addChange(String fieldName, BigDecimal origValue, BigDecimal curValue, List<AuditChange> changes) {

		

		

		if ( amountChanged(curValue,origValue) ) { 			 

			AuditChange change = new AuditChange();

			change.setFieldName(fieldName);

			change.setOrigValue(origValue!= null ? origValue.toString(): null );

			change.setNewValue(curValue!= null ? curValue.toString(): null);

			changes.add(change);

		}

	}	

	

	private void addChange(String fieldName, Date origValue, Date curValue, List<AuditChange> changes) {

		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MMM-dd");

		

		if ( (origValue == null && curValue != null) || 

		 	 (origValue != null && ! origValue.equals(curValue)) ) {

			AuditChange change = new AuditChange();

			change.setFieldName(fieldName);

			change.setOrigValue(origValue!=null ? sdf.format(origValue) : null);

			change.setNewValue(curValue!=null ? sdf.format(curValue) : null);

			changes.add(change);

		}

	}

	

	private boolean amountChanged(BigDecimal origValue, BigDecimal curValue) {

		

		double origAmount = origValue != null ? origValue.doubleValue(): 0.0;

		double curAmount = curValue != null ? curValue.doubleValue(): 0.0;

		

		return ( Math.abs(curAmount-origAmount) > 0.005 );

	}

}

