package gov.va.fnod.soa_web.ws;

import java.util.Date;
import java.util.logging.Level;
import java.util.logging.Logger;
import gov.va.fnod.model.UserPrivilege;
import gov.va.fnod.model.fnoddata.CaseLink;
import gov.va.fnod.model.fnoddata.CaseType;
import gov.va.fnod.model.fnoddata.SourceSystem;
import gov.va.fnod.security.authorization.SecurityContext;
import gov.va.fnod.security.exception.AuthenticationException;
import gov.va.fnod.service.CaseSession;
import gov.va.fnod.service.LookupSession;
import gov.va.fnod.service.UserAuthenticationSession;
import gov.va.fnod.service.UserAuthorizationSession;
import gov.va.fnod.soa_common.model.fnod.Authentication;
import gov.va.fnod.soa_common.model.fnod.FnodCase;
import gov.va.fnod.soa_common.model.fnod.FnodPayload;
import gov.va.fnod.soa_web.exception.CustomSOAPFaultException;

import javax.ejb.EJB;
import javax.jws.WebResult;
import javax.jws.WebService;
import javax.jws.soap.SOAPBinding;

import org.apache.commons.lang.StringUtils;

@WebService(name = "FnodPayloadService", targetNamespace = "http://fnodcaseservice.fnod.domain", 
serviceName="FnodPayloadService", portName="FnodPayload", wsdlLocation="WEB-INF/wsdl/FnodPayload.wsdl")
@SOAPBinding(parameterStyle=SOAPBinding.ParameterStyle.BARE)
public class FnodPayloadService  {
	private static final Logger logr = Logger.getLogger(FnodPayloadService.class.getName());
	
	@EJB
	private LookupSession lookupSession;
	
	@EJB
	private CaseSession caseSession;
	private CustomSOAPFaultException soapFaultException;

	@EJB
	private UserAuthenticationSession userAuthenticSession;
	
	@EJB
	private UserAuthorizationSession userAuthorizationSession;

	@WebResult(name = "fnodCaseId")
	public long createFnodCase(FnodPayload fnodPayload) {
		
		FnodCase fnodCase = fnodPayload.getFnodCase();
		if (fnodCase == null) {
			soapFaultException = new CustomSOAPFaultException();
			soapFaultException.createSOAPFaultExceptionClient("Illegal Argument: The FnodCase element cannnot be null.");
		}
		// Authenticate request
		Authentication authentication = fnodCase.getAuthentication();
		if (authentication == null) {
			soapFaultException = new CustomSOAPFaultException();
			soapFaultException.createSOAPFaultExceptionClient("Illegal Argument: The Authentication element cannot be null.");
		}
		// Try and login
		try {
			SecurityContext<UserPrivilege> flagAppSecContext = userAuthorizationSession.createSystemContext(
					"FNOD_APP_LOGIN", UserPrivilege.FNOD_LOAD, UserPrivilege.CONNECT);
			userAuthenticSession.connect(authentication.getUsername(), authentication.getPassword(),
					flagAppSecContext);
		} catch (AuthenticationException ae) {
			logr.log(Level.WARNING, "{0} failed a FLAG_APP password login - {1}: {2}", new Object[]{authentication.getUsername(), new Date(), ae.getMessage()});
			soapFaultException = new CustomSOAPFaultException();
			soapFaultException.createSOAPFaultExceptionClient("Not Authorized: The username/password combination is not authorized in the system");
		}
		
		// Query CaseType by caseTypeCd provided by typeOfCase in FnodPayload object.
		LookupModelObjectsBean lookupCaseType = new LookupModelObjectsBean(lookupSession);
		String caseTypeCd = fnodPayload.getFnodCase().getCaseTypeCd().value();
		if (StringUtils.isBlank(caseTypeCd)) {
			soapFaultException = new CustomSOAPFaultException();
			soapFaultException.createSOAPFaultExceptionClient("Illegal Argument: The caseTypeCd element in the FnodPayload.xml cannnot be null or empty.");
		}
		CaseType caseType = lookupCaseType.getCaseType(caseTypeCd);
		if (caseType == null) {
			soapFaultException = new CustomSOAPFaultException();
			soapFaultException.createSOAPFaultExceptionClient("Bad Request: The CaseType could not be found by the provided typeOfCase element in the FnodPayload.xml.");
		}
		
		// Query SourceSystem by sourceSystemCd provided by sourceSystem in FnodPayload object.
		LookupModelObjectsBean lookupSourceSystem = new LookupModelObjectsBean(lookupSession);
		String sourceSystemCd = fnodPayload.getFnodCase().getSourceSystemCd().value();
		if (StringUtils.isBlank(sourceSystemCd)) {
			soapFaultException = new CustomSOAPFaultException();
			soapFaultException.createSOAPFaultExceptionClient("Illegal Argument: The sourceSystemCd element in the FnodPayload.xml cannnot be null or empty.");
		}
		SourceSystem sourceSystem = lookupSourceSystem.getSourceSystem(sourceSystemCd);
		if (sourceSystem == null) {
			soapFaultException = new CustomSOAPFaultException();
			soapFaultException.createSOAPFaultExceptionClient("Bad Request: The SourceSystem could not be found by the provided sourceSystem element in the FnodPayload.xml.");
		}
		
		// Map FnodPayload to Model
		FnodPayloadModelMapping fnodPayloadModelMapping = new FnodPayloadModelMapping();
		CaseLink caseLink = fnodPayloadModelMapping.mapFnodToModel(fnodPayload, caseType, sourceSystem);
		
		// Save FNOD Case
		long fnodCaseId = caseSession.addPendingCase(caseLink);
		if(fnodCaseId == 0 ) {
			soapFaultException = new CustomSOAPFaultException();
			soapFaultException.createSOAPFaultExceptionClient("Internal Server: The FNOD Case could not be created.");
		}

		return fnodCaseId;
	}
	
}